/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.pcaconnectorad.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Defines the attributes of the private key.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PrivateKeyAttributesV3 implements SdkPojo, Serializable,
        ToCopyableBuilder<PrivateKeyAttributesV3.Builder, PrivateKeyAttributesV3> {
    private static final SdkField<String> ALGORITHM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Algorithm").getter(getter(PrivateKeyAttributesV3::algorithmAsString)).setter(setter(Builder::algorithm))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Algorithm").build()).build();

    private static final SdkField<List<String>> CRYPTO_PROVIDERS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("CryptoProviders")
            .getter(getter(PrivateKeyAttributesV3::cryptoProviders))
            .setter(setter(Builder::cryptoProviders))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CryptoProviders").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> KEY_SPEC_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KeySpec").getter(getter(PrivateKeyAttributesV3::keySpecAsString)).setter(setter(Builder::keySpec))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeySpec").build()).build();

    private static final SdkField<KeyUsageProperty> KEY_USAGE_PROPERTY_FIELD = SdkField
            .<KeyUsageProperty> builder(MarshallingType.SDK_POJO).memberName("KeyUsageProperty")
            .getter(getter(PrivateKeyAttributesV3::keyUsageProperty)).setter(setter(Builder::keyUsageProperty))
            .constructor(KeyUsageProperty::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyUsageProperty").build()).build();

    private static final SdkField<Integer> MINIMAL_KEY_LENGTH_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MinimalKeyLength").getter(getter(PrivateKeyAttributesV3::minimalKeyLength))
            .setter(setter(Builder::minimalKeyLength))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MinimalKeyLength").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ALGORITHM_FIELD,
            CRYPTO_PROVIDERS_FIELD, KEY_SPEC_FIELD, KEY_USAGE_PROPERTY_FIELD, MINIMAL_KEY_LENGTH_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String algorithm;

    private final List<String> cryptoProviders;

    private final String keySpec;

    private final KeyUsageProperty keyUsageProperty;

    private final Integer minimalKeyLength;

    private PrivateKeyAttributesV3(BuilderImpl builder) {
        this.algorithm = builder.algorithm;
        this.cryptoProviders = builder.cryptoProviders;
        this.keySpec = builder.keySpec;
        this.keyUsageProperty = builder.keyUsageProperty;
        this.minimalKeyLength = builder.minimalKeyLength;
    }

    /**
     * <p>
     * Defines the algorithm used to generate the private key.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #algorithm} will
     * return {@link PrivateKeyAlgorithm#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #algorithmAsString}.
     * </p>
     * 
     * @return Defines the algorithm used to generate the private key.
     * @see PrivateKeyAlgorithm
     */
    public final PrivateKeyAlgorithm algorithm() {
        return PrivateKeyAlgorithm.fromValue(algorithm);
    }

    /**
     * <p>
     * Defines the algorithm used to generate the private key.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #algorithm} will
     * return {@link PrivateKeyAlgorithm#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #algorithmAsString}.
     * </p>
     * 
     * @return Defines the algorithm used to generate the private key.
     * @see PrivateKeyAlgorithm
     */
    public final String algorithmAsString() {
        return algorithm;
    }

    /**
     * For responses, this returns true if the service returned a value for the CryptoProviders property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasCryptoProviders() {
        return cryptoProviders != null && !(cryptoProviders instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Defines the cryptographic providers used to generate the private key.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCryptoProviders} method.
     * </p>
     * 
     * @return Defines the cryptographic providers used to generate the private key.
     */
    public final List<String> cryptoProviders() {
        return cryptoProviders;
    }

    /**
     * <p>
     * Defines the purpose of the private key. Set it to "KEY_EXCHANGE" or "SIGNATURE" value.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #keySpec} will
     * return {@link KeySpec#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #keySpecAsString}.
     * </p>
     * 
     * @return Defines the purpose of the private key. Set it to "KEY_EXCHANGE" or "SIGNATURE" value.
     * @see KeySpec
     */
    public final KeySpec keySpec() {
        return KeySpec.fromValue(keySpec);
    }

    /**
     * <p>
     * Defines the purpose of the private key. Set it to "KEY_EXCHANGE" or "SIGNATURE" value.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #keySpec} will
     * return {@link KeySpec#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #keySpecAsString}.
     * </p>
     * 
     * @return Defines the purpose of the private key. Set it to "KEY_EXCHANGE" or "SIGNATURE" value.
     * @see KeySpec
     */
    public final String keySpecAsString() {
        return keySpec;
    }

    /**
     * <p>
     * The key usage property defines the purpose of the private key contained in the certificate. You can specify
     * specific purposes using property flags or all by using property type ALL.
     * </p>
     * 
     * @return The key usage property defines the purpose of the private key contained in the certificate. You can
     *         specify specific purposes using property flags or all by using property type ALL.
     */
    public final KeyUsageProperty keyUsageProperty() {
        return keyUsageProperty;
    }

    /**
     * <p>
     * Set the minimum key length of the private key.
     * </p>
     * 
     * @return Set the minimum key length of the private key.
     */
    public final Integer minimalKeyLength() {
        return minimalKeyLength;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(algorithmAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasCryptoProviders() ? cryptoProviders() : null);
        hashCode = 31 * hashCode + Objects.hashCode(keySpecAsString());
        hashCode = 31 * hashCode + Objects.hashCode(keyUsageProperty());
        hashCode = 31 * hashCode + Objects.hashCode(minimalKeyLength());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PrivateKeyAttributesV3)) {
            return false;
        }
        PrivateKeyAttributesV3 other = (PrivateKeyAttributesV3) obj;
        return Objects.equals(algorithmAsString(), other.algorithmAsString())
                && hasCryptoProviders() == other.hasCryptoProviders()
                && Objects.equals(cryptoProviders(), other.cryptoProviders())
                && Objects.equals(keySpecAsString(), other.keySpecAsString())
                && Objects.equals(keyUsageProperty(), other.keyUsageProperty())
                && Objects.equals(minimalKeyLength(), other.minimalKeyLength());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PrivateKeyAttributesV3").add("Algorithm", algorithmAsString())
                .add("CryptoProviders", hasCryptoProviders() ? cryptoProviders() : null).add("KeySpec", keySpecAsString())
                .add("KeyUsageProperty", keyUsageProperty()).add("MinimalKeyLength", minimalKeyLength()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Algorithm":
            return Optional.ofNullable(clazz.cast(algorithmAsString()));
        case "CryptoProviders":
            return Optional.ofNullable(clazz.cast(cryptoProviders()));
        case "KeySpec":
            return Optional.ofNullable(clazz.cast(keySpecAsString()));
        case "KeyUsageProperty":
            return Optional.ofNullable(clazz.cast(keyUsageProperty()));
        case "MinimalKeyLength":
            return Optional.ofNullable(clazz.cast(minimalKeyLength()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Algorithm", ALGORITHM_FIELD);
        map.put("CryptoProviders", CRYPTO_PROVIDERS_FIELD);
        map.put("KeySpec", KEY_SPEC_FIELD);
        map.put("KeyUsageProperty", KEY_USAGE_PROPERTY_FIELD);
        map.put("MinimalKeyLength", MINIMAL_KEY_LENGTH_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<PrivateKeyAttributesV3, T> g) {
        return obj -> g.apply((PrivateKeyAttributesV3) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PrivateKeyAttributesV3> {
        /**
         * <p>
         * Defines the algorithm used to generate the private key.
         * </p>
         * 
         * @param algorithm
         *        Defines the algorithm used to generate the private key.
         * @see PrivateKeyAlgorithm
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PrivateKeyAlgorithm
         */
        Builder algorithm(String algorithm);

        /**
         * <p>
         * Defines the algorithm used to generate the private key.
         * </p>
         * 
         * @param algorithm
         *        Defines the algorithm used to generate the private key.
         * @see PrivateKeyAlgorithm
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PrivateKeyAlgorithm
         */
        Builder algorithm(PrivateKeyAlgorithm algorithm);

        /**
         * <p>
         * Defines the cryptographic providers used to generate the private key.
         * </p>
         * 
         * @param cryptoProviders
         *        Defines the cryptographic providers used to generate the private key.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cryptoProviders(Collection<String> cryptoProviders);

        /**
         * <p>
         * Defines the cryptographic providers used to generate the private key.
         * </p>
         * 
         * @param cryptoProviders
         *        Defines the cryptographic providers used to generate the private key.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cryptoProviders(String... cryptoProviders);

        /**
         * <p>
         * Defines the purpose of the private key. Set it to "KEY_EXCHANGE" or "SIGNATURE" value.
         * </p>
         * 
         * @param keySpec
         *        Defines the purpose of the private key. Set it to "KEY_EXCHANGE" or "SIGNATURE" value.
         * @see KeySpec
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see KeySpec
         */
        Builder keySpec(String keySpec);

        /**
         * <p>
         * Defines the purpose of the private key. Set it to "KEY_EXCHANGE" or "SIGNATURE" value.
         * </p>
         * 
         * @param keySpec
         *        Defines the purpose of the private key. Set it to "KEY_EXCHANGE" or "SIGNATURE" value.
         * @see KeySpec
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see KeySpec
         */
        Builder keySpec(KeySpec keySpec);

        /**
         * <p>
         * The key usage property defines the purpose of the private key contained in the certificate. You can specify
         * specific purposes using property flags or all by using property type ALL.
         * </p>
         * 
         * @param keyUsageProperty
         *        The key usage property defines the purpose of the private key contained in the certificate. You can
         *        specify specific purposes using property flags or all by using property type ALL.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keyUsageProperty(KeyUsageProperty keyUsageProperty);

        /**
         * <p>
         * The key usage property defines the purpose of the private key contained in the certificate. You can specify
         * specific purposes using property flags or all by using property type ALL.
         * </p>
         * This is a convenience method that creates an instance of the {@link KeyUsageProperty.Builder} avoiding the
         * need to create one manually via {@link KeyUsageProperty#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link KeyUsageProperty.Builder#build()} is called immediately and its
         * result is passed to {@link #keyUsageProperty(KeyUsageProperty)}.
         * 
         * @param keyUsageProperty
         *        a consumer that will call methods on {@link KeyUsageProperty.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #keyUsageProperty(KeyUsageProperty)
         */
        default Builder keyUsageProperty(Consumer<KeyUsageProperty.Builder> keyUsageProperty) {
            return keyUsageProperty(KeyUsageProperty.builder().applyMutation(keyUsageProperty).build());
        }

        /**
         * <p>
         * Set the minimum key length of the private key.
         * </p>
         * 
         * @param minimalKeyLength
         *        Set the minimum key length of the private key.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minimalKeyLength(Integer minimalKeyLength);
    }

    static final class BuilderImpl implements Builder {
        private String algorithm;

        private List<String> cryptoProviders = DefaultSdkAutoConstructList.getInstance();

        private String keySpec;

        private KeyUsageProperty keyUsageProperty;

        private Integer minimalKeyLength;

        private BuilderImpl() {
        }

        private BuilderImpl(PrivateKeyAttributesV3 model) {
            algorithm(model.algorithm);
            cryptoProviders(model.cryptoProviders);
            keySpec(model.keySpec);
            keyUsageProperty(model.keyUsageProperty);
            minimalKeyLength(model.minimalKeyLength);
        }

        public final String getAlgorithm() {
            return algorithm;
        }

        public final void setAlgorithm(String algorithm) {
            this.algorithm = algorithm;
        }

        @Override
        public final Builder algorithm(String algorithm) {
            this.algorithm = algorithm;
            return this;
        }

        @Override
        public final Builder algorithm(PrivateKeyAlgorithm algorithm) {
            this.algorithm(algorithm == null ? null : algorithm.toString());
            return this;
        }

        public final Collection<String> getCryptoProviders() {
            if (cryptoProviders instanceof SdkAutoConstructList) {
                return null;
            }
            return cryptoProviders;
        }

        public final void setCryptoProviders(Collection<String> cryptoProviders) {
            this.cryptoProviders = CryptoProvidersListCopier.copy(cryptoProviders);
        }

        @Override
        public final Builder cryptoProviders(Collection<String> cryptoProviders) {
            this.cryptoProviders = CryptoProvidersListCopier.copy(cryptoProviders);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder cryptoProviders(String... cryptoProviders) {
            cryptoProviders(Arrays.asList(cryptoProviders));
            return this;
        }

        public final String getKeySpec() {
            return keySpec;
        }

        public final void setKeySpec(String keySpec) {
            this.keySpec = keySpec;
        }

        @Override
        public final Builder keySpec(String keySpec) {
            this.keySpec = keySpec;
            return this;
        }

        @Override
        public final Builder keySpec(KeySpec keySpec) {
            this.keySpec(keySpec == null ? null : keySpec.toString());
            return this;
        }

        public final KeyUsageProperty.Builder getKeyUsageProperty() {
            return keyUsageProperty != null ? keyUsageProperty.toBuilder() : null;
        }

        public final void setKeyUsageProperty(KeyUsageProperty.BuilderImpl keyUsageProperty) {
            this.keyUsageProperty = keyUsageProperty != null ? keyUsageProperty.build() : null;
        }

        @Override
        public final Builder keyUsageProperty(KeyUsageProperty keyUsageProperty) {
            this.keyUsageProperty = keyUsageProperty;
            return this;
        }

        public final Integer getMinimalKeyLength() {
            return minimalKeyLength;
        }

        public final void setMinimalKeyLength(Integer minimalKeyLength) {
            this.minimalKeyLength = minimalKeyLength;
        }

        @Override
        public final Builder minimalKeyLength(Integer minimalKeyLength) {
            this.minimalKeyLength = minimalKeyLength;
            return this;
        }

        @Override
        public PrivateKeyAttributesV3 build() {
            return new PrivateKeyAttributesV3(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
