/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.pi;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.pi.model.CreatePerformanceAnalysisReportRequest;
import software.amazon.awssdk.services.pi.model.CreatePerformanceAnalysisReportResponse;
import software.amazon.awssdk.services.pi.model.DeletePerformanceAnalysisReportRequest;
import software.amazon.awssdk.services.pi.model.DeletePerformanceAnalysisReportResponse;
import software.amazon.awssdk.services.pi.model.DescribeDimensionKeysRequest;
import software.amazon.awssdk.services.pi.model.DescribeDimensionKeysResponse;
import software.amazon.awssdk.services.pi.model.GetDimensionKeyDetailsRequest;
import software.amazon.awssdk.services.pi.model.GetDimensionKeyDetailsResponse;
import software.amazon.awssdk.services.pi.model.GetPerformanceAnalysisReportRequest;
import software.amazon.awssdk.services.pi.model.GetPerformanceAnalysisReportResponse;
import software.amazon.awssdk.services.pi.model.GetResourceMetadataRequest;
import software.amazon.awssdk.services.pi.model.GetResourceMetadataResponse;
import software.amazon.awssdk.services.pi.model.GetResourceMetricsRequest;
import software.amazon.awssdk.services.pi.model.GetResourceMetricsResponse;
import software.amazon.awssdk.services.pi.model.InternalServiceErrorException;
import software.amazon.awssdk.services.pi.model.InvalidArgumentException;
import software.amazon.awssdk.services.pi.model.ListAvailableResourceDimensionsRequest;
import software.amazon.awssdk.services.pi.model.ListAvailableResourceDimensionsResponse;
import software.amazon.awssdk.services.pi.model.ListAvailableResourceMetricsRequest;
import software.amazon.awssdk.services.pi.model.ListAvailableResourceMetricsResponse;
import software.amazon.awssdk.services.pi.model.ListPerformanceAnalysisReportsRequest;
import software.amazon.awssdk.services.pi.model.ListPerformanceAnalysisReportsResponse;
import software.amazon.awssdk.services.pi.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.pi.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.pi.model.NotAuthorizedException;
import software.amazon.awssdk.services.pi.model.PiException;
import software.amazon.awssdk.services.pi.model.TagResourceRequest;
import software.amazon.awssdk.services.pi.model.TagResourceResponse;
import software.amazon.awssdk.services.pi.model.UntagResourceRequest;
import software.amazon.awssdk.services.pi.model.UntagResourceResponse;
import software.amazon.awssdk.services.pi.paginators.DescribeDimensionKeysIterable;
import software.amazon.awssdk.services.pi.paginators.GetResourceMetricsIterable;
import software.amazon.awssdk.services.pi.paginators.ListAvailableResourceDimensionsIterable;
import software.amazon.awssdk.services.pi.paginators.ListAvailableResourceMetricsIterable;
import software.amazon.awssdk.services.pi.paginators.ListPerformanceAnalysisReportsIterable;

/**
 * Service client for accessing AWS PI. This can be created using the static {@link #builder()} method.
 *
 * <fullname>Amazon RDS Performance Insights</fullname>
 * <p>
 * Amazon RDS Performance Insights enables you to monitor and explore different dimensions of database load based on
 * data captured from a running DB instance. The guide provides detailed information about Performance Insights data
 * types, parameters and errors.
 * </p>
 * <p>
 * When Performance Insights is enabled, the Amazon RDS Performance Insights API provides visibility into the
 * performance of your DB instance. Amazon CloudWatch provides the authoritative source for Amazon Web Services
 * service-vended monitoring metrics. Performance Insights offers a domain-specific view of DB load.
 * </p>
 * <p>
 * DB load is measured as average active sessions. Performance Insights provides the data to API consumers as a
 * two-dimensional time-series dataset. The time dimension provides DB load data for each time point in the queried time
 * range. Each time point decomposes overall load in relation to the requested dimensions, measured at that time point.
 * Examples include SQL, Wait event, User, and Host.
 * </p>
 * <ul>
 * <li>
 * <p>
 * To learn more about Performance Insights and Amazon Aurora DB instances, go to the <i> <a
 * href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_PerfInsights.html"> Amazon Aurora User
 * Guide</a> </i>.
 * </p>
 * </li>
 * <li>
 * <p>
 * To learn more about Performance Insights and Amazon RDS DB instances, go to the <i> <a
 * href="https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_PerfInsights.html"> Amazon RDS User Guide</a> </i>.
 * </p>
 * </li>
 * <li>
 * <p>
 * To learn more about Performance Insights and Amazon DocumentDB clusters, go to the <i> <a
 * href="https://docs.aws.amazon.com/documentdb/latest/developerguide/performance-insights.html"> Amazon DocumentDB
 * Developer Guide</a> </i>.
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface PiClient extends AwsClient {
    String SERVICE_NAME = "pi";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "pi";

    /**
     * <p>
     * Creates a new performance analysis report for a specific time period for the DB instance.
     * </p>
     *
     * @param createPerformanceAnalysisReportRequest
     * @return Result of the CreatePerformanceAnalysisReport operation returned by the service.
     * @throws InvalidArgumentException
     *         One of the arguments provided is invalid for this request.
     * @throws InternalServiceErrorException
     *         The request failed due to an unknown error.
     * @throws NotAuthorizedException
     *         The user is not authorized to perform this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PiClient.CreatePerformanceAnalysisReport
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/CreatePerformanceAnalysisReport"
     *      target="_top">AWS API Documentation</a>
     */
    default CreatePerformanceAnalysisReportResponse createPerformanceAnalysisReport(
            CreatePerformanceAnalysisReportRequest createPerformanceAnalysisReportRequest) throws InvalidArgumentException,
            InternalServiceErrorException, NotAuthorizedException, AwsServiceException, SdkClientException, PiException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a new performance analysis report for a specific time period for the DB instance.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreatePerformanceAnalysisReportRequest.Builder}
     * avoiding the need to create one manually via {@link CreatePerformanceAnalysisReportRequest#builder()}
     * </p>
     *
     * @param createPerformanceAnalysisReportRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.pi.model.CreatePerformanceAnalysisReportRequest.Builder} to create
     *        a request.
     * @return Result of the CreatePerformanceAnalysisReport operation returned by the service.
     * @throws InvalidArgumentException
     *         One of the arguments provided is invalid for this request.
     * @throws InternalServiceErrorException
     *         The request failed due to an unknown error.
     * @throws NotAuthorizedException
     *         The user is not authorized to perform this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PiClient.CreatePerformanceAnalysisReport
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/CreatePerformanceAnalysisReport"
     *      target="_top">AWS API Documentation</a>
     */
    default CreatePerformanceAnalysisReportResponse createPerformanceAnalysisReport(
            Consumer<CreatePerformanceAnalysisReportRequest.Builder> createPerformanceAnalysisReportRequest)
            throws InvalidArgumentException, InternalServiceErrorException, NotAuthorizedException, AwsServiceException,
            SdkClientException, PiException {
        return createPerformanceAnalysisReport(CreatePerformanceAnalysisReportRequest.builder()
                .applyMutation(createPerformanceAnalysisReportRequest).build());
    }

    /**
     * <p>
     * Deletes a performance analysis report.
     * </p>
     *
     * @param deletePerformanceAnalysisReportRequest
     * @return Result of the DeletePerformanceAnalysisReport operation returned by the service.
     * @throws InvalidArgumentException
     *         One of the arguments provided is invalid for this request.
     * @throws InternalServiceErrorException
     *         The request failed due to an unknown error.
     * @throws NotAuthorizedException
     *         The user is not authorized to perform this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PiClient.DeletePerformanceAnalysisReport
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/DeletePerformanceAnalysisReport"
     *      target="_top">AWS API Documentation</a>
     */
    default DeletePerformanceAnalysisReportResponse deletePerformanceAnalysisReport(
            DeletePerformanceAnalysisReportRequest deletePerformanceAnalysisReportRequest) throws InvalidArgumentException,
            InternalServiceErrorException, NotAuthorizedException, AwsServiceException, SdkClientException, PiException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes a performance analysis report.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeletePerformanceAnalysisReportRequest.Builder}
     * avoiding the need to create one manually via {@link DeletePerformanceAnalysisReportRequest#builder()}
     * </p>
     *
     * @param deletePerformanceAnalysisReportRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.pi.model.DeletePerformanceAnalysisReportRequest.Builder} to create
     *        a request.
     * @return Result of the DeletePerformanceAnalysisReport operation returned by the service.
     * @throws InvalidArgumentException
     *         One of the arguments provided is invalid for this request.
     * @throws InternalServiceErrorException
     *         The request failed due to an unknown error.
     * @throws NotAuthorizedException
     *         The user is not authorized to perform this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PiClient.DeletePerformanceAnalysisReport
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/DeletePerformanceAnalysisReport"
     *      target="_top">AWS API Documentation</a>
     */
    default DeletePerformanceAnalysisReportResponse deletePerformanceAnalysisReport(
            Consumer<DeletePerformanceAnalysisReportRequest.Builder> deletePerformanceAnalysisReportRequest)
            throws InvalidArgumentException, InternalServiceErrorException, NotAuthorizedException, AwsServiceException,
            SdkClientException, PiException {
        return deletePerformanceAnalysisReport(DeletePerformanceAnalysisReportRequest.builder()
                .applyMutation(deletePerformanceAnalysisReportRequest).build());
    }

    /**
     * <p>
     * For a specific time period, retrieve the top <code>N</code> dimension keys for a metric.
     * </p>
     * <note>
     * <p>
     * Each response element returns a maximum of 500 bytes. For larger elements, such as SQL statements, only the first
     * 500 bytes are returned.
     * </p>
     * </note>
     *
     * @param describeDimensionKeysRequest
     * @return Result of the DescribeDimensionKeys operation returned by the service.
     * @throws InvalidArgumentException
     *         One of the arguments provided is invalid for this request.
     * @throws InternalServiceErrorException
     *         The request failed due to an unknown error.
     * @throws NotAuthorizedException
     *         The user is not authorized to perform this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PiClient.DescribeDimensionKeys
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/DescribeDimensionKeys" target="_top">AWS API
     *      Documentation</a>
     */
    default DescribeDimensionKeysResponse describeDimensionKeys(DescribeDimensionKeysRequest describeDimensionKeysRequest)
            throws InvalidArgumentException, InternalServiceErrorException, NotAuthorizedException, AwsServiceException,
            SdkClientException, PiException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * For a specific time period, retrieve the top <code>N</code> dimension keys for a metric.
     * </p>
     * <note>
     * <p>
     * Each response element returns a maximum of 500 bytes. For larger elements, such as SQL statements, only the first
     * 500 bytes are returned.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeDimensionKeysRequest.Builder} avoiding the
     * need to create one manually via {@link DescribeDimensionKeysRequest#builder()}
     * </p>
     *
     * @param describeDimensionKeysRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.pi.model.DescribeDimensionKeysRequest.Builder} to create a request.
     * @return Result of the DescribeDimensionKeys operation returned by the service.
     * @throws InvalidArgumentException
     *         One of the arguments provided is invalid for this request.
     * @throws InternalServiceErrorException
     *         The request failed due to an unknown error.
     * @throws NotAuthorizedException
     *         The user is not authorized to perform this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PiClient.DescribeDimensionKeys
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/DescribeDimensionKeys" target="_top">AWS API
     *      Documentation</a>
     */
    default DescribeDimensionKeysResponse describeDimensionKeys(
            Consumer<DescribeDimensionKeysRequest.Builder> describeDimensionKeysRequest) throws InvalidArgumentException,
            InternalServiceErrorException, NotAuthorizedException, AwsServiceException, SdkClientException, PiException {
        return describeDimensionKeys(DescribeDimensionKeysRequest.builder().applyMutation(describeDimensionKeysRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #describeDimensionKeys(software.amazon.awssdk.services.pi.model.DescribeDimensionKeysRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pi.paginators.DescribeDimensionKeysIterable responses = client.describeDimensionKeysPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.pi.paginators.DescribeDimensionKeysIterable responses = client
     *             .describeDimensionKeysPaginator(request);
     *     for (software.amazon.awssdk.services.pi.model.DescribeDimensionKeysResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pi.paginators.DescribeDimensionKeysIterable responses = client.describeDimensionKeysPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeDimensionKeys(software.amazon.awssdk.services.pi.model.DescribeDimensionKeysRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeDimensionKeysRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidArgumentException
     *         One of the arguments provided is invalid for this request.
     * @throws InternalServiceErrorException
     *         The request failed due to an unknown error.
     * @throws NotAuthorizedException
     *         The user is not authorized to perform this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PiClient.DescribeDimensionKeys
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/DescribeDimensionKeys" target="_top">AWS API
     *      Documentation</a>
     */
    default DescribeDimensionKeysIterable describeDimensionKeysPaginator(DescribeDimensionKeysRequest describeDimensionKeysRequest)
            throws InvalidArgumentException, InternalServiceErrorException, NotAuthorizedException, AwsServiceException,
            SdkClientException, PiException {
        return new DescribeDimensionKeysIterable(this, describeDimensionKeysRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #describeDimensionKeys(software.amazon.awssdk.services.pi.model.DescribeDimensionKeysRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pi.paginators.DescribeDimensionKeysIterable responses = client.describeDimensionKeysPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.pi.paginators.DescribeDimensionKeysIterable responses = client
     *             .describeDimensionKeysPaginator(request);
     *     for (software.amazon.awssdk.services.pi.model.DescribeDimensionKeysResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pi.paginators.DescribeDimensionKeysIterable responses = client.describeDimensionKeysPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeDimensionKeys(software.amazon.awssdk.services.pi.model.DescribeDimensionKeysRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeDimensionKeysRequest.Builder} avoiding the
     * need to create one manually via {@link DescribeDimensionKeysRequest#builder()}
     * </p>
     *
     * @param describeDimensionKeysRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.pi.model.DescribeDimensionKeysRequest.Builder} to create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidArgumentException
     *         One of the arguments provided is invalid for this request.
     * @throws InternalServiceErrorException
     *         The request failed due to an unknown error.
     * @throws NotAuthorizedException
     *         The user is not authorized to perform this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PiClient.DescribeDimensionKeys
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/DescribeDimensionKeys" target="_top">AWS API
     *      Documentation</a>
     */
    default DescribeDimensionKeysIterable describeDimensionKeysPaginator(
            Consumer<DescribeDimensionKeysRequest.Builder> describeDimensionKeysRequest) throws InvalidArgumentException,
            InternalServiceErrorException, NotAuthorizedException, AwsServiceException, SdkClientException, PiException {
        return describeDimensionKeysPaginator(DescribeDimensionKeysRequest.builder().applyMutation(describeDimensionKeysRequest)
                .build());
    }

    /**
     * <p>
     * Get the attributes of the specified dimension group for a DB instance or data source. For example, if you specify
     * a SQL ID, <code>GetDimensionKeyDetails</code> retrieves the full text of the dimension
     * <code>db.sql.statement</code> associated with this ID. This operation is useful because
     * <code>GetResourceMetrics</code> and <code>DescribeDimensionKeys</code> don't support retrieval of large SQL
     * statement text, lock snapshots, and execution plans.
     * </p>
     *
     * @param getDimensionKeyDetailsRequest
     * @return Result of the GetDimensionKeyDetails operation returned by the service.
     * @throws InvalidArgumentException
     *         One of the arguments provided is invalid for this request.
     * @throws InternalServiceErrorException
     *         The request failed due to an unknown error.
     * @throws NotAuthorizedException
     *         The user is not authorized to perform this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PiClient.GetDimensionKeyDetails
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/GetDimensionKeyDetails" target="_top">AWS API
     *      Documentation</a>
     */
    default GetDimensionKeyDetailsResponse getDimensionKeyDetails(GetDimensionKeyDetailsRequest getDimensionKeyDetailsRequest)
            throws InvalidArgumentException, InternalServiceErrorException, NotAuthorizedException, AwsServiceException,
            SdkClientException, PiException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Get the attributes of the specified dimension group for a DB instance or data source. For example, if you specify
     * a SQL ID, <code>GetDimensionKeyDetails</code> retrieves the full text of the dimension
     * <code>db.sql.statement</code> associated with this ID. This operation is useful because
     * <code>GetResourceMetrics</code> and <code>DescribeDimensionKeys</code> don't support retrieval of large SQL
     * statement text, lock snapshots, and execution plans.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetDimensionKeyDetailsRequest.Builder} avoiding the
     * need to create one manually via {@link GetDimensionKeyDetailsRequest#builder()}
     * </p>
     *
     * @param getDimensionKeyDetailsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.pi.model.GetDimensionKeyDetailsRequest.Builder} to create a
     *        request.
     * @return Result of the GetDimensionKeyDetails operation returned by the service.
     * @throws InvalidArgumentException
     *         One of the arguments provided is invalid for this request.
     * @throws InternalServiceErrorException
     *         The request failed due to an unknown error.
     * @throws NotAuthorizedException
     *         The user is not authorized to perform this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PiClient.GetDimensionKeyDetails
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/GetDimensionKeyDetails" target="_top">AWS API
     *      Documentation</a>
     */
    default GetDimensionKeyDetailsResponse getDimensionKeyDetails(
            Consumer<GetDimensionKeyDetailsRequest.Builder> getDimensionKeyDetailsRequest) throws InvalidArgumentException,
            InternalServiceErrorException, NotAuthorizedException, AwsServiceException, SdkClientException, PiException {
        return getDimensionKeyDetails(GetDimensionKeyDetailsRequest.builder().applyMutation(getDimensionKeyDetailsRequest)
                .build());
    }

    /**
     * <p>
     * Retrieves the report including the report ID, status, time details, and the insights with recommendations. The
     * report status can be <code>RUNNING</code>, <code>SUCCEEDED</code>, or <code>FAILED</code>. The insights include
     * the <code>description</code> and <code>recommendation</code> fields.
     * </p>
     *
     * @param getPerformanceAnalysisReportRequest
     * @return Result of the GetPerformanceAnalysisReport operation returned by the service.
     * @throws InvalidArgumentException
     *         One of the arguments provided is invalid for this request.
     * @throws InternalServiceErrorException
     *         The request failed due to an unknown error.
     * @throws NotAuthorizedException
     *         The user is not authorized to perform this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PiClient.GetPerformanceAnalysisReport
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/GetPerformanceAnalysisReport"
     *      target="_top">AWS API Documentation</a>
     */
    default GetPerformanceAnalysisReportResponse getPerformanceAnalysisReport(
            GetPerformanceAnalysisReportRequest getPerformanceAnalysisReportRequest) throws InvalidArgumentException,
            InternalServiceErrorException, NotAuthorizedException, AwsServiceException, SdkClientException, PiException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves the report including the report ID, status, time details, and the insights with recommendations. The
     * report status can be <code>RUNNING</code>, <code>SUCCEEDED</code>, or <code>FAILED</code>. The insights include
     * the <code>description</code> and <code>recommendation</code> fields.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetPerformanceAnalysisReportRequest.Builder}
     * avoiding the need to create one manually via {@link GetPerformanceAnalysisReportRequest#builder()}
     * </p>
     *
     * @param getPerformanceAnalysisReportRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.pi.model.GetPerformanceAnalysisReportRequest.Builder} to create a
     *        request.
     * @return Result of the GetPerformanceAnalysisReport operation returned by the service.
     * @throws InvalidArgumentException
     *         One of the arguments provided is invalid for this request.
     * @throws InternalServiceErrorException
     *         The request failed due to an unknown error.
     * @throws NotAuthorizedException
     *         The user is not authorized to perform this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PiClient.GetPerformanceAnalysisReport
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/GetPerformanceAnalysisReport"
     *      target="_top">AWS API Documentation</a>
     */
    default GetPerformanceAnalysisReportResponse getPerformanceAnalysisReport(
            Consumer<GetPerformanceAnalysisReportRequest.Builder> getPerformanceAnalysisReportRequest)
            throws InvalidArgumentException, InternalServiceErrorException, NotAuthorizedException, AwsServiceException,
            SdkClientException, PiException {
        return getPerformanceAnalysisReport(GetPerformanceAnalysisReportRequest.builder()
                .applyMutation(getPerformanceAnalysisReportRequest).build());
    }

    /**
     * <p>
     * Retrieve the metadata for different features. For example, the metadata might indicate that a feature is turned
     * on or off on a specific DB instance.
     * </p>
     *
     * @param getResourceMetadataRequest
     * @return Result of the GetResourceMetadata operation returned by the service.
     * @throws InvalidArgumentException
     *         One of the arguments provided is invalid for this request.
     * @throws InternalServiceErrorException
     *         The request failed due to an unknown error.
     * @throws NotAuthorizedException
     *         The user is not authorized to perform this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PiClient.GetResourceMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/GetResourceMetadata" target="_top">AWS API
     *      Documentation</a>
     */
    default GetResourceMetadataResponse getResourceMetadata(GetResourceMetadataRequest getResourceMetadataRequest)
            throws InvalidArgumentException, InternalServiceErrorException, NotAuthorizedException, AwsServiceException,
            SdkClientException, PiException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieve the metadata for different features. For example, the metadata might indicate that a feature is turned
     * on or off on a specific DB instance.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetResourceMetadataRequest.Builder} avoiding the
     * need to create one manually via {@link GetResourceMetadataRequest#builder()}
     * </p>
     *
     * @param getResourceMetadataRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.pi.model.GetResourceMetadataRequest.Builder} to create a request.
     * @return Result of the GetResourceMetadata operation returned by the service.
     * @throws InvalidArgumentException
     *         One of the arguments provided is invalid for this request.
     * @throws InternalServiceErrorException
     *         The request failed due to an unknown error.
     * @throws NotAuthorizedException
     *         The user is not authorized to perform this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PiClient.GetResourceMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/GetResourceMetadata" target="_top">AWS API
     *      Documentation</a>
     */
    default GetResourceMetadataResponse getResourceMetadata(
            Consumer<GetResourceMetadataRequest.Builder> getResourceMetadataRequest) throws InvalidArgumentException,
            InternalServiceErrorException, NotAuthorizedException, AwsServiceException, SdkClientException, PiException {
        return getResourceMetadata(GetResourceMetadataRequest.builder().applyMutation(getResourceMetadataRequest).build());
    }

    /**
     * <p>
     * Retrieve Performance Insights metrics for a set of data sources over a time period. You can provide specific
     * dimension groups and dimensions, and provide filtering criteria for each group. You must specify an aggregate
     * function for each metric.
     * </p>
     * <note>
     * <p>
     * Each response element returns a maximum of 500 bytes. For larger elements, such as SQL statements, only the first
     * 500 bytes are returned.
     * </p>
     * </note>
     *
     * @param getResourceMetricsRequest
     * @return Result of the GetResourceMetrics operation returned by the service.
     * @throws InvalidArgumentException
     *         One of the arguments provided is invalid for this request.
     * @throws InternalServiceErrorException
     *         The request failed due to an unknown error.
     * @throws NotAuthorizedException
     *         The user is not authorized to perform this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PiClient.GetResourceMetrics
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/GetResourceMetrics" target="_top">AWS API
     *      Documentation</a>
     */
    default GetResourceMetricsResponse getResourceMetrics(GetResourceMetricsRequest getResourceMetricsRequest)
            throws InvalidArgumentException, InternalServiceErrorException, NotAuthorizedException, AwsServiceException,
            SdkClientException, PiException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieve Performance Insights metrics for a set of data sources over a time period. You can provide specific
     * dimension groups and dimensions, and provide filtering criteria for each group. You must specify an aggregate
     * function for each metric.
     * </p>
     * <note>
     * <p>
     * Each response element returns a maximum of 500 bytes. For larger elements, such as SQL statements, only the first
     * 500 bytes are returned.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetResourceMetricsRequest.Builder} avoiding the
     * need to create one manually via {@link GetResourceMetricsRequest#builder()}
     * </p>
     *
     * @param getResourceMetricsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.pi.model.GetResourceMetricsRequest.Builder} to create a request.
     * @return Result of the GetResourceMetrics operation returned by the service.
     * @throws InvalidArgumentException
     *         One of the arguments provided is invalid for this request.
     * @throws InternalServiceErrorException
     *         The request failed due to an unknown error.
     * @throws NotAuthorizedException
     *         The user is not authorized to perform this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PiClient.GetResourceMetrics
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/GetResourceMetrics" target="_top">AWS API
     *      Documentation</a>
     */
    default GetResourceMetricsResponse getResourceMetrics(Consumer<GetResourceMetricsRequest.Builder> getResourceMetricsRequest)
            throws InvalidArgumentException, InternalServiceErrorException, NotAuthorizedException, AwsServiceException,
            SdkClientException, PiException {
        return getResourceMetrics(GetResourceMetricsRequest.builder().applyMutation(getResourceMetricsRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #getResourceMetrics(software.amazon.awssdk.services.pi.model.GetResourceMetricsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pi.paginators.GetResourceMetricsIterable responses = client.getResourceMetricsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.pi.paginators.GetResourceMetricsIterable responses = client
     *             .getResourceMetricsPaginator(request);
     *     for (software.amazon.awssdk.services.pi.model.GetResourceMetricsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pi.paginators.GetResourceMetricsIterable responses = client.getResourceMetricsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getResourceMetrics(software.amazon.awssdk.services.pi.model.GetResourceMetricsRequest)} operation.</b>
     * </p>
     *
     * @param getResourceMetricsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidArgumentException
     *         One of the arguments provided is invalid for this request.
     * @throws InternalServiceErrorException
     *         The request failed due to an unknown error.
     * @throws NotAuthorizedException
     *         The user is not authorized to perform this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PiClient.GetResourceMetrics
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/GetResourceMetrics" target="_top">AWS API
     *      Documentation</a>
     */
    default GetResourceMetricsIterable getResourceMetricsPaginator(GetResourceMetricsRequest getResourceMetricsRequest)
            throws InvalidArgumentException, InternalServiceErrorException, NotAuthorizedException, AwsServiceException,
            SdkClientException, PiException {
        return new GetResourceMetricsIterable(this, getResourceMetricsRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #getResourceMetrics(software.amazon.awssdk.services.pi.model.GetResourceMetricsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pi.paginators.GetResourceMetricsIterable responses = client.getResourceMetricsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.pi.paginators.GetResourceMetricsIterable responses = client
     *             .getResourceMetricsPaginator(request);
     *     for (software.amazon.awssdk.services.pi.model.GetResourceMetricsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pi.paginators.GetResourceMetricsIterable responses = client.getResourceMetricsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getResourceMetrics(software.amazon.awssdk.services.pi.model.GetResourceMetricsRequest)} operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetResourceMetricsRequest.Builder} avoiding the
     * need to create one manually via {@link GetResourceMetricsRequest#builder()}
     * </p>
     *
     * @param getResourceMetricsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.pi.model.GetResourceMetricsRequest.Builder} to create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidArgumentException
     *         One of the arguments provided is invalid for this request.
     * @throws InternalServiceErrorException
     *         The request failed due to an unknown error.
     * @throws NotAuthorizedException
     *         The user is not authorized to perform this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PiClient.GetResourceMetrics
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/GetResourceMetrics" target="_top">AWS API
     *      Documentation</a>
     */
    default GetResourceMetricsIterable getResourceMetricsPaginator(
            Consumer<GetResourceMetricsRequest.Builder> getResourceMetricsRequest) throws InvalidArgumentException,
            InternalServiceErrorException, NotAuthorizedException, AwsServiceException, SdkClientException, PiException {
        return getResourceMetricsPaginator(GetResourceMetricsRequest.builder().applyMutation(getResourceMetricsRequest).build());
    }

    /**
     * <p>
     * Retrieve the dimensions that can be queried for each specified metric type on a specified DB instance.
     * </p>
     *
     * @param listAvailableResourceDimensionsRequest
     * @return Result of the ListAvailableResourceDimensions operation returned by the service.
     * @throws InvalidArgumentException
     *         One of the arguments provided is invalid for this request.
     * @throws InternalServiceErrorException
     *         The request failed due to an unknown error.
     * @throws NotAuthorizedException
     *         The user is not authorized to perform this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PiClient.ListAvailableResourceDimensions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/ListAvailableResourceDimensions"
     *      target="_top">AWS API Documentation</a>
     */
    default ListAvailableResourceDimensionsResponse listAvailableResourceDimensions(
            ListAvailableResourceDimensionsRequest listAvailableResourceDimensionsRequest) throws InvalidArgumentException,
            InternalServiceErrorException, NotAuthorizedException, AwsServiceException, SdkClientException, PiException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieve the dimensions that can be queried for each specified metric type on a specified DB instance.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListAvailableResourceDimensionsRequest.Builder}
     * avoiding the need to create one manually via {@link ListAvailableResourceDimensionsRequest#builder()}
     * </p>
     *
     * @param listAvailableResourceDimensionsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.pi.model.ListAvailableResourceDimensionsRequest.Builder} to create
     *        a request.
     * @return Result of the ListAvailableResourceDimensions operation returned by the service.
     * @throws InvalidArgumentException
     *         One of the arguments provided is invalid for this request.
     * @throws InternalServiceErrorException
     *         The request failed due to an unknown error.
     * @throws NotAuthorizedException
     *         The user is not authorized to perform this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PiClient.ListAvailableResourceDimensions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/ListAvailableResourceDimensions"
     *      target="_top">AWS API Documentation</a>
     */
    default ListAvailableResourceDimensionsResponse listAvailableResourceDimensions(
            Consumer<ListAvailableResourceDimensionsRequest.Builder> listAvailableResourceDimensionsRequest)
            throws InvalidArgumentException, InternalServiceErrorException, NotAuthorizedException, AwsServiceException,
            SdkClientException, PiException {
        return listAvailableResourceDimensions(ListAvailableResourceDimensionsRequest.builder()
                .applyMutation(listAvailableResourceDimensionsRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listAvailableResourceDimensions(software.amazon.awssdk.services.pi.model.ListAvailableResourceDimensionsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pi.paginators.ListAvailableResourceDimensionsIterable responses = client.listAvailableResourceDimensionsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.pi.paginators.ListAvailableResourceDimensionsIterable responses = client
     *             .listAvailableResourceDimensionsPaginator(request);
     *     for (software.amazon.awssdk.services.pi.model.ListAvailableResourceDimensionsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pi.paginators.ListAvailableResourceDimensionsIterable responses = client.listAvailableResourceDimensionsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAvailableResourceDimensions(software.amazon.awssdk.services.pi.model.ListAvailableResourceDimensionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listAvailableResourceDimensionsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidArgumentException
     *         One of the arguments provided is invalid for this request.
     * @throws InternalServiceErrorException
     *         The request failed due to an unknown error.
     * @throws NotAuthorizedException
     *         The user is not authorized to perform this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PiClient.ListAvailableResourceDimensions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/ListAvailableResourceDimensions"
     *      target="_top">AWS API Documentation</a>
     */
    default ListAvailableResourceDimensionsIterable listAvailableResourceDimensionsPaginator(
            ListAvailableResourceDimensionsRequest listAvailableResourceDimensionsRequest) throws InvalidArgumentException,
            InternalServiceErrorException, NotAuthorizedException, AwsServiceException, SdkClientException, PiException {
        return new ListAvailableResourceDimensionsIterable(this, listAvailableResourceDimensionsRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listAvailableResourceDimensions(software.amazon.awssdk.services.pi.model.ListAvailableResourceDimensionsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pi.paginators.ListAvailableResourceDimensionsIterable responses = client.listAvailableResourceDimensionsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.pi.paginators.ListAvailableResourceDimensionsIterable responses = client
     *             .listAvailableResourceDimensionsPaginator(request);
     *     for (software.amazon.awssdk.services.pi.model.ListAvailableResourceDimensionsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pi.paginators.ListAvailableResourceDimensionsIterable responses = client.listAvailableResourceDimensionsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAvailableResourceDimensions(software.amazon.awssdk.services.pi.model.ListAvailableResourceDimensionsRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListAvailableResourceDimensionsRequest.Builder}
     * avoiding the need to create one manually via {@link ListAvailableResourceDimensionsRequest#builder()}
     * </p>
     *
     * @param listAvailableResourceDimensionsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.pi.model.ListAvailableResourceDimensionsRequest.Builder} to create
     *        a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidArgumentException
     *         One of the arguments provided is invalid for this request.
     * @throws InternalServiceErrorException
     *         The request failed due to an unknown error.
     * @throws NotAuthorizedException
     *         The user is not authorized to perform this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PiClient.ListAvailableResourceDimensions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/ListAvailableResourceDimensions"
     *      target="_top">AWS API Documentation</a>
     */
    default ListAvailableResourceDimensionsIterable listAvailableResourceDimensionsPaginator(
            Consumer<ListAvailableResourceDimensionsRequest.Builder> listAvailableResourceDimensionsRequest)
            throws InvalidArgumentException, InternalServiceErrorException, NotAuthorizedException, AwsServiceException,
            SdkClientException, PiException {
        return listAvailableResourceDimensionsPaginator(ListAvailableResourceDimensionsRequest.builder()
                .applyMutation(listAvailableResourceDimensionsRequest).build());
    }

    /**
     * <p>
     * Retrieve metrics of the specified types that can be queried for a specified DB instance.
     * </p>
     *
     * @param listAvailableResourceMetricsRequest
     * @return Result of the ListAvailableResourceMetrics operation returned by the service.
     * @throws InvalidArgumentException
     *         One of the arguments provided is invalid for this request.
     * @throws InternalServiceErrorException
     *         The request failed due to an unknown error.
     * @throws NotAuthorizedException
     *         The user is not authorized to perform this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PiClient.ListAvailableResourceMetrics
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/ListAvailableResourceMetrics"
     *      target="_top">AWS API Documentation</a>
     */
    default ListAvailableResourceMetricsResponse listAvailableResourceMetrics(
            ListAvailableResourceMetricsRequest listAvailableResourceMetricsRequest) throws InvalidArgumentException,
            InternalServiceErrorException, NotAuthorizedException, AwsServiceException, SdkClientException, PiException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieve metrics of the specified types that can be queried for a specified DB instance.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListAvailableResourceMetricsRequest.Builder}
     * avoiding the need to create one manually via {@link ListAvailableResourceMetricsRequest#builder()}
     * </p>
     *
     * @param listAvailableResourceMetricsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.pi.model.ListAvailableResourceMetricsRequest.Builder} to create a
     *        request.
     * @return Result of the ListAvailableResourceMetrics operation returned by the service.
     * @throws InvalidArgumentException
     *         One of the arguments provided is invalid for this request.
     * @throws InternalServiceErrorException
     *         The request failed due to an unknown error.
     * @throws NotAuthorizedException
     *         The user is not authorized to perform this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PiClient.ListAvailableResourceMetrics
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/ListAvailableResourceMetrics"
     *      target="_top">AWS API Documentation</a>
     */
    default ListAvailableResourceMetricsResponse listAvailableResourceMetrics(
            Consumer<ListAvailableResourceMetricsRequest.Builder> listAvailableResourceMetricsRequest)
            throws InvalidArgumentException, InternalServiceErrorException, NotAuthorizedException, AwsServiceException,
            SdkClientException, PiException {
        return listAvailableResourceMetrics(ListAvailableResourceMetricsRequest.builder()
                .applyMutation(listAvailableResourceMetricsRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listAvailableResourceMetrics(software.amazon.awssdk.services.pi.model.ListAvailableResourceMetricsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pi.paginators.ListAvailableResourceMetricsIterable responses = client.listAvailableResourceMetricsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.pi.paginators.ListAvailableResourceMetricsIterable responses = client
     *             .listAvailableResourceMetricsPaginator(request);
     *     for (software.amazon.awssdk.services.pi.model.ListAvailableResourceMetricsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pi.paginators.ListAvailableResourceMetricsIterable responses = client.listAvailableResourceMetricsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAvailableResourceMetrics(software.amazon.awssdk.services.pi.model.ListAvailableResourceMetricsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listAvailableResourceMetricsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidArgumentException
     *         One of the arguments provided is invalid for this request.
     * @throws InternalServiceErrorException
     *         The request failed due to an unknown error.
     * @throws NotAuthorizedException
     *         The user is not authorized to perform this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PiClient.ListAvailableResourceMetrics
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/ListAvailableResourceMetrics"
     *      target="_top">AWS API Documentation</a>
     */
    default ListAvailableResourceMetricsIterable listAvailableResourceMetricsPaginator(
            ListAvailableResourceMetricsRequest listAvailableResourceMetricsRequest) throws InvalidArgumentException,
            InternalServiceErrorException, NotAuthorizedException, AwsServiceException, SdkClientException, PiException {
        return new ListAvailableResourceMetricsIterable(this, listAvailableResourceMetricsRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listAvailableResourceMetrics(software.amazon.awssdk.services.pi.model.ListAvailableResourceMetricsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pi.paginators.ListAvailableResourceMetricsIterable responses = client.listAvailableResourceMetricsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.pi.paginators.ListAvailableResourceMetricsIterable responses = client
     *             .listAvailableResourceMetricsPaginator(request);
     *     for (software.amazon.awssdk.services.pi.model.ListAvailableResourceMetricsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pi.paginators.ListAvailableResourceMetricsIterable responses = client.listAvailableResourceMetricsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAvailableResourceMetrics(software.amazon.awssdk.services.pi.model.ListAvailableResourceMetricsRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListAvailableResourceMetricsRequest.Builder}
     * avoiding the need to create one manually via {@link ListAvailableResourceMetricsRequest#builder()}
     * </p>
     *
     * @param listAvailableResourceMetricsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.pi.model.ListAvailableResourceMetricsRequest.Builder} to create a
     *        request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidArgumentException
     *         One of the arguments provided is invalid for this request.
     * @throws InternalServiceErrorException
     *         The request failed due to an unknown error.
     * @throws NotAuthorizedException
     *         The user is not authorized to perform this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PiClient.ListAvailableResourceMetrics
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/ListAvailableResourceMetrics"
     *      target="_top">AWS API Documentation</a>
     */
    default ListAvailableResourceMetricsIterable listAvailableResourceMetricsPaginator(
            Consumer<ListAvailableResourceMetricsRequest.Builder> listAvailableResourceMetricsRequest)
            throws InvalidArgumentException, InternalServiceErrorException, NotAuthorizedException, AwsServiceException,
            SdkClientException, PiException {
        return listAvailableResourceMetricsPaginator(ListAvailableResourceMetricsRequest.builder()
                .applyMutation(listAvailableResourceMetricsRequest).build());
    }

    /**
     * <p>
     * Lists all the analysis reports created for the DB instance. The reports are sorted based on the start time of
     * each report.
     * </p>
     *
     * @param listPerformanceAnalysisReportsRequest
     * @return Result of the ListPerformanceAnalysisReports operation returned by the service.
     * @throws InvalidArgumentException
     *         One of the arguments provided is invalid for this request.
     * @throws InternalServiceErrorException
     *         The request failed due to an unknown error.
     * @throws NotAuthorizedException
     *         The user is not authorized to perform this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PiClient.ListPerformanceAnalysisReports
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/ListPerformanceAnalysisReports"
     *      target="_top">AWS API Documentation</a>
     */
    default ListPerformanceAnalysisReportsResponse listPerformanceAnalysisReports(
            ListPerformanceAnalysisReportsRequest listPerformanceAnalysisReportsRequest) throws InvalidArgumentException,
            InternalServiceErrorException, NotAuthorizedException, AwsServiceException, SdkClientException, PiException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists all the analysis reports created for the DB instance. The reports are sorted based on the start time of
     * each report.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListPerformanceAnalysisReportsRequest.Builder}
     * avoiding the need to create one manually via {@link ListPerformanceAnalysisReportsRequest#builder()}
     * </p>
     *
     * @param listPerformanceAnalysisReportsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.pi.model.ListPerformanceAnalysisReportsRequest.Builder} to create a
     *        request.
     * @return Result of the ListPerformanceAnalysisReports operation returned by the service.
     * @throws InvalidArgumentException
     *         One of the arguments provided is invalid for this request.
     * @throws InternalServiceErrorException
     *         The request failed due to an unknown error.
     * @throws NotAuthorizedException
     *         The user is not authorized to perform this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PiClient.ListPerformanceAnalysisReports
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/ListPerformanceAnalysisReports"
     *      target="_top">AWS API Documentation</a>
     */
    default ListPerformanceAnalysisReportsResponse listPerformanceAnalysisReports(
            Consumer<ListPerformanceAnalysisReportsRequest.Builder> listPerformanceAnalysisReportsRequest)
            throws InvalidArgumentException, InternalServiceErrorException, NotAuthorizedException, AwsServiceException,
            SdkClientException, PiException {
        return listPerformanceAnalysisReports(ListPerformanceAnalysisReportsRequest.builder()
                .applyMutation(listPerformanceAnalysisReportsRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listPerformanceAnalysisReports(software.amazon.awssdk.services.pi.model.ListPerformanceAnalysisReportsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pi.paginators.ListPerformanceAnalysisReportsIterable responses = client.listPerformanceAnalysisReportsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.pi.paginators.ListPerformanceAnalysisReportsIterable responses = client
     *             .listPerformanceAnalysisReportsPaginator(request);
     *     for (software.amazon.awssdk.services.pi.model.ListPerformanceAnalysisReportsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pi.paginators.ListPerformanceAnalysisReportsIterable responses = client.listPerformanceAnalysisReportsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPerformanceAnalysisReports(software.amazon.awssdk.services.pi.model.ListPerformanceAnalysisReportsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listPerformanceAnalysisReportsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidArgumentException
     *         One of the arguments provided is invalid for this request.
     * @throws InternalServiceErrorException
     *         The request failed due to an unknown error.
     * @throws NotAuthorizedException
     *         The user is not authorized to perform this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PiClient.ListPerformanceAnalysisReports
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/ListPerformanceAnalysisReports"
     *      target="_top">AWS API Documentation</a>
     */
    default ListPerformanceAnalysisReportsIterable listPerformanceAnalysisReportsPaginator(
            ListPerformanceAnalysisReportsRequest listPerformanceAnalysisReportsRequest) throws InvalidArgumentException,
            InternalServiceErrorException, NotAuthorizedException, AwsServiceException, SdkClientException, PiException {
        return new ListPerformanceAnalysisReportsIterable(this, listPerformanceAnalysisReportsRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listPerformanceAnalysisReports(software.amazon.awssdk.services.pi.model.ListPerformanceAnalysisReportsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pi.paginators.ListPerformanceAnalysisReportsIterable responses = client.listPerformanceAnalysisReportsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.pi.paginators.ListPerformanceAnalysisReportsIterable responses = client
     *             .listPerformanceAnalysisReportsPaginator(request);
     *     for (software.amazon.awssdk.services.pi.model.ListPerformanceAnalysisReportsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pi.paginators.ListPerformanceAnalysisReportsIterable responses = client.listPerformanceAnalysisReportsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPerformanceAnalysisReports(software.amazon.awssdk.services.pi.model.ListPerformanceAnalysisReportsRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListPerformanceAnalysisReportsRequest.Builder}
     * avoiding the need to create one manually via {@link ListPerformanceAnalysisReportsRequest#builder()}
     * </p>
     *
     * @param listPerformanceAnalysisReportsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.pi.model.ListPerformanceAnalysisReportsRequest.Builder} to create a
     *        request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidArgumentException
     *         One of the arguments provided is invalid for this request.
     * @throws InternalServiceErrorException
     *         The request failed due to an unknown error.
     * @throws NotAuthorizedException
     *         The user is not authorized to perform this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PiClient.ListPerformanceAnalysisReports
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/ListPerformanceAnalysisReports"
     *      target="_top">AWS API Documentation</a>
     */
    default ListPerformanceAnalysisReportsIterable listPerformanceAnalysisReportsPaginator(
            Consumer<ListPerformanceAnalysisReportsRequest.Builder> listPerformanceAnalysisReportsRequest)
            throws InvalidArgumentException, InternalServiceErrorException, NotAuthorizedException, AwsServiceException,
            SdkClientException, PiException {
        return listPerformanceAnalysisReportsPaginator(ListPerformanceAnalysisReportsRequest.builder()
                .applyMutation(listPerformanceAnalysisReportsRequest).build());
    }

    /**
     * <p>
     * Retrieves all the metadata tags associated with Amazon RDS Performance Insights resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws InvalidArgumentException
     *         One of the arguments provided is invalid for this request.
     * @throws InternalServiceErrorException
     *         The request failed due to an unknown error.
     * @throws NotAuthorizedException
     *         The user is not authorized to perform this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PiClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    default ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws InvalidArgumentException, InternalServiceErrorException, NotAuthorizedException, AwsServiceException,
            SdkClientException, PiException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves all the metadata tags associated with Amazon RDS Performance Insights resource.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListTagsForResourceRequest.Builder} avoiding the
     * need to create one manually via {@link ListTagsForResourceRequest#builder()}
     * </p>
     *
     * @param listTagsForResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.pi.model.ListTagsForResourceRequest.Builder} to create a request.
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws InvalidArgumentException
     *         One of the arguments provided is invalid for this request.
     * @throws InternalServiceErrorException
     *         The request failed due to an unknown error.
     * @throws NotAuthorizedException
     *         The user is not authorized to perform this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PiClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    default ListTagsForResourceResponse listTagsForResource(
            Consumer<ListTagsForResourceRequest.Builder> listTagsForResourceRequest) throws InvalidArgumentException,
            InternalServiceErrorException, NotAuthorizedException, AwsServiceException, SdkClientException, PiException {
        return listTagsForResource(ListTagsForResourceRequest.builder().applyMutation(listTagsForResourceRequest).build());
    }

    /**
     * <p>
     * Adds metadata tags to the Amazon RDS Performance Insights resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws InvalidArgumentException
     *         One of the arguments provided is invalid for this request.
     * @throws InternalServiceErrorException
     *         The request failed due to an unknown error.
     * @throws NotAuthorizedException
     *         The user is not authorized to perform this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PiClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws InvalidArgumentException,
            InternalServiceErrorException, NotAuthorizedException, AwsServiceException, SdkClientException, PiException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Adds metadata tags to the Amazon RDS Performance Insights resource.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link TagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link TagResourceRequest#builder()}
     * </p>
     *
     * @param tagResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.pi.model.TagResourceRequest.Builder} to create a request.
     * @return Result of the TagResource operation returned by the service.
     * @throws InvalidArgumentException
     *         One of the arguments provided is invalid for this request.
     * @throws InternalServiceErrorException
     *         The request failed due to an unknown error.
     * @throws NotAuthorizedException
     *         The user is not authorized to perform this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PiClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default TagResourceResponse tagResource(Consumer<TagResourceRequest.Builder> tagResourceRequest)
            throws InvalidArgumentException, InternalServiceErrorException, NotAuthorizedException, AwsServiceException,
            SdkClientException, PiException {
        return tagResource(TagResourceRequest.builder().applyMutation(tagResourceRequest).build());
    }

    /**
     * <p>
     * Deletes the metadata tags from the Amazon RDS Performance Insights resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws InvalidArgumentException
     *         One of the arguments provided is invalid for this request.
     * @throws InternalServiceErrorException
     *         The request failed due to an unknown error.
     * @throws NotAuthorizedException
     *         The user is not authorized to perform this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PiClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws InvalidArgumentException,
            InternalServiceErrorException, NotAuthorizedException, AwsServiceException, SdkClientException, PiException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes the metadata tags from the Amazon RDS Performance Insights resource.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UntagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link UntagResourceRequest#builder()}
     * </p>
     *
     * @param untagResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.pi.model.UntagResourceRequest.Builder} to create a request.
     * @return Result of the UntagResource operation returned by the service.
     * @throws InvalidArgumentException
     *         One of the arguments provided is invalid for this request.
     * @throws InternalServiceErrorException
     *         The request failed due to an unknown error.
     * @throws NotAuthorizedException
     *         The user is not authorized to perform this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PiClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default UntagResourceResponse untagResource(Consumer<UntagResourceRequest.Builder> untagResourceRequest)
            throws InvalidArgumentException, InternalServiceErrorException, NotAuthorizedException, AwsServiceException,
            SdkClientException, PiException {
        return untagResource(UntagResourceRequest.builder().applyMutation(untagResourceRequest).build());
    }

    /**
     * Create a {@link PiClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static PiClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link PiClient}.
     */
    static PiClientBuilder builder() {
        return new DefaultPiClientBuilder();
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of(SERVICE_METADATA_ID);
    }

    @Override
    default PiServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }
}
