/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.pinpoint.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides information about the type and the names of attributes that were removed from all the endpoints that are
 * associated with an application.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AttributesResource implements SdkPojo, Serializable,
        ToCopyableBuilder<AttributesResource.Builder, AttributesResource> {
    private static final SdkField<String> APPLICATION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ApplicationId").getter(getter(AttributesResource::applicationId)).setter(setter(Builder::applicationId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ApplicationId").build()).build();

    private static final SdkField<String> ATTRIBUTE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AttributeType").getter(getter(AttributesResource::attributeType)).setter(setter(Builder::attributeType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AttributeType").build()).build();

    private static final SdkField<List<String>> ATTRIBUTES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Attributes")
            .getter(getter(AttributesResource::attributes))
            .setter(setter(Builder::attributes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Attributes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(APPLICATION_ID_FIELD,
            ATTRIBUTE_TYPE_FIELD, ATTRIBUTES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String applicationId;

    private final String attributeType;

    private final List<String> attributes;

    private AttributesResource(BuilderImpl builder) {
        this.applicationId = builder.applicationId;
        this.attributeType = builder.attributeType;
        this.attributes = builder.attributes;
    }

    /**
     * <p>
     * The unique identifier for the application.
     * </p>
     * 
     * @return The unique identifier for the application.
     */
    public final String applicationId() {
        return applicationId;
    }

    /**
     * <p>
     * The type of attribute or attributes that were removed from the endpoints. Valid values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * endpoint-custom-attributes - Custom attributes that describe endpoints.
     * </p>
     * </li>
     * <li>
     * <p>
     * endpoint-metric-attributes - Custom metrics that your app reports to Amazon Pinpoint for endpoints.
     * </p>
     * </li>
     * <li>
     * <p>
     * endpoint-user-attributes - Custom attributes that describe users.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The type of attribute or attributes that were removed from the endpoints. Valid values are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         endpoint-custom-attributes - Custom attributes that describe endpoints.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         endpoint-metric-attributes - Custom metrics that your app reports to Amazon Pinpoint for endpoints.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         endpoint-user-attributes - Custom attributes that describe users.
     *         </p>
     *         </li>
     */
    public final String attributeType() {
        return attributeType;
    }

    /**
     * For responses, this returns true if the service returned a value for the Attributes property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasAttributes() {
        return attributes != null && !(attributes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array that specifies the names of the attributes that were removed from the endpoints.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAttributes} method.
     * </p>
     * 
     * @return An array that specifies the names of the attributes that were removed from the endpoints.
     */
    public final List<String> attributes() {
        return attributes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(applicationId());
        hashCode = 31 * hashCode + Objects.hashCode(attributeType());
        hashCode = 31 * hashCode + Objects.hashCode(hasAttributes() ? attributes() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AttributesResource)) {
            return false;
        }
        AttributesResource other = (AttributesResource) obj;
        return Objects.equals(applicationId(), other.applicationId()) && Objects.equals(attributeType(), other.attributeType())
                && hasAttributes() == other.hasAttributes() && Objects.equals(attributes(), other.attributes());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AttributesResource").add("ApplicationId", applicationId()).add("AttributeType", attributeType())
                .add("Attributes", hasAttributes() ? attributes() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ApplicationId":
            return Optional.ofNullable(clazz.cast(applicationId()));
        case "AttributeType":
            return Optional.ofNullable(clazz.cast(attributeType()));
        case "Attributes":
            return Optional.ofNullable(clazz.cast(attributes()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("ApplicationId", APPLICATION_ID_FIELD);
        map.put("AttributeType", ATTRIBUTE_TYPE_FIELD);
        map.put("Attributes", ATTRIBUTES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<AttributesResource, T> g) {
        return obj -> g.apply((AttributesResource) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AttributesResource> {
        /**
         * <p>
         * The unique identifier for the application.
         * </p>
         * 
         * @param applicationId
         *        The unique identifier for the application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder applicationId(String applicationId);

        /**
         * <p>
         * The type of attribute or attributes that were removed from the endpoints. Valid values are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * endpoint-custom-attributes - Custom attributes that describe endpoints.
         * </p>
         * </li>
         * <li>
         * <p>
         * endpoint-metric-attributes - Custom metrics that your app reports to Amazon Pinpoint for endpoints.
         * </p>
         * </li>
         * <li>
         * <p>
         * endpoint-user-attributes - Custom attributes that describe users.
         * </p>
         * </li>
         * </ul>
         * 
         * @param attributeType
         *        The type of attribute or attributes that were removed from the endpoints. Valid values are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        endpoint-custom-attributes - Custom attributes that describe endpoints.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        endpoint-metric-attributes - Custom metrics that your app reports to Amazon Pinpoint for endpoints.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        endpoint-user-attributes - Custom attributes that describe users.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributeType(String attributeType);

        /**
         * <p>
         * An array that specifies the names of the attributes that were removed from the endpoints.
         * </p>
         * 
         * @param attributes
         *        An array that specifies the names of the attributes that were removed from the endpoints.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributes(Collection<String> attributes);

        /**
         * <p>
         * An array that specifies the names of the attributes that were removed from the endpoints.
         * </p>
         * 
         * @param attributes
         *        An array that specifies the names of the attributes that were removed from the endpoints.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributes(String... attributes);
    }

    static final class BuilderImpl implements Builder {
        private String applicationId;

        private String attributeType;

        private List<String> attributes = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(AttributesResource model) {
            applicationId(model.applicationId);
            attributeType(model.attributeType);
            attributes(model.attributes);
        }

        public final String getApplicationId() {
            return applicationId;
        }

        public final void setApplicationId(String applicationId) {
            this.applicationId = applicationId;
        }

        @Override
        public final Builder applicationId(String applicationId) {
            this.applicationId = applicationId;
            return this;
        }

        public final String getAttributeType() {
            return attributeType;
        }

        public final void setAttributeType(String attributeType) {
            this.attributeType = attributeType;
        }

        @Override
        public final Builder attributeType(String attributeType) {
            this.attributeType = attributeType;
            return this;
        }

        public final Collection<String> getAttributes() {
            if (attributes instanceof SdkAutoConstructList) {
                return null;
            }
            return attributes;
        }

        public final void setAttributes(Collection<String> attributes) {
            this.attributes = ListOf__stringCopier.copy(attributes);
        }

        @Override
        public final Builder attributes(Collection<String> attributes) {
            this.attributes = ListOf__stringCopier.copy(attributes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder attributes(String... attributes) {
            attributes(Arrays.asList(attributes));
            return this;
        }

        @Override
        public AttributesResource build() {
            return new AttributesResource(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
