/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.quicksight.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The structure of a VPC connection.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class VPCConnection implements SdkPojo, Serializable, ToCopyableBuilder<VPCConnection.Builder, VPCConnection> {
    private static final SdkField<String> VPC_CONNECTION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VPCConnectionId").getter(getter(VPCConnection::vpcConnectionId))
            .setter(setter(Builder::vpcConnectionId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VPCConnectionId").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Arn")
            .getter(getter(VPCConnection::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Arn").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(VPCConnection::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> VPC_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("VPCId")
            .getter(getter(VPCConnection::vpcId)).setter(setter(Builder::vpcId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VPCId").build()).build();

    private static final SdkField<List<String>> SECURITY_GROUP_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SecurityGroupIds")
            .getter(getter(VPCConnection::securityGroupIds))
            .setter(setter(Builder::securityGroupIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityGroupIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> DNS_RESOLVERS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("DnsResolvers")
            .getter(getter(VPCConnection::dnsResolvers))
            .setter(setter(Builder::dnsResolvers))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DnsResolvers").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(VPCConnection::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> AVAILABILITY_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AvailabilityStatus").getter(getter(VPCConnection::availabilityStatusAsString))
            .setter(setter(Builder::availabilityStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvailabilityStatus").build())
            .build();

    private static final SdkField<List<NetworkInterface>> NETWORK_INTERFACES_FIELD = SdkField
            .<List<NetworkInterface>> builder(MarshallingType.LIST)
            .memberName("NetworkInterfaces")
            .getter(getter(VPCConnection::networkInterfaces))
            .setter(setter(Builder::networkInterfaces))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NetworkInterfaces").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<NetworkInterface> builder(MarshallingType.SDK_POJO)
                                            .constructor(NetworkInterface::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RoleArn").getter(getter(VPCConnection::roleArn)).setter(setter(Builder::roleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RoleArn").build()).build();

    private static final SdkField<Instant> CREATED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedTime").getter(getter(VPCConnection::createdTime)).setter(setter(Builder::createdTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedTime").build()).build();

    private static final SdkField<Instant> LAST_UPDATED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastUpdatedTime").getter(getter(VPCConnection::lastUpdatedTime))
            .setter(setter(Builder::lastUpdatedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastUpdatedTime").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(VPC_CONNECTION_ID_FIELD,
            ARN_FIELD, NAME_FIELD, VPC_ID_FIELD, SECURITY_GROUP_IDS_FIELD, DNS_RESOLVERS_FIELD, STATUS_FIELD,
            AVAILABILITY_STATUS_FIELD, NETWORK_INTERFACES_FIELD, ROLE_ARN_FIELD, CREATED_TIME_FIELD, LAST_UPDATED_TIME_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String vpcConnectionId;

    private final String arn;

    private final String name;

    private final String vpcId;

    private final List<String> securityGroupIds;

    private final List<String> dnsResolvers;

    private final String status;

    private final String availabilityStatus;

    private final List<NetworkInterface> networkInterfaces;

    private final String roleArn;

    private final Instant createdTime;

    private final Instant lastUpdatedTime;

    private VPCConnection(BuilderImpl builder) {
        this.vpcConnectionId = builder.vpcConnectionId;
        this.arn = builder.arn;
        this.name = builder.name;
        this.vpcId = builder.vpcId;
        this.securityGroupIds = builder.securityGroupIds;
        this.dnsResolvers = builder.dnsResolvers;
        this.status = builder.status;
        this.availabilityStatus = builder.availabilityStatus;
        this.networkInterfaces = builder.networkInterfaces;
        this.roleArn = builder.roleArn;
        this.createdTime = builder.createdTime;
        this.lastUpdatedTime = builder.lastUpdatedTime;
    }

    /**
     * <p>
     * The ID of the VPC connection that you're creating. This ID is a unique identifier for each Amazon Web Services
     * Region in an Amazon Web Services account.
     * </p>
     * 
     * @return The ID of the VPC connection that you're creating. This ID is a unique identifier for each Amazon Web
     *         Services Region in an Amazon Web Services account.
     */
    public final String vpcConnectionId() {
        return vpcConnectionId;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the VPC connection.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the VPC connection.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The display name for the VPC connection.
     * </p>
     * 
     * @return The display name for the VPC connection.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The Amazon EC2 VPC ID associated with the VPC connection.
     * </p>
     * 
     * @return The Amazon EC2 VPC ID associated with the VPC connection.
     */
    public final String vpcId() {
        return vpcId;
    }

    /**
     * For responses, this returns true if the service returned a value for the SecurityGroupIds property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSecurityGroupIds() {
        return securityGroupIds != null && !(securityGroupIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Amazon EC2 security group IDs associated with the VPC connection.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSecurityGroupIds} method.
     * </p>
     * 
     * @return The Amazon EC2 security group IDs associated with the VPC connection.
     */
    public final List<String> securityGroupIds() {
        return securityGroupIds;
    }

    /**
     * For responses, this returns true if the service returned a value for the DnsResolvers property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasDnsResolvers() {
        return dnsResolvers != null && !(dnsResolvers instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of IP addresses of DNS resolver endpoints for the VPC connection.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDnsResolvers} method.
     * </p>
     * 
     * @return A list of IP addresses of DNS resolver endpoints for the VPC connection.
     */
    public final List<String> dnsResolvers() {
        return dnsResolvers;
    }

    /**
     * <p>
     * The status of the VPC connection.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link VPCConnectionResourceStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the VPC connection.
     * @see VPCConnectionResourceStatus
     */
    public final VPCConnectionResourceStatus status() {
        return VPCConnectionResourceStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the VPC connection.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link VPCConnectionResourceStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the VPC connection.
     * @see VPCConnectionResourceStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The availability status of the VPC connection.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #availabilityStatus} will return {@link VPCConnectionAvailabilityStatus#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #availabilityStatusAsString}.
     * </p>
     * 
     * @return The availability status of the VPC connection.
     * @see VPCConnectionAvailabilityStatus
     */
    public final VPCConnectionAvailabilityStatus availabilityStatus() {
        return VPCConnectionAvailabilityStatus.fromValue(availabilityStatus);
    }

    /**
     * <p>
     * The availability status of the VPC connection.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #availabilityStatus} will return {@link VPCConnectionAvailabilityStatus#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #availabilityStatusAsString}.
     * </p>
     * 
     * @return The availability status of the VPC connection.
     * @see VPCConnectionAvailabilityStatus
     */
    public final String availabilityStatusAsString() {
        return availabilityStatus;
    }

    /**
     * For responses, this returns true if the service returned a value for the NetworkInterfaces property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasNetworkInterfaces() {
        return networkInterfaces != null && !(networkInterfaces instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of network interfaces.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasNetworkInterfaces} method.
     * </p>
     * 
     * @return A list of network interfaces.
     */
    public final List<NetworkInterface> networkInterfaces() {
        return networkInterfaces;
    }

    /**
     * <p>
     * The ARN of the IAM role associated with the VPC connection.
     * </p>
     * 
     * @return The ARN of the IAM role associated with the VPC connection.
     */
    public final String roleArn() {
        return roleArn;
    }

    /**
     * <p>
     * The time that the VPC connection was created.
     * </p>
     * 
     * @return The time that the VPC connection was created.
     */
    public final Instant createdTime() {
        return createdTime;
    }

    /**
     * <p>
     * The time that the VPC connection was last updated.
     * </p>
     * 
     * @return The time that the VPC connection was last updated.
     */
    public final Instant lastUpdatedTime() {
        return lastUpdatedTime;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(vpcConnectionId());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(vpcId());
        hashCode = 31 * hashCode + Objects.hashCode(hasSecurityGroupIds() ? securityGroupIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasDnsResolvers() ? dnsResolvers() : null);
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(availabilityStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasNetworkInterfaces() ? networkInterfaces() : null);
        hashCode = 31 * hashCode + Objects.hashCode(roleArn());
        hashCode = 31 * hashCode + Objects.hashCode(createdTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdatedTime());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof VPCConnection)) {
            return false;
        }
        VPCConnection other = (VPCConnection) obj;
        return Objects.equals(vpcConnectionId(), other.vpcConnectionId()) && Objects.equals(arn(), other.arn())
                && Objects.equals(name(), other.name()) && Objects.equals(vpcId(), other.vpcId())
                && hasSecurityGroupIds() == other.hasSecurityGroupIds()
                && Objects.equals(securityGroupIds(), other.securityGroupIds()) && hasDnsResolvers() == other.hasDnsResolvers()
                && Objects.equals(dnsResolvers(), other.dnsResolvers())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(availabilityStatusAsString(), other.availabilityStatusAsString())
                && hasNetworkInterfaces() == other.hasNetworkInterfaces()
                && Objects.equals(networkInterfaces(), other.networkInterfaces()) && Objects.equals(roleArn(), other.roleArn())
                && Objects.equals(createdTime(), other.createdTime())
                && Objects.equals(lastUpdatedTime(), other.lastUpdatedTime());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("VPCConnection").add("VPCConnectionId", vpcConnectionId()).add("Arn", arn()).add("Name", name())
                .add("VPCId", vpcId()).add("SecurityGroupIds", hasSecurityGroupIds() ? securityGroupIds() : null)
                .add("DnsResolvers", hasDnsResolvers() ? dnsResolvers() : null).add("Status", statusAsString())
                .add("AvailabilityStatus", availabilityStatusAsString())
                .add("NetworkInterfaces", hasNetworkInterfaces() ? networkInterfaces() : null).add("RoleArn", roleArn())
                .add("CreatedTime", createdTime()).add("LastUpdatedTime", lastUpdatedTime()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "VPCConnectionId":
            return Optional.ofNullable(clazz.cast(vpcConnectionId()));
        case "Arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "VPCId":
            return Optional.ofNullable(clazz.cast(vpcId()));
        case "SecurityGroupIds":
            return Optional.ofNullable(clazz.cast(securityGroupIds()));
        case "DnsResolvers":
            return Optional.ofNullable(clazz.cast(dnsResolvers()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "AvailabilityStatus":
            return Optional.ofNullable(clazz.cast(availabilityStatusAsString()));
        case "NetworkInterfaces":
            return Optional.ofNullable(clazz.cast(networkInterfaces()));
        case "RoleArn":
            return Optional.ofNullable(clazz.cast(roleArn()));
        case "CreatedTime":
            return Optional.ofNullable(clazz.cast(createdTime()));
        case "LastUpdatedTime":
            return Optional.ofNullable(clazz.cast(lastUpdatedTime()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("VPCConnectionId", VPC_CONNECTION_ID_FIELD);
        map.put("Arn", ARN_FIELD);
        map.put("Name", NAME_FIELD);
        map.put("VPCId", VPC_ID_FIELD);
        map.put("SecurityGroupIds", SECURITY_GROUP_IDS_FIELD);
        map.put("DnsResolvers", DNS_RESOLVERS_FIELD);
        map.put("Status", STATUS_FIELD);
        map.put("AvailabilityStatus", AVAILABILITY_STATUS_FIELD);
        map.put("NetworkInterfaces", NETWORK_INTERFACES_FIELD);
        map.put("RoleArn", ROLE_ARN_FIELD);
        map.put("CreatedTime", CREATED_TIME_FIELD);
        map.put("LastUpdatedTime", LAST_UPDATED_TIME_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<VPCConnection, T> g) {
        return obj -> g.apply((VPCConnection) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, VPCConnection> {
        /**
         * <p>
         * The ID of the VPC connection that you're creating. This ID is a unique identifier for each Amazon Web
         * Services Region in an Amazon Web Services account.
         * </p>
         * 
         * @param vpcConnectionId
         *        The ID of the VPC connection that you're creating. This ID is a unique identifier for each Amazon Web
         *        Services Region in an Amazon Web Services account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcConnectionId(String vpcConnectionId);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the VPC connection.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the VPC connection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The display name for the VPC connection.
         * </p>
         * 
         * @param name
         *        The display name for the VPC connection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The Amazon EC2 VPC ID associated with the VPC connection.
         * </p>
         * 
         * @param vpcId
         *        The Amazon EC2 VPC ID associated with the VPC connection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcId(String vpcId);

        /**
         * <p>
         * The Amazon EC2 security group IDs associated with the VPC connection.
         * </p>
         * 
         * @param securityGroupIds
         *        The Amazon EC2 security group IDs associated with the VPC connection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroupIds(Collection<String> securityGroupIds);

        /**
         * <p>
         * The Amazon EC2 security group IDs associated with the VPC connection.
         * </p>
         * 
         * @param securityGroupIds
         *        The Amazon EC2 security group IDs associated with the VPC connection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroupIds(String... securityGroupIds);

        /**
         * <p>
         * A list of IP addresses of DNS resolver endpoints for the VPC connection.
         * </p>
         * 
         * @param dnsResolvers
         *        A list of IP addresses of DNS resolver endpoints for the VPC connection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsResolvers(Collection<String> dnsResolvers);

        /**
         * <p>
         * A list of IP addresses of DNS resolver endpoints for the VPC connection.
         * </p>
         * 
         * @param dnsResolvers
         *        A list of IP addresses of DNS resolver endpoints for the VPC connection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsResolvers(String... dnsResolvers);

        /**
         * <p>
         * The status of the VPC connection.
         * </p>
         * 
         * @param status
         *        The status of the VPC connection.
         * @see VPCConnectionResourceStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VPCConnectionResourceStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the VPC connection.
         * </p>
         * 
         * @param status
         *        The status of the VPC connection.
         * @see VPCConnectionResourceStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VPCConnectionResourceStatus
         */
        Builder status(VPCConnectionResourceStatus status);

        /**
         * <p>
         * The availability status of the VPC connection.
         * </p>
         * 
         * @param availabilityStatus
         *        The availability status of the VPC connection.
         * @see VPCConnectionAvailabilityStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VPCConnectionAvailabilityStatus
         */
        Builder availabilityStatus(String availabilityStatus);

        /**
         * <p>
         * The availability status of the VPC connection.
         * </p>
         * 
         * @param availabilityStatus
         *        The availability status of the VPC connection.
         * @see VPCConnectionAvailabilityStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VPCConnectionAvailabilityStatus
         */
        Builder availabilityStatus(VPCConnectionAvailabilityStatus availabilityStatus);

        /**
         * <p>
         * A list of network interfaces.
         * </p>
         * 
         * @param networkInterfaces
         *        A list of network interfaces.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkInterfaces(Collection<NetworkInterface> networkInterfaces);

        /**
         * <p>
         * A list of network interfaces.
         * </p>
         * 
         * @param networkInterfaces
         *        A list of network interfaces.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkInterfaces(NetworkInterface... networkInterfaces);

        /**
         * <p>
         * A list of network interfaces.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.quicksight.model.NetworkInterface.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.quicksight.model.NetworkInterface#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.quicksight.model.NetworkInterface.Builder#build()} is called
         * immediately and its result is passed to {@link #networkInterfaces(List<NetworkInterface>)}.
         * 
         * @param networkInterfaces
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.quicksight.model.NetworkInterface.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #networkInterfaces(java.util.Collection<NetworkInterface>)
         */
        Builder networkInterfaces(Consumer<NetworkInterface.Builder>... networkInterfaces);

        /**
         * <p>
         * The ARN of the IAM role associated with the VPC connection.
         * </p>
         * 
         * @param roleArn
         *        The ARN of the IAM role associated with the VPC connection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleArn(String roleArn);

        /**
         * <p>
         * The time that the VPC connection was created.
         * </p>
         * 
         * @param createdTime
         *        The time that the VPC connection was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdTime(Instant createdTime);

        /**
         * <p>
         * The time that the VPC connection was last updated.
         * </p>
         * 
         * @param lastUpdatedTime
         *        The time that the VPC connection was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdatedTime(Instant lastUpdatedTime);
    }

    static final class BuilderImpl implements Builder {
        private String vpcConnectionId;

        private String arn;

        private String name;

        private String vpcId;

        private List<String> securityGroupIds = DefaultSdkAutoConstructList.getInstance();

        private List<String> dnsResolvers = DefaultSdkAutoConstructList.getInstance();

        private String status;

        private String availabilityStatus;

        private List<NetworkInterface> networkInterfaces = DefaultSdkAutoConstructList.getInstance();

        private String roleArn;

        private Instant createdTime;

        private Instant lastUpdatedTime;

        private BuilderImpl() {
        }

        private BuilderImpl(VPCConnection model) {
            vpcConnectionId(model.vpcConnectionId);
            arn(model.arn);
            name(model.name);
            vpcId(model.vpcId);
            securityGroupIds(model.securityGroupIds);
            dnsResolvers(model.dnsResolvers);
            status(model.status);
            availabilityStatus(model.availabilityStatus);
            networkInterfaces(model.networkInterfaces);
            roleArn(model.roleArn);
            createdTime(model.createdTime);
            lastUpdatedTime(model.lastUpdatedTime);
        }

        public final String getVpcConnectionId() {
            return vpcConnectionId;
        }

        public final void setVpcConnectionId(String vpcConnectionId) {
            this.vpcConnectionId = vpcConnectionId;
        }

        @Override
        public final Builder vpcConnectionId(String vpcConnectionId) {
            this.vpcConnectionId = vpcConnectionId;
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getVpcId() {
            return vpcId;
        }

        public final void setVpcId(String vpcId) {
            this.vpcId = vpcId;
        }

        @Override
        public final Builder vpcId(String vpcId) {
            this.vpcId = vpcId;
            return this;
        }

        public final Collection<String> getSecurityGroupIds() {
            if (securityGroupIds instanceof SdkAutoConstructList) {
                return null;
            }
            return securityGroupIds;
        }

        public final void setSecurityGroupIds(Collection<String> securityGroupIds) {
            this.securityGroupIds = SecurityGroupIdListCopier.copy(securityGroupIds);
        }

        @Override
        public final Builder securityGroupIds(Collection<String> securityGroupIds) {
            this.securityGroupIds = SecurityGroupIdListCopier.copy(securityGroupIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder securityGroupIds(String... securityGroupIds) {
            securityGroupIds(Arrays.asList(securityGroupIds));
            return this;
        }

        public final Collection<String> getDnsResolvers() {
            if (dnsResolvers instanceof SdkAutoConstructList) {
                return null;
            }
            return dnsResolvers;
        }

        public final void setDnsResolvers(Collection<String> dnsResolvers) {
            this.dnsResolvers = StringListCopier.copy(dnsResolvers);
        }

        @Override
        public final Builder dnsResolvers(Collection<String> dnsResolvers) {
            this.dnsResolvers = StringListCopier.copy(dnsResolvers);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dnsResolvers(String... dnsResolvers) {
            dnsResolvers(Arrays.asList(dnsResolvers));
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(VPCConnectionResourceStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getAvailabilityStatus() {
            return availabilityStatus;
        }

        public final void setAvailabilityStatus(String availabilityStatus) {
            this.availabilityStatus = availabilityStatus;
        }

        @Override
        public final Builder availabilityStatus(String availabilityStatus) {
            this.availabilityStatus = availabilityStatus;
            return this;
        }

        @Override
        public final Builder availabilityStatus(VPCConnectionAvailabilityStatus availabilityStatus) {
            this.availabilityStatus(availabilityStatus == null ? null : availabilityStatus.toString());
            return this;
        }

        public final List<NetworkInterface.Builder> getNetworkInterfaces() {
            List<NetworkInterface.Builder> result = NetworkInterfaceListCopier.copyToBuilder(this.networkInterfaces);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setNetworkInterfaces(Collection<NetworkInterface.BuilderImpl> networkInterfaces) {
            this.networkInterfaces = NetworkInterfaceListCopier.copyFromBuilder(networkInterfaces);
        }

        @Override
        public final Builder networkInterfaces(Collection<NetworkInterface> networkInterfaces) {
            this.networkInterfaces = NetworkInterfaceListCopier.copy(networkInterfaces);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder networkInterfaces(NetworkInterface... networkInterfaces) {
            networkInterfaces(Arrays.asList(networkInterfaces));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder networkInterfaces(Consumer<NetworkInterface.Builder>... networkInterfaces) {
            networkInterfaces(Stream.of(networkInterfaces).map(c -> NetworkInterface.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getRoleArn() {
            return roleArn;
        }

        public final void setRoleArn(String roleArn) {
            this.roleArn = roleArn;
        }

        @Override
        public final Builder roleArn(String roleArn) {
            this.roleArn = roleArn;
            return this;
        }

        public final Instant getCreatedTime() {
            return createdTime;
        }

        public final void setCreatedTime(Instant createdTime) {
            this.createdTime = createdTime;
        }

        @Override
        public final Builder createdTime(Instant createdTime) {
            this.createdTime = createdTime;
            return this;
        }

        public final Instant getLastUpdatedTime() {
            return lastUpdatedTime;
        }

        public final void setLastUpdatedTime(Instant lastUpdatedTime) {
            this.lastUpdatedTime = lastUpdatedTime;
        }

        @Override
        public final Builder lastUpdatedTime(Instant lastUpdatedTime) {
            this.lastUpdatedTime = lastUpdatedTime;
            return this;
        }

        @Override
        public VPCConnection build() {
            return new VPCConnection(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
