/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rekognition.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A description of an Amazon Rekognition Custom Labels project. For more information, see <a>DescribeProjects</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ProjectDescription implements SdkPojo, Serializable,
        ToCopyableBuilder<ProjectDescription.Builder, ProjectDescription> {
    private static final SdkField<String> PROJECT_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ProjectArn").getter(getter(ProjectDescription::projectArn)).setter(setter(Builder::projectArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProjectArn").build()).build();

    private static final SdkField<Instant> CREATION_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationTimestamp").getter(getter(ProjectDescription::creationTimestamp))
            .setter(setter(Builder::creationTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTimestamp").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(ProjectDescription::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<List<DatasetMetadata>> DATASETS_FIELD = SdkField
            .<List<DatasetMetadata>> builder(MarshallingType.LIST)
            .memberName("Datasets")
            .getter(getter(ProjectDescription::datasets))
            .setter(setter(Builder::datasets))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Datasets").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<DatasetMetadata> builder(MarshallingType.SDK_POJO)
                                            .constructor(DatasetMetadata::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> FEATURE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Feature")
            .getter(getter(ProjectDescription::featureAsString)).setter(setter(Builder::feature))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Feature").build()).build();

    private static final SdkField<String> AUTO_UPDATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AutoUpdate").getter(getter(ProjectDescription::autoUpdateAsString)).setter(setter(Builder::autoUpdate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AutoUpdate").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PROJECT_ARN_FIELD,
            CREATION_TIMESTAMP_FIELD, STATUS_FIELD, DATASETS_FIELD, FEATURE_FIELD, AUTO_UPDATE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String projectArn;

    private final Instant creationTimestamp;

    private final String status;

    private final List<DatasetMetadata> datasets;

    private final String feature;

    private final String autoUpdate;

    private ProjectDescription(BuilderImpl builder) {
        this.projectArn = builder.projectArn;
        this.creationTimestamp = builder.creationTimestamp;
        this.status = builder.status;
        this.datasets = builder.datasets;
        this.feature = builder.feature;
        this.autoUpdate = builder.autoUpdate;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the project.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the project.
     */
    public final String projectArn() {
        return projectArn;
    }

    /**
     * <p>
     * The Unix timestamp for the date and time that the project was created.
     * </p>
     * 
     * @return The Unix timestamp for the date and time that the project was created.
     */
    public final Instant creationTimestamp() {
        return creationTimestamp;
    }

    /**
     * <p>
     * The current status of the project.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ProjectStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the project.
     * @see ProjectStatus
     */
    public final ProjectStatus status() {
        return ProjectStatus.fromValue(status);
    }

    /**
     * <p>
     * The current status of the project.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ProjectStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the project.
     * @see ProjectStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * For responses, this returns true if the service returned a value for the Datasets property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasDatasets() {
        return datasets != null && !(datasets instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Information about the training and test datasets in the project.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDatasets} method.
     * </p>
     * 
     * @return Information about the training and test datasets in the project.
     */
    public final List<DatasetMetadata> datasets() {
        return datasets;
    }

    /**
     * <p>
     * Specifies the project that is being customized.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #feature} will
     * return {@link CustomizationFeature#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #featureAsString}.
     * </p>
     * 
     * @return Specifies the project that is being customized.
     * @see CustomizationFeature
     */
    public final CustomizationFeature feature() {
        return CustomizationFeature.fromValue(feature);
    }

    /**
     * <p>
     * Specifies the project that is being customized.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #feature} will
     * return {@link CustomizationFeature#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #featureAsString}.
     * </p>
     * 
     * @return Specifies the project that is being customized.
     * @see CustomizationFeature
     */
    public final String featureAsString() {
        return feature;
    }

    /**
     * <p>
     * Indicates whether automatic retraining will be attempted for the versions of the project. Applies only to
     * adapters.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #autoUpdate} will
     * return {@link ProjectAutoUpdate#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #autoUpdateAsString}.
     * </p>
     * 
     * @return Indicates whether automatic retraining will be attempted for the versions of the project. Applies only to
     *         adapters.
     * @see ProjectAutoUpdate
     */
    public final ProjectAutoUpdate autoUpdate() {
        return ProjectAutoUpdate.fromValue(autoUpdate);
    }

    /**
     * <p>
     * Indicates whether automatic retraining will be attempted for the versions of the project. Applies only to
     * adapters.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #autoUpdate} will
     * return {@link ProjectAutoUpdate#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #autoUpdateAsString}.
     * </p>
     * 
     * @return Indicates whether automatic retraining will be attempted for the versions of the project. Applies only to
     *         adapters.
     * @see ProjectAutoUpdate
     */
    public final String autoUpdateAsString() {
        return autoUpdate;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(projectArn());
        hashCode = 31 * hashCode + Objects.hashCode(creationTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasDatasets() ? datasets() : null);
        hashCode = 31 * hashCode + Objects.hashCode(featureAsString());
        hashCode = 31 * hashCode + Objects.hashCode(autoUpdateAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ProjectDescription)) {
            return false;
        }
        ProjectDescription other = (ProjectDescription) obj;
        return Objects.equals(projectArn(), other.projectArn()) && Objects.equals(creationTimestamp(), other.creationTimestamp())
                && Objects.equals(statusAsString(), other.statusAsString()) && hasDatasets() == other.hasDatasets()
                && Objects.equals(datasets(), other.datasets()) && Objects.equals(featureAsString(), other.featureAsString())
                && Objects.equals(autoUpdateAsString(), other.autoUpdateAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ProjectDescription").add("ProjectArn", projectArn())
                .add("CreationTimestamp", creationTimestamp()).add("Status", statusAsString())
                .add("Datasets", hasDatasets() ? datasets() : null).add("Feature", featureAsString())
                .add("AutoUpdate", autoUpdateAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ProjectArn":
            return Optional.ofNullable(clazz.cast(projectArn()));
        case "CreationTimestamp":
            return Optional.ofNullable(clazz.cast(creationTimestamp()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "Datasets":
            return Optional.ofNullable(clazz.cast(datasets()));
        case "Feature":
            return Optional.ofNullable(clazz.cast(featureAsString()));
        case "AutoUpdate":
            return Optional.ofNullable(clazz.cast(autoUpdateAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ProjectDescription, T> g) {
        return obj -> g.apply((ProjectDescription) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ProjectDescription> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the project.
         * </p>
         * 
         * @param projectArn
         *        The Amazon Resource Name (ARN) of the project.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder projectArn(String projectArn);

        /**
         * <p>
         * The Unix timestamp for the date and time that the project was created.
         * </p>
         * 
         * @param creationTimestamp
         *        The Unix timestamp for the date and time that the project was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTimestamp(Instant creationTimestamp);

        /**
         * <p>
         * The current status of the project.
         * </p>
         * 
         * @param status
         *        The current status of the project.
         * @see ProjectStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProjectStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The current status of the project.
         * </p>
         * 
         * @param status
         *        The current status of the project.
         * @see ProjectStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProjectStatus
         */
        Builder status(ProjectStatus status);

        /**
         * <p>
         * Information about the training and test datasets in the project.
         * </p>
         * 
         * @param datasets
         *        Information about the training and test datasets in the project.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder datasets(Collection<DatasetMetadata> datasets);

        /**
         * <p>
         * Information about the training and test datasets in the project.
         * </p>
         * 
         * @param datasets
         *        Information about the training and test datasets in the project.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder datasets(DatasetMetadata... datasets);

        /**
         * <p>
         * Information about the training and test datasets in the project.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.rekognition.model.DatasetMetadata.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.rekognition.model.DatasetMetadata#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.rekognition.model.DatasetMetadata.Builder#build()} is called
         * immediately and its result is passed to {@link #datasets(List<DatasetMetadata>)}.
         * 
         * @param datasets
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.rekognition.model.DatasetMetadata.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #datasets(java.util.Collection<DatasetMetadata>)
         */
        Builder datasets(Consumer<DatasetMetadata.Builder>... datasets);

        /**
         * <p>
         * Specifies the project that is being customized.
         * </p>
         * 
         * @param feature
         *        Specifies the project that is being customized.
         * @see CustomizationFeature
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CustomizationFeature
         */
        Builder feature(String feature);

        /**
         * <p>
         * Specifies the project that is being customized.
         * </p>
         * 
         * @param feature
         *        Specifies the project that is being customized.
         * @see CustomizationFeature
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CustomizationFeature
         */
        Builder feature(CustomizationFeature feature);

        /**
         * <p>
         * Indicates whether automatic retraining will be attempted for the versions of the project. Applies only to
         * adapters.
         * </p>
         * 
         * @param autoUpdate
         *        Indicates whether automatic retraining will be attempted for the versions of the project. Applies only
         *        to adapters.
         * @see ProjectAutoUpdate
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProjectAutoUpdate
         */
        Builder autoUpdate(String autoUpdate);

        /**
         * <p>
         * Indicates whether automatic retraining will be attempted for the versions of the project. Applies only to
         * adapters.
         * </p>
         * 
         * @param autoUpdate
         *        Indicates whether automatic retraining will be attempted for the versions of the project. Applies only
         *        to adapters.
         * @see ProjectAutoUpdate
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProjectAutoUpdate
         */
        Builder autoUpdate(ProjectAutoUpdate autoUpdate);
    }

    static final class BuilderImpl implements Builder {
        private String projectArn;

        private Instant creationTimestamp;

        private String status;

        private List<DatasetMetadata> datasets = DefaultSdkAutoConstructList.getInstance();

        private String feature;

        private String autoUpdate;

        private BuilderImpl() {
        }

        private BuilderImpl(ProjectDescription model) {
            projectArn(model.projectArn);
            creationTimestamp(model.creationTimestamp);
            status(model.status);
            datasets(model.datasets);
            feature(model.feature);
            autoUpdate(model.autoUpdate);
        }

        public final String getProjectArn() {
            return projectArn;
        }

        public final void setProjectArn(String projectArn) {
            this.projectArn = projectArn;
        }

        @Override
        public final Builder projectArn(String projectArn) {
            this.projectArn = projectArn;
            return this;
        }

        public final Instant getCreationTimestamp() {
            return creationTimestamp;
        }

        public final void setCreationTimestamp(Instant creationTimestamp) {
            this.creationTimestamp = creationTimestamp;
        }

        @Override
        public final Builder creationTimestamp(Instant creationTimestamp) {
            this.creationTimestamp = creationTimestamp;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(ProjectStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final List<DatasetMetadata.Builder> getDatasets() {
            List<DatasetMetadata.Builder> result = DatasetMetadataListCopier.copyToBuilder(this.datasets);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setDatasets(Collection<DatasetMetadata.BuilderImpl> datasets) {
            this.datasets = DatasetMetadataListCopier.copyFromBuilder(datasets);
        }

        @Override
        public final Builder datasets(Collection<DatasetMetadata> datasets) {
            this.datasets = DatasetMetadataListCopier.copy(datasets);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder datasets(DatasetMetadata... datasets) {
            datasets(Arrays.asList(datasets));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder datasets(Consumer<DatasetMetadata.Builder>... datasets) {
            datasets(Stream.of(datasets).map(c -> DatasetMetadata.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getFeature() {
            return feature;
        }

        public final void setFeature(String feature) {
            this.feature = feature;
        }

        @Override
        public final Builder feature(String feature) {
            this.feature = feature;
            return this;
        }

        @Override
        public final Builder feature(CustomizationFeature feature) {
            this.feature(feature == null ? null : feature.toString());
            return this;
        }

        public final String getAutoUpdate() {
            return autoUpdate;
        }

        public final void setAutoUpdate(String autoUpdate) {
            this.autoUpdate = autoUpdate;
        }

        @Override
        public final Builder autoUpdate(String autoUpdate) {
            this.autoUpdate = autoUpdate;
            return this;
        }

        @Override
        public final Builder autoUpdate(ProjectAutoUpdate autoUpdate) {
            this.autoUpdate(autoUpdate == null ? null : autoUpdate.toString());
            return this;
        }

        @Override
        public ProjectDescription build() {
            return new ProjectDescription(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
