/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3tables;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.s3tables.internal.S3TablesServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.s3tables.model.AccessDeniedException;
import software.amazon.awssdk.services.s3tables.model.BadRequestException;
import software.amazon.awssdk.services.s3tables.model.ConflictException;
import software.amazon.awssdk.services.s3tables.model.CreateNamespaceRequest;
import software.amazon.awssdk.services.s3tables.model.CreateNamespaceResponse;
import software.amazon.awssdk.services.s3tables.model.CreateTableBucketRequest;
import software.amazon.awssdk.services.s3tables.model.CreateTableBucketResponse;
import software.amazon.awssdk.services.s3tables.model.CreateTableRequest;
import software.amazon.awssdk.services.s3tables.model.CreateTableResponse;
import software.amazon.awssdk.services.s3tables.model.DeleteNamespaceRequest;
import software.amazon.awssdk.services.s3tables.model.DeleteNamespaceResponse;
import software.amazon.awssdk.services.s3tables.model.DeleteTableBucketPolicyRequest;
import software.amazon.awssdk.services.s3tables.model.DeleteTableBucketPolicyResponse;
import software.amazon.awssdk.services.s3tables.model.DeleteTableBucketRequest;
import software.amazon.awssdk.services.s3tables.model.DeleteTableBucketResponse;
import software.amazon.awssdk.services.s3tables.model.DeleteTablePolicyRequest;
import software.amazon.awssdk.services.s3tables.model.DeleteTablePolicyResponse;
import software.amazon.awssdk.services.s3tables.model.DeleteTableRequest;
import software.amazon.awssdk.services.s3tables.model.DeleteTableResponse;
import software.amazon.awssdk.services.s3tables.model.ForbiddenException;
import software.amazon.awssdk.services.s3tables.model.GetNamespaceRequest;
import software.amazon.awssdk.services.s3tables.model.GetNamespaceResponse;
import software.amazon.awssdk.services.s3tables.model.GetTableBucketMaintenanceConfigurationRequest;
import software.amazon.awssdk.services.s3tables.model.GetTableBucketMaintenanceConfigurationResponse;
import software.amazon.awssdk.services.s3tables.model.GetTableBucketPolicyRequest;
import software.amazon.awssdk.services.s3tables.model.GetTableBucketPolicyResponse;
import software.amazon.awssdk.services.s3tables.model.GetTableBucketRequest;
import software.amazon.awssdk.services.s3tables.model.GetTableBucketResponse;
import software.amazon.awssdk.services.s3tables.model.GetTableMaintenanceConfigurationRequest;
import software.amazon.awssdk.services.s3tables.model.GetTableMaintenanceConfigurationResponse;
import software.amazon.awssdk.services.s3tables.model.GetTableMaintenanceJobStatusRequest;
import software.amazon.awssdk.services.s3tables.model.GetTableMaintenanceJobStatusResponse;
import software.amazon.awssdk.services.s3tables.model.GetTableMetadataLocationRequest;
import software.amazon.awssdk.services.s3tables.model.GetTableMetadataLocationResponse;
import software.amazon.awssdk.services.s3tables.model.GetTablePolicyRequest;
import software.amazon.awssdk.services.s3tables.model.GetTablePolicyResponse;
import software.amazon.awssdk.services.s3tables.model.GetTableRequest;
import software.amazon.awssdk.services.s3tables.model.GetTableResponse;
import software.amazon.awssdk.services.s3tables.model.InternalServerErrorException;
import software.amazon.awssdk.services.s3tables.model.ListNamespacesRequest;
import software.amazon.awssdk.services.s3tables.model.ListNamespacesResponse;
import software.amazon.awssdk.services.s3tables.model.ListTableBucketsRequest;
import software.amazon.awssdk.services.s3tables.model.ListTableBucketsResponse;
import software.amazon.awssdk.services.s3tables.model.ListTablesRequest;
import software.amazon.awssdk.services.s3tables.model.ListTablesResponse;
import software.amazon.awssdk.services.s3tables.model.NotFoundException;
import software.amazon.awssdk.services.s3tables.model.PutTableBucketMaintenanceConfigurationRequest;
import software.amazon.awssdk.services.s3tables.model.PutTableBucketMaintenanceConfigurationResponse;
import software.amazon.awssdk.services.s3tables.model.PutTableBucketPolicyRequest;
import software.amazon.awssdk.services.s3tables.model.PutTableBucketPolicyResponse;
import software.amazon.awssdk.services.s3tables.model.PutTableMaintenanceConfigurationRequest;
import software.amazon.awssdk.services.s3tables.model.PutTableMaintenanceConfigurationResponse;
import software.amazon.awssdk.services.s3tables.model.PutTablePolicyRequest;
import software.amazon.awssdk.services.s3tables.model.PutTablePolicyResponse;
import software.amazon.awssdk.services.s3tables.model.RenameTableRequest;
import software.amazon.awssdk.services.s3tables.model.RenameTableResponse;
import software.amazon.awssdk.services.s3tables.model.S3TablesException;
import software.amazon.awssdk.services.s3tables.model.TooManyRequestsException;
import software.amazon.awssdk.services.s3tables.model.UpdateTableMetadataLocationRequest;
import software.amazon.awssdk.services.s3tables.model.UpdateTableMetadataLocationResponse;
import software.amazon.awssdk.services.s3tables.transform.CreateNamespaceRequestMarshaller;
import software.amazon.awssdk.services.s3tables.transform.CreateTableBucketRequestMarshaller;
import software.amazon.awssdk.services.s3tables.transform.CreateTableRequestMarshaller;
import software.amazon.awssdk.services.s3tables.transform.DeleteNamespaceRequestMarshaller;
import software.amazon.awssdk.services.s3tables.transform.DeleteTableBucketPolicyRequestMarshaller;
import software.amazon.awssdk.services.s3tables.transform.DeleteTableBucketRequestMarshaller;
import software.amazon.awssdk.services.s3tables.transform.DeleteTablePolicyRequestMarshaller;
import software.amazon.awssdk.services.s3tables.transform.DeleteTableRequestMarshaller;
import software.amazon.awssdk.services.s3tables.transform.GetNamespaceRequestMarshaller;
import software.amazon.awssdk.services.s3tables.transform.GetTableBucketMaintenanceConfigurationRequestMarshaller;
import software.amazon.awssdk.services.s3tables.transform.GetTableBucketPolicyRequestMarshaller;
import software.amazon.awssdk.services.s3tables.transform.GetTableBucketRequestMarshaller;
import software.amazon.awssdk.services.s3tables.transform.GetTableMaintenanceConfigurationRequestMarshaller;
import software.amazon.awssdk.services.s3tables.transform.GetTableMaintenanceJobStatusRequestMarshaller;
import software.amazon.awssdk.services.s3tables.transform.GetTableMetadataLocationRequestMarshaller;
import software.amazon.awssdk.services.s3tables.transform.GetTablePolicyRequestMarshaller;
import software.amazon.awssdk.services.s3tables.transform.GetTableRequestMarshaller;
import software.amazon.awssdk.services.s3tables.transform.ListNamespacesRequestMarshaller;
import software.amazon.awssdk.services.s3tables.transform.ListTableBucketsRequestMarshaller;
import software.amazon.awssdk.services.s3tables.transform.ListTablesRequestMarshaller;
import software.amazon.awssdk.services.s3tables.transform.PutTableBucketMaintenanceConfigurationRequestMarshaller;
import software.amazon.awssdk.services.s3tables.transform.PutTableBucketPolicyRequestMarshaller;
import software.amazon.awssdk.services.s3tables.transform.PutTableMaintenanceConfigurationRequestMarshaller;
import software.amazon.awssdk.services.s3tables.transform.PutTablePolicyRequestMarshaller;
import software.amazon.awssdk.services.s3tables.transform.RenameTableRequestMarshaller;
import software.amazon.awssdk.services.s3tables.transform.UpdateTableMetadataLocationRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link S3TablesAsyncClient}.
 *
 * @see S3TablesAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultS3TablesAsyncClient implements S3TablesAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultS3TablesAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultS3TablesAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a namespace. A namespace is a logical grouping of tables within your table bucket, which you can use to
     * organize tables. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-tables-namespace-create.html">Create a
     * namespace</a> in the <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3tables:CreateNamespace</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     *
     * @param createNamespaceRequest
     * @return A Java Future containing the result of the CreateNamespace operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerErrorException The request failed due to an internal server error.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request.</li>
     *         <li>NotFoundException The request was rejected because the specified resource could not be found.</li>
     *         <li>TooManyRequestsException The limit on the number of requests per second was exceeded.</li>
     *         <li>ConflictException The request failed because there is a conflict with a previous write. You can retry
     *         the request.</li>
     *         <li>BadRequestException The request is invalid or malformed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3TablesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3TablesAsyncClient.CreateNamespace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3tables-2018-05-10/CreateNamespace" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateNamespaceResponse> createNamespace(CreateNamespaceRequest createNamespaceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createNamespaceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createNamespaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Tables");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateNamespace");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateNamespaceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateNamespaceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateNamespaceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateNamespaceRequest, CreateNamespaceResponse>()
                            .withOperationName("CreateNamespace").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateNamespaceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createNamespaceRequest));
            CompletableFuture<CreateNamespaceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new table associated with the given namespace in a table bucket. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-tables-create.html">Creating an Amazon S3
     * table</a> in the <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3tables:CreateTable</code> permission to use this operation.
     * </p>
     * <note>
     * <p>
     * Additionally, you must have the <code>s3tables:PutTableData</code> permission to use this operation with the
     * optional <code>metadata</code> request parameter.
     * </p>
     * </note></dd>
     * </dl>
     *
     * @param createTableRequest
     * @return A Java Future containing the result of the CreateTable operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerErrorException The request failed due to an internal server error.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request.</li>
     *         <li>NotFoundException The request was rejected because the specified resource could not be found.</li>
     *         <li>TooManyRequestsException The limit on the number of requests per second was exceeded.</li>
     *         <li>ConflictException The request failed because there is a conflict with a previous write. You can retry
     *         the request.</li>
     *         <li>BadRequestException The request is invalid or malformed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3TablesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3TablesAsyncClient.CreateTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3tables-2018-05-10/CreateTable" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateTableResponse> createTable(CreateTableRequest createTableRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createTableRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createTableRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Tables");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateTable");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateTableResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateTableResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateTableResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateTableRequest, CreateTableResponse>()
                            .withOperationName("CreateTable").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateTableRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createTableRequest));
            CompletableFuture<CreateTableResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a table bucket. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-tables-buckets-create.html">Creating a table
     * bucket</a> in the <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3tables:CreateTableBucket</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     *
     * @param createTableBucketRequest
     * @return A Java Future containing the result of the CreateTableBucket operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerErrorException The request failed due to an internal server error.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request.</li>
     *         <li>NotFoundException The request was rejected because the specified resource could not be found.</li>
     *         <li>TooManyRequestsException The limit on the number of requests per second was exceeded.</li>
     *         <li>ConflictException The request failed because there is a conflict with a previous write. You can retry
     *         the request.</li>
     *         <li>BadRequestException The request is invalid or malformed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3TablesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3TablesAsyncClient.CreateTableBucket
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3tables-2018-05-10/CreateTableBucket" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateTableBucketResponse> createTableBucket(CreateTableBucketRequest createTableBucketRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createTableBucketRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createTableBucketRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Tables");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateTableBucket");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateTableBucketResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateTableBucketResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateTableBucketResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateTableBucketRequest, CreateTableBucketResponse>()
                            .withOperationName("CreateTableBucket").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateTableBucketRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createTableBucketRequest));
            CompletableFuture<CreateTableBucketResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a namespace. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-tables-namespace-delete.html">Delete a
     * namespace</a> in the <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3tables:DeleteNamespace</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     *
     * @param deleteNamespaceRequest
     * @return A Java Future containing the result of the DeleteNamespace operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerErrorException The request failed due to an internal server error.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request.</li>
     *         <li>NotFoundException The request was rejected because the specified resource could not be found.</li>
     *         <li>TooManyRequestsException The limit on the number of requests per second was exceeded.</li>
     *         <li>ConflictException The request failed because there is a conflict with a previous write. You can retry
     *         the request.</li>
     *         <li>BadRequestException The request is invalid or malformed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3TablesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3TablesAsyncClient.DeleteNamespace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3tables-2018-05-10/DeleteNamespace" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteNamespaceResponse> deleteNamespace(DeleteNamespaceRequest deleteNamespaceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteNamespaceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteNamespaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Tables");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteNamespace");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteNamespaceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteNamespaceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteNamespaceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteNamespaceRequest, DeleteNamespaceResponse>()
                            .withOperationName("DeleteNamespace").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteNamespaceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteNamespaceRequest));
            CompletableFuture<DeleteNamespaceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a table. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-tables-delete.html">Deleting an Amazon S3
     * table</a> in the <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3tables:DeleteTable</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     *
     * @param deleteTableRequest
     * @return A Java Future containing the result of the DeleteTable operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerErrorException The request failed due to an internal server error.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request.</li>
     *         <li>NotFoundException The request was rejected because the specified resource could not be found.</li>
     *         <li>TooManyRequestsException The limit on the number of requests per second was exceeded.</li>
     *         <li>ConflictException The request failed because there is a conflict with a previous write. You can retry
     *         the request.</li>
     *         <li>BadRequestException The request is invalid or malformed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3TablesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3TablesAsyncClient.DeleteTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3tables-2018-05-10/DeleteTable" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteTableResponse> deleteTable(DeleteTableRequest deleteTableRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteTableRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTableRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Tables");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTable");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteTableResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteTableResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteTableResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteTableRequest, DeleteTableResponse>()
                            .withOperationName("DeleteTable").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteTableRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteTableRequest));
            CompletableFuture<DeleteTableResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a table bucket. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-tables-buckets-delete.html">Deleting a table
     * bucket</a> in the <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3tables:DeleteTableBucket</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     *
     * @param deleteTableBucketRequest
     * @return A Java Future containing the result of the DeleteTableBucket operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerErrorException The request failed due to an internal server error.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request.</li>
     *         <li>NotFoundException The request was rejected because the specified resource could not be found.</li>
     *         <li>TooManyRequestsException The limit on the number of requests per second was exceeded.</li>
     *         <li>ConflictException The request failed because there is a conflict with a previous write. You can retry
     *         the request.</li>
     *         <li>BadRequestException The request is invalid or malformed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3TablesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3TablesAsyncClient.DeleteTableBucket
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3tables-2018-05-10/DeleteTableBucket" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteTableBucketResponse> deleteTableBucket(DeleteTableBucketRequest deleteTableBucketRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteTableBucketRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTableBucketRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Tables");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTableBucket");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteTableBucketResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteTableBucketResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteTableBucketResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteTableBucketRequest, DeleteTableBucketResponse>()
                            .withOperationName("DeleteTableBucket").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteTableBucketRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteTableBucketRequest));
            CompletableFuture<DeleteTableBucketResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a table bucket policy. For more information, see <a href=
     * "https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-tables-bucket-policy.html#table-bucket-policy-delete"
     * >Deleting a table bucket policy</a> in the <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3tables:DeleteTableBucketPolicy</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     *
     * @param deleteTableBucketPolicyRequest
     * @return A Java Future containing the result of the DeleteTableBucketPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerErrorException The request failed due to an internal server error.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request.</li>
     *         <li>NotFoundException The request was rejected because the specified resource could not be found.</li>
     *         <li>TooManyRequestsException The limit on the number of requests per second was exceeded.</li>
     *         <li>ConflictException The request failed because there is a conflict with a previous write. You can retry
     *         the request.</li>
     *         <li>BadRequestException The request is invalid or malformed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3TablesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3TablesAsyncClient.DeleteTableBucketPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3tables-2018-05-10/DeleteTableBucketPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteTableBucketPolicyResponse> deleteTableBucketPolicy(
            DeleteTableBucketPolicyRequest deleteTableBucketPolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteTableBucketPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTableBucketPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Tables");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTableBucketPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteTableBucketPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteTableBucketPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteTableBucketPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteTableBucketPolicyRequest, DeleteTableBucketPolicyResponse>()
                            .withOperationName("DeleteTableBucketPolicy").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteTableBucketPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteTableBucketPolicyRequest));
            CompletableFuture<DeleteTableBucketPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a table policy. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-tables-table-policy.html#table-policy-delete"
     * >Deleting a table policy</a> in the <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3tables:DeleteTablePolicy</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     *
     * @param deleteTablePolicyRequest
     * @return A Java Future containing the result of the DeleteTablePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerErrorException The request failed due to an internal server error.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request.</li>
     *         <li>NotFoundException The request was rejected because the specified resource could not be found.</li>
     *         <li>TooManyRequestsException The limit on the number of requests per second was exceeded.</li>
     *         <li>ConflictException The request failed because there is a conflict with a previous write. You can retry
     *         the request.</li>
     *         <li>BadRequestException The request is invalid or malformed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3TablesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3TablesAsyncClient.DeleteTablePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3tables-2018-05-10/DeleteTablePolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteTablePolicyResponse> deleteTablePolicy(DeleteTablePolicyRequest deleteTablePolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteTablePolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTablePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Tables");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTablePolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteTablePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteTablePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteTablePolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteTablePolicyRequest, DeleteTablePolicyResponse>()
                            .withOperationName("DeleteTablePolicy").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteTablePolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteTablePolicyRequest));
            CompletableFuture<DeleteTablePolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets details about a namespace. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-tables-namespace.html">Table namespaces</a> in the
     * <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3tables:GetNamespace</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     *
     * @param getNamespaceRequest
     * @return A Java Future containing the result of the GetNamespace operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerErrorException The request failed due to an internal server error.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request.</li>
     *         <li>NotFoundException The request was rejected because the specified resource could not be found.</li>
     *         <li>AccessDeniedException The action cannot be performed because you do not have the required permission.
     *         </li>
     *         <li>TooManyRequestsException The limit on the number of requests per second was exceeded.</li>
     *         <li>ConflictException The request failed because there is a conflict with a previous write. You can retry
     *         the request.</li>
     *         <li>BadRequestException The request is invalid or malformed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3TablesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3TablesAsyncClient.GetNamespace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3tables-2018-05-10/GetNamespace" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetNamespaceResponse> getNamespace(GetNamespaceRequest getNamespaceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getNamespaceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getNamespaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Tables");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetNamespace");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetNamespaceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetNamespaceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetNamespaceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetNamespaceRequest, GetNamespaceResponse>()
                            .withOperationName("GetNamespace").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetNamespaceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getNamespaceRequest));
            CompletableFuture<GetNamespaceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets details about a table. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-tables-tables.html">S3 Tables</a> in the <i>Amazon
     * Simple Storage Service User Guide</i>.
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3tables:GetTable</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     *
     * @param getTableRequest
     * @return A Java Future containing the result of the GetTable operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerErrorException The request failed due to an internal server error.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request.</li>
     *         <li>NotFoundException The request was rejected because the specified resource could not be found.</li>
     *         <li>AccessDeniedException The action cannot be performed because you do not have the required permission.
     *         </li>
     *         <li>TooManyRequestsException The limit on the number of requests per second was exceeded.</li>
     *         <li>ConflictException The request failed because there is a conflict with a previous write. You can retry
     *         the request.</li>
     *         <li>BadRequestException The request is invalid or malformed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3TablesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3TablesAsyncClient.GetTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3tables-2018-05-10/GetTable" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetTableResponse> getTable(GetTableRequest getTableRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTableRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTableRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Tables");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTable");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTableResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetTableResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetTableResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTableRequest, GetTableResponse>().withOperationName("GetTable")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetTableRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(getTableRequest));
            CompletableFuture<GetTableResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets details on a table bucket. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-tables-buckets-details.html">Viewing details about
     * an Amazon S3 table bucket</a> in the <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3tables:GetTableBucket</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     *
     * @param getTableBucketRequest
     * @return A Java Future containing the result of the GetTableBucket operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerErrorException The request failed due to an internal server error.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request.</li>
     *         <li>NotFoundException The request was rejected because the specified resource could not be found.</li>
     *         <li>AccessDeniedException The action cannot be performed because you do not have the required permission.
     *         </li>
     *         <li>TooManyRequestsException The limit on the number of requests per second was exceeded.</li>
     *         <li>ConflictException The request failed because there is a conflict with a previous write. You can retry
     *         the request.</li>
     *         <li>BadRequestException The request is invalid or malformed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3TablesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3TablesAsyncClient.GetTableBucket
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3tables-2018-05-10/GetTableBucket" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetTableBucketResponse> getTableBucket(GetTableBucketRequest getTableBucketRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTableBucketRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTableBucketRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Tables");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTableBucket");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTableBucketResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetTableBucketResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetTableBucketResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTableBucketRequest, GetTableBucketResponse>()
                            .withOperationName("GetTableBucket").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetTableBucketRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getTableBucketRequest));
            CompletableFuture<GetTableBucketResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets details about a maintenance configuration for a given table bucket. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-table-buckets-maintenance.html">Amazon S3 table
     * bucket maintenance</a> in the <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3tables:GetTableBucketMaintenanceConfiguration</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     *
     * @param getTableBucketMaintenanceConfigurationRequest
     * @return A Java Future containing the result of the GetTableBucketMaintenanceConfiguration operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerErrorException The request failed due to an internal server error.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request.</li>
     *         <li>NotFoundException The request was rejected because the specified resource could not be found.</li>
     *         <li>TooManyRequestsException The limit on the number of requests per second was exceeded.</li>
     *         <li>ConflictException The request failed because there is a conflict with a previous write. You can retry
     *         the request.</li>
     *         <li>BadRequestException The request is invalid or malformed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3TablesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3TablesAsyncClient.GetTableBucketMaintenanceConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3tables-2018-05-10/GetTableBucketMaintenanceConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetTableBucketMaintenanceConfigurationResponse> getTableBucketMaintenanceConfiguration(
            GetTableBucketMaintenanceConfigurationRequest getTableBucketMaintenanceConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTableBucketMaintenanceConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getTableBucketMaintenanceConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Tables");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTableBucketMaintenanceConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTableBucketMaintenanceConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetTableBucketMaintenanceConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetTableBucketMaintenanceConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTableBucketMaintenanceConfigurationRequest, GetTableBucketMaintenanceConfigurationResponse>()
                            .withOperationName("GetTableBucketMaintenanceConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetTableBucketMaintenanceConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getTableBucketMaintenanceConfigurationRequest));
            CompletableFuture<GetTableBucketMaintenanceConfigurationResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets details about a table bucket policy. For more information, see <a href=
     * "https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-tables-bucket-policy.html#table-bucket-policy-get"
     * >Viewing a table bucket policy</a> in the <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3tables:GetTableBucketPolicy</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     *
     * @param getTableBucketPolicyRequest
     * @return A Java Future containing the result of the GetTableBucketPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerErrorException The request failed due to an internal server error.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request.</li>
     *         <li>NotFoundException The request was rejected because the specified resource could not be found.</li>
     *         <li>TooManyRequestsException The limit on the number of requests per second was exceeded.</li>
     *         <li>ConflictException The request failed because there is a conflict with a previous write. You can retry
     *         the request.</li>
     *         <li>BadRequestException The request is invalid or malformed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3TablesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3TablesAsyncClient.GetTableBucketPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3tables-2018-05-10/GetTableBucketPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetTableBucketPolicyResponse> getTableBucketPolicy(
            GetTableBucketPolicyRequest getTableBucketPolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTableBucketPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTableBucketPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Tables");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTableBucketPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTableBucketPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetTableBucketPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetTableBucketPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTableBucketPolicyRequest, GetTableBucketPolicyResponse>()
                            .withOperationName("GetTableBucketPolicy").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetTableBucketPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getTableBucketPolicyRequest));
            CompletableFuture<GetTableBucketPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets details about the maintenance configuration of a table. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-tables-maintenance.html">S3 Tables maintenance</a>
     * in the <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3tables:GetTableMaintenanceConfiguration</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     *
     * @param getTableMaintenanceConfigurationRequest
     * @return A Java Future containing the result of the GetTableMaintenanceConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerErrorException The request failed due to an internal server error.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request.</li>
     *         <li>NotFoundException The request was rejected because the specified resource could not be found.</li>
     *         <li>TooManyRequestsException The limit on the number of requests per second was exceeded.</li>
     *         <li>ConflictException The request failed because there is a conflict with a previous write. You can retry
     *         the request.</li>
     *         <li>BadRequestException The request is invalid or malformed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3TablesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3TablesAsyncClient.GetTableMaintenanceConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3tables-2018-05-10/GetTableMaintenanceConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetTableMaintenanceConfigurationResponse> getTableMaintenanceConfiguration(
            GetTableMaintenanceConfigurationRequest getTableMaintenanceConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTableMaintenanceConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getTableMaintenanceConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Tables");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTableMaintenanceConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTableMaintenanceConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetTableMaintenanceConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetTableMaintenanceConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTableMaintenanceConfigurationRequest, GetTableMaintenanceConfigurationResponse>()
                            .withOperationName("GetTableMaintenanceConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetTableMaintenanceConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getTableMaintenanceConfigurationRequest));
            CompletableFuture<GetTableMaintenanceConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the status of a maintenance job for a table. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-tables-maintenance.html">S3 Tables maintenance</a>
     * in the <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3tables:GetTableMaintenanceJobStatus</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     *
     * @param getTableMaintenanceJobStatusRequest
     * @return A Java Future containing the result of the GetTableMaintenanceJobStatus operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerErrorException The request failed due to an internal server error.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request.</li>
     *         <li>NotFoundException The request was rejected because the specified resource could not be found.</li>
     *         <li>TooManyRequestsException The limit on the number of requests per second was exceeded.</li>
     *         <li>ConflictException The request failed because there is a conflict with a previous write. You can retry
     *         the request.</li>
     *         <li>BadRequestException The request is invalid or malformed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3TablesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3TablesAsyncClient.GetTableMaintenanceJobStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3tables-2018-05-10/GetTableMaintenanceJobStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetTableMaintenanceJobStatusResponse> getTableMaintenanceJobStatus(
            GetTableMaintenanceJobStatusRequest getTableMaintenanceJobStatusRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTableMaintenanceJobStatusRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTableMaintenanceJobStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Tables");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTableMaintenanceJobStatus");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTableMaintenanceJobStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetTableMaintenanceJobStatusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetTableMaintenanceJobStatusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTableMaintenanceJobStatusRequest, GetTableMaintenanceJobStatusResponse>()
                            .withOperationName("GetTableMaintenanceJobStatus").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetTableMaintenanceJobStatusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getTableMaintenanceJobStatusRequest));
            CompletableFuture<GetTableMaintenanceJobStatusResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the location of the table metadata.
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3tables:GetTableMetadataLocation</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     *
     * @param getTableMetadataLocationRequest
     * @return A Java Future containing the result of the GetTableMetadataLocation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerErrorException The request failed due to an internal server error.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request.</li>
     *         <li>NotFoundException The request was rejected because the specified resource could not be found.</li>
     *         <li>TooManyRequestsException The limit on the number of requests per second was exceeded.</li>
     *         <li>ConflictException The request failed because there is a conflict with a previous write. You can retry
     *         the request.</li>
     *         <li>BadRequestException The request is invalid or malformed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3TablesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3TablesAsyncClient.GetTableMetadataLocation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3tables-2018-05-10/GetTableMetadataLocation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetTableMetadataLocationResponse> getTableMetadataLocation(
            GetTableMetadataLocationRequest getTableMetadataLocationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTableMetadataLocationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTableMetadataLocationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Tables");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTableMetadataLocation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTableMetadataLocationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetTableMetadataLocationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetTableMetadataLocationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTableMetadataLocationRequest, GetTableMetadataLocationResponse>()
                            .withOperationName("GetTableMetadataLocation").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetTableMetadataLocationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getTableMetadataLocationRequest));
            CompletableFuture<GetTableMetadataLocationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets details about a table policy. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-tables-table-policy.html#table-policy-get">Viewing
     * a table policy</a> in the <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3tables:GetTablePolicy</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     *
     * @param getTablePolicyRequest
     * @return A Java Future containing the result of the GetTablePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerErrorException The request failed due to an internal server error.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request.</li>
     *         <li>NotFoundException The request was rejected because the specified resource could not be found.</li>
     *         <li>TooManyRequestsException The limit on the number of requests per second was exceeded.</li>
     *         <li>ConflictException The request failed because there is a conflict with a previous write. You can retry
     *         the request.</li>
     *         <li>BadRequestException The request is invalid or malformed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3TablesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3TablesAsyncClient.GetTablePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3tables-2018-05-10/GetTablePolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetTablePolicyResponse> getTablePolicy(GetTablePolicyRequest getTablePolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTablePolicyRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTablePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Tables");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTablePolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTablePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetTablePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetTablePolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTablePolicyRequest, GetTablePolicyResponse>()
                            .withOperationName("GetTablePolicy").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetTablePolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getTablePolicyRequest));
            CompletableFuture<GetTablePolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the namespaces within a table bucket. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-tables-namespace.html">Table namespaces</a> in the
     * <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3tables:ListNamespaces</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     *
     * @param listNamespacesRequest
     * @return A Java Future containing the result of the ListNamespaces operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerErrorException The request failed due to an internal server error.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request.</li>
     *         <li>NotFoundException The request was rejected because the specified resource could not be found.</li>
     *         <li>AccessDeniedException The action cannot be performed because you do not have the required permission.
     *         </li>
     *         <li>TooManyRequestsException The limit on the number of requests per second was exceeded.</li>
     *         <li>ConflictException The request failed because there is a conflict with a previous write. You can retry
     *         the request.</li>
     *         <li>BadRequestException The request is invalid or malformed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3TablesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3TablesAsyncClient.ListNamespaces
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3tables-2018-05-10/ListNamespaces" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListNamespacesResponse> listNamespaces(ListNamespacesRequest listNamespacesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listNamespacesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listNamespacesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Tables");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListNamespaces");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListNamespacesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListNamespacesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListNamespacesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListNamespacesRequest, ListNamespacesResponse>()
                            .withOperationName("ListNamespaces").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListNamespacesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listNamespacesRequest));
            CompletableFuture<ListNamespacesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists table buckets for your account. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-tables-buckets.html">S3 Table buckets</a> in the
     * <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3tables:ListTableBuckets</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     *
     * @param listTableBucketsRequest
     * @return A Java Future containing the result of the ListTableBuckets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerErrorException The request failed due to an internal server error.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request.</li>
     *         <li>NotFoundException The request was rejected because the specified resource could not be found.</li>
     *         <li>AccessDeniedException The action cannot be performed because you do not have the required permission.
     *         </li>
     *         <li>TooManyRequestsException The limit on the number of requests per second was exceeded.</li>
     *         <li>ConflictException The request failed because there is a conflict with a previous write. You can retry
     *         the request.</li>
     *         <li>BadRequestException The request is invalid or malformed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3TablesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3TablesAsyncClient.ListTableBuckets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3tables-2018-05-10/ListTableBuckets" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTableBucketsResponse> listTableBuckets(ListTableBucketsRequest listTableBucketsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTableBucketsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTableBucketsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Tables");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTableBuckets");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTableBucketsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTableBucketsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTableBucketsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTableBucketsRequest, ListTableBucketsResponse>()
                            .withOperationName("ListTableBuckets").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTableBucketsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTableBucketsRequest));
            CompletableFuture<ListTableBucketsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List tables in the given table bucket. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-tables-tables.html">S3 Tables</a> in the <i>Amazon
     * Simple Storage Service User Guide</i>.
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3tables:ListTables</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     *
     * @param listTablesRequest
     * @return A Java Future containing the result of the ListTables operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerErrorException The request failed due to an internal server error.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request.</li>
     *         <li>NotFoundException The request was rejected because the specified resource could not be found.</li>
     *         <li>TooManyRequestsException The limit on the number of requests per second was exceeded.</li>
     *         <li>ConflictException The request failed because there is a conflict with a previous write. You can retry
     *         the request.</li>
     *         <li>BadRequestException The request is invalid or malformed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3TablesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3TablesAsyncClient.ListTables
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3tables-2018-05-10/ListTables" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTablesResponse> listTables(ListTablesRequest listTablesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTablesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTablesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Tables");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTables");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTablesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListTablesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTablesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTablesRequest, ListTablesResponse>().withOperationName("ListTables")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTablesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTablesRequest));
            CompletableFuture<ListTablesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new maintenance configuration or replaces an existing maintenance configuration for a table bucket. For
     * more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-table-buckets-maintenance.html">Amazon S3 table
     * bucket maintenance</a> in the <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3tables:PutTableBucketMaintenanceConfiguration</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     *
     * @param putTableBucketMaintenanceConfigurationRequest
     * @return A Java Future containing the result of the PutTableBucketMaintenanceConfiguration operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerErrorException The request failed due to an internal server error.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request.</li>
     *         <li>NotFoundException The request was rejected because the specified resource could not be found.</li>
     *         <li>TooManyRequestsException The limit on the number of requests per second was exceeded.</li>
     *         <li>ConflictException The request failed because there is a conflict with a previous write. You can retry
     *         the request.</li>
     *         <li>BadRequestException The request is invalid or malformed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3TablesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3TablesAsyncClient.PutTableBucketMaintenanceConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3tables-2018-05-10/PutTableBucketMaintenanceConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutTableBucketMaintenanceConfigurationResponse> putTableBucketMaintenanceConfiguration(
            PutTableBucketMaintenanceConfigurationRequest putTableBucketMaintenanceConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putTableBucketMaintenanceConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                putTableBucketMaintenanceConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Tables");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutTableBucketMaintenanceConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutTableBucketMaintenanceConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, PutTableBucketMaintenanceConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutTableBucketMaintenanceConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutTableBucketMaintenanceConfigurationRequest, PutTableBucketMaintenanceConfigurationResponse>()
                            .withOperationName("PutTableBucketMaintenanceConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutTableBucketMaintenanceConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putTableBucketMaintenanceConfigurationRequest));
            CompletableFuture<PutTableBucketMaintenanceConfigurationResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new maintenance configuration or replaces an existing table bucket policy for a table bucket. For more
     * information, see <a href=
     * "https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-tables-bucket-policy.html#table-bucket-policy-add"
     * >Adding a table bucket policy</a> in the <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3tables:PutTableBucketPolicy</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     *
     * @param putTableBucketPolicyRequest
     * @return A Java Future containing the result of the PutTableBucketPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerErrorException The request failed due to an internal server error.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request.</li>
     *         <li>NotFoundException The request was rejected because the specified resource could not be found.</li>
     *         <li>TooManyRequestsException The limit on the number of requests per second was exceeded.</li>
     *         <li>ConflictException The request failed because there is a conflict with a previous write. You can retry
     *         the request.</li>
     *         <li>BadRequestException The request is invalid or malformed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3TablesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3TablesAsyncClient.PutTableBucketPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3tables-2018-05-10/PutTableBucketPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<PutTableBucketPolicyResponse> putTableBucketPolicy(
            PutTableBucketPolicyRequest putTableBucketPolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putTableBucketPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putTableBucketPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Tables");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutTableBucketPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutTableBucketPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutTableBucketPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutTableBucketPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutTableBucketPolicyRequest, PutTableBucketPolicyResponse>()
                            .withOperationName("PutTableBucketPolicy").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutTableBucketPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putTableBucketPolicyRequest));
            CompletableFuture<PutTableBucketPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new maintenance configuration or replaces an existing maintenance configuration for a table. For more
     * information, see <a href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-tables-maintenance.html">S3
     * Tables maintenance</a> in the <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3tables:PutTableMaintenanceConfiguration</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     *
     * @param putTableMaintenanceConfigurationRequest
     * @return A Java Future containing the result of the PutTableMaintenanceConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerErrorException The request failed due to an internal server error.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request.</li>
     *         <li>NotFoundException The request was rejected because the specified resource could not be found.</li>
     *         <li>TooManyRequestsException The limit on the number of requests per second was exceeded.</li>
     *         <li>ConflictException The request failed because there is a conflict with a previous write. You can retry
     *         the request.</li>
     *         <li>BadRequestException The request is invalid or malformed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3TablesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3TablesAsyncClient.PutTableMaintenanceConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3tables-2018-05-10/PutTableMaintenanceConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutTableMaintenanceConfigurationResponse> putTableMaintenanceConfiguration(
            PutTableMaintenanceConfigurationRequest putTableMaintenanceConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putTableMaintenanceConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                putTableMaintenanceConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Tables");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutTableMaintenanceConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutTableMaintenanceConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, PutTableMaintenanceConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutTableMaintenanceConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutTableMaintenanceConfigurationRequest, PutTableMaintenanceConfigurationResponse>()
                            .withOperationName("PutTableMaintenanceConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutTableMaintenanceConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putTableMaintenanceConfigurationRequest));
            CompletableFuture<PutTableMaintenanceConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new maintenance configuration or replaces an existing table policy for a table. For more information,
     * see <a href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-tables-table-policy.html#table-policy-add">
     * Adding a table policy</a> in the <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3tables:PutTablePolicy</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     *
     * @param putTablePolicyRequest
     * @return A Java Future containing the result of the PutTablePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerErrorException The request failed due to an internal server error.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request.</li>
     *         <li>NotFoundException The request was rejected because the specified resource could not be found.</li>
     *         <li>TooManyRequestsException The limit on the number of requests per second was exceeded.</li>
     *         <li>ConflictException The request failed because there is a conflict with a previous write. You can retry
     *         the request.</li>
     *         <li>BadRequestException The request is invalid or malformed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3TablesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3TablesAsyncClient.PutTablePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3tables-2018-05-10/PutTablePolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutTablePolicyResponse> putTablePolicy(PutTablePolicyRequest putTablePolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putTablePolicyRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putTablePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Tables");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutTablePolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutTablePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutTablePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutTablePolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutTablePolicyRequest, PutTablePolicyResponse>()
                            .withOperationName("PutTablePolicy").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutTablePolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putTablePolicyRequest));
            CompletableFuture<PutTablePolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Renames a table or a namespace. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-tables-tables.html">S3 Tables</a> in the <i>Amazon
     * Simple Storage Service User Guide</i>.
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3tables:RenameTable</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     *
     * @param renameTableRequest
     * @return A Java Future containing the result of the RenameTable operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerErrorException The request failed due to an internal server error.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request.</li>
     *         <li>NotFoundException The request was rejected because the specified resource could not be found.</li>
     *         <li>TooManyRequestsException The limit on the number of requests per second was exceeded.</li>
     *         <li>ConflictException The request failed because there is a conflict with a previous write. You can retry
     *         the request.</li>
     *         <li>BadRequestException The request is invalid or malformed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3TablesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3TablesAsyncClient.RenameTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3tables-2018-05-10/RenameTable" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<RenameTableResponse> renameTable(RenameTableRequest renameTableRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(renameTableRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, renameTableRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Tables");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RenameTable");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RenameTableResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    RenameTableResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RenameTableResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RenameTableRequest, RenameTableResponse>()
                            .withOperationName("RenameTable").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new RenameTableRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(renameTableRequest));
            CompletableFuture<RenameTableResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the metadata location for a table. The metadata location of a table must be an S3 URI that begins with
     * the table's warehouse location. The metadata location for an Apache Iceberg table must end with
     * <code>.metadata.json</code>, or if the metadata file is Gzip-compressed, <code>.metadata.json.gz</code>.
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3tables:UpdateTableMetadataLocation</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     *
     * @param updateTableMetadataLocationRequest
     * @return A Java Future containing the result of the UpdateTableMetadataLocation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerErrorException The request failed due to an internal server error.</li>
     *         <li>ForbiddenException The caller isn't authorized to make the request.</li>
     *         <li>NotFoundException The request was rejected because the specified resource could not be found.</li>
     *         <li>TooManyRequestsException The limit on the number of requests per second was exceeded.</li>
     *         <li>ConflictException The request failed because there is a conflict with a previous write. You can retry
     *         the request.</li>
     *         <li>BadRequestException The request is invalid or malformed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3TablesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3TablesAsyncClient.UpdateTableMetadataLocation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3tables-2018-05-10/UpdateTableMetadataLocation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateTableMetadataLocationResponse> updateTableMetadataLocation(
            UpdateTableMetadataLocationRequest updateTableMetadataLocationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateTableMetadataLocationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateTableMetadataLocationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Tables");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateTableMetadataLocation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateTableMetadataLocationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateTableMetadataLocationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateTableMetadataLocationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateTableMetadataLocationRequest, UpdateTableMetadataLocationResponse>()
                            .withOperationName("UpdateTableMetadataLocation").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateTableMetadataLocationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateTableMetadataLocationRequest));
            CompletableFuture<UpdateTableMetadataLocationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final S3TablesServiceClientConfiguration serviceClientConfiguration() {
        return new S3TablesServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(S3TablesException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ForbiddenException")
                                .exceptionBuilderSupplier(ForbiddenException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerErrorException")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        S3TablesServiceClientConfigurationBuilder serviceConfigBuilder = new S3TablesServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
