/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.tnb;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.tnb.internal.TnbServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.tnb.model.AccessDeniedException;
import software.amazon.awssdk.services.tnb.model.CancelSolNetworkOperationRequest;
import software.amazon.awssdk.services.tnb.model.CancelSolNetworkOperationResponse;
import software.amazon.awssdk.services.tnb.model.CreateSolFunctionPackageRequest;
import software.amazon.awssdk.services.tnb.model.CreateSolFunctionPackageResponse;
import software.amazon.awssdk.services.tnb.model.CreateSolNetworkInstanceRequest;
import software.amazon.awssdk.services.tnb.model.CreateSolNetworkInstanceResponse;
import software.amazon.awssdk.services.tnb.model.CreateSolNetworkPackageRequest;
import software.amazon.awssdk.services.tnb.model.CreateSolNetworkPackageResponse;
import software.amazon.awssdk.services.tnb.model.DeleteSolFunctionPackageRequest;
import software.amazon.awssdk.services.tnb.model.DeleteSolFunctionPackageResponse;
import software.amazon.awssdk.services.tnb.model.DeleteSolNetworkInstanceRequest;
import software.amazon.awssdk.services.tnb.model.DeleteSolNetworkInstanceResponse;
import software.amazon.awssdk.services.tnb.model.DeleteSolNetworkPackageRequest;
import software.amazon.awssdk.services.tnb.model.DeleteSolNetworkPackageResponse;
import software.amazon.awssdk.services.tnb.model.GetSolFunctionInstanceRequest;
import software.amazon.awssdk.services.tnb.model.GetSolFunctionInstanceResponse;
import software.amazon.awssdk.services.tnb.model.GetSolFunctionPackageContentRequest;
import software.amazon.awssdk.services.tnb.model.GetSolFunctionPackageContentResponse;
import software.amazon.awssdk.services.tnb.model.GetSolFunctionPackageDescriptorRequest;
import software.amazon.awssdk.services.tnb.model.GetSolFunctionPackageDescriptorResponse;
import software.amazon.awssdk.services.tnb.model.GetSolFunctionPackageRequest;
import software.amazon.awssdk.services.tnb.model.GetSolFunctionPackageResponse;
import software.amazon.awssdk.services.tnb.model.GetSolNetworkInstanceRequest;
import software.amazon.awssdk.services.tnb.model.GetSolNetworkInstanceResponse;
import software.amazon.awssdk.services.tnb.model.GetSolNetworkOperationRequest;
import software.amazon.awssdk.services.tnb.model.GetSolNetworkOperationResponse;
import software.amazon.awssdk.services.tnb.model.GetSolNetworkPackageContentRequest;
import software.amazon.awssdk.services.tnb.model.GetSolNetworkPackageContentResponse;
import software.amazon.awssdk.services.tnb.model.GetSolNetworkPackageDescriptorRequest;
import software.amazon.awssdk.services.tnb.model.GetSolNetworkPackageDescriptorResponse;
import software.amazon.awssdk.services.tnb.model.GetSolNetworkPackageRequest;
import software.amazon.awssdk.services.tnb.model.GetSolNetworkPackageResponse;
import software.amazon.awssdk.services.tnb.model.InstantiateSolNetworkInstanceRequest;
import software.amazon.awssdk.services.tnb.model.InstantiateSolNetworkInstanceResponse;
import software.amazon.awssdk.services.tnb.model.InternalServerException;
import software.amazon.awssdk.services.tnb.model.ListSolFunctionInstancesRequest;
import software.amazon.awssdk.services.tnb.model.ListSolFunctionInstancesResponse;
import software.amazon.awssdk.services.tnb.model.ListSolFunctionPackagesRequest;
import software.amazon.awssdk.services.tnb.model.ListSolFunctionPackagesResponse;
import software.amazon.awssdk.services.tnb.model.ListSolNetworkInstancesRequest;
import software.amazon.awssdk.services.tnb.model.ListSolNetworkInstancesResponse;
import software.amazon.awssdk.services.tnb.model.ListSolNetworkOperationsRequest;
import software.amazon.awssdk.services.tnb.model.ListSolNetworkOperationsResponse;
import software.amazon.awssdk.services.tnb.model.ListSolNetworkPackagesRequest;
import software.amazon.awssdk.services.tnb.model.ListSolNetworkPackagesResponse;
import software.amazon.awssdk.services.tnb.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.tnb.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.tnb.model.PutSolFunctionPackageContentRequest;
import software.amazon.awssdk.services.tnb.model.PutSolFunctionPackageContentResponse;
import software.amazon.awssdk.services.tnb.model.PutSolNetworkPackageContentRequest;
import software.amazon.awssdk.services.tnb.model.PutSolNetworkPackageContentResponse;
import software.amazon.awssdk.services.tnb.model.ResourceNotFoundException;
import software.amazon.awssdk.services.tnb.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.tnb.model.TagResourceRequest;
import software.amazon.awssdk.services.tnb.model.TagResourceResponse;
import software.amazon.awssdk.services.tnb.model.TerminateSolNetworkInstanceRequest;
import software.amazon.awssdk.services.tnb.model.TerminateSolNetworkInstanceResponse;
import software.amazon.awssdk.services.tnb.model.ThrottlingException;
import software.amazon.awssdk.services.tnb.model.TnbException;
import software.amazon.awssdk.services.tnb.model.UntagResourceRequest;
import software.amazon.awssdk.services.tnb.model.UntagResourceResponse;
import software.amazon.awssdk.services.tnb.model.UpdateSolFunctionPackageRequest;
import software.amazon.awssdk.services.tnb.model.UpdateSolFunctionPackageResponse;
import software.amazon.awssdk.services.tnb.model.UpdateSolNetworkInstanceRequest;
import software.amazon.awssdk.services.tnb.model.UpdateSolNetworkInstanceResponse;
import software.amazon.awssdk.services.tnb.model.UpdateSolNetworkPackageRequest;
import software.amazon.awssdk.services.tnb.model.UpdateSolNetworkPackageResponse;
import software.amazon.awssdk.services.tnb.model.ValidateSolFunctionPackageContentRequest;
import software.amazon.awssdk.services.tnb.model.ValidateSolFunctionPackageContentResponse;
import software.amazon.awssdk.services.tnb.model.ValidateSolNetworkPackageContentRequest;
import software.amazon.awssdk.services.tnb.model.ValidateSolNetworkPackageContentResponse;
import software.amazon.awssdk.services.tnb.model.ValidationException;
import software.amazon.awssdk.services.tnb.transform.CancelSolNetworkOperationRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.CreateSolFunctionPackageRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.CreateSolNetworkInstanceRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.CreateSolNetworkPackageRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.DeleteSolFunctionPackageRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.DeleteSolNetworkInstanceRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.DeleteSolNetworkPackageRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.GetSolFunctionInstanceRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.GetSolFunctionPackageContentRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.GetSolFunctionPackageDescriptorRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.GetSolFunctionPackageRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.GetSolNetworkInstanceRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.GetSolNetworkOperationRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.GetSolNetworkPackageContentRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.GetSolNetworkPackageDescriptorRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.GetSolNetworkPackageRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.InstantiateSolNetworkInstanceRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.ListSolFunctionInstancesRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.ListSolFunctionPackagesRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.ListSolNetworkInstancesRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.ListSolNetworkOperationsRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.ListSolNetworkPackagesRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.PutSolFunctionPackageContentRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.PutSolNetworkPackageContentRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.TerminateSolNetworkInstanceRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.UpdateSolFunctionPackageRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.UpdateSolNetworkInstanceRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.UpdateSolNetworkPackageRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.ValidateSolFunctionPackageContentRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.ValidateSolNetworkPackageContentRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link TnbAsyncClient}.
 *
 * @see TnbAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultTnbAsyncClient implements TnbAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultTnbAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultTnbAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Cancels a network operation.
     * </p>
     * <p>
     * A network operation is any operation that is done to your network, such as network instance instantiation or
     * termination.
     * </p>
     *
     * @param cancelSolNetworkOperationRequest
     * @return A Java Future containing the result of the CancelSolNetworkOperation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException Unexpected error occurred. Problem on the server.</li>
     *         <li>ThrottlingException Exception caused by throttling.</li>
     *         <li>ValidationException Unable to process the request because the client provided input failed to satisfy
     *         request constraints.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>AccessDeniedException Insufficient permissions to make request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TnbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample TnbAsyncClient.CancelSolNetworkOperation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/CancelSolNetworkOperation" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CancelSolNetworkOperationResponse> cancelSolNetworkOperation(
            CancelSolNetworkOperationRequest cancelSolNetworkOperationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(cancelSolNetworkOperationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelSolNetworkOperationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelSolNetworkOperation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CancelSolNetworkOperationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CancelSolNetworkOperationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CancelSolNetworkOperationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CancelSolNetworkOperationRequest, CancelSolNetworkOperationResponse>()
                            .withOperationName("CancelSolNetworkOperation").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CancelSolNetworkOperationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(cancelSolNetworkOperationRequest));
            CompletableFuture<CancelSolNetworkOperationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a function package.
     * </p>
     * <p>
     * A function package is a .zip file in CSAR (Cloud Service Archive) format that contains a network function (an
     * ETSI standard telecommunication application) and function package descriptor that uses the TOSCA standard to
     * describe how the network functions should run on your network. For more information, see <a
     * href="https://docs.aws.amazon.com/tnb/latest/ug/function-packages.html">Function packages</a> in the <i>Amazon
     * Web Services Telco Network Builder User Guide</i>.
     * </p>
     * <p>
     * Creating a function package is the first step for creating a network in AWS TNB. This request creates an empty
     * container with an ID. The next step is to upload the actual CSAR zip file into that empty container. To upload
     * function package content, see <a
     * href="https://docs.aws.amazon.com/tnb/latest/APIReference/API_PutSolFunctionPackageContent.html"
     * >PutSolFunctionPackageContent</a>.
     * </p>
     *
     * @param createSolFunctionPackageRequest
     * @return A Java Future containing the result of the CreateSolFunctionPackage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException Unexpected error occurred. Problem on the server.</li>
     *         <li>ServiceQuotaExceededException Service quotas have been exceeded.</li>
     *         <li>ThrottlingException Exception caused by throttling.</li>
     *         <li>ValidationException Unable to process the request because the client provided input failed to satisfy
     *         request constraints.</li>
     *         <li>AccessDeniedException Insufficient permissions to make request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TnbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample TnbAsyncClient.CreateSolFunctionPackage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/CreateSolFunctionPackage" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateSolFunctionPackageResponse> createSolFunctionPackage(
            CreateSolFunctionPackageRequest createSolFunctionPackageRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createSolFunctionPackageRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSolFunctionPackageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSolFunctionPackage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateSolFunctionPackageResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateSolFunctionPackageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateSolFunctionPackageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateSolFunctionPackageRequest, CreateSolFunctionPackageResponse>()
                            .withOperationName("CreateSolFunctionPackage").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateSolFunctionPackageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createSolFunctionPackageRequest));
            CompletableFuture<CreateSolFunctionPackageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a network instance.
     * </p>
     * <p>
     * A network instance is a single network created in Amazon Web Services TNB that can be deployed and on which
     * life-cycle operations (like terminate, update, and delete) can be performed. Creating a network instance is the
     * third step after creating a network package. For more information about network instances, <a
     * href="https://docs.aws.amazon.com/tnb/latest/ug/network-instances.html">Network instances</a> in the <i>Amazon
     * Web Services Telco Network Builder User Guide</i>.
     * </p>
     * <p>
     * Once you create a network instance, you can instantiate it. To instantiate a network, see <a
     * href="https://docs.aws.amazon.com/tnb/latest/APIReference/API_InstantiateSolNetworkInstance.html"
     * >InstantiateSolNetworkInstance</a>.
     * </p>
     *
     * @param createSolNetworkInstanceRequest
     * @return A Java Future containing the result of the CreateSolNetworkInstance operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException Unexpected error occurred. Problem on the server.</li>
     *         <li>ServiceQuotaExceededException Service quotas have been exceeded.</li>
     *         <li>ThrottlingException Exception caused by throttling.</li>
     *         <li>ValidationException Unable to process the request because the client provided input failed to satisfy
     *         request constraints.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>AccessDeniedException Insufficient permissions to make request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TnbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample TnbAsyncClient.CreateSolNetworkInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/CreateSolNetworkInstance" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateSolNetworkInstanceResponse> createSolNetworkInstance(
            CreateSolNetworkInstanceRequest createSolNetworkInstanceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createSolNetworkInstanceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSolNetworkInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSolNetworkInstance");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateSolNetworkInstanceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateSolNetworkInstanceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateSolNetworkInstanceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateSolNetworkInstanceRequest, CreateSolNetworkInstanceResponse>()
                            .withOperationName("CreateSolNetworkInstance").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateSolNetworkInstanceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createSolNetworkInstanceRequest));
            CompletableFuture<CreateSolNetworkInstanceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a network package.
     * </p>
     * <p>
     * A network package is a .zip file in CSAR (Cloud Service Archive) format defines the function packages you want to
     * deploy and the Amazon Web Services infrastructure you want to deploy them on. For more information, see <a
     * href="https://docs.aws.amazon.com/tnb/latest/ug/network-instances.html">Network instances</a> in the <i>Amazon
     * Web Services Telco Network Builder User Guide</i>.
     * </p>
     * <p>
     * A network package consists of a network service descriptor (NSD) file (required) and any additional files
     * (optional), such as scripts specific to your needs. For example, if you have multiple function packages in your
     * network package, you can use the NSD to define which network functions should run in certain VPCs, subnets, or
     * EKS clusters.
     * </p>
     * <p>
     * This request creates an empty network package container with an ID. Once you create a network package, you can
     * upload the network package content using <a
     * href="https://docs.aws.amazon.com/tnb/latest/APIReference/API_PutSolNetworkPackageContent.html"
     * >PutSolNetworkPackageContent</a>.
     * </p>
     *
     * @param createSolNetworkPackageRequest
     * @return A Java Future containing the result of the CreateSolNetworkPackage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException Unexpected error occurred. Problem on the server.</li>
     *         <li>ServiceQuotaExceededException Service quotas have been exceeded.</li>
     *         <li>ThrottlingException Exception caused by throttling.</li>
     *         <li>ValidationException Unable to process the request because the client provided input failed to satisfy
     *         request constraints.</li>
     *         <li>AccessDeniedException Insufficient permissions to make request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TnbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample TnbAsyncClient.CreateSolNetworkPackage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/CreateSolNetworkPackage" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateSolNetworkPackageResponse> createSolNetworkPackage(
            CreateSolNetworkPackageRequest createSolNetworkPackageRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createSolNetworkPackageRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSolNetworkPackageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSolNetworkPackage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateSolNetworkPackageResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateSolNetworkPackageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateSolNetworkPackageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateSolNetworkPackageRequest, CreateSolNetworkPackageResponse>()
                            .withOperationName("CreateSolNetworkPackage").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateSolNetworkPackageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createSolNetworkPackageRequest));
            CompletableFuture<CreateSolNetworkPackageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a function package.
     * </p>
     * <p>
     * A function package is a .zip file in CSAR (Cloud Service Archive) format that contains a network function (an
     * ETSI standard telecommunication application) and function package descriptor that uses the TOSCA standard to
     * describe how the network functions should run on your network.
     * </p>
     * <p>
     * To delete a function package, the package must be in a disabled state. To disable a function package, see <a
     * href=
     * "https://docs.aws.amazon.com/tnb/latest/APIReference/API_UpdateSolFunctionPackage.html">UpdateSolFunctionPackage
     * </a>.
     * </p>
     *
     * @param deleteSolFunctionPackageRequest
     * @return A Java Future containing the result of the DeleteSolFunctionPackage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException Unexpected error occurred. Problem on the server.</li>
     *         <li>ThrottlingException Exception caused by throttling.</li>
     *         <li>ValidationException Unable to process the request because the client provided input failed to satisfy
     *         request constraints.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>AccessDeniedException Insufficient permissions to make request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TnbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample TnbAsyncClient.DeleteSolFunctionPackage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/DeleteSolFunctionPackage" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSolFunctionPackageResponse> deleteSolFunctionPackage(
            DeleteSolFunctionPackageRequest deleteSolFunctionPackageRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteSolFunctionPackageRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSolFunctionPackageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSolFunctionPackage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSolFunctionPackageResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteSolFunctionPackageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteSolFunctionPackageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSolFunctionPackageRequest, DeleteSolFunctionPackageResponse>()
                            .withOperationName("DeleteSolFunctionPackage").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteSolFunctionPackageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteSolFunctionPackageRequest));
            CompletableFuture<DeleteSolFunctionPackageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a network instance.
     * </p>
     * <p>
     * A network instance is a single network created in Amazon Web Services TNB that can be deployed and on which
     * life-cycle operations (like terminate, update, and delete) can be performed.
     * </p>
     * <p>
     * To delete a network instance, the instance must be in a stopped or terminated state. To terminate a network
     * instance, see <a href="https://docs.aws.amazon.com/tnb/latest/APIReference/API_TerminateSolNetworkInstance.html">
     * TerminateSolNetworkInstance</a>.
     * </p>
     *
     * @param deleteSolNetworkInstanceRequest
     * @return A Java Future containing the result of the DeleteSolNetworkInstance operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException Unexpected error occurred. Problem on the server.</li>
     *         <li>ThrottlingException Exception caused by throttling.</li>
     *         <li>ValidationException Unable to process the request because the client provided input failed to satisfy
     *         request constraints.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>AccessDeniedException Insufficient permissions to make request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TnbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample TnbAsyncClient.DeleteSolNetworkInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/DeleteSolNetworkInstance" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSolNetworkInstanceResponse> deleteSolNetworkInstance(
            DeleteSolNetworkInstanceRequest deleteSolNetworkInstanceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteSolNetworkInstanceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSolNetworkInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSolNetworkInstance");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSolNetworkInstanceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteSolNetworkInstanceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteSolNetworkInstanceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSolNetworkInstanceRequest, DeleteSolNetworkInstanceResponse>()
                            .withOperationName("DeleteSolNetworkInstance").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteSolNetworkInstanceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteSolNetworkInstanceRequest));
            CompletableFuture<DeleteSolNetworkInstanceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes network package.
     * </p>
     * <p>
     * A network package is a .zip file in CSAR (Cloud Service Archive) format defines the function packages you want to
     * deploy and the Amazon Web Services infrastructure you want to deploy them on.
     * </p>
     * <p>
     * To delete a network package, the package must be in a disable state. To disable a network package, see <a
     * href="https://docs.aws.amazon.com/tnb/latest/APIReference/API_UpdateSolNetworkPackage.html"
     * >UpdateSolNetworkPackage</a>.
     * </p>
     *
     * @param deleteSolNetworkPackageRequest
     * @return A Java Future containing the result of the DeleteSolNetworkPackage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException Unexpected error occurred. Problem on the server.</li>
     *         <li>ThrottlingException Exception caused by throttling.</li>
     *         <li>ValidationException Unable to process the request because the client provided input failed to satisfy
     *         request constraints.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>AccessDeniedException Insufficient permissions to make request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TnbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample TnbAsyncClient.DeleteSolNetworkPackage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/DeleteSolNetworkPackage" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSolNetworkPackageResponse> deleteSolNetworkPackage(
            DeleteSolNetworkPackageRequest deleteSolNetworkPackageRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteSolNetworkPackageRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSolNetworkPackageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSolNetworkPackage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSolNetworkPackageResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteSolNetworkPackageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteSolNetworkPackageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSolNetworkPackageRequest, DeleteSolNetworkPackageResponse>()
                            .withOperationName("DeleteSolNetworkPackage").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteSolNetworkPackageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteSolNetworkPackageRequest));
            CompletableFuture<DeleteSolNetworkPackageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the details of a network function instance, including the instantiation state and metadata from the function
     * package descriptor in the network function package.
     * </p>
     * <p>
     * A network function instance is a function in a function package .
     * </p>
     *
     * @param getSolFunctionInstanceRequest
     * @return A Java Future containing the result of the GetSolFunctionInstance operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException Unexpected error occurred. Problem on the server.</li>
     *         <li>ThrottlingException Exception caused by throttling.</li>
     *         <li>ValidationException Unable to process the request because the client provided input failed to satisfy
     *         request constraints.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>AccessDeniedException Insufficient permissions to make request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TnbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample TnbAsyncClient.GetSolFunctionInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/GetSolFunctionInstance" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetSolFunctionInstanceResponse> getSolFunctionInstance(
            GetSolFunctionInstanceRequest getSolFunctionInstanceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSolFunctionInstanceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSolFunctionInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSolFunctionInstance");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSolFunctionInstanceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetSolFunctionInstanceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSolFunctionInstanceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSolFunctionInstanceRequest, GetSolFunctionInstanceResponse>()
                            .withOperationName("GetSolFunctionInstance").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetSolFunctionInstanceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getSolFunctionInstanceRequest));
            CompletableFuture<GetSolFunctionInstanceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the details of an individual function package, such as the operational state and whether the package is in
     * use.
     * </p>
     * <p>
     * A function package is a .zip file in CSAR (Cloud Service Archive) format that contains a network function (an
     * ETSI standard telecommunication application) and function package descriptor that uses the TOSCA standard to
     * describe how the network functions should run on your network..
     * </p>
     *
     * @param getSolFunctionPackageRequest
     * @return A Java Future containing the result of the GetSolFunctionPackage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException Unexpected error occurred. Problem on the server.</li>
     *         <li>ThrottlingException Exception caused by throttling.</li>
     *         <li>ValidationException Unable to process the request because the client provided input failed to satisfy
     *         request constraints.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>AccessDeniedException Insufficient permissions to make request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TnbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample TnbAsyncClient.GetSolFunctionPackage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/GetSolFunctionPackage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetSolFunctionPackageResponse> getSolFunctionPackage(
            GetSolFunctionPackageRequest getSolFunctionPackageRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSolFunctionPackageRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSolFunctionPackageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSolFunctionPackage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSolFunctionPackageResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetSolFunctionPackageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSolFunctionPackageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSolFunctionPackageRequest, GetSolFunctionPackageResponse>()
                            .withOperationName("GetSolFunctionPackage").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetSolFunctionPackageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getSolFunctionPackageRequest));
            CompletableFuture<GetSolFunctionPackageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the contents of a function package.
     * </p>
     * <p>
     * A function package is a .zip file in CSAR (Cloud Service Archive) format that contains a network function (an
     * ETSI standard telecommunication application) and function package descriptor that uses the TOSCA standard to
     * describe how the network functions should run on your network.
     * </p>
     *
     * @param getSolFunctionPackageContentRequest
     * @return A Java Future containing the result of the GetSolFunctionPackageContent operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException Unexpected error occurred. Problem on the server.</li>
     *         <li>ThrottlingException Exception caused by throttling.</li>
     *         <li>ValidationException Unable to process the request because the client provided input failed to satisfy
     *         request constraints.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>AccessDeniedException Insufficient permissions to make request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TnbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample TnbAsyncClient.GetSolFunctionPackageContent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/GetSolFunctionPackageContent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetSolFunctionPackageContentResponse> getSolFunctionPackageContent(
            GetSolFunctionPackageContentRequest getSolFunctionPackageContentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSolFunctionPackageContentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSolFunctionPackageContentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSolFunctionPackageContent");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(false).build();

            HttpResponseHandler<GetSolFunctionPackageContentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetSolFunctionPackageContentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSolFunctionPackageContentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSolFunctionPackageContentRequest, GetSolFunctionPackageContentResponse>()
                            .withOperationName("GetSolFunctionPackageContent").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetSolFunctionPackageContentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getSolFunctionPackageContentRequest));
            CompletableFuture<GetSolFunctionPackageContentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a function package descriptor in a function package.
     * </p>
     * <p>
     * A function package descriptor is a .yaml file in a function package that uses the TOSCA standard to describe how
     * the network function in the function package should run on your network.
     * </p>
     * <p>
     * A function package is a .zip file in CSAR (Cloud Service Archive) format that contains a network function (an
     * ETSI standard telecommunication application) and function package descriptor that uses the TOSCA standard to
     * describe how the network functions should run on your network.
     * </p>
     *
     * @param getSolFunctionPackageDescriptorRequest
     * @return A Java Future containing the result of the GetSolFunctionPackageDescriptor operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException Unexpected error occurred. Problem on the server.</li>
     *         <li>ThrottlingException Exception caused by throttling.</li>
     *         <li>ValidationException Unable to process the request because the client provided input failed to satisfy
     *         request constraints.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>AccessDeniedException Insufficient permissions to make request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TnbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample TnbAsyncClient.GetSolFunctionPackageDescriptor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/GetSolFunctionPackageDescriptor"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetSolFunctionPackageDescriptorResponse> getSolFunctionPackageDescriptor(
            GetSolFunctionPackageDescriptorRequest getSolFunctionPackageDescriptorRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSolFunctionPackageDescriptorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getSolFunctionPackageDescriptorRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSolFunctionPackageDescriptor");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(false).build();

            HttpResponseHandler<GetSolFunctionPackageDescriptorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetSolFunctionPackageDescriptorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSolFunctionPackageDescriptorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSolFunctionPackageDescriptorRequest, GetSolFunctionPackageDescriptorResponse>()
                            .withOperationName("GetSolFunctionPackageDescriptor").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetSolFunctionPackageDescriptorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getSolFunctionPackageDescriptorRequest));
            CompletableFuture<GetSolFunctionPackageDescriptorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the details of the network instance.
     * </p>
     * <p>
     * A network instance is a single network created in Amazon Web Services TNB that can be deployed and on which
     * life-cycle operations (like terminate, update, and delete) can be performed.
     * </p>
     *
     * @param getSolNetworkInstanceRequest
     * @return A Java Future containing the result of the GetSolNetworkInstance operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException Unexpected error occurred. Problem on the server.</li>
     *         <li>ThrottlingException Exception caused by throttling.</li>
     *         <li>ValidationException Unable to process the request because the client provided input failed to satisfy
     *         request constraints.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>AccessDeniedException Insufficient permissions to make request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TnbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample TnbAsyncClient.GetSolNetworkInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/GetSolNetworkInstance" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetSolNetworkInstanceResponse> getSolNetworkInstance(
            GetSolNetworkInstanceRequest getSolNetworkInstanceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSolNetworkInstanceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSolNetworkInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSolNetworkInstance");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSolNetworkInstanceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetSolNetworkInstanceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSolNetworkInstanceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSolNetworkInstanceRequest, GetSolNetworkInstanceResponse>()
                            .withOperationName("GetSolNetworkInstance").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetSolNetworkInstanceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getSolNetworkInstanceRequest));
            CompletableFuture<GetSolNetworkInstanceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the details of a network operation, including the tasks involved in the network operation and the status of
     * the tasks.
     * </p>
     * <p>
     * A network operation is any operation that is done to your network, such as network instance instantiation or
     * termination.
     * </p>
     *
     * @param getSolNetworkOperationRequest
     * @return A Java Future containing the result of the GetSolNetworkOperation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException Unexpected error occurred. Problem on the server.</li>
     *         <li>ThrottlingException Exception caused by throttling.</li>
     *         <li>ValidationException Unable to process the request because the client provided input failed to satisfy
     *         request constraints.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>AccessDeniedException Insufficient permissions to make request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TnbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample TnbAsyncClient.GetSolNetworkOperation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/GetSolNetworkOperation" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetSolNetworkOperationResponse> getSolNetworkOperation(
            GetSolNetworkOperationRequest getSolNetworkOperationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSolNetworkOperationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSolNetworkOperationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSolNetworkOperation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSolNetworkOperationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetSolNetworkOperationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSolNetworkOperationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSolNetworkOperationRequest, GetSolNetworkOperationResponse>()
                            .withOperationName("GetSolNetworkOperation").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetSolNetworkOperationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getSolNetworkOperationRequest));
            CompletableFuture<GetSolNetworkOperationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the details of a network package.
     * </p>
     * <p>
     * A network package is a .zip file in CSAR (Cloud Service Archive) format defines the function packages you want to
     * deploy and the Amazon Web Services infrastructure you want to deploy them on.
     * </p>
     *
     * @param getSolNetworkPackageRequest
     * @return A Java Future containing the result of the GetSolNetworkPackage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException Unexpected error occurred. Problem on the server.</li>
     *         <li>ThrottlingException Exception caused by throttling.</li>
     *         <li>ValidationException Unable to process the request because the client provided input failed to satisfy
     *         request constraints.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>AccessDeniedException Insufficient permissions to make request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TnbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample TnbAsyncClient.GetSolNetworkPackage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/GetSolNetworkPackage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetSolNetworkPackageResponse> getSolNetworkPackage(
            GetSolNetworkPackageRequest getSolNetworkPackageRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSolNetworkPackageRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSolNetworkPackageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSolNetworkPackage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSolNetworkPackageResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetSolNetworkPackageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSolNetworkPackageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSolNetworkPackageRequest, GetSolNetworkPackageResponse>()
                            .withOperationName("GetSolNetworkPackage").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetSolNetworkPackageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getSolNetworkPackageRequest));
            CompletableFuture<GetSolNetworkPackageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the contents of a network package.
     * </p>
     * <p>
     * A network package is a .zip file in CSAR (Cloud Service Archive) format defines the function packages you want to
     * deploy and the Amazon Web Services infrastructure you want to deploy them on.
     * </p>
     *
     * @param getSolNetworkPackageContentRequest
     * @return A Java Future containing the result of the GetSolNetworkPackageContent operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException Unexpected error occurred. Problem on the server.</li>
     *         <li>ThrottlingException Exception caused by throttling.</li>
     *         <li>ValidationException Unable to process the request because the client provided input failed to satisfy
     *         request constraints.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>AccessDeniedException Insufficient permissions to make request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TnbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample TnbAsyncClient.GetSolNetworkPackageContent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/GetSolNetworkPackageContent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetSolNetworkPackageContentResponse> getSolNetworkPackageContent(
            GetSolNetworkPackageContentRequest getSolNetworkPackageContentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSolNetworkPackageContentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSolNetworkPackageContentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSolNetworkPackageContent");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(false).build();

            HttpResponseHandler<GetSolNetworkPackageContentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetSolNetworkPackageContentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSolNetworkPackageContentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSolNetworkPackageContentRequest, GetSolNetworkPackageContentResponse>()
                            .withOperationName("GetSolNetworkPackageContent").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetSolNetworkPackageContentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getSolNetworkPackageContentRequest));
            CompletableFuture<GetSolNetworkPackageContentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the content of the network service descriptor.
     * </p>
     * <p>
     * A network service descriptor is a .yaml file in a network package that uses the TOSCA standard to describe the
     * network functions you want to deploy and the Amazon Web Services infrastructure you want to deploy the network
     * functions on.
     * </p>
     *
     * @param getSolNetworkPackageDescriptorRequest
     * @return A Java Future containing the result of the GetSolNetworkPackageDescriptor operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException Unexpected error occurred. Problem on the server.</li>
     *         <li>ThrottlingException Exception caused by throttling.</li>
     *         <li>ValidationException Unable to process the request because the client provided input failed to satisfy
     *         request constraints.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>AccessDeniedException Insufficient permissions to make request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TnbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample TnbAsyncClient.GetSolNetworkPackageDescriptor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/GetSolNetworkPackageDescriptor"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetSolNetworkPackageDescriptorResponse> getSolNetworkPackageDescriptor(
            GetSolNetworkPackageDescriptorRequest getSolNetworkPackageDescriptorRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSolNetworkPackageDescriptorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getSolNetworkPackageDescriptorRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSolNetworkPackageDescriptor");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(false).build();

            HttpResponseHandler<GetSolNetworkPackageDescriptorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetSolNetworkPackageDescriptorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSolNetworkPackageDescriptorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSolNetworkPackageDescriptorRequest, GetSolNetworkPackageDescriptorResponse>()
                            .withOperationName("GetSolNetworkPackageDescriptor").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetSolNetworkPackageDescriptorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getSolNetworkPackageDescriptorRequest));
            CompletableFuture<GetSolNetworkPackageDescriptorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Instantiates a network instance.
     * </p>
     * <p>
     * A network instance is a single network created in Amazon Web Services TNB that can be deployed and on which
     * life-cycle operations (like terminate, update, and delete) can be performed.
     * </p>
     * <p>
     * Before you can instantiate a network instance, you have to create a network instance. For more information, see
     * <a href="https://docs.aws.amazon.com/tnb/latest/APIReference/API_CreateSolNetworkInstance.html">
     * CreateSolNetworkInstance</a>.
     * </p>
     *
     * @param instantiateSolNetworkInstanceRequest
     * @return A Java Future containing the result of the InstantiateSolNetworkInstance operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException Unexpected error occurred. Problem on the server.</li>
     *         <li>ServiceQuotaExceededException Service quotas have been exceeded.</li>
     *         <li>ThrottlingException Exception caused by throttling.</li>
     *         <li>ValidationException Unable to process the request because the client provided input failed to satisfy
     *         request constraints.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>AccessDeniedException Insufficient permissions to make request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TnbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample TnbAsyncClient.InstantiateSolNetworkInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/InstantiateSolNetworkInstance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<InstantiateSolNetworkInstanceResponse> instantiateSolNetworkInstance(
            InstantiateSolNetworkInstanceRequest instantiateSolNetworkInstanceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(instantiateSolNetworkInstanceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                instantiateSolNetworkInstanceRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "InstantiateSolNetworkInstance");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<InstantiateSolNetworkInstanceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, InstantiateSolNetworkInstanceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<InstantiateSolNetworkInstanceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<InstantiateSolNetworkInstanceRequest, InstantiateSolNetworkInstanceResponse>()
                            .withOperationName("InstantiateSolNetworkInstance").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new InstantiateSolNetworkInstanceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(instantiateSolNetworkInstanceRequest));
            CompletableFuture<InstantiateSolNetworkInstanceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists network function instances.
     * </p>
     * <p>
     * A network function instance is a function in a function package .
     * </p>
     *
     * @param listSolFunctionInstancesRequest
     * @return A Java Future containing the result of the ListSolFunctionInstances operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException Unexpected error occurred. Problem on the server.</li>
     *         <li>ThrottlingException Exception caused by throttling.</li>
     *         <li>ValidationException Unable to process the request because the client provided input failed to satisfy
     *         request constraints.</li>
     *         <li>AccessDeniedException Insufficient permissions to make request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TnbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample TnbAsyncClient.ListSolFunctionInstances
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/ListSolFunctionInstances" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSolFunctionInstancesResponse> listSolFunctionInstances(
            ListSolFunctionInstancesRequest listSolFunctionInstancesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSolFunctionInstancesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSolFunctionInstancesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSolFunctionInstances");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSolFunctionInstancesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListSolFunctionInstancesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSolFunctionInstancesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSolFunctionInstancesRequest, ListSolFunctionInstancesResponse>()
                            .withOperationName("ListSolFunctionInstances").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListSolFunctionInstancesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listSolFunctionInstancesRequest));
            CompletableFuture<ListSolFunctionInstancesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists information about function packages.
     * </p>
     * <p>
     * A function package is a .zip file in CSAR (Cloud Service Archive) format that contains a network function (an
     * ETSI standard telecommunication application) and function package descriptor that uses the TOSCA standard to
     * describe how the network functions should run on your network.
     * </p>
     *
     * @param listSolFunctionPackagesRequest
     * @return A Java Future containing the result of the ListSolFunctionPackages operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException Unexpected error occurred. Problem on the server.</li>
     *         <li>ThrottlingException Exception caused by throttling.</li>
     *         <li>ValidationException Unable to process the request because the client provided input failed to satisfy
     *         request constraints.</li>
     *         <li>AccessDeniedException Insufficient permissions to make request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TnbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample TnbAsyncClient.ListSolFunctionPackages
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/ListSolFunctionPackages" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSolFunctionPackagesResponse> listSolFunctionPackages(
            ListSolFunctionPackagesRequest listSolFunctionPackagesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSolFunctionPackagesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSolFunctionPackagesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSolFunctionPackages");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSolFunctionPackagesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListSolFunctionPackagesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSolFunctionPackagesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSolFunctionPackagesRequest, ListSolFunctionPackagesResponse>()
                            .withOperationName("ListSolFunctionPackages").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListSolFunctionPackagesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listSolFunctionPackagesRequest));
            CompletableFuture<ListSolFunctionPackagesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists your network instances.
     * </p>
     * <p>
     * A network instance is a single network created in Amazon Web Services TNB that can be deployed and on which
     * life-cycle operations (like terminate, update, and delete) can be performed.
     * </p>
     *
     * @param listSolNetworkInstancesRequest
     * @return A Java Future containing the result of the ListSolNetworkInstances operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException Unexpected error occurred. Problem on the server.</li>
     *         <li>ThrottlingException Exception caused by throttling.</li>
     *         <li>ValidationException Unable to process the request because the client provided input failed to satisfy
     *         request constraints.</li>
     *         <li>AccessDeniedException Insufficient permissions to make request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TnbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample TnbAsyncClient.ListSolNetworkInstances
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/ListSolNetworkInstances" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSolNetworkInstancesResponse> listSolNetworkInstances(
            ListSolNetworkInstancesRequest listSolNetworkInstancesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSolNetworkInstancesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSolNetworkInstancesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSolNetworkInstances");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSolNetworkInstancesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListSolNetworkInstancesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSolNetworkInstancesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSolNetworkInstancesRequest, ListSolNetworkInstancesResponse>()
                            .withOperationName("ListSolNetworkInstances").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListSolNetworkInstancesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listSolNetworkInstancesRequest));
            CompletableFuture<ListSolNetworkInstancesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists details for a network operation, including when the operation started and the status of the operation.
     * </p>
     * <p>
     * A network operation is any operation that is done to your network, such as network instance instantiation or
     * termination.
     * </p>
     *
     * @param listSolNetworkOperationsRequest
     * @return A Java Future containing the result of the ListSolNetworkOperations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException Unexpected error occurred. Problem on the server.</li>
     *         <li>ThrottlingException Exception caused by throttling.</li>
     *         <li>ValidationException Unable to process the request because the client provided input failed to satisfy
     *         request constraints.</li>
     *         <li>AccessDeniedException Insufficient permissions to make request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TnbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample TnbAsyncClient.ListSolNetworkOperations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/ListSolNetworkOperations" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSolNetworkOperationsResponse> listSolNetworkOperations(
            ListSolNetworkOperationsRequest listSolNetworkOperationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSolNetworkOperationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSolNetworkOperationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSolNetworkOperations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSolNetworkOperationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListSolNetworkOperationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSolNetworkOperationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSolNetworkOperationsRequest, ListSolNetworkOperationsResponse>()
                            .withOperationName("ListSolNetworkOperations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListSolNetworkOperationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listSolNetworkOperationsRequest));
            CompletableFuture<ListSolNetworkOperationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists network packages.
     * </p>
     * <p>
     * A network package is a .zip file in CSAR (Cloud Service Archive) format defines the function packages you want to
     * deploy and the Amazon Web Services infrastructure you want to deploy them on.
     * </p>
     *
     * @param listSolNetworkPackagesRequest
     * @return A Java Future containing the result of the ListSolNetworkPackages operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException Unexpected error occurred. Problem on the server.</li>
     *         <li>ThrottlingException Exception caused by throttling.</li>
     *         <li>ValidationException Unable to process the request because the client provided input failed to satisfy
     *         request constraints.</li>
     *         <li>AccessDeniedException Insufficient permissions to make request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TnbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample TnbAsyncClient.ListSolNetworkPackages
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/ListSolNetworkPackages" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSolNetworkPackagesResponse> listSolNetworkPackages(
            ListSolNetworkPackagesRequest listSolNetworkPackagesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSolNetworkPackagesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSolNetworkPackagesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSolNetworkPackages");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSolNetworkPackagesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListSolNetworkPackagesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSolNetworkPackagesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSolNetworkPackagesRequest, ListSolNetworkPackagesResponse>()
                            .withOperationName("ListSolNetworkPackages").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListSolNetworkPackagesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listSolNetworkPackagesRequest));
            CompletableFuture<ListSolNetworkPackagesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists tags for AWS TNB resources.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException Unexpected error occurred. Problem on the server.</li>
     *         <li>ThrottlingException Exception caused by throttling.</li>
     *         <li>ValidationException Unable to process the request because the client provided input failed to satisfy
     *         request constraints.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>AccessDeniedException Insufficient permissions to make request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TnbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample TnbAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Uploads the contents of a function package.
     * </p>
     * <p>
     * A function package is a .zip file in CSAR (Cloud Service Archive) format that contains a network function (an
     * ETSI standard telecommunication application) and function package descriptor that uses the TOSCA standard to
     * describe how the network functions should run on your network.
     * </p>
     *
     * @param putSolFunctionPackageContentRequest
     * @return A Java Future containing the result of the PutSolFunctionPackageContent operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException Unexpected error occurred. Problem on the server.</li>
     *         <li>ThrottlingException Exception caused by throttling.</li>
     *         <li>ValidationException Unable to process the request because the client provided input failed to satisfy
     *         request constraints.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>AccessDeniedException Insufficient permissions to make request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TnbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample TnbAsyncClient.PutSolFunctionPackageContent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/PutSolFunctionPackageContent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutSolFunctionPackageContentResponse> putSolFunctionPackageContent(
            PutSolFunctionPackageContentRequest putSolFunctionPackageContentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putSolFunctionPackageContentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putSolFunctionPackageContentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutSolFunctionPackageContent");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutSolFunctionPackageContentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutSolFunctionPackageContentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutSolFunctionPackageContentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutSolFunctionPackageContentRequest, PutSolFunctionPackageContentResponse>()
                            .withOperationName("PutSolFunctionPackageContent").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutSolFunctionPackageContentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putSolFunctionPackageContentRequest));
            CompletableFuture<PutSolFunctionPackageContentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Uploads the contents of a network package.
     * </p>
     * <p>
     * A network package is a .zip file in CSAR (Cloud Service Archive) format defines the function packages you want to
     * deploy and the Amazon Web Services infrastructure you want to deploy them on.
     * </p>
     *
     * @param putSolNetworkPackageContentRequest
     * @return A Java Future containing the result of the PutSolNetworkPackageContent operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException Unexpected error occurred. Problem on the server.</li>
     *         <li>ThrottlingException Exception caused by throttling.</li>
     *         <li>ValidationException Unable to process the request because the client provided input failed to satisfy
     *         request constraints.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>AccessDeniedException Insufficient permissions to make request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TnbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample TnbAsyncClient.PutSolNetworkPackageContent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/PutSolNetworkPackageContent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutSolNetworkPackageContentResponse> putSolNetworkPackageContent(
            PutSolNetworkPackageContentRequest putSolNetworkPackageContentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putSolNetworkPackageContentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putSolNetworkPackageContentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutSolNetworkPackageContent");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutSolNetworkPackageContentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutSolNetworkPackageContentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutSolNetworkPackageContentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutSolNetworkPackageContentRequest, PutSolNetworkPackageContentResponse>()
                            .withOperationName("PutSolNetworkPackageContent").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutSolNetworkPackageContentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putSolNetworkPackageContentRequest));
            CompletableFuture<PutSolNetworkPackageContentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Tags an AWS TNB resource.
     * </p>
     * <p>
     * A tag is a label that you assign to an Amazon Web Services resource. Each tag consists of a key and an optional
     * value. You can use tags to search and filter your resources or track your Amazon Web Services costs.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException Unexpected error occurred. Problem on the server.</li>
     *         <li>ThrottlingException Exception caused by throttling.</li>
     *         <li>ValidationException Unable to process the request because the client provided input failed to satisfy
     *         request constraints.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>AccessDeniedException Insufficient permissions to make request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TnbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample TnbAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Terminates a network instance.
     * </p>
     * <p>
     * A network instance is a single network created in Amazon Web Services TNB that can be deployed and on which
     * life-cycle operations (like terminate, update, and delete) can be performed.
     * </p>
     * <p>
     * You must terminate a network instance before you can delete it.
     * </p>
     *
     * @param terminateSolNetworkInstanceRequest
     * @return A Java Future containing the result of the TerminateSolNetworkInstance operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException Unexpected error occurred. Problem on the server.</li>
     *         <li>ServiceQuotaExceededException Service quotas have been exceeded.</li>
     *         <li>ThrottlingException Exception caused by throttling.</li>
     *         <li>ValidationException Unable to process the request because the client provided input failed to satisfy
     *         request constraints.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>AccessDeniedException Insufficient permissions to make request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TnbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample TnbAsyncClient.TerminateSolNetworkInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/TerminateSolNetworkInstance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<TerminateSolNetworkInstanceResponse> terminateSolNetworkInstance(
            TerminateSolNetworkInstanceRequest terminateSolNetworkInstanceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(terminateSolNetworkInstanceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, terminateSolNetworkInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TerminateSolNetworkInstance");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TerminateSolNetworkInstanceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, TerminateSolNetworkInstanceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TerminateSolNetworkInstanceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TerminateSolNetworkInstanceRequest, TerminateSolNetworkInstanceResponse>()
                            .withOperationName("TerminateSolNetworkInstance").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TerminateSolNetworkInstanceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(terminateSolNetworkInstanceRequest));
            CompletableFuture<TerminateSolNetworkInstanceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Untags an AWS TNB resource.
     * </p>
     * <p>
     * A tag is a label that you assign to an Amazon Web Services resource. Each tag consists of a key and an optional
     * value. You can use tags to search and filter your resources or track your Amazon Web Services costs.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException Unexpected error occurred. Problem on the server.</li>
     *         <li>ThrottlingException Exception caused by throttling.</li>
     *         <li>ValidationException Unable to process the request because the client provided input failed to satisfy
     *         request constraints.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>AccessDeniedException Insufficient permissions to make request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TnbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample TnbAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the operational state of function package.
     * </p>
     * <p>
     * A function package is a .zip file in CSAR (Cloud Service Archive) format that contains a network function (an
     * ETSI standard telecommunication application) and function package descriptor that uses the TOSCA standard to
     * describe how the network functions should run on your network.
     * </p>
     *
     * @param updateSolFunctionPackageRequest
     * @return A Java Future containing the result of the UpdateSolFunctionPackage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException Unexpected error occurred. Problem on the server.</li>
     *         <li>ThrottlingException Exception caused by throttling.</li>
     *         <li>ValidationException Unable to process the request because the client provided input failed to satisfy
     *         request constraints.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>AccessDeniedException Insufficient permissions to make request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TnbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample TnbAsyncClient.UpdateSolFunctionPackage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/UpdateSolFunctionPackage" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateSolFunctionPackageResponse> updateSolFunctionPackage(
            UpdateSolFunctionPackageRequest updateSolFunctionPackageRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateSolFunctionPackageRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSolFunctionPackageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSolFunctionPackage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateSolFunctionPackageResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateSolFunctionPackageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateSolFunctionPackageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateSolFunctionPackageRequest, UpdateSolFunctionPackageResponse>()
                            .withOperationName("UpdateSolFunctionPackage").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateSolFunctionPackageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateSolFunctionPackageRequest));
            CompletableFuture<UpdateSolFunctionPackageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Update a network instance.
     * </p>
     * <p>
     * A network instance is a single network created in Amazon Web Services TNB that can be deployed and on which
     * life-cycle operations (like terminate, update, and delete) can be performed.
     * </p>
     * <p>
     * Choose the <i>updateType</i> parameter to target the necessary update of the network instance.
     * </p>
     *
     * @param updateSolNetworkInstanceRequest
     * @return A Java Future containing the result of the UpdateSolNetworkInstance operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException Unexpected error occurred. Problem on the server.</li>
     *         <li>ServiceQuotaExceededException Service quotas have been exceeded.</li>
     *         <li>ThrottlingException Exception caused by throttling.</li>
     *         <li>ValidationException Unable to process the request because the client provided input failed to satisfy
     *         request constraints.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>AccessDeniedException Insufficient permissions to make request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TnbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample TnbAsyncClient.UpdateSolNetworkInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/UpdateSolNetworkInstance" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateSolNetworkInstanceResponse> updateSolNetworkInstance(
            UpdateSolNetworkInstanceRequest updateSolNetworkInstanceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateSolNetworkInstanceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSolNetworkInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSolNetworkInstance");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateSolNetworkInstanceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateSolNetworkInstanceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateSolNetworkInstanceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateSolNetworkInstanceRequest, UpdateSolNetworkInstanceResponse>()
                            .withOperationName("UpdateSolNetworkInstance").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateSolNetworkInstanceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateSolNetworkInstanceRequest));
            CompletableFuture<UpdateSolNetworkInstanceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the operational state of a network package.
     * </p>
     * <p>
     * A network package is a .zip file in CSAR (Cloud Service Archive) format defines the function packages you want to
     * deploy and the Amazon Web Services infrastructure you want to deploy them on.
     * </p>
     * <p>
     * A network service descriptor is a .yaml file in a network package that uses the TOSCA standard to describe the
     * network functions you want to deploy and the Amazon Web Services infrastructure you want to deploy the network
     * functions on.
     * </p>
     *
     * @param updateSolNetworkPackageRequest
     * @return A Java Future containing the result of the UpdateSolNetworkPackage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException Unexpected error occurred. Problem on the server.</li>
     *         <li>ThrottlingException Exception caused by throttling.</li>
     *         <li>ValidationException Unable to process the request because the client provided input failed to satisfy
     *         request constraints.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>AccessDeniedException Insufficient permissions to make request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TnbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample TnbAsyncClient.UpdateSolNetworkPackage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/UpdateSolNetworkPackage" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateSolNetworkPackageResponse> updateSolNetworkPackage(
            UpdateSolNetworkPackageRequest updateSolNetworkPackageRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateSolNetworkPackageRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSolNetworkPackageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSolNetworkPackage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateSolNetworkPackageResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateSolNetworkPackageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateSolNetworkPackageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateSolNetworkPackageRequest, UpdateSolNetworkPackageResponse>()
                            .withOperationName("UpdateSolNetworkPackage").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateSolNetworkPackageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateSolNetworkPackageRequest));
            CompletableFuture<UpdateSolNetworkPackageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Validates function package content. This can be used as a dry run before uploading function package content with
     * <a href="https://docs.aws.amazon.com/tnb/latest/APIReference/API_PutSolFunctionPackageContent.html">
     * PutSolFunctionPackageContent</a>.
     * </p>
     * <p>
     * A function package is a .zip file in CSAR (Cloud Service Archive) format that contains a network function (an
     * ETSI standard telecommunication application) and function package descriptor that uses the TOSCA standard to
     * describe how the network functions should run on your network.
     * </p>
     *
     * @param validateSolFunctionPackageContentRequest
     * @return A Java Future containing the result of the ValidateSolFunctionPackageContent operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException Unexpected error occurred. Problem on the server.</li>
     *         <li>ThrottlingException Exception caused by throttling.</li>
     *         <li>ValidationException Unable to process the request because the client provided input failed to satisfy
     *         request constraints.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>AccessDeniedException Insufficient permissions to make request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TnbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample TnbAsyncClient.ValidateSolFunctionPackageContent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/ValidateSolFunctionPackageContent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ValidateSolFunctionPackageContentResponse> validateSolFunctionPackageContent(
            ValidateSolFunctionPackageContentRequest validateSolFunctionPackageContentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(validateSolFunctionPackageContentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                validateSolFunctionPackageContentRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ValidateSolFunctionPackageContent");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ValidateSolFunctionPackageContentResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ValidateSolFunctionPackageContentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ValidateSolFunctionPackageContentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ValidateSolFunctionPackageContentRequest, ValidateSolFunctionPackageContentResponse>()
                            .withOperationName("ValidateSolFunctionPackageContent").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ValidateSolFunctionPackageContentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(validateSolFunctionPackageContentRequest));
            CompletableFuture<ValidateSolFunctionPackageContentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Validates network package content. This can be used as a dry run before uploading network package content with <a
     * href="https://docs.aws.amazon.com/tnb/latest/APIReference/API_PutSolNetworkPackageContent.html">
     * PutSolNetworkPackageContent</a>.
     * </p>
     * <p>
     * A network package is a .zip file in CSAR (Cloud Service Archive) format defines the function packages you want to
     * deploy and the Amazon Web Services infrastructure you want to deploy them on.
     * </p>
     *
     * @param validateSolNetworkPackageContentRequest
     * @return A Java Future containing the result of the ValidateSolNetworkPackageContent operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException Unexpected error occurred. Problem on the server.</li>
     *         <li>ThrottlingException Exception caused by throttling.</li>
     *         <li>ValidationException Unable to process the request because the client provided input failed to satisfy
     *         request constraints.</li>
     *         <li>ResourceNotFoundException Request references a resource that doesn't exist.</li>
     *         <li>AccessDeniedException Insufficient permissions to make request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>TnbException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample TnbAsyncClient.ValidateSolNetworkPackageContent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/ValidateSolNetworkPackageContent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ValidateSolNetworkPackageContentResponse> validateSolNetworkPackageContent(
            ValidateSolNetworkPackageContentRequest validateSolNetworkPackageContentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(validateSolNetworkPackageContentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                validateSolNetworkPackageContentRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ValidateSolNetworkPackageContent");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ValidateSolNetworkPackageContentResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ValidateSolNetworkPackageContentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ValidateSolNetworkPackageContentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ValidateSolNetworkPackageContentRequest, ValidateSolNetworkPackageContentResponse>()
                            .withOperationName("ValidateSolNetworkPackageContent").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ValidateSolNetworkPackageContentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(validateSolNetworkPackageContentRequest));
            CompletableFuture<ValidateSolNetworkPackageContentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final TnbServiceClientConfiguration serviceClientConfiguration() {
        return new TnbServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(TnbException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        TnbServiceClientConfigurationBuilder serviceConfigBuilder = new TnbServiceClientConfigurationBuilder(configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
