/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.voiceid;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.voiceid.internal.VoiceIdServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.voiceid.model.AccessDeniedException;
import software.amazon.awssdk.services.voiceid.model.AssociateFraudsterRequest;
import software.amazon.awssdk.services.voiceid.model.AssociateFraudsterResponse;
import software.amazon.awssdk.services.voiceid.model.ConflictException;
import software.amazon.awssdk.services.voiceid.model.CreateDomainRequest;
import software.amazon.awssdk.services.voiceid.model.CreateDomainResponse;
import software.amazon.awssdk.services.voiceid.model.CreateWatchlistRequest;
import software.amazon.awssdk.services.voiceid.model.CreateWatchlistResponse;
import software.amazon.awssdk.services.voiceid.model.DeleteDomainRequest;
import software.amazon.awssdk.services.voiceid.model.DeleteDomainResponse;
import software.amazon.awssdk.services.voiceid.model.DeleteFraudsterRequest;
import software.amazon.awssdk.services.voiceid.model.DeleteFraudsterResponse;
import software.amazon.awssdk.services.voiceid.model.DeleteSpeakerRequest;
import software.amazon.awssdk.services.voiceid.model.DeleteSpeakerResponse;
import software.amazon.awssdk.services.voiceid.model.DeleteWatchlistRequest;
import software.amazon.awssdk.services.voiceid.model.DeleteWatchlistResponse;
import software.amazon.awssdk.services.voiceid.model.DescribeDomainRequest;
import software.amazon.awssdk.services.voiceid.model.DescribeDomainResponse;
import software.amazon.awssdk.services.voiceid.model.DescribeFraudsterRegistrationJobRequest;
import software.amazon.awssdk.services.voiceid.model.DescribeFraudsterRegistrationJobResponse;
import software.amazon.awssdk.services.voiceid.model.DescribeFraudsterRequest;
import software.amazon.awssdk.services.voiceid.model.DescribeFraudsterResponse;
import software.amazon.awssdk.services.voiceid.model.DescribeSpeakerEnrollmentJobRequest;
import software.amazon.awssdk.services.voiceid.model.DescribeSpeakerEnrollmentJobResponse;
import software.amazon.awssdk.services.voiceid.model.DescribeSpeakerRequest;
import software.amazon.awssdk.services.voiceid.model.DescribeSpeakerResponse;
import software.amazon.awssdk.services.voiceid.model.DescribeWatchlistRequest;
import software.amazon.awssdk.services.voiceid.model.DescribeWatchlistResponse;
import software.amazon.awssdk.services.voiceid.model.DisassociateFraudsterRequest;
import software.amazon.awssdk.services.voiceid.model.DisassociateFraudsterResponse;
import software.amazon.awssdk.services.voiceid.model.EvaluateSessionRequest;
import software.amazon.awssdk.services.voiceid.model.EvaluateSessionResponse;
import software.amazon.awssdk.services.voiceid.model.InternalServerException;
import software.amazon.awssdk.services.voiceid.model.ListDomainsRequest;
import software.amazon.awssdk.services.voiceid.model.ListDomainsResponse;
import software.amazon.awssdk.services.voiceid.model.ListFraudsterRegistrationJobsRequest;
import software.amazon.awssdk.services.voiceid.model.ListFraudsterRegistrationJobsResponse;
import software.amazon.awssdk.services.voiceid.model.ListFraudstersRequest;
import software.amazon.awssdk.services.voiceid.model.ListFraudstersResponse;
import software.amazon.awssdk.services.voiceid.model.ListSpeakerEnrollmentJobsRequest;
import software.amazon.awssdk.services.voiceid.model.ListSpeakerEnrollmentJobsResponse;
import software.amazon.awssdk.services.voiceid.model.ListSpeakersRequest;
import software.amazon.awssdk.services.voiceid.model.ListSpeakersResponse;
import software.amazon.awssdk.services.voiceid.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.voiceid.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.voiceid.model.ListWatchlistsRequest;
import software.amazon.awssdk.services.voiceid.model.ListWatchlistsResponse;
import software.amazon.awssdk.services.voiceid.model.OptOutSpeakerRequest;
import software.amazon.awssdk.services.voiceid.model.OptOutSpeakerResponse;
import software.amazon.awssdk.services.voiceid.model.ResourceNotFoundException;
import software.amazon.awssdk.services.voiceid.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.voiceid.model.StartFraudsterRegistrationJobRequest;
import software.amazon.awssdk.services.voiceid.model.StartFraudsterRegistrationJobResponse;
import software.amazon.awssdk.services.voiceid.model.StartSpeakerEnrollmentJobRequest;
import software.amazon.awssdk.services.voiceid.model.StartSpeakerEnrollmentJobResponse;
import software.amazon.awssdk.services.voiceid.model.TagResourceRequest;
import software.amazon.awssdk.services.voiceid.model.TagResourceResponse;
import software.amazon.awssdk.services.voiceid.model.ThrottlingException;
import software.amazon.awssdk.services.voiceid.model.UntagResourceRequest;
import software.amazon.awssdk.services.voiceid.model.UntagResourceResponse;
import software.amazon.awssdk.services.voiceid.model.UpdateDomainRequest;
import software.amazon.awssdk.services.voiceid.model.UpdateDomainResponse;
import software.amazon.awssdk.services.voiceid.model.UpdateWatchlistRequest;
import software.amazon.awssdk.services.voiceid.model.UpdateWatchlistResponse;
import software.amazon.awssdk.services.voiceid.model.ValidationException;
import software.amazon.awssdk.services.voiceid.model.VoiceIdException;
import software.amazon.awssdk.services.voiceid.transform.AssociateFraudsterRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.CreateDomainRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.CreateWatchlistRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.DeleteDomainRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.DeleteFraudsterRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.DeleteSpeakerRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.DeleteWatchlistRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.DescribeDomainRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.DescribeFraudsterRegistrationJobRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.DescribeFraudsterRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.DescribeSpeakerEnrollmentJobRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.DescribeSpeakerRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.DescribeWatchlistRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.DisassociateFraudsterRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.EvaluateSessionRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.ListDomainsRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.ListFraudsterRegistrationJobsRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.ListFraudstersRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.ListSpeakerEnrollmentJobsRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.ListSpeakersRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.ListWatchlistsRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.OptOutSpeakerRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.StartFraudsterRegistrationJobRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.StartSpeakerEnrollmentJobRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.UpdateDomainRequestMarshaller;
import software.amazon.awssdk.services.voiceid.transform.UpdateWatchlistRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link VoiceIdAsyncClient}.
 *
 * @see VoiceIdAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultVoiceIdAsyncClient implements VoiceIdAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultVoiceIdAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultVoiceIdAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Associates the fraudsters with the watchlist specified in the same domain.
     * </p>
     *
     * @param associateFraudsterRequest
     * @return A Java Future containing the result of the AssociateFraudster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The request exceeded the service quota. Refer to <a
     *         href="https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html#voiceid-quotas"
     *         >Voice ID Service Quotas</a> and try your request again.</li>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>ConflictException The request failed due to a conflict. Check the <code>ConflictType</code> and error
     *         message for more details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.AssociateFraudster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/AssociateFraudster" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateFraudsterResponse> associateFraudster(AssociateFraudsterRequest associateFraudsterRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(associateFraudsterRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateFraudsterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateFraudster");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateFraudsterResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociateFraudsterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateFraudsterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateFraudsterRequest, AssociateFraudsterResponse>()
                            .withOperationName("AssociateFraudster").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new AssociateFraudsterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(associateFraudsterRequest));
            CompletableFuture<AssociateFraudsterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a domain that contains all Amazon Connect Voice ID data, such as speakers, fraudsters, customer audio,
     * and voiceprints. Every domain is created with a default watchlist that fraudsters can be a part of.
     * </p>
     *
     * @param createDomainRequest
     * @return A Java Future containing the result of the CreateDomain operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The request exceeded the service quota. Refer to <a
     *         href="https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html#voiceid-quotas"
     *         >Voice ID Service Quotas</a> and try your request again.</li>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>ConflictException The request failed due to a conflict. Check the <code>ConflictType</code> and error
     *         message for more details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.CreateDomain
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/CreateDomain" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDomainResponse> createDomain(CreateDomainRequest createDomainRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createDomainRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDomainRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDomain");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDomainResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateDomainResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDomainResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDomainRequest, CreateDomainResponse>()
                            .withOperationName("CreateDomain").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateDomainRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createDomainRequest));
            CompletableFuture<CreateDomainResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a watchlist that fraudsters can be a part of.
     * </p>
     *
     * @param createWatchlistRequest
     * @return A Java Future containing the result of the CreateWatchlist operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The request exceeded the service quota. Refer to <a
     *         href="https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html#voiceid-quotas"
     *         >Voice ID Service Quotas</a> and try your request again.</li>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>ConflictException The request failed due to a conflict. Check the <code>ConflictType</code> and error
     *         message for more details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.CreateWatchlist
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/CreateWatchlist" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateWatchlistResponse> createWatchlist(CreateWatchlistRequest createWatchlistRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createWatchlistRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createWatchlistRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateWatchlist");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateWatchlistResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateWatchlistResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateWatchlistResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateWatchlistRequest, CreateWatchlistResponse>()
                            .withOperationName("CreateWatchlist").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateWatchlistRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createWatchlistRequest));
            CompletableFuture<CreateWatchlistResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified domain from Voice ID.
     * </p>
     *
     * @param deleteDomainRequest
     * @return A Java Future containing the result of the DeleteDomain operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>ConflictException The request failed due to a conflict. Check the <code>ConflictType</code> and error
     *         message for more details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.DeleteDomain
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/DeleteDomain" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDomainResponse> deleteDomain(DeleteDomainRequest deleteDomainRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteDomainRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDomainRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDomain");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDomainResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteDomainResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDomainResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDomainRequest, DeleteDomainResponse>()
                            .withOperationName("DeleteDomain").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteDomainRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteDomainRequest));
            CompletableFuture<DeleteDomainResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified fraudster from Voice ID. This action disassociates the fraudster from any watchlists it is
     * a part of.
     * </p>
     *
     * @param deleteFraudsterRequest
     * @return A Java Future containing the result of the DeleteFraudster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>ConflictException The request failed due to a conflict. Check the <code>ConflictType</code> and error
     *         message for more details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.DeleteFraudster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/DeleteFraudster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteFraudsterResponse> deleteFraudster(DeleteFraudsterRequest deleteFraudsterRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteFraudsterRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteFraudsterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteFraudster");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteFraudsterResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteFraudsterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteFraudsterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteFraudsterRequest, DeleteFraudsterResponse>()
                            .withOperationName("DeleteFraudster").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteFraudsterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteFraudsterRequest));
            CompletableFuture<DeleteFraudsterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified speaker from Voice ID.
     * </p>
     *
     * @param deleteSpeakerRequest
     * @return A Java Future containing the result of the DeleteSpeaker operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>ConflictException The request failed due to a conflict. Check the <code>ConflictType</code> and error
     *         message for more details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.DeleteSpeaker
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/DeleteSpeaker" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSpeakerResponse> deleteSpeaker(DeleteSpeakerRequest deleteSpeakerRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteSpeakerRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSpeakerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSpeaker");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSpeakerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteSpeakerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteSpeakerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSpeakerRequest, DeleteSpeakerResponse>()
                            .withOperationName("DeleteSpeaker").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteSpeakerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteSpeakerRequest));
            CompletableFuture<DeleteSpeakerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified watchlist from Voice ID. This API throws an exception when there are fraudsters in the
     * watchlist that you are trying to delete. You must delete the fraudsters, and then delete the watchlist. Every
     * domain has a default watchlist which cannot be deleted.
     * </p>
     *
     * @param deleteWatchlistRequest
     * @return A Java Future containing the result of the DeleteWatchlist operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>ConflictException The request failed due to a conflict. Check the <code>ConflictType</code> and error
     *         message for more details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.DeleteWatchlist
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/DeleteWatchlist" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteWatchlistResponse> deleteWatchlist(DeleteWatchlistRequest deleteWatchlistRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteWatchlistRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteWatchlistRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteWatchlist");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteWatchlistResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteWatchlistResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteWatchlistResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteWatchlistRequest, DeleteWatchlistResponse>()
                            .withOperationName("DeleteWatchlist").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteWatchlistRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteWatchlistRequest));
            CompletableFuture<DeleteWatchlistResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the specified domain.
     * </p>
     *
     * @param describeDomainRequest
     * @return A Java Future containing the result of the DescribeDomain operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.DescribeDomain
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/DescribeDomain" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeDomainResponse> describeDomain(DescribeDomainRequest describeDomainRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeDomainRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDomainRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDomain");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDomainResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDomainResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDomainResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDomainRequest, DescribeDomainResponse>()
                            .withOperationName("DescribeDomain").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeDomainRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeDomainRequest));
            CompletableFuture<DescribeDomainResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the specified fraudster.
     * </p>
     *
     * @param describeFraudsterRequest
     * @return A Java Future containing the result of the DescribeFraudster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.DescribeFraudster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/DescribeFraudster" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeFraudsterResponse> describeFraudster(DescribeFraudsterRequest describeFraudsterRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeFraudsterRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeFraudsterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeFraudster");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeFraudsterResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeFraudsterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeFraudsterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeFraudsterRequest, DescribeFraudsterResponse>()
                            .withOperationName("DescribeFraudster").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeFraudsterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeFraudsterRequest));
            CompletableFuture<DescribeFraudsterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the specified fraudster registration job.
     * </p>
     *
     * @param describeFraudsterRegistrationJobRequest
     * @return A Java Future containing the result of the DescribeFraudsterRegistrationJob operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.DescribeFraudsterRegistrationJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/DescribeFraudsterRegistrationJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeFraudsterRegistrationJobResponse> describeFraudsterRegistrationJob(
            DescribeFraudsterRegistrationJobRequest describeFraudsterRegistrationJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeFraudsterRegistrationJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeFraudsterRegistrationJobRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeFraudsterRegistrationJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeFraudsterRegistrationJobResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeFraudsterRegistrationJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeFraudsterRegistrationJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeFraudsterRegistrationJobRequest, DescribeFraudsterRegistrationJobResponse>()
                            .withOperationName("DescribeFraudsterRegistrationJob").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeFraudsterRegistrationJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeFraudsterRegistrationJobRequest));
            CompletableFuture<DescribeFraudsterRegistrationJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the specified speaker.
     * </p>
     *
     * @param describeSpeakerRequest
     * @return A Java Future containing the result of the DescribeSpeaker operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.DescribeSpeaker
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/DescribeSpeaker" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeSpeakerResponse> describeSpeaker(DescribeSpeakerRequest describeSpeakerRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeSpeakerRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeSpeakerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeSpeaker");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeSpeakerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeSpeakerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeSpeakerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeSpeakerRequest, DescribeSpeakerResponse>()
                            .withOperationName("DescribeSpeaker").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeSpeakerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeSpeakerRequest));
            CompletableFuture<DescribeSpeakerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the specified speaker enrollment job.
     * </p>
     *
     * @param describeSpeakerEnrollmentJobRequest
     * @return A Java Future containing the result of the DescribeSpeakerEnrollmentJob operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.DescribeSpeakerEnrollmentJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/DescribeSpeakerEnrollmentJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeSpeakerEnrollmentJobResponse> describeSpeakerEnrollmentJob(
            DescribeSpeakerEnrollmentJobRequest describeSpeakerEnrollmentJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeSpeakerEnrollmentJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeSpeakerEnrollmentJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeSpeakerEnrollmentJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeSpeakerEnrollmentJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeSpeakerEnrollmentJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeSpeakerEnrollmentJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeSpeakerEnrollmentJobRequest, DescribeSpeakerEnrollmentJobResponse>()
                            .withOperationName("DescribeSpeakerEnrollmentJob").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeSpeakerEnrollmentJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeSpeakerEnrollmentJobRequest));
            CompletableFuture<DescribeSpeakerEnrollmentJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the specified watchlist.
     * </p>
     *
     * @param describeWatchlistRequest
     * @return A Java Future containing the result of the DescribeWatchlist operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.DescribeWatchlist
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/DescribeWatchlist" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeWatchlistResponse> describeWatchlist(DescribeWatchlistRequest describeWatchlistRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeWatchlistRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeWatchlistRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeWatchlist");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeWatchlistResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeWatchlistResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeWatchlistResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeWatchlistRequest, DescribeWatchlistResponse>()
                            .withOperationName("DescribeWatchlist").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeWatchlistRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeWatchlistRequest));
            CompletableFuture<DescribeWatchlistResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disassociates the fraudsters from the watchlist specified. Voice ID always expects a fraudster to be a part of at
     * least one watchlist. If you try to disassociate a fraudster from its only watchlist, a
     * <code>ValidationException</code> is thrown.
     * </p>
     *
     * @param disassociateFraudsterRequest
     * @return A Java Future containing the result of the DisassociateFraudster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>ConflictException The request failed due to a conflict. Check the <code>ConflictType</code> and error
     *         message for more details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.DisassociateFraudster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/DisassociateFraudster"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateFraudsterResponse> disassociateFraudster(
            DisassociateFraudsterRequest disassociateFraudsterRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disassociateFraudsterRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateFraudsterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateFraudster");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateFraudsterResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateFraudsterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateFraudsterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateFraudsterRequest, DisassociateFraudsterResponse>()
                            .withOperationName("DisassociateFraudster").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DisassociateFraudsterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(disassociateFraudsterRequest));
            CompletableFuture<DisassociateFraudsterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Evaluates a specified session based on audio data accumulated during a streaming Amazon Connect Voice ID call.
     * </p>
     *
     * @param evaluateSessionRequest
     * @return A Java Future containing the result of the EvaluateSession operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>ConflictException The request failed due to a conflict. Check the <code>ConflictType</code> and error
     *         message for more details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.EvaluateSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/EvaluateSession" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<EvaluateSessionResponse> evaluateSession(EvaluateSessionRequest evaluateSessionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(evaluateSessionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, evaluateSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EvaluateSession");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<EvaluateSessionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, EvaluateSessionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<EvaluateSessionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<EvaluateSessionRequest, EvaluateSessionResponse>()
                            .withOperationName("EvaluateSession").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new EvaluateSessionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(evaluateSessionRequest));
            CompletableFuture<EvaluateSessionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all the domains in the Amazon Web Services account.
     * </p>
     *
     * @param listDomainsRequest
     * @return A Java Future containing the result of the ListDomains operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.ListDomains
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/ListDomains" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListDomainsResponse> listDomains(ListDomainsRequest listDomainsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDomainsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDomainsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDomains");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDomainsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListDomainsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDomainsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDomainsRequest, ListDomainsResponse>()
                            .withOperationName("ListDomains").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListDomainsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listDomainsRequest));
            CompletableFuture<ListDomainsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all the fraudster registration jobs in the domain with the given <code>JobStatus</code>. If
     * <code>JobStatus</code> is not provided, this lists all fraudster registration jobs in the given domain.
     * </p>
     *
     * @param listFraudsterRegistrationJobsRequest
     * @return A Java Future containing the result of the ListFraudsterRegistrationJobs operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.ListFraudsterRegistrationJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/ListFraudsterRegistrationJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListFraudsterRegistrationJobsResponse> listFraudsterRegistrationJobs(
            ListFraudsterRegistrationJobsRequest listFraudsterRegistrationJobsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listFraudsterRegistrationJobsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listFraudsterRegistrationJobsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListFraudsterRegistrationJobs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListFraudsterRegistrationJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListFraudsterRegistrationJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListFraudsterRegistrationJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListFraudsterRegistrationJobsRequest, ListFraudsterRegistrationJobsResponse>()
                            .withOperationName("ListFraudsterRegistrationJobs").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListFraudsterRegistrationJobsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listFraudsterRegistrationJobsRequest));
            CompletableFuture<ListFraudsterRegistrationJobsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all fraudsters in a specified watchlist or domain.
     * </p>
     *
     * @param listFraudstersRequest
     * @return A Java Future containing the result of the ListFraudsters operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.ListFraudsters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/ListFraudsters" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListFraudstersResponse> listFraudsters(ListFraudstersRequest listFraudstersRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listFraudstersRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listFraudstersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListFraudsters");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListFraudstersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListFraudstersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListFraudstersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListFraudstersRequest, ListFraudstersResponse>()
                            .withOperationName("ListFraudsters").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListFraudstersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listFraudstersRequest));
            CompletableFuture<ListFraudstersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all the speaker enrollment jobs in the domain with the specified <code>JobStatus</code>. If
     * <code>JobStatus</code> is not provided, this lists all jobs with all possible speaker enrollment job statuses.
     * </p>
     *
     * @param listSpeakerEnrollmentJobsRequest
     * @return A Java Future containing the result of the ListSpeakerEnrollmentJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.ListSpeakerEnrollmentJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/ListSpeakerEnrollmentJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSpeakerEnrollmentJobsResponse> listSpeakerEnrollmentJobs(
            ListSpeakerEnrollmentJobsRequest listSpeakerEnrollmentJobsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSpeakerEnrollmentJobsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSpeakerEnrollmentJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSpeakerEnrollmentJobs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSpeakerEnrollmentJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListSpeakerEnrollmentJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSpeakerEnrollmentJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSpeakerEnrollmentJobsRequest, ListSpeakerEnrollmentJobsResponse>()
                            .withOperationName("ListSpeakerEnrollmentJobs").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListSpeakerEnrollmentJobsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listSpeakerEnrollmentJobsRequest));
            CompletableFuture<ListSpeakerEnrollmentJobsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all speakers in a specified domain.
     * </p>
     *
     * @param listSpeakersRequest
     * @return A Java Future containing the result of the ListSpeakers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.ListSpeakers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/ListSpeakers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListSpeakersResponse> listSpeakers(ListSpeakersRequest listSpeakersRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSpeakersRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSpeakersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSpeakers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSpeakersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListSpeakersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSpeakersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSpeakersRequest, ListSpeakersResponse>()
                            .withOperationName("ListSpeakers").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListSpeakersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listSpeakersRequest));
            CompletableFuture<ListSpeakersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all tags associated with a specified Voice ID resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all watchlists in a specified domain.
     * </p>
     *
     * @param listWatchlistsRequest
     * @return A Java Future containing the result of the ListWatchlists operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.ListWatchlists
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/ListWatchlists" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListWatchlistsResponse> listWatchlists(ListWatchlistsRequest listWatchlistsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listWatchlistsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listWatchlistsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListWatchlists");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListWatchlistsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListWatchlistsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListWatchlistsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListWatchlistsRequest, ListWatchlistsResponse>()
                            .withOperationName("ListWatchlists").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListWatchlistsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listWatchlistsRequest));
            CompletableFuture<ListWatchlistsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Opts out a speaker from Voice ID. A speaker can be opted out regardless of whether or not they already exist in
     * Voice ID. If they don't yet exist, a new speaker is created in an opted out state. If they already exist, their
     * existing status is overridden and they are opted out. Enrollment and evaluation authentication requests are
     * rejected for opted out speakers, and opted out speakers have no voice embeddings stored in Voice ID.
     * </p>
     *
     * @param optOutSpeakerRequest
     * @return A Java Future containing the result of the OptOutSpeaker operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The request exceeded the service quota. Refer to <a
     *         href="https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html#voiceid-quotas"
     *         >Voice ID Service Quotas</a> and try your request again.</li>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>ConflictException The request failed due to a conflict. Check the <code>ConflictType</code> and error
     *         message for more details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.OptOutSpeaker
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/OptOutSpeaker" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<OptOutSpeakerResponse> optOutSpeaker(OptOutSpeakerRequest optOutSpeakerRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(optOutSpeakerRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, optOutSpeakerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "OptOutSpeaker");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<OptOutSpeakerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    OptOutSpeakerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<OptOutSpeakerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<OptOutSpeakerRequest, OptOutSpeakerResponse>()
                            .withOperationName("OptOutSpeaker").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new OptOutSpeakerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(optOutSpeakerRequest));
            CompletableFuture<OptOutSpeakerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts a new batch fraudster registration job using provided details.
     * </p>
     *
     * @param startFraudsterRegistrationJobRequest
     * @return A Java Future containing the result of the StartFraudsterRegistrationJob operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The request exceeded the service quota. Refer to <a
     *         href="https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html#voiceid-quotas"
     *         >Voice ID Service Quotas</a> and try your request again.</li>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>ConflictException The request failed due to a conflict. Check the <code>ConflictType</code> and error
     *         message for more details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.StartFraudsterRegistrationJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/StartFraudsterRegistrationJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartFraudsterRegistrationJobResponse> startFraudsterRegistrationJob(
            StartFraudsterRegistrationJobRequest startFraudsterRegistrationJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startFraudsterRegistrationJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                startFraudsterRegistrationJobRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartFraudsterRegistrationJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartFraudsterRegistrationJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartFraudsterRegistrationJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartFraudsterRegistrationJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartFraudsterRegistrationJobRequest, StartFraudsterRegistrationJobResponse>()
                            .withOperationName("StartFraudsterRegistrationJob").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartFraudsterRegistrationJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startFraudsterRegistrationJobRequest));
            CompletableFuture<StartFraudsterRegistrationJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts a new batch speaker enrollment job using specified details.
     * </p>
     *
     * @param startSpeakerEnrollmentJobRequest
     * @return A Java Future containing the result of the StartSpeakerEnrollmentJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The request exceeded the service quota. Refer to <a
     *         href="https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html#voiceid-quotas"
     *         >Voice ID Service Quotas</a> and try your request again.</li>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>ConflictException The request failed due to a conflict. Check the <code>ConflictType</code> and error
     *         message for more details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.StartSpeakerEnrollmentJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/StartSpeakerEnrollmentJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartSpeakerEnrollmentJobResponse> startSpeakerEnrollmentJob(
            StartSpeakerEnrollmentJobRequest startSpeakerEnrollmentJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startSpeakerEnrollmentJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startSpeakerEnrollmentJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartSpeakerEnrollmentJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartSpeakerEnrollmentJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartSpeakerEnrollmentJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartSpeakerEnrollmentJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartSpeakerEnrollmentJobRequest, StartSpeakerEnrollmentJobResponse>()
                            .withOperationName("StartSpeakerEnrollmentJob").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartSpeakerEnrollmentJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startSpeakerEnrollmentJobRequest));
            CompletableFuture<StartSpeakerEnrollmentJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Tags a Voice ID resource with the provided list of tags.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>ConflictException The request failed due to a conflict. Check the <code>ConflictType</code> and error
     *         message for more details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes specified tags from a specified Amazon Connect Voice ID resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>ConflictException The request failed due to a conflict. Check the <code>ConflictType</code> and error
     *         message for more details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the specified domain. This API has clobber behavior, and clears and replaces all attributes. If an
     * optional field, such as 'Description' is not provided, it is removed from the domain.
     * </p>
     *
     * @param updateDomainRequest
     * @return A Java Future containing the result of the UpdateDomain operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>ConflictException The request failed due to a conflict. Check the <code>ConflictType</code> and error
     *         message for more details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.UpdateDomain
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/UpdateDomain" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDomainResponse> updateDomain(UpdateDomainRequest updateDomainRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateDomainRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDomainRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDomain");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDomainResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateDomainResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDomainResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDomainRequest, UpdateDomainResponse>()
                            .withOperationName("UpdateDomain").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateDomainRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateDomainRequest));
            CompletableFuture<UpdateDomainResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the specified watchlist. Every domain has a default watchlist which cannot be updated.
     * </p>
     *
     * @param updateWatchlistRequest
     * @return A Java Future containing the result of the UpdateWatchlist operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the <code>ResourceType</code>
     *         and error message for more details.</li>
     *         <li>ValidationException The request failed one or more validations; check the error message for more
     *         details.</li>
     *         <li>ConflictException The request failed due to a conflict. Check the <code>ConflictType</code> and error
     *         message for more details.</li>
     *         <li>InternalServerException The request failed due to an unknown error on the server side.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please slow down your request
     *         rate. Refer to <a href=
     *         "https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html##voiceid-api-quotas"
     *         > Amazon Connect Voice ID Service API throttling quotas </a> and try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Check the error
     *         message and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>VoiceIdException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample VoiceIdAsyncClient.UpdateWatchlist
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/voice-id-2021-09-27/UpdateWatchlist" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateWatchlistResponse> updateWatchlist(UpdateWatchlistRequest updateWatchlistRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateWatchlistRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateWatchlistRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Voice ID");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateWatchlist");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateWatchlistResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateWatchlistResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateWatchlistResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateWatchlistRequest, UpdateWatchlistResponse>()
                            .withOperationName("UpdateWatchlist").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateWatchlistRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateWatchlistRequest));
            CompletableFuture<UpdateWatchlistResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final VoiceIdServiceClientConfiguration serviceClientConfiguration() {
        return new VoiceIdServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(VoiceIdException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        VoiceIdServiceClientConfigurationBuilder serviceConfigBuilder = new VoiceIdServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
