/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.wafv2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A rule statement used to run the rules that are defined in a managed rule group. To use this, provide the vendor name
 * and the name of the rule group in this statement. You can retrieve the required names by calling
 * <a>ListAvailableManagedRuleGroups</a>.
 * </p>
 * <p>
 * You cannot nest a <code>ManagedRuleGroupStatement</code>, for example for use inside a <code>NotStatement</code> or
 * <code>OrStatement</code>. You cannot use a managed rule group inside another rule group. You can only reference a
 * managed rule group as a top-level statement within a rule that you define in a web ACL.
 * </p>
 * <note>
 * <p>
 * You are charged additional fees when you use the WAF Bot Control managed rule group
 * <code>AWSManagedRulesBotControlRuleSet</code>, the WAF Fraud Control account takeover prevention (ATP) managed rule
 * group <code>AWSManagedRulesATPRuleSet</code>, or the WAF Fraud Control account creation fraud prevention (ACFP)
 * managed rule group <code>AWSManagedRulesACFPRuleSet</code>. For more information, see <a
 * href="http://aws.amazon.com/waf/pricing/">WAF Pricing</a>.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ManagedRuleGroupStatement implements SdkPojo, Serializable,
        ToCopyableBuilder<ManagedRuleGroupStatement.Builder, ManagedRuleGroupStatement> {
    private static final SdkField<String> VENDOR_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VendorName").getter(getter(ManagedRuleGroupStatement::vendorName)).setter(setter(Builder::vendorName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VendorName").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(ManagedRuleGroupStatement::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Version")
            .getter(getter(ManagedRuleGroupStatement::version)).setter(setter(Builder::version))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Version").build()).build();

    private static final SdkField<List<ExcludedRule>> EXCLUDED_RULES_FIELD = SdkField
            .<List<ExcludedRule>> builder(MarshallingType.LIST)
            .memberName("ExcludedRules")
            .getter(getter(ManagedRuleGroupStatement::excludedRules))
            .setter(setter(Builder::excludedRules))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExcludedRules").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ExcludedRule> builder(MarshallingType.SDK_POJO)
                                            .constructor(ExcludedRule::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Statement> SCOPE_DOWN_STATEMENT_FIELD = SdkField.<Statement> builder(MarshallingType.SDK_POJO)
            .memberName("ScopeDownStatement").getter(getter(ManagedRuleGroupStatement::scopeDownStatement))
            .setter(setter(Builder::scopeDownStatement)).constructor(Statement::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScopeDownStatement").build())
            .build();

    private static final SdkField<List<ManagedRuleGroupConfig>> MANAGED_RULE_GROUP_CONFIGS_FIELD = SdkField
            .<List<ManagedRuleGroupConfig>> builder(MarshallingType.LIST)
            .memberName("ManagedRuleGroupConfigs")
            .getter(getter(ManagedRuleGroupStatement::managedRuleGroupConfigs))
            .setter(setter(Builder::managedRuleGroupConfigs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ManagedRuleGroupConfigs").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ManagedRuleGroupConfig> builder(MarshallingType.SDK_POJO)
                                            .constructor(ManagedRuleGroupConfig::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<RuleActionOverride>> RULE_ACTION_OVERRIDES_FIELD = SdkField
            .<List<RuleActionOverride>> builder(MarshallingType.LIST)
            .memberName("RuleActionOverrides")
            .getter(getter(ManagedRuleGroupStatement::ruleActionOverrides))
            .setter(setter(Builder::ruleActionOverrides))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RuleActionOverrides").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<RuleActionOverride> builder(MarshallingType.SDK_POJO)
                                            .constructor(RuleActionOverride::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(VENDOR_NAME_FIELD, NAME_FIELD,
            VERSION_FIELD, EXCLUDED_RULES_FIELD, SCOPE_DOWN_STATEMENT_FIELD, MANAGED_RULE_GROUP_CONFIGS_FIELD,
            RULE_ACTION_OVERRIDES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String vendorName;

    private final String name;

    private final String version;

    private final List<ExcludedRule> excludedRules;

    private final Statement scopeDownStatement;

    private final List<ManagedRuleGroupConfig> managedRuleGroupConfigs;

    private final List<RuleActionOverride> ruleActionOverrides;

    private ManagedRuleGroupStatement(BuilderImpl builder) {
        this.vendorName = builder.vendorName;
        this.name = builder.name;
        this.version = builder.version;
        this.excludedRules = builder.excludedRules;
        this.scopeDownStatement = builder.scopeDownStatement;
        this.managedRuleGroupConfigs = builder.managedRuleGroupConfigs;
        this.ruleActionOverrides = builder.ruleActionOverrides;
    }

    /**
     * <p>
     * The name of the managed rule group vendor. You use this, along with the rule group name, to identify a rule
     * group.
     * </p>
     * 
     * @return The name of the managed rule group vendor. You use this, along with the rule group name, to identify a
     *         rule group.
     */
    public final String vendorName() {
        return vendorName;
    }

    /**
     * <p>
     * The name of the managed rule group. You use this, along with the vendor name, to identify the rule group.
     * </p>
     * 
     * @return The name of the managed rule group. You use this, along with the vendor name, to identify the rule group.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The version of the managed rule group to use. If you specify this, the version setting is fixed until you change
     * it. If you don't specify this, WAF uses the vendor's default version, and then keeps the version at the vendor's
     * default when the vendor updates the managed rule group settings.
     * </p>
     * 
     * @return The version of the managed rule group to use. If you specify this, the version setting is fixed until you
     *         change it. If you don't specify this, WAF uses the vendor's default version, and then keeps the version
     *         at the vendor's default when the vendor updates the managed rule group settings.
     */
    public final String version() {
        return version;
    }

    /**
     * For responses, this returns true if the service returned a value for the ExcludedRules property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasExcludedRules() {
        return excludedRules != null && !(excludedRules instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Rules in the referenced rule group whose actions are set to <code>Count</code>.
     * </p>
     * <note>
     * <p>
     * Instead of this option, use <code>RuleActionOverrides</code>. It accepts any valid action setting, including
     * <code>Count</code>.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasExcludedRules} method.
     * </p>
     * 
     * @return Rules in the referenced rule group whose actions are set to <code>Count</code>. </p> <note>
     *         <p>
     *         Instead of this option, use <code>RuleActionOverrides</code>. It accepts any valid action setting,
     *         including <code>Count</code>.
     *         </p>
     */
    public final List<ExcludedRule> excludedRules() {
        return excludedRules;
    }

    /**
     * <p>
     * An optional nested statement that narrows the scope of the web requests that are evaluated by the managed rule
     * group. Requests are only evaluated by the rule group if they match the scope-down statement. You can use any
     * nestable <a>Statement</a> in the scope-down statement, and you can nest statements at any level, the same as you
     * can for a rule statement.
     * </p>
     * 
     * @return An optional nested statement that narrows the scope of the web requests that are evaluated by the managed
     *         rule group. Requests are only evaluated by the rule group if they match the scope-down statement. You can
     *         use any nestable <a>Statement</a> in the scope-down statement, and you can nest statements at any level,
     *         the same as you can for a rule statement.
     */
    public final Statement scopeDownStatement() {
        return scopeDownStatement;
    }

    /**
     * For responses, this returns true if the service returned a value for the ManagedRuleGroupConfigs property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasManagedRuleGroupConfigs() {
        return managedRuleGroupConfigs != null && !(managedRuleGroupConfigs instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Additional information that's used by a managed rule group. Many managed rule groups don't require this.
     * </p>
     * <p>
     * The rule groups used for intelligent threat mitigation require additional configuration:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Use the <code>AWSManagedRulesACFPRuleSet</code> configuration object to configure the account creation fraud
     * prevention managed rule group. The configuration includes the registration and sign-up pages of your application
     * and the locations in the account creation request payload of data, such as the user email and phone number
     * fields.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use the <code>AWSManagedRulesATPRuleSet</code> configuration object to configure the account takeover prevention
     * managed rule group. The configuration includes the sign-in page of your application and the locations in the
     * login request payload of data such as the username and password.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use the <code>AWSManagedRulesBotControlRuleSet</code> configuration object to configure the protection level that
     * you want the Bot Control rule group to use.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasManagedRuleGroupConfigs} method.
     * </p>
     * 
     * @return Additional information that's used by a managed rule group. Many managed rule groups don't require
     *         this.</p>
     *         <p>
     *         The rule groups used for intelligent threat mitigation require additional configuration:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Use the <code>AWSManagedRulesACFPRuleSet</code> configuration object to configure the account creation
     *         fraud prevention managed rule group. The configuration includes the registration and sign-up pages of
     *         your application and the locations in the account creation request payload of data, such as the user
     *         email and phone number fields.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Use the <code>AWSManagedRulesATPRuleSet</code> configuration object to configure the account takeover
     *         prevention managed rule group. The configuration includes the sign-in page of your application and the
     *         locations in the login request payload of data such as the username and password.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Use the <code>AWSManagedRulesBotControlRuleSet</code> configuration object to configure the protection
     *         level that you want the Bot Control rule group to use.
     *         </p>
     *         </li>
     */
    public final List<ManagedRuleGroupConfig> managedRuleGroupConfigs() {
        return managedRuleGroupConfigs;
    }

    /**
     * For responses, this returns true if the service returned a value for the RuleActionOverrides property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasRuleActionOverrides() {
        return ruleActionOverrides != null && !(ruleActionOverrides instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Action settings to use in the place of the rule actions that are configured inside the rule group. You specify
     * one override for each rule whose action you want to change.
     * </p>
     * <note>
     * <p>
     * Take care to verify the rule names in your overrides. If you provide a rule name that doesn't match the name of
     * any rule in the rule group, WAF doesn't return an error and doesn't apply the override setting.
     * </p>
     * </note>
     * <p>
     * You can use overrides for testing, for example you can override all of rule actions to <code>Count</code> and
     * then monitor the resulting count metrics to understand how the rule group would handle your web traffic. You can
     * also permanently override some or all actions, to modify how the rule group manages your web traffic.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRuleActionOverrides} method.
     * </p>
     * 
     * @return Action settings to use in the place of the rule actions that are configured inside the rule group. You
     *         specify one override for each rule whose action you want to change. </p> <note>
     *         <p>
     *         Take care to verify the rule names in your overrides. If you provide a rule name that doesn't match the
     *         name of any rule in the rule group, WAF doesn't return an error and doesn't apply the override setting.
     *         </p>
     *         </note>
     *         <p>
     *         You can use overrides for testing, for example you can override all of rule actions to <code>Count</code>
     *         and then monitor the resulting count metrics to understand how the rule group would handle your web
     *         traffic. You can also permanently override some or all actions, to modify how the rule group manages your
     *         web traffic.
     */
    public final List<RuleActionOverride> ruleActionOverrides() {
        return ruleActionOverrides;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(vendorName());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(version());
        hashCode = 31 * hashCode + Objects.hashCode(hasExcludedRules() ? excludedRules() : null);
        hashCode = 31 * hashCode + Objects.hashCode(scopeDownStatement());
        hashCode = 31 * hashCode + Objects.hashCode(hasManagedRuleGroupConfigs() ? managedRuleGroupConfigs() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasRuleActionOverrides() ? ruleActionOverrides() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ManagedRuleGroupStatement)) {
            return false;
        }
        ManagedRuleGroupStatement other = (ManagedRuleGroupStatement) obj;
        return Objects.equals(vendorName(), other.vendorName()) && Objects.equals(name(), other.name())
                && Objects.equals(version(), other.version()) && hasExcludedRules() == other.hasExcludedRules()
                && Objects.equals(excludedRules(), other.excludedRules())
                && Objects.equals(scopeDownStatement(), other.scopeDownStatement())
                && hasManagedRuleGroupConfigs() == other.hasManagedRuleGroupConfigs()
                && Objects.equals(managedRuleGroupConfigs(), other.managedRuleGroupConfigs())
                && hasRuleActionOverrides() == other.hasRuleActionOverrides()
                && Objects.equals(ruleActionOverrides(), other.ruleActionOverrides());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ManagedRuleGroupStatement").add("VendorName", vendorName()).add("Name", name())
                .add("Version", version()).add("ExcludedRules", hasExcludedRules() ? excludedRules() : null)
                .add("ScopeDownStatement", scopeDownStatement())
                .add("ManagedRuleGroupConfigs", hasManagedRuleGroupConfigs() ? managedRuleGroupConfigs() : null)
                .add("RuleActionOverrides", hasRuleActionOverrides() ? ruleActionOverrides() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "VendorName":
            return Optional.ofNullable(clazz.cast(vendorName()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Version":
            return Optional.ofNullable(clazz.cast(version()));
        case "ExcludedRules":
            return Optional.ofNullable(clazz.cast(excludedRules()));
        case "ScopeDownStatement":
            return Optional.ofNullable(clazz.cast(scopeDownStatement()));
        case "ManagedRuleGroupConfigs":
            return Optional.ofNullable(clazz.cast(managedRuleGroupConfigs()));
        case "RuleActionOverrides":
            return Optional.ofNullable(clazz.cast(ruleActionOverrides()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("VendorName", VENDOR_NAME_FIELD);
        map.put("Name", NAME_FIELD);
        map.put("Version", VERSION_FIELD);
        map.put("ExcludedRules", EXCLUDED_RULES_FIELD);
        map.put("ScopeDownStatement", SCOPE_DOWN_STATEMENT_FIELD);
        map.put("ManagedRuleGroupConfigs", MANAGED_RULE_GROUP_CONFIGS_FIELD);
        map.put("RuleActionOverrides", RULE_ACTION_OVERRIDES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ManagedRuleGroupStatement, T> g) {
        return obj -> g.apply((ManagedRuleGroupStatement) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ManagedRuleGroupStatement> {
        /**
         * <p>
         * The name of the managed rule group vendor. You use this, along with the rule group name, to identify a rule
         * group.
         * </p>
         * 
         * @param vendorName
         *        The name of the managed rule group vendor. You use this, along with the rule group name, to identify a
         *        rule group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vendorName(String vendorName);

        /**
         * <p>
         * The name of the managed rule group. You use this, along with the vendor name, to identify the rule group.
         * </p>
         * 
         * @param name
         *        The name of the managed rule group. You use this, along with the vendor name, to identify the rule
         *        group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The version of the managed rule group to use. If you specify this, the version setting is fixed until you
         * change it. If you don't specify this, WAF uses the vendor's default version, and then keeps the version at
         * the vendor's default when the vendor updates the managed rule group settings.
         * </p>
         * 
         * @param version
         *        The version of the managed rule group to use. If you specify this, the version setting is fixed until
         *        you change it. If you don't specify this, WAF uses the vendor's default version, and then keeps the
         *        version at the vendor's default when the vendor updates the managed rule group settings.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(String version);

        /**
         * <p>
         * Rules in the referenced rule group whose actions are set to <code>Count</code>.
         * </p>
         * <note>
         * <p>
         * Instead of this option, use <code>RuleActionOverrides</code>. It accepts any valid action setting, including
         * <code>Count</code>.
         * </p>
         * </note>
         * 
         * @param excludedRules
         *        Rules in the referenced rule group whose actions are set to <code>Count</code>. </p> <note>
         *        <p>
         *        Instead of this option, use <code>RuleActionOverrides</code>. It accepts any valid action setting,
         *        including <code>Count</code>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder excludedRules(Collection<ExcludedRule> excludedRules);

        /**
         * <p>
         * Rules in the referenced rule group whose actions are set to <code>Count</code>.
         * </p>
         * <note>
         * <p>
         * Instead of this option, use <code>RuleActionOverrides</code>. It accepts any valid action setting, including
         * <code>Count</code>.
         * </p>
         * </note>
         * 
         * @param excludedRules
         *        Rules in the referenced rule group whose actions are set to <code>Count</code>. </p> <note>
         *        <p>
         *        Instead of this option, use <code>RuleActionOverrides</code>. It accepts any valid action setting,
         *        including <code>Count</code>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder excludedRules(ExcludedRule... excludedRules);

        /**
         * <p>
         * Rules in the referenced rule group whose actions are set to <code>Count</code>.
         * </p>
         * <note>
         * <p>
         * Instead of this option, use <code>RuleActionOverrides</code>. It accepts any valid action setting, including
         * <code>Count</code>.
         * </p>
         * </note> This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.wafv2.model.ExcludedRule.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.wafv2.model.ExcludedRule#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.wafv2.model.ExcludedRule.Builder#build()} is called immediately and
         * its result is passed to {@link #excludedRules(List<ExcludedRule>)}.
         * 
         * @param excludedRules
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.wafv2.model.ExcludedRule.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #excludedRules(java.util.Collection<ExcludedRule>)
         */
        Builder excludedRules(Consumer<ExcludedRule.Builder>... excludedRules);

        /**
         * <p>
         * An optional nested statement that narrows the scope of the web requests that are evaluated by the managed
         * rule group. Requests are only evaluated by the rule group if they match the scope-down statement. You can use
         * any nestable <a>Statement</a> in the scope-down statement, and you can nest statements at any level, the same
         * as you can for a rule statement.
         * </p>
         * 
         * @param scopeDownStatement
         *        An optional nested statement that narrows the scope of the web requests that are evaluated by the
         *        managed rule group. Requests are only evaluated by the rule group if they match the scope-down
         *        statement. You can use any nestable <a>Statement</a> in the scope-down statement, and you can nest
         *        statements at any level, the same as you can for a rule statement.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scopeDownStatement(Statement scopeDownStatement);

        /**
         * <p>
         * An optional nested statement that narrows the scope of the web requests that are evaluated by the managed
         * rule group. Requests are only evaluated by the rule group if they match the scope-down statement. You can use
         * any nestable <a>Statement</a> in the scope-down statement, and you can nest statements at any level, the same
         * as you can for a rule statement.
         * </p>
         * This is a convenience method that creates an instance of the {@link Statement.Builder} avoiding the need to
         * create one manually via {@link Statement#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Statement.Builder#build()} is called immediately and its result
         * is passed to {@link #scopeDownStatement(Statement)}.
         * 
         * @param scopeDownStatement
         *        a consumer that will call methods on {@link Statement.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #scopeDownStatement(Statement)
         */
        default Builder scopeDownStatement(Consumer<Statement.Builder> scopeDownStatement) {
            return scopeDownStatement(Statement.builder().applyMutation(scopeDownStatement).build());
        }

        /**
         * <p>
         * Additional information that's used by a managed rule group. Many managed rule groups don't require this.
         * </p>
         * <p>
         * The rule groups used for intelligent threat mitigation require additional configuration:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Use the <code>AWSManagedRulesACFPRuleSet</code> configuration object to configure the account creation fraud
         * prevention managed rule group. The configuration includes the registration and sign-up pages of your
         * application and the locations in the account creation request payload of data, such as the user email and
         * phone number fields.
         * </p>
         * </li>
         * <li>
         * <p>
         * Use the <code>AWSManagedRulesATPRuleSet</code> configuration object to configure the account takeover
         * prevention managed rule group. The configuration includes the sign-in page of your application and the
         * locations in the login request payload of data such as the username and password.
         * </p>
         * </li>
         * <li>
         * <p>
         * Use the <code>AWSManagedRulesBotControlRuleSet</code> configuration object to configure the protection level
         * that you want the Bot Control rule group to use.
         * </p>
         * </li>
         * </ul>
         * 
         * @param managedRuleGroupConfigs
         *        Additional information that's used by a managed rule group. Many managed rule groups don't require
         *        this.</p>
         *        <p>
         *        The rule groups used for intelligent threat mitigation require additional configuration:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Use the <code>AWSManagedRulesACFPRuleSet</code> configuration object to configure the account creation
         *        fraud prevention managed rule group. The configuration includes the registration and sign-up pages of
         *        your application and the locations in the account creation request payload of data, such as the user
         *        email and phone number fields.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Use the <code>AWSManagedRulesATPRuleSet</code> configuration object to configure the account takeover
         *        prevention managed rule group. The configuration includes the sign-in page of your application and the
         *        locations in the login request payload of data such as the username and password.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Use the <code>AWSManagedRulesBotControlRuleSet</code> configuration object to configure the protection
         *        level that you want the Bot Control rule group to use.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder managedRuleGroupConfigs(Collection<ManagedRuleGroupConfig> managedRuleGroupConfigs);

        /**
         * <p>
         * Additional information that's used by a managed rule group. Many managed rule groups don't require this.
         * </p>
         * <p>
         * The rule groups used for intelligent threat mitigation require additional configuration:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Use the <code>AWSManagedRulesACFPRuleSet</code> configuration object to configure the account creation fraud
         * prevention managed rule group. The configuration includes the registration and sign-up pages of your
         * application and the locations in the account creation request payload of data, such as the user email and
         * phone number fields.
         * </p>
         * </li>
         * <li>
         * <p>
         * Use the <code>AWSManagedRulesATPRuleSet</code> configuration object to configure the account takeover
         * prevention managed rule group. The configuration includes the sign-in page of your application and the
         * locations in the login request payload of data such as the username and password.
         * </p>
         * </li>
         * <li>
         * <p>
         * Use the <code>AWSManagedRulesBotControlRuleSet</code> configuration object to configure the protection level
         * that you want the Bot Control rule group to use.
         * </p>
         * </li>
         * </ul>
         * 
         * @param managedRuleGroupConfigs
         *        Additional information that's used by a managed rule group. Many managed rule groups don't require
         *        this.</p>
         *        <p>
         *        The rule groups used for intelligent threat mitigation require additional configuration:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Use the <code>AWSManagedRulesACFPRuleSet</code> configuration object to configure the account creation
         *        fraud prevention managed rule group. The configuration includes the registration and sign-up pages of
         *        your application and the locations in the account creation request payload of data, such as the user
         *        email and phone number fields.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Use the <code>AWSManagedRulesATPRuleSet</code> configuration object to configure the account takeover
         *        prevention managed rule group. The configuration includes the sign-in page of your application and the
         *        locations in the login request payload of data such as the username and password.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Use the <code>AWSManagedRulesBotControlRuleSet</code> configuration object to configure the protection
         *        level that you want the Bot Control rule group to use.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder managedRuleGroupConfigs(ManagedRuleGroupConfig... managedRuleGroupConfigs);

        /**
         * <p>
         * Additional information that's used by a managed rule group. Many managed rule groups don't require this.
         * </p>
         * <p>
         * The rule groups used for intelligent threat mitigation require additional configuration:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Use the <code>AWSManagedRulesACFPRuleSet</code> configuration object to configure the account creation fraud
         * prevention managed rule group. The configuration includes the registration and sign-up pages of your
         * application and the locations in the account creation request payload of data, such as the user email and
         * phone number fields.
         * </p>
         * </li>
         * <li>
         * <p>
         * Use the <code>AWSManagedRulesATPRuleSet</code> configuration object to configure the account takeover
         * prevention managed rule group. The configuration includes the sign-in page of your application and the
         * locations in the login request payload of data such as the username and password.
         * </p>
         * </li>
         * <li>
         * <p>
         * Use the <code>AWSManagedRulesBotControlRuleSet</code> configuration object to configure the protection level
         * that you want the Bot Control rule group to use.
         * </p>
         * </li>
         * </ul>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.wafv2.model.ManagedRuleGroupConfig.Builder} avoiding the need to
         * create one manually via {@link software.amazon.awssdk.services.wafv2.model.ManagedRuleGroupConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.wafv2.model.ManagedRuleGroupConfig.Builder#build()} is called
         * immediately and its result is passed to {@link #managedRuleGroupConfigs(List<ManagedRuleGroupConfig>)}.
         * 
         * @param managedRuleGroupConfigs
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.wafv2.model.ManagedRuleGroupConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #managedRuleGroupConfigs(java.util.Collection<ManagedRuleGroupConfig>)
         */
        Builder managedRuleGroupConfigs(Consumer<ManagedRuleGroupConfig.Builder>... managedRuleGroupConfigs);

        /**
         * <p>
         * Action settings to use in the place of the rule actions that are configured inside the rule group. You
         * specify one override for each rule whose action you want to change.
         * </p>
         * <note>
         * <p>
         * Take care to verify the rule names in your overrides. If you provide a rule name that doesn't match the name
         * of any rule in the rule group, WAF doesn't return an error and doesn't apply the override setting.
         * </p>
         * </note>
         * <p>
         * You can use overrides for testing, for example you can override all of rule actions to <code>Count</code> and
         * then monitor the resulting count metrics to understand how the rule group would handle your web traffic. You
         * can also permanently override some or all actions, to modify how the rule group manages your web traffic.
         * </p>
         * 
         * @param ruleActionOverrides
         *        Action settings to use in the place of the rule actions that are configured inside the rule group. You
         *        specify one override for each rule whose action you want to change. </p> <note>
         *        <p>
         *        Take care to verify the rule names in your overrides. If you provide a rule name that doesn't match
         *        the name of any rule in the rule group, WAF doesn't return an error and doesn't apply the override
         *        setting.
         *        </p>
         *        </note>
         *        <p>
         *        You can use overrides for testing, for example you can override all of rule actions to
         *        <code>Count</code> and then monitor the resulting count metrics to understand how the rule group would
         *        handle your web traffic. You can also permanently override some or all actions, to modify how the rule
         *        group manages your web traffic.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ruleActionOverrides(Collection<RuleActionOverride> ruleActionOverrides);

        /**
         * <p>
         * Action settings to use in the place of the rule actions that are configured inside the rule group. You
         * specify one override for each rule whose action you want to change.
         * </p>
         * <note>
         * <p>
         * Take care to verify the rule names in your overrides. If you provide a rule name that doesn't match the name
         * of any rule in the rule group, WAF doesn't return an error and doesn't apply the override setting.
         * </p>
         * </note>
         * <p>
         * You can use overrides for testing, for example you can override all of rule actions to <code>Count</code> and
         * then monitor the resulting count metrics to understand how the rule group would handle your web traffic. You
         * can also permanently override some or all actions, to modify how the rule group manages your web traffic.
         * </p>
         * 
         * @param ruleActionOverrides
         *        Action settings to use in the place of the rule actions that are configured inside the rule group. You
         *        specify one override for each rule whose action you want to change. </p> <note>
         *        <p>
         *        Take care to verify the rule names in your overrides. If you provide a rule name that doesn't match
         *        the name of any rule in the rule group, WAF doesn't return an error and doesn't apply the override
         *        setting.
         *        </p>
         *        </note>
         *        <p>
         *        You can use overrides for testing, for example you can override all of rule actions to
         *        <code>Count</code> and then monitor the resulting count metrics to understand how the rule group would
         *        handle your web traffic. You can also permanently override some or all actions, to modify how the rule
         *        group manages your web traffic.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ruleActionOverrides(RuleActionOverride... ruleActionOverrides);

        /**
         * <p>
         * Action settings to use in the place of the rule actions that are configured inside the rule group. You
         * specify one override for each rule whose action you want to change.
         * </p>
         * <note>
         * <p>
         * Take care to verify the rule names in your overrides. If you provide a rule name that doesn't match the name
         * of any rule in the rule group, WAF doesn't return an error and doesn't apply the override setting.
         * </p>
         * </note>
         * <p>
         * You can use overrides for testing, for example you can override all of rule actions to <code>Count</code> and
         * then monitor the resulting count metrics to understand how the rule group would handle your web traffic. You
         * can also permanently override some or all actions, to modify how the rule group manages your web traffic.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.wafv2.model.RuleActionOverride.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.wafv2.model.RuleActionOverride#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.wafv2.model.RuleActionOverride.Builder#build()} is called immediately
         * and its result is passed to {@link #ruleActionOverrides(List<RuleActionOverride>)}.
         * 
         * @param ruleActionOverrides
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.wafv2.model.RuleActionOverride.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #ruleActionOverrides(java.util.Collection<RuleActionOverride>)
         */
        Builder ruleActionOverrides(Consumer<RuleActionOverride.Builder>... ruleActionOverrides);
    }

    static final class BuilderImpl implements Builder {
        private String vendorName;

        private String name;

        private String version;

        private List<ExcludedRule> excludedRules = DefaultSdkAutoConstructList.getInstance();

        private Statement scopeDownStatement;

        private List<ManagedRuleGroupConfig> managedRuleGroupConfigs = DefaultSdkAutoConstructList.getInstance();

        private List<RuleActionOverride> ruleActionOverrides = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ManagedRuleGroupStatement model) {
            vendorName(model.vendorName);
            name(model.name);
            version(model.version);
            excludedRules(model.excludedRules);
            scopeDownStatement(model.scopeDownStatement);
            managedRuleGroupConfigs(model.managedRuleGroupConfigs);
            ruleActionOverrides(model.ruleActionOverrides);
        }

        public final String getVendorName() {
            return vendorName;
        }

        public final void setVendorName(String vendorName) {
            this.vendorName = vendorName;
        }

        @Override
        public final Builder vendorName(String vendorName) {
            this.vendorName = vendorName;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getVersion() {
            return version;
        }

        public final void setVersion(String version) {
            this.version = version;
        }

        @Override
        public final Builder version(String version) {
            this.version = version;
            return this;
        }

        public final List<ExcludedRule.Builder> getExcludedRules() {
            List<ExcludedRule.Builder> result = ExcludedRulesCopier.copyToBuilder(this.excludedRules);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setExcludedRules(Collection<ExcludedRule.BuilderImpl> excludedRules) {
            this.excludedRules = ExcludedRulesCopier.copyFromBuilder(excludedRules);
        }

        @Override
        public final Builder excludedRules(Collection<ExcludedRule> excludedRules) {
            this.excludedRules = ExcludedRulesCopier.copy(excludedRules);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder excludedRules(ExcludedRule... excludedRules) {
            excludedRules(Arrays.asList(excludedRules));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder excludedRules(Consumer<ExcludedRule.Builder>... excludedRules) {
            excludedRules(Stream.of(excludedRules).map(c -> ExcludedRule.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Statement.Builder getScopeDownStatement() {
            return scopeDownStatement != null ? scopeDownStatement.toBuilder() : null;
        }

        public final void setScopeDownStatement(Statement.BuilderImpl scopeDownStatement) {
            this.scopeDownStatement = scopeDownStatement != null ? scopeDownStatement.build() : null;
        }

        @Override
        public final Builder scopeDownStatement(Statement scopeDownStatement) {
            this.scopeDownStatement = scopeDownStatement;
            return this;
        }

        public final List<ManagedRuleGroupConfig.Builder> getManagedRuleGroupConfigs() {
            List<ManagedRuleGroupConfig.Builder> result = ManagedRuleGroupConfigsCopier
                    .copyToBuilder(this.managedRuleGroupConfigs);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setManagedRuleGroupConfigs(Collection<ManagedRuleGroupConfig.BuilderImpl> managedRuleGroupConfigs) {
            this.managedRuleGroupConfigs = ManagedRuleGroupConfigsCopier.copyFromBuilder(managedRuleGroupConfigs);
        }

        @Override
        public final Builder managedRuleGroupConfigs(Collection<ManagedRuleGroupConfig> managedRuleGroupConfigs) {
            this.managedRuleGroupConfigs = ManagedRuleGroupConfigsCopier.copy(managedRuleGroupConfigs);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder managedRuleGroupConfigs(ManagedRuleGroupConfig... managedRuleGroupConfigs) {
            managedRuleGroupConfigs(Arrays.asList(managedRuleGroupConfigs));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder managedRuleGroupConfigs(Consumer<ManagedRuleGroupConfig.Builder>... managedRuleGroupConfigs) {
            managedRuleGroupConfigs(Stream.of(managedRuleGroupConfigs)
                    .map(c -> ManagedRuleGroupConfig.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<RuleActionOverride.Builder> getRuleActionOverrides() {
            List<RuleActionOverride.Builder> result = RuleActionOverridesCopier.copyToBuilder(this.ruleActionOverrides);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setRuleActionOverrides(Collection<RuleActionOverride.BuilderImpl> ruleActionOverrides) {
            this.ruleActionOverrides = RuleActionOverridesCopier.copyFromBuilder(ruleActionOverrides);
        }

        @Override
        public final Builder ruleActionOverrides(Collection<RuleActionOverride> ruleActionOverrides) {
            this.ruleActionOverrides = RuleActionOverridesCopier.copy(ruleActionOverrides);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ruleActionOverrides(RuleActionOverride... ruleActionOverrides) {
            ruleActionOverrides(Arrays.asList(ruleActionOverrides));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ruleActionOverrides(Consumer<RuleActionOverride.Builder>... ruleActionOverrides) {
            ruleActionOverrides(Stream.of(ruleActionOverrides).map(c -> RuleActionOverride.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public ManagedRuleGroupStatement build() {
            return new ManagedRuleGroupStatement(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
