/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.wafv2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A rate-based rule counts incoming requests and rate limits requests when they are coming at too fast a rate. The rule
 * categorizes requests according to your aggregation criteria, collects them into aggregation instances, and counts and
 * rate limits the requests for each instance.
 * </p>
 * <note>
 * <p>
 * If you change any of these settings in a rule that's currently in use, the change resets the rule's rate limiting
 * counts. This can pause the rule's rate limiting activities for up to a minute.
 * </p>
 * </note>
 * <p>
 * You can specify individual aggregation keys, like IP address or HTTP method. You can also specify aggregation key
 * combinations, like IP address and HTTP method, or HTTP method, query argument, and cookie.
 * </p>
 * <p>
 * Each unique set of values for the aggregation keys that you specify is a separate aggregation instance, with the
 * value from each key contributing to the aggregation instance definition.
 * </p>
 * <p>
 * For example, assume the rule evaluates web requests with the following IP address and HTTP method values:
 * </p>
 * <ul>
 * <li>
 * <p>
 * IP address 10.1.1.1, HTTP method POST
 * </p>
 * </li>
 * <li>
 * <p>
 * IP address 10.1.1.1, HTTP method GET
 * </p>
 * </li>
 * <li>
 * <p>
 * IP address 127.0.0.0, HTTP method POST
 * </p>
 * </li>
 * <li>
 * <p>
 * IP address 10.1.1.1, HTTP method GET
 * </p>
 * </li>
 * </ul>
 * <p>
 * The rule would create different aggregation instances according to your aggregation criteria, for example:
 * </p>
 * <ul>
 * <li>
 * <p>
 * If the aggregation criteria is just the IP address, then each individual address is an aggregation instance, and WAF
 * counts requests separately for each. The aggregation instances and request counts for our example would be the
 * following:
 * </p>
 * <ul>
 * <li>
 * <p>
 * IP address 10.1.1.1: count 3
 * </p>
 * </li>
 * <li>
 * <p>
 * IP address 127.0.0.0: count 1
 * </p>
 * </li>
 * </ul>
 * </li>
 * <li>
 * <p>
 * If the aggregation criteria is HTTP method, then each individual HTTP method is an aggregation instance. The
 * aggregation instances and request counts for our example would be the following:
 * </p>
 * <ul>
 * <li>
 * <p>
 * HTTP method POST: count 2
 * </p>
 * </li>
 * <li>
 * <p>
 * HTTP method GET: count 2
 * </p>
 * </li>
 * </ul>
 * </li>
 * <li>
 * <p>
 * If the aggregation criteria is IP address and HTTP method, then each IP address and each HTTP method would contribute
 * to the combined aggregation instance. The aggregation instances and request counts for our example would be the
 * following:
 * </p>
 * <ul>
 * <li>
 * <p>
 * IP address 10.1.1.1, HTTP method POST: count 1
 * </p>
 * </li>
 * <li>
 * <p>
 * IP address 10.1.1.1, HTTP method GET: count 2
 * </p>
 * </li>
 * <li>
 * <p>
 * IP address 127.0.0.0, HTTP method POST: count 1
 * </p>
 * </li>
 * </ul>
 * </li>
 * </ul>
 * <p>
 * For any n-tuple of aggregation keys, each unique combination of values for the keys defines a separate aggregation
 * instance, which WAF counts and rate-limits individually.
 * </p>
 * <p>
 * You can optionally nest another statement inside the rate-based statement, to narrow the scope of the rule so that it
 * only counts and rate limits requests that match the nested statement. You can use this nested scope-down statement in
 * conjunction with your aggregation key specifications or you can just count and rate limit all requests that match the
 * scope-down statement, without additional aggregation. When you choose to just manage all requests that match a
 * scope-down statement, the aggregation instance is singular for the rule.
 * </p>
 * <p>
 * You cannot nest a <code>RateBasedStatement</code> inside another statement, for example inside a
 * <code>NotStatement</code> or <code>OrStatement</code>. You can define a <code>RateBasedStatement</code> inside a web
 * ACL and inside a rule group.
 * </p>
 * <p>
 * For additional information about the options, see <a
 * href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-rate-based-rules.html">Rate limiting web requests
 * using rate-based rules</a> in the <i>WAF Developer Guide</i>.
 * </p>
 * <p>
 * If you only aggregate on the individual IP address or forwarded IP address, you can retrieve the list of IP addresses
 * that WAF is currently rate limiting for a rule through the API call <code>GetRateBasedStatementManagedKeys</code>.
 * This option is not available for other aggregation configurations.
 * </p>
 * <p>
 * WAF tracks and manages web requests separately for each instance of a rate-based rule that you use. For example, if
 * you provide the same rate-based rule settings in two web ACLs, each of the two rule statements represents a separate
 * instance of the rate-based rule and gets its own tracking and management by WAF. If you define a rate-based rule
 * inside a rule group, and then use that rule group in multiple places, each use creates a separate instance of the
 * rate-based rule that gets its own tracking and management by WAF.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RateBasedStatement implements SdkPojo, Serializable,
        ToCopyableBuilder<RateBasedStatement.Builder, RateBasedStatement> {
    private static final SdkField<Long> LIMIT_FIELD = SdkField.<Long> builder(MarshallingType.LONG).memberName("Limit")
            .getter(getter(RateBasedStatement::limit)).setter(setter(Builder::limit))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Limit").build()).build();

    private static final SdkField<Long> EVALUATION_WINDOW_SEC_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("EvaluationWindowSec").getter(getter(RateBasedStatement::evaluationWindowSec))
            .setter(setter(Builder::evaluationWindowSec))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EvaluationWindowSec").build())
            .build();

    private static final SdkField<String> AGGREGATE_KEY_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AggregateKeyType").getter(getter(RateBasedStatement::aggregateKeyTypeAsString))
            .setter(setter(Builder::aggregateKeyType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AggregateKeyType").build()).build();

    private static final SdkField<Statement> SCOPE_DOWN_STATEMENT_FIELD = SdkField.<Statement> builder(MarshallingType.SDK_POJO)
            .memberName("ScopeDownStatement").getter(getter(RateBasedStatement::scopeDownStatement))
            .setter(setter(Builder::scopeDownStatement)).constructor(Statement::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScopeDownStatement").build())
            .build();

    private static final SdkField<ForwardedIPConfig> FORWARDED_IP_CONFIG_FIELD = SdkField
            .<ForwardedIPConfig> builder(MarshallingType.SDK_POJO).memberName("ForwardedIPConfig")
            .getter(getter(RateBasedStatement::forwardedIPConfig)).setter(setter(Builder::forwardedIPConfig))
            .constructor(ForwardedIPConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ForwardedIPConfig").build()).build();

    private static final SdkField<List<RateBasedStatementCustomKey>> CUSTOM_KEYS_FIELD = SdkField
            .<List<RateBasedStatementCustomKey>> builder(MarshallingType.LIST)
            .memberName("CustomKeys")
            .getter(getter(RateBasedStatement::customKeys))
            .setter(setter(Builder::customKeys))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomKeys").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<RateBasedStatementCustomKey> builder(MarshallingType.SDK_POJO)
                                            .constructor(RateBasedStatementCustomKey::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(LIMIT_FIELD,
            EVALUATION_WINDOW_SEC_FIELD, AGGREGATE_KEY_TYPE_FIELD, SCOPE_DOWN_STATEMENT_FIELD, FORWARDED_IP_CONFIG_FIELD,
            CUSTOM_KEYS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Long limit;

    private final Long evaluationWindowSec;

    private final String aggregateKeyType;

    private final Statement scopeDownStatement;

    private final ForwardedIPConfig forwardedIPConfig;

    private final List<RateBasedStatementCustomKey> customKeys;

    private RateBasedStatement(BuilderImpl builder) {
        this.limit = builder.limit;
        this.evaluationWindowSec = builder.evaluationWindowSec;
        this.aggregateKeyType = builder.aggregateKeyType;
        this.scopeDownStatement = builder.scopeDownStatement;
        this.forwardedIPConfig = builder.forwardedIPConfig;
        this.customKeys = builder.customKeys;
    }

    /**
     * <p>
     * The limit on requests during the specified evaluation window for a single aggregation instance for the rate-based
     * rule. If the rate-based statement includes a <code>ScopeDownStatement</code>, this limit is applied only to the
     * requests that match the statement.
     * </p>
     * <p>
     * Examples:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you aggregate on just the IP address, this is the limit on requests from any single IP address.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you aggregate on the HTTP method and the query argument name "city", then this is the limit on requests for
     * any single method, city pair.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The limit on requests during the specified evaluation window for a single aggregation instance for the
     *         rate-based rule. If the rate-based statement includes a <code>ScopeDownStatement</code>, this limit is
     *         applied only to the requests that match the statement.</p>
     *         <p>
     *         Examples:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If you aggregate on just the IP address, this is the limit on requests from any single IP address.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you aggregate on the HTTP method and the query argument name "city", then this is the limit on
     *         requests for any single method, city pair.
     *         </p>
     *         </li>
     */
    public final Long limit() {
        return limit;
    }

    /**
     * <p>
     * The amount of time, in seconds, that WAF should include in its request counts, looking back from the current
     * time. For example, for a setting of 120, when WAF checks the rate, it counts the requests for the 2 minutes
     * immediately preceding the current time. Valid settings are 60, 120, 300, and 600.
     * </p>
     * <p>
     * This setting doesn't determine how often WAF checks the rate, but how far back it looks each time it checks. WAF
     * checks the rate about every 10 seconds.
     * </p>
     * <p>
     * Default: <code>300</code> (5 minutes)
     * </p>
     * 
     * @return The amount of time, in seconds, that WAF should include in its request counts, looking back from the
     *         current time. For example, for a setting of 120, when WAF checks the rate, it counts the requests for the
     *         2 minutes immediately preceding the current time. Valid settings are 60, 120, 300, and 600. </p>
     *         <p>
     *         This setting doesn't determine how often WAF checks the rate, but how far back it looks each time it
     *         checks. WAF checks the rate about every 10 seconds.
     *         </p>
     *         <p>
     *         Default: <code>300</code> (5 minutes)
     */
    public final Long evaluationWindowSec() {
        return evaluationWindowSec;
    }

    /**
     * <p>
     * Setting that indicates how to aggregate the request counts.
     * </p>
     * <note>
     * <p>
     * Web requests that are missing any of the components specified in the aggregation keys are omitted from the
     * rate-based rule evaluation and handling.
     * </p>
     * </note>
     * <ul>
     * <li>
     * <p>
     * <code>CONSTANT</code> - Count and limit the requests that match the rate-based rule's scope-down statement. With
     * this option, the counted requests aren't further aggregated. The scope-down statement is the only specification
     * used. When the count of all requests that satisfy the scope-down statement goes over the limit, WAF applies the
     * rule action to all requests that satisfy the scope-down statement.
     * </p>
     * <p>
     * With this option, you must configure the <code>ScopeDownStatement</code> property.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CUSTOM_KEYS</code> - Aggregate the request counts using one or more web request components as the aggregate
     * keys.
     * </p>
     * <p>
     * With this option, you must specify the aggregate keys in the <code>CustomKeys</code> property.
     * </p>
     * <p>
     * To aggregate on only the IP address or only the forwarded IP address, don't use custom keys. Instead, set the
     * aggregate key type to <code>IP</code> or <code>FORWARDED_IP</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FORWARDED_IP</code> - Aggregate the request counts on the first IP address in an HTTP header.
     * </p>
     * <p>
     * With this option, you must specify the header to use in the <code>ForwardedIPConfig</code> property.
     * </p>
     * <p>
     * To aggregate on a combination of the forwarded IP address with other aggregate keys, use <code>CUSTOM_KEYS</code>
     * .
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IP</code> - Aggregate the request counts on the IP address from the web request origin.
     * </p>
     * <p>
     * To aggregate on a combination of the IP address with other aggregate keys, use <code>CUSTOM_KEYS</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #aggregateKeyType}
     * will return {@link RateBasedStatementAggregateKeyType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #aggregateKeyTypeAsString}.
     * </p>
     * 
     * @return Setting that indicates how to aggregate the request counts. </p> <note>
     *         <p>
     *         Web requests that are missing any of the components specified in the aggregation keys are omitted from
     *         the rate-based rule evaluation and handling.
     *         </p>
     *         </note>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CONSTANT</code> - Count and limit the requests that match the rate-based rule's scope-down
     *         statement. With this option, the counted requests aren't further aggregated. The scope-down statement is
     *         the only specification used. When the count of all requests that satisfy the scope-down statement goes
     *         over the limit, WAF applies the rule action to all requests that satisfy the scope-down statement.
     *         </p>
     *         <p>
     *         With this option, you must configure the <code>ScopeDownStatement</code> property.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CUSTOM_KEYS</code> - Aggregate the request counts using one or more web request components as the
     *         aggregate keys.
     *         </p>
     *         <p>
     *         With this option, you must specify the aggregate keys in the <code>CustomKeys</code> property.
     *         </p>
     *         <p>
     *         To aggregate on only the IP address or only the forwarded IP address, don't use custom keys. Instead, set
     *         the aggregate key type to <code>IP</code> or <code>FORWARDED_IP</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FORWARDED_IP</code> - Aggregate the request counts on the first IP address in an HTTP header.
     *         </p>
     *         <p>
     *         With this option, you must specify the header to use in the <code>ForwardedIPConfig</code> property.
     *         </p>
     *         <p>
     *         To aggregate on a combination of the forwarded IP address with other aggregate keys, use
     *         <code>CUSTOM_KEYS</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IP</code> - Aggregate the request counts on the IP address from the web request origin.
     *         </p>
     *         <p>
     *         To aggregate on a combination of the IP address with other aggregate keys, use <code>CUSTOM_KEYS</code>.
     *         </p>
     *         </li>
     * @see RateBasedStatementAggregateKeyType
     */
    public final RateBasedStatementAggregateKeyType aggregateKeyType() {
        return RateBasedStatementAggregateKeyType.fromValue(aggregateKeyType);
    }

    /**
     * <p>
     * Setting that indicates how to aggregate the request counts.
     * </p>
     * <note>
     * <p>
     * Web requests that are missing any of the components specified in the aggregation keys are omitted from the
     * rate-based rule evaluation and handling.
     * </p>
     * </note>
     * <ul>
     * <li>
     * <p>
     * <code>CONSTANT</code> - Count and limit the requests that match the rate-based rule's scope-down statement. With
     * this option, the counted requests aren't further aggregated. The scope-down statement is the only specification
     * used. When the count of all requests that satisfy the scope-down statement goes over the limit, WAF applies the
     * rule action to all requests that satisfy the scope-down statement.
     * </p>
     * <p>
     * With this option, you must configure the <code>ScopeDownStatement</code> property.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CUSTOM_KEYS</code> - Aggregate the request counts using one or more web request components as the aggregate
     * keys.
     * </p>
     * <p>
     * With this option, you must specify the aggregate keys in the <code>CustomKeys</code> property.
     * </p>
     * <p>
     * To aggregate on only the IP address or only the forwarded IP address, don't use custom keys. Instead, set the
     * aggregate key type to <code>IP</code> or <code>FORWARDED_IP</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FORWARDED_IP</code> - Aggregate the request counts on the first IP address in an HTTP header.
     * </p>
     * <p>
     * With this option, you must specify the header to use in the <code>ForwardedIPConfig</code> property.
     * </p>
     * <p>
     * To aggregate on a combination of the forwarded IP address with other aggregate keys, use <code>CUSTOM_KEYS</code>
     * .
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IP</code> - Aggregate the request counts on the IP address from the web request origin.
     * </p>
     * <p>
     * To aggregate on a combination of the IP address with other aggregate keys, use <code>CUSTOM_KEYS</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #aggregateKeyType}
     * will return {@link RateBasedStatementAggregateKeyType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #aggregateKeyTypeAsString}.
     * </p>
     * 
     * @return Setting that indicates how to aggregate the request counts. </p> <note>
     *         <p>
     *         Web requests that are missing any of the components specified in the aggregation keys are omitted from
     *         the rate-based rule evaluation and handling.
     *         </p>
     *         </note>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CONSTANT</code> - Count and limit the requests that match the rate-based rule's scope-down
     *         statement. With this option, the counted requests aren't further aggregated. The scope-down statement is
     *         the only specification used. When the count of all requests that satisfy the scope-down statement goes
     *         over the limit, WAF applies the rule action to all requests that satisfy the scope-down statement.
     *         </p>
     *         <p>
     *         With this option, you must configure the <code>ScopeDownStatement</code> property.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CUSTOM_KEYS</code> - Aggregate the request counts using one or more web request components as the
     *         aggregate keys.
     *         </p>
     *         <p>
     *         With this option, you must specify the aggregate keys in the <code>CustomKeys</code> property.
     *         </p>
     *         <p>
     *         To aggregate on only the IP address or only the forwarded IP address, don't use custom keys. Instead, set
     *         the aggregate key type to <code>IP</code> or <code>FORWARDED_IP</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FORWARDED_IP</code> - Aggregate the request counts on the first IP address in an HTTP header.
     *         </p>
     *         <p>
     *         With this option, you must specify the header to use in the <code>ForwardedIPConfig</code> property.
     *         </p>
     *         <p>
     *         To aggregate on a combination of the forwarded IP address with other aggregate keys, use
     *         <code>CUSTOM_KEYS</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IP</code> - Aggregate the request counts on the IP address from the web request origin.
     *         </p>
     *         <p>
     *         To aggregate on a combination of the IP address with other aggregate keys, use <code>CUSTOM_KEYS</code>.
     *         </p>
     *         </li>
     * @see RateBasedStatementAggregateKeyType
     */
    public final String aggregateKeyTypeAsString() {
        return aggregateKeyType;
    }

    /**
     * <p>
     * An optional nested statement that narrows the scope of the web requests that are evaluated and managed by the
     * rate-based statement. When you use a scope-down statement, the rate-based rule only tracks and rate limits
     * requests that match the scope-down statement. You can use any nestable <a>Statement</a> in the scope-down
     * statement, and you can nest statements at any level, the same as you can for a rule statement.
     * </p>
     * 
     * @return An optional nested statement that narrows the scope of the web requests that are evaluated and managed by
     *         the rate-based statement. When you use a scope-down statement, the rate-based rule only tracks and rate
     *         limits requests that match the scope-down statement. You can use any nestable <a>Statement</a> in the
     *         scope-down statement, and you can nest statements at any level, the same as you can for a rule statement.
     */
    public final Statement scopeDownStatement() {
        return scopeDownStatement;
    }

    /**
     * <p>
     * The configuration for inspecting IP addresses in an HTTP header that you specify, instead of using the IP address
     * that's reported by the web request origin. Commonly, this is the X-Forwarded-For (XFF) header, but you can
     * specify any header name.
     * </p>
     * <note>
     * <p>
     * If the specified header isn't present in the request, WAF doesn't apply the rule to the web request at all.
     * </p>
     * </note>
     * <p>
     * This is required if you specify a forwarded IP in the rule's aggregate key settings.
     * </p>
     * 
     * @return The configuration for inspecting IP addresses in an HTTP header that you specify, instead of using the IP
     *         address that's reported by the web request origin. Commonly, this is the X-Forwarded-For (XFF) header,
     *         but you can specify any header name. </p> <note>
     *         <p>
     *         If the specified header isn't present in the request, WAF doesn't apply the rule to the web request at
     *         all.
     *         </p>
     *         </note>
     *         <p>
     *         This is required if you specify a forwarded IP in the rule's aggregate key settings.
     */
    public final ForwardedIPConfig forwardedIPConfig() {
        return forwardedIPConfig;
    }

    /**
     * For responses, this returns true if the service returned a value for the CustomKeys property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasCustomKeys() {
        return customKeys != null && !(customKeys instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specifies the aggregate keys to use in a rate-base rule.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCustomKeys} method.
     * </p>
     * 
     * @return Specifies the aggregate keys to use in a rate-base rule.
     */
    public final List<RateBasedStatementCustomKey> customKeys() {
        return customKeys;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(limit());
        hashCode = 31 * hashCode + Objects.hashCode(evaluationWindowSec());
        hashCode = 31 * hashCode + Objects.hashCode(aggregateKeyTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(scopeDownStatement());
        hashCode = 31 * hashCode + Objects.hashCode(forwardedIPConfig());
        hashCode = 31 * hashCode + Objects.hashCode(hasCustomKeys() ? customKeys() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RateBasedStatement)) {
            return false;
        }
        RateBasedStatement other = (RateBasedStatement) obj;
        return Objects.equals(limit(), other.limit()) && Objects.equals(evaluationWindowSec(), other.evaluationWindowSec())
                && Objects.equals(aggregateKeyTypeAsString(), other.aggregateKeyTypeAsString())
                && Objects.equals(scopeDownStatement(), other.scopeDownStatement())
                && Objects.equals(forwardedIPConfig(), other.forwardedIPConfig()) && hasCustomKeys() == other.hasCustomKeys()
                && Objects.equals(customKeys(), other.customKeys());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RateBasedStatement").add("Limit", limit()).add("EvaluationWindowSec", evaluationWindowSec())
                .add("AggregateKeyType", aggregateKeyTypeAsString()).add("ScopeDownStatement", scopeDownStatement())
                .add("ForwardedIPConfig", forwardedIPConfig()).add("CustomKeys", hasCustomKeys() ? customKeys() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Limit":
            return Optional.ofNullable(clazz.cast(limit()));
        case "EvaluationWindowSec":
            return Optional.ofNullable(clazz.cast(evaluationWindowSec()));
        case "AggregateKeyType":
            return Optional.ofNullable(clazz.cast(aggregateKeyTypeAsString()));
        case "ScopeDownStatement":
            return Optional.ofNullable(clazz.cast(scopeDownStatement()));
        case "ForwardedIPConfig":
            return Optional.ofNullable(clazz.cast(forwardedIPConfig()));
        case "CustomKeys":
            return Optional.ofNullable(clazz.cast(customKeys()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Limit", LIMIT_FIELD);
        map.put("EvaluationWindowSec", EVALUATION_WINDOW_SEC_FIELD);
        map.put("AggregateKeyType", AGGREGATE_KEY_TYPE_FIELD);
        map.put("ScopeDownStatement", SCOPE_DOWN_STATEMENT_FIELD);
        map.put("ForwardedIPConfig", FORWARDED_IP_CONFIG_FIELD);
        map.put("CustomKeys", CUSTOM_KEYS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<RateBasedStatement, T> g) {
        return obj -> g.apply((RateBasedStatement) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RateBasedStatement> {
        /**
         * <p>
         * The limit on requests during the specified evaluation window for a single aggregation instance for the
         * rate-based rule. If the rate-based statement includes a <code>ScopeDownStatement</code>, this limit is
         * applied only to the requests that match the statement.
         * </p>
         * <p>
         * Examples:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If you aggregate on just the IP address, this is the limit on requests from any single IP address.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you aggregate on the HTTP method and the query argument name "city", then this is the limit on requests
         * for any single method, city pair.
         * </p>
         * </li>
         * </ul>
         * 
         * @param limit
         *        The limit on requests during the specified evaluation window for a single aggregation instance for the
         *        rate-based rule. If the rate-based statement includes a <code>ScopeDownStatement</code>, this limit is
         *        applied only to the requests that match the statement.</p>
         *        <p>
         *        Examples:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If you aggregate on just the IP address, this is the limit on requests from any single IP address.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you aggregate on the HTTP method and the query argument name "city", then this is the limit on
         *        requests for any single method, city pair.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder limit(Long limit);

        /**
         * <p>
         * The amount of time, in seconds, that WAF should include in its request counts, looking back from the current
         * time. For example, for a setting of 120, when WAF checks the rate, it counts the requests for the 2 minutes
         * immediately preceding the current time. Valid settings are 60, 120, 300, and 600.
         * </p>
         * <p>
         * This setting doesn't determine how often WAF checks the rate, but how far back it looks each time it checks.
         * WAF checks the rate about every 10 seconds.
         * </p>
         * <p>
         * Default: <code>300</code> (5 minutes)
         * </p>
         * 
         * @param evaluationWindowSec
         *        The amount of time, in seconds, that WAF should include in its request counts, looking back from the
         *        current time. For example, for a setting of 120, when WAF checks the rate, it counts the requests for
         *        the 2 minutes immediately preceding the current time. Valid settings are 60, 120, 300, and 600. </p>
         *        <p>
         *        This setting doesn't determine how often WAF checks the rate, but how far back it looks each time it
         *        checks. WAF checks the rate about every 10 seconds.
         *        </p>
         *        <p>
         *        Default: <code>300</code> (5 minutes)
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder evaluationWindowSec(Long evaluationWindowSec);

        /**
         * <p>
         * Setting that indicates how to aggregate the request counts.
         * </p>
         * <note>
         * <p>
         * Web requests that are missing any of the components specified in the aggregation keys are omitted from the
         * rate-based rule evaluation and handling.
         * </p>
         * </note>
         * <ul>
         * <li>
         * <p>
         * <code>CONSTANT</code> - Count and limit the requests that match the rate-based rule's scope-down statement.
         * With this option, the counted requests aren't further aggregated. The scope-down statement is the only
         * specification used. When the count of all requests that satisfy the scope-down statement goes over the limit,
         * WAF applies the rule action to all requests that satisfy the scope-down statement.
         * </p>
         * <p>
         * With this option, you must configure the <code>ScopeDownStatement</code> property.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CUSTOM_KEYS</code> - Aggregate the request counts using one or more web request components as the
         * aggregate keys.
         * </p>
         * <p>
         * With this option, you must specify the aggregate keys in the <code>CustomKeys</code> property.
         * </p>
         * <p>
         * To aggregate on only the IP address or only the forwarded IP address, don't use custom keys. Instead, set the
         * aggregate key type to <code>IP</code> or <code>FORWARDED_IP</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FORWARDED_IP</code> - Aggregate the request counts on the first IP address in an HTTP header.
         * </p>
         * <p>
         * With this option, you must specify the header to use in the <code>ForwardedIPConfig</code> property.
         * </p>
         * <p>
         * To aggregate on a combination of the forwarded IP address with other aggregate keys, use
         * <code>CUSTOM_KEYS</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IP</code> - Aggregate the request counts on the IP address from the web request origin.
         * </p>
         * <p>
         * To aggregate on a combination of the IP address with other aggregate keys, use <code>CUSTOM_KEYS</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param aggregateKeyType
         *        Setting that indicates how to aggregate the request counts. </p> <note>
         *        <p>
         *        Web requests that are missing any of the components specified in the aggregation keys are omitted from
         *        the rate-based rule evaluation and handling.
         *        </p>
         *        </note>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CONSTANT</code> - Count and limit the requests that match the rate-based rule's scope-down
         *        statement. With this option, the counted requests aren't further aggregated. The scope-down statement
         *        is the only specification used. When the count of all requests that satisfy the scope-down statement
         *        goes over the limit, WAF applies the rule action to all requests that satisfy the scope-down
         *        statement.
         *        </p>
         *        <p>
         *        With this option, you must configure the <code>ScopeDownStatement</code> property.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CUSTOM_KEYS</code> - Aggregate the request counts using one or more web request components as
         *        the aggregate keys.
         *        </p>
         *        <p>
         *        With this option, you must specify the aggregate keys in the <code>CustomKeys</code> property.
         *        </p>
         *        <p>
         *        To aggregate on only the IP address or only the forwarded IP address, don't use custom keys. Instead,
         *        set the aggregate key type to <code>IP</code> or <code>FORWARDED_IP</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FORWARDED_IP</code> - Aggregate the request counts on the first IP address in an HTTP header.
         *        </p>
         *        <p>
         *        With this option, you must specify the header to use in the <code>ForwardedIPConfig</code> property.
         *        </p>
         *        <p>
         *        To aggregate on a combination of the forwarded IP address with other aggregate keys, use
         *        <code>CUSTOM_KEYS</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IP</code> - Aggregate the request counts on the IP address from the web request origin.
         *        </p>
         *        <p>
         *        To aggregate on a combination of the IP address with other aggregate keys, use
         *        <code>CUSTOM_KEYS</code>.
         *        </p>
         *        </li>
         * @see RateBasedStatementAggregateKeyType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RateBasedStatementAggregateKeyType
         */
        Builder aggregateKeyType(String aggregateKeyType);

        /**
         * <p>
         * Setting that indicates how to aggregate the request counts.
         * </p>
         * <note>
         * <p>
         * Web requests that are missing any of the components specified in the aggregation keys are omitted from the
         * rate-based rule evaluation and handling.
         * </p>
         * </note>
         * <ul>
         * <li>
         * <p>
         * <code>CONSTANT</code> - Count and limit the requests that match the rate-based rule's scope-down statement.
         * With this option, the counted requests aren't further aggregated. The scope-down statement is the only
         * specification used. When the count of all requests that satisfy the scope-down statement goes over the limit,
         * WAF applies the rule action to all requests that satisfy the scope-down statement.
         * </p>
         * <p>
         * With this option, you must configure the <code>ScopeDownStatement</code> property.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CUSTOM_KEYS</code> - Aggregate the request counts using one or more web request components as the
         * aggregate keys.
         * </p>
         * <p>
         * With this option, you must specify the aggregate keys in the <code>CustomKeys</code> property.
         * </p>
         * <p>
         * To aggregate on only the IP address or only the forwarded IP address, don't use custom keys. Instead, set the
         * aggregate key type to <code>IP</code> or <code>FORWARDED_IP</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FORWARDED_IP</code> - Aggregate the request counts on the first IP address in an HTTP header.
         * </p>
         * <p>
         * With this option, you must specify the header to use in the <code>ForwardedIPConfig</code> property.
         * </p>
         * <p>
         * To aggregate on a combination of the forwarded IP address with other aggregate keys, use
         * <code>CUSTOM_KEYS</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IP</code> - Aggregate the request counts on the IP address from the web request origin.
         * </p>
         * <p>
         * To aggregate on a combination of the IP address with other aggregate keys, use <code>CUSTOM_KEYS</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param aggregateKeyType
         *        Setting that indicates how to aggregate the request counts. </p> <note>
         *        <p>
         *        Web requests that are missing any of the components specified in the aggregation keys are omitted from
         *        the rate-based rule evaluation and handling.
         *        </p>
         *        </note>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CONSTANT</code> - Count and limit the requests that match the rate-based rule's scope-down
         *        statement. With this option, the counted requests aren't further aggregated. The scope-down statement
         *        is the only specification used. When the count of all requests that satisfy the scope-down statement
         *        goes over the limit, WAF applies the rule action to all requests that satisfy the scope-down
         *        statement.
         *        </p>
         *        <p>
         *        With this option, you must configure the <code>ScopeDownStatement</code> property.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CUSTOM_KEYS</code> - Aggregate the request counts using one or more web request components as
         *        the aggregate keys.
         *        </p>
         *        <p>
         *        With this option, you must specify the aggregate keys in the <code>CustomKeys</code> property.
         *        </p>
         *        <p>
         *        To aggregate on only the IP address or only the forwarded IP address, don't use custom keys. Instead,
         *        set the aggregate key type to <code>IP</code> or <code>FORWARDED_IP</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FORWARDED_IP</code> - Aggregate the request counts on the first IP address in an HTTP header.
         *        </p>
         *        <p>
         *        With this option, you must specify the header to use in the <code>ForwardedIPConfig</code> property.
         *        </p>
         *        <p>
         *        To aggregate on a combination of the forwarded IP address with other aggregate keys, use
         *        <code>CUSTOM_KEYS</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IP</code> - Aggregate the request counts on the IP address from the web request origin.
         *        </p>
         *        <p>
         *        To aggregate on a combination of the IP address with other aggregate keys, use
         *        <code>CUSTOM_KEYS</code>.
         *        </p>
         *        </li>
         * @see RateBasedStatementAggregateKeyType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RateBasedStatementAggregateKeyType
         */
        Builder aggregateKeyType(RateBasedStatementAggregateKeyType aggregateKeyType);

        /**
         * <p>
         * An optional nested statement that narrows the scope of the web requests that are evaluated and managed by the
         * rate-based statement. When you use a scope-down statement, the rate-based rule only tracks and rate limits
         * requests that match the scope-down statement. You can use any nestable <a>Statement</a> in the scope-down
         * statement, and you can nest statements at any level, the same as you can for a rule statement.
         * </p>
         * 
         * @param scopeDownStatement
         *        An optional nested statement that narrows the scope of the web requests that are evaluated and managed
         *        by the rate-based statement. When you use a scope-down statement, the rate-based rule only tracks and
         *        rate limits requests that match the scope-down statement. You can use any nestable <a>Statement</a> in
         *        the scope-down statement, and you can nest statements at any level, the same as you can for a rule
         *        statement.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scopeDownStatement(Statement scopeDownStatement);

        /**
         * <p>
         * An optional nested statement that narrows the scope of the web requests that are evaluated and managed by the
         * rate-based statement. When you use a scope-down statement, the rate-based rule only tracks and rate limits
         * requests that match the scope-down statement. You can use any nestable <a>Statement</a> in the scope-down
         * statement, and you can nest statements at any level, the same as you can for a rule statement.
         * </p>
         * This is a convenience method that creates an instance of the {@link Statement.Builder} avoiding the need to
         * create one manually via {@link Statement#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Statement.Builder#build()} is called immediately and its result
         * is passed to {@link #scopeDownStatement(Statement)}.
         * 
         * @param scopeDownStatement
         *        a consumer that will call methods on {@link Statement.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #scopeDownStatement(Statement)
         */
        default Builder scopeDownStatement(Consumer<Statement.Builder> scopeDownStatement) {
            return scopeDownStatement(Statement.builder().applyMutation(scopeDownStatement).build());
        }

        /**
         * <p>
         * The configuration for inspecting IP addresses in an HTTP header that you specify, instead of using the IP
         * address that's reported by the web request origin. Commonly, this is the X-Forwarded-For (XFF) header, but
         * you can specify any header name.
         * </p>
         * <note>
         * <p>
         * If the specified header isn't present in the request, WAF doesn't apply the rule to the web request at all.
         * </p>
         * </note>
         * <p>
         * This is required if you specify a forwarded IP in the rule's aggregate key settings.
         * </p>
         * 
         * @param forwardedIPConfig
         *        The configuration for inspecting IP addresses in an HTTP header that you specify, instead of using the
         *        IP address that's reported by the web request origin. Commonly, this is the X-Forwarded-For (XFF)
         *        header, but you can specify any header name. </p> <note>
         *        <p>
         *        If the specified header isn't present in the request, WAF doesn't apply the rule to the web request at
         *        all.
         *        </p>
         *        </note>
         *        <p>
         *        This is required if you specify a forwarded IP in the rule's aggregate key settings.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder forwardedIPConfig(ForwardedIPConfig forwardedIPConfig);

        /**
         * <p>
         * The configuration for inspecting IP addresses in an HTTP header that you specify, instead of using the IP
         * address that's reported by the web request origin. Commonly, this is the X-Forwarded-For (XFF) header, but
         * you can specify any header name.
         * </p>
         * <note>
         * <p>
         * If the specified header isn't present in the request, WAF doesn't apply the rule to the web request at all.
         * </p>
         * </note>
         * <p>
         * This is required if you specify a forwarded IP in the rule's aggregate key settings.
         * </p>
         * This is a convenience method that creates an instance of the {@link ForwardedIPConfig.Builder} avoiding the
         * need to create one manually via {@link ForwardedIPConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ForwardedIPConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #forwardedIPConfig(ForwardedIPConfig)}.
         * 
         * @param forwardedIPConfig
         *        a consumer that will call methods on {@link ForwardedIPConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #forwardedIPConfig(ForwardedIPConfig)
         */
        default Builder forwardedIPConfig(Consumer<ForwardedIPConfig.Builder> forwardedIPConfig) {
            return forwardedIPConfig(ForwardedIPConfig.builder().applyMutation(forwardedIPConfig).build());
        }

        /**
         * <p>
         * Specifies the aggregate keys to use in a rate-base rule.
         * </p>
         * 
         * @param customKeys
         *        Specifies the aggregate keys to use in a rate-base rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customKeys(Collection<RateBasedStatementCustomKey> customKeys);

        /**
         * <p>
         * Specifies the aggregate keys to use in a rate-base rule.
         * </p>
         * 
         * @param customKeys
         *        Specifies the aggregate keys to use in a rate-base rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customKeys(RateBasedStatementCustomKey... customKeys);

        /**
         * <p>
         * Specifies the aggregate keys to use in a rate-base rule.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.wafv2.model.RateBasedStatementCustomKey.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.wafv2.model.RateBasedStatementCustomKey#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.wafv2.model.RateBasedStatementCustomKey.Builder#build()} is called
         * immediately and its result is passed to {@link #customKeys(List<RateBasedStatementCustomKey>)}.
         * 
         * @param customKeys
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.wafv2.model.RateBasedStatementCustomKey.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #customKeys(java.util.Collection<RateBasedStatementCustomKey>)
         */
        Builder customKeys(Consumer<RateBasedStatementCustomKey.Builder>... customKeys);
    }

    static final class BuilderImpl implements Builder {
        private Long limit;

        private Long evaluationWindowSec;

        private String aggregateKeyType;

        private Statement scopeDownStatement;

        private ForwardedIPConfig forwardedIPConfig;

        private List<RateBasedStatementCustomKey> customKeys = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(RateBasedStatement model) {
            limit(model.limit);
            evaluationWindowSec(model.evaluationWindowSec);
            aggregateKeyType(model.aggregateKeyType);
            scopeDownStatement(model.scopeDownStatement);
            forwardedIPConfig(model.forwardedIPConfig);
            customKeys(model.customKeys);
        }

        public final Long getLimit() {
            return limit;
        }

        public final void setLimit(Long limit) {
            this.limit = limit;
        }

        @Override
        public final Builder limit(Long limit) {
            this.limit = limit;
            return this;
        }

        public final Long getEvaluationWindowSec() {
            return evaluationWindowSec;
        }

        public final void setEvaluationWindowSec(Long evaluationWindowSec) {
            this.evaluationWindowSec = evaluationWindowSec;
        }

        @Override
        public final Builder evaluationWindowSec(Long evaluationWindowSec) {
            this.evaluationWindowSec = evaluationWindowSec;
            return this;
        }

        public final String getAggregateKeyType() {
            return aggregateKeyType;
        }

        public final void setAggregateKeyType(String aggregateKeyType) {
            this.aggregateKeyType = aggregateKeyType;
        }

        @Override
        public final Builder aggregateKeyType(String aggregateKeyType) {
            this.aggregateKeyType = aggregateKeyType;
            return this;
        }

        @Override
        public final Builder aggregateKeyType(RateBasedStatementAggregateKeyType aggregateKeyType) {
            this.aggregateKeyType(aggregateKeyType == null ? null : aggregateKeyType.toString());
            return this;
        }

        public final Statement.Builder getScopeDownStatement() {
            return scopeDownStatement != null ? scopeDownStatement.toBuilder() : null;
        }

        public final void setScopeDownStatement(Statement.BuilderImpl scopeDownStatement) {
            this.scopeDownStatement = scopeDownStatement != null ? scopeDownStatement.build() : null;
        }

        @Override
        public final Builder scopeDownStatement(Statement scopeDownStatement) {
            this.scopeDownStatement = scopeDownStatement;
            return this;
        }

        public final ForwardedIPConfig.Builder getForwardedIPConfig() {
            return forwardedIPConfig != null ? forwardedIPConfig.toBuilder() : null;
        }

        public final void setForwardedIPConfig(ForwardedIPConfig.BuilderImpl forwardedIPConfig) {
            this.forwardedIPConfig = forwardedIPConfig != null ? forwardedIPConfig.build() : null;
        }

        @Override
        public final Builder forwardedIPConfig(ForwardedIPConfig forwardedIPConfig) {
            this.forwardedIPConfig = forwardedIPConfig;
            return this;
        }

        public final List<RateBasedStatementCustomKey.Builder> getCustomKeys() {
            List<RateBasedStatementCustomKey.Builder> result = RateBasedStatementCustomKeysCopier.copyToBuilder(this.customKeys);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setCustomKeys(Collection<RateBasedStatementCustomKey.BuilderImpl> customKeys) {
            this.customKeys = RateBasedStatementCustomKeysCopier.copyFromBuilder(customKeys);
        }

        @Override
        public final Builder customKeys(Collection<RateBasedStatementCustomKey> customKeys) {
            this.customKeys = RateBasedStatementCustomKeysCopier.copy(customKeys);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder customKeys(RateBasedStatementCustomKey... customKeys) {
            customKeys(Arrays.asList(customKeys));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder customKeys(Consumer<RateBasedStatementCustomKey.Builder>... customKeys) {
            customKeys(Stream.of(customKeys).map(c -> RateBasedStatementCustomKey.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public RateBasedStatement build() {
            return new RateBasedStatement(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
