// Copyright (c) 2018 WSO2 Inc. (http://www.wso2.org) All Rights Reserved.
//
// WSO2 Inc. licenses this file to you under the Apache License,
// Version 2.0 (the "License"); you may not use this file except
// in compliance with the License.
// You may obtain a copy of the License at
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.


import ballerina/log;
import ballerina/auth;

@Description {value:"Representation of JWT Auth handler for HTTP traffic"}
@Field {value:"jwtAuthenticator: JWTAuthenticator instance"}
@Field {value:"name: Authentication handler name"}
public type HttpJwtAuthnHandler object {
    public {
        string name;
        auth:JWTAuthProvider jwtAuthenticator;
    }
    public new (jwtAuthenticator) {
        name = "jwt";
    }
    public function canHandle (Request req) returns (boolean);
    public function handle (Request req) returns (boolean);
};

@Description {value:"Intercepts a HTTP request for authentication"}
@Param {value:"req: Request object"}
@Return {value:"boolean: true if authentication is a success, else false"}
public function HttpJwtAuthnHandler::canHandle (Request req) returns (boolean) {
    string authHeader;
    try {
        authHeader = req.getHeader(AUTH_HEADER);
    } catch (error e) {
        log:printDebug("Error in retrieving header " + AUTH_HEADER + ": " + e.message);
        return false;
    }
    if (authHeader != null && authHeader.hasPrefix(AUTH_SCHEME_BEARER)) {
        string[] authHeaderComponents = authHeader.split(" ");
        if (lengthof authHeaderComponents == 2) {
            string[] jwtComponents = authHeaderComponents[1].split("\\.");
            if (lengthof jwtComponents == 3) {
                return true;
            }
        }
    }
    return false;
}

@Description {value:"Checks if the provided HTTP request can be authenticated with JWT authentication"}
@Param {value:"req: Request object"}
@Return {value:"boolean: true if its possible to authenticate with JWT auth, else false"}
public function HttpJwtAuthnHandler::handle (Request req) returns (boolean) {
    string jwtToken = extractJWTToken(req);
    var isAuthenticated = self.jwtAuthenticator.authenticate(jwtToken);
    match isAuthenticated {
        boolean authenticated => {
            return authenticated;
        }
        error err => {
            log:printError("Error while validating JWT token ", err = err);
            return false;
        }
    }
}

function extractJWTToken (Request req) returns (string) {
    string authHeader = req.getHeader(AUTH_HEADER);
    string[] authHeaderComponents = authHeader.split(" ");
    return authHeaderComponents[1];
}
