/*
 * Decompiled with CFR 0.152.
 */
package com.adobe.testing.s3mock.domain;

import com.adobe.testing.s3mock.domain.Bucket;
import com.adobe.testing.s3mock.domain.MultipartUploadInfo;
import com.adobe.testing.s3mock.domain.S3Object;
import com.adobe.testing.s3mock.dto.CopyObjectResult;
import com.adobe.testing.s3mock.dto.MultipartUpload;
import com.adobe.testing.s3mock.dto.Owner;
import com.adobe.testing.s3mock.util.AwsChunkDecodingInputStream;
import com.adobe.testing.s3mock.util.HashUtil;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.DirectoryStream;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.BasicFileAttributes;
import java.security.DigestInputStream;
import java.security.DigestOutputStream;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.codec.binary.Hex;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.io.input.BoundedInputStream;
import org.apache.log4j.Logger;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

@Component
public class FileStore {
    private static final SimpleDateFormat S3_OBJECT_DATE_FORMAT = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss'.000Z'");
    private static final String META_FILE = "metadata";
    private static final String DATA_FILE = "fileData";
    private static final String PART_SUFFIX = ".part";
    private static final String DEFAULT_CONTENT_TYPE = "application/octet-stream";
    private static final Logger LOG = Logger.getLogger(FileStore.class);
    private final File rootFolder;
    private final ObjectMapper objectMapper;
    private final Map<String, MultipartUploadInfo> uploadIdToInfo = new ConcurrentHashMap<String, MultipartUploadInfo>();

    public FileStore() throws IOException {
        this(new File(FileUtils.getTempDirectory(), "s3mockFileStore" + new Date().getTime()));
    }

    FileStore(File rootFolder) {
        this.rootFolder = rootFolder;
        rootFolder.deleteOnExit();
        rootFolder.mkdir();
        this.objectMapper = new ObjectMapper();
    }

    public Bucket createBucket(String bucketName) throws IOException {
        File newBucket = new File(this.rootFolder, bucketName);
        FileUtils.forceMkdir((File)newBucket);
        return this.bucketFromPath(newBucket.toPath());
    }

    public List<Bucket> listBuckets() {
        DirectoryStream.Filter<Path> filter = file -> Files.isDirectory(file, new LinkOption[0]);
        return this.findBucketsByFilter(filter);
    }

    public Bucket getBucket(String bucketName) {
        DirectoryStream.Filter<Path> filter = file -> Files.isDirectory(file, new LinkOption[0]) && file.getFileName().endsWith(bucketName);
        List<Bucket> buckets = this.findBucketsByFilter(filter);
        return buckets.size() > 0 ? buckets.get(0) : null;
    }

    private List<Bucket> findBucketsByFilter(DirectoryStream.Filter<Path> filter) {
        ArrayList<Bucket> buckets = new ArrayList<Bucket>();
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(this.rootFolder.toPath(), filter);){
            for (Path path : stream) {
                buckets.add(this.bucketFromPath(path));
            }
        }
        catch (IOException e) {
            LOG.error((Object)"Could not Iterate over Bucket-Folders", (Throwable)e);
        }
        return buckets;
    }

    private Bucket bucketFromPath(Path path) {
        Bucket result = null;
        try {
            BasicFileAttributes attributes = Files.readAttributes(path, BasicFileAttributes.class, new LinkOption[0]);
            result = new Bucket(path, path.getFileName().toString(), S3_OBJECT_DATE_FORMAT.format(new Date(attributes.creationTime().toMillis())));
        }
        catch (IOException e) {
            LOG.error((Object)"File can not be read!", (Throwable)e);
        }
        return result;
    }

    public S3Object putS3Object(String bucketName, String fileName, String contentType, InputStream dataStream, boolean useV4Signing) throws IOException {
        S3Object s3Object = new S3Object();
        s3Object.setName(fileName);
        s3Object.setContentType(contentType);
        Bucket theBucket = this.getBucketOrCreateNewOne(bucketName);
        File objectRootFolder = this.createObjectRootFolder(theBucket, s3Object.getName());
        File dataFile = this.inputStreamToFile(this.wrapStream(dataStream, useV4Signing), objectRootFolder.toPath().resolve(DATA_FILE));
        s3Object.setDataFile(dataFile);
        s3Object.setSize(Long.toString(dataFile.length()));
        BasicFileAttributes attributes = Files.readAttributes(dataFile.toPath(), BasicFileAttributes.class, new LinkOption[0]);
        s3Object.setCreationDate(S3_OBJECT_DATE_FORMAT.format(new Date(attributes.creationTime().toMillis())));
        s3Object.setModificationDate(S3_OBJECT_DATE_FORMAT.format(new Date(attributes.lastModifiedTime().toMillis())));
        s3Object.setLastModified(attributes.lastModifiedTime().toMillis());
        s3Object.setMd5(this.digest(null, dataFile));
        this.objectMapper.writeValue(new File(objectRootFolder, META_FILE), (Object)s3Object);
        return s3Object;
    }

    private InputStream wrapStream(InputStream dataStream, boolean useV4Signing) {
        InputStream inStream = useV4Signing ? new AwsChunkDecodingInputStream(dataStream) : dataStream;
        return inStream;
    }

    public S3Object putS3ObjectWithKMSEncryption(String bucketName, String fileName, String contentType, InputStream dataStream, boolean useV4Signing, String encryption, String kmsKeyId) throws IOException {
        S3Object s3Object = new S3Object();
        s3Object.setName(fileName);
        s3Object.setContentType(contentType);
        s3Object.setEncrypted(true);
        s3Object.setKmsEncryption(encryption);
        s3Object.setKmsEncryptionKeyId(kmsKeyId);
        Bucket theBucket = this.getBucketOrCreateNewOne(bucketName);
        File objectRootFolder = this.createObjectRootFolder(theBucket, s3Object.getName());
        File dataFile = this.inputStreamToFile(this.wrapStream(dataStream, useV4Signing), objectRootFolder.toPath().resolve(DATA_FILE));
        s3Object.setDataFile(dataFile);
        s3Object.setSize(Long.toString(dataFile.length()));
        BasicFileAttributes attributes = Files.readAttributes(dataFile.toPath(), BasicFileAttributes.class, new LinkOption[0]);
        s3Object.setCreationDate(S3_OBJECT_DATE_FORMAT.format(new Date(attributes.creationTime().toMillis())));
        s3Object.setModificationDate(S3_OBJECT_DATE_FORMAT.format(new Date(attributes.lastModifiedTime().toMillis())));
        s3Object.setLastModified(attributes.lastModifiedTime().toMillis());
        s3Object.setMd5(this.digest(kmsKeyId, dataFile));
        this.objectMapper.writeValue(new File(objectRootFolder, META_FILE), (Object)s3Object);
        return s3Object;
    }

    private Bucket getBucketOrCreateNewOne(String bucketName) throws IOException {
        Bucket theBucket = this.getBucket(bucketName);
        if (theBucket == null) {
            theBucket = this.createBucket(bucketName);
        }
        return theBucket;
    }

    private File createObjectRootFolder(Bucket theBucket, String objectName) {
        Path bucketPath = theBucket.getPath();
        File objectRootFolder = new File(bucketPath.toFile(), objectName);
        objectRootFolder.mkdirs();
        return objectRootFolder;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private File inputStreamToFile(InputStream inputStream, Path filePath) {
        FileOutputStream outputStream = null;
        File targetFile = filePath.toFile();
        try {
            int read;
            if (!targetFile.exists()) {
                targetFile.createNewFile();
            }
            outputStream = new FileOutputStream(targetFile);
            byte[] bytes = new byte[1024];
            while ((read = inputStream.read(bytes)) != -1) {
                ((OutputStream)outputStream).write(bytes, 0, read);
            }
        }
        catch (IOException e) {
            LOG.error((Object)"Wasn't able to store file on disk!", (Throwable)e);
        }
        finally {
            if (inputStream != null) {
                try {
                    inputStream.close();
                }
                catch (IOException e) {
                    LOG.error((Object)"InputStream can not be closed!", (Throwable)e);
                }
            }
            if (outputStream != null) {
                try {
                    ((OutputStream)outputStream).close();
                }
                catch (IOException e) {
                    LOG.error((Object)"OutputStream can not be closed!", (Throwable)e);
                }
            }
        }
        return targetFile;
    }

    public S3Object getS3Object(String bucketName, String objectName) {
        Bucket theBucket = this.getBucket(Objects.requireNonNull(bucketName, "bucketName == null"));
        S3Object theObject = null;
        Path metaPath = theBucket.getPath().resolve(objectName + "/" + META_FILE);
        if (Files.exists(metaPath, new LinkOption[0])) {
            try {
                theObject = (S3Object)this.objectMapper.readValue(metaPath.toFile(), S3Object.class);
                theObject.setDataFile(theBucket.getPath().resolve(objectName + "/" + DATA_FILE).toFile());
            }
            catch (IOException e) {
                LOG.error((Object)"File can not be read", (Throwable)e);
                e.printStackTrace();
            }
        }
        return theObject;
    }

    public List<S3Object> getS3Objects(String bucketName, String prefix) throws IOException {
        Bucket theBucket = this.getBucket(Objects.requireNonNull(bucketName, "bucketName == null"));
        ArrayList<S3Object> resultObjects = new ArrayList<S3Object>();
        Stream<Path> directoryHierarchy = Files.walk(theBucket.getPath(), new FileVisitOption[0]);
        Set collect = directoryHierarchy.filter(path -> path.toFile().isDirectory()).map(path -> theBucket.getPath().relativize((Path)path)).filter(path -> StringUtils.isEmpty((Object)prefix) || path.startsWith(prefix)).collect(Collectors.toSet());
        for (Path path2 : collect) {
            S3Object s3Object = this.getS3Object(bucketName, path2.toString());
            if (s3Object == null) continue;
            resultObjects.add(s3Object);
        }
        return resultObjects;
    }

    public CopyObjectResult copyS3Object(String sourceBucketName, String sourceObjectName, String destinationBucketName, String destinationObjectName) throws IOException {
        S3Object sourceObject = this.getS3Object(sourceBucketName, sourceObjectName);
        if (sourceObject == null) {
            return null;
        }
        S3Object copiedObject = this.putS3Object(destinationBucketName, destinationObjectName, sourceObject.getContentType(), new FileInputStream(sourceObject.getDataFile()), false);
        return new CopyObjectResult(copiedObject.getModificationDate(), copiedObject.getMd5());
    }

    public CopyObjectResult copyS3ObjectEncrypted(String sourceBucketName, String sourceObjectName, String destinationBucketName, String destinationObjectName, String encryption, String kmsKeyId) throws IOException {
        S3Object sourceObject = this.getS3Object(sourceBucketName, sourceObjectName);
        if (sourceObject == null) {
            return null;
        }
        S3Object copiedObject = this.putS3ObjectWithKMSEncryption(destinationBucketName, destinationObjectName, sourceObject.getContentType(), new FileInputStream(sourceObject.getDataFile()), false, encryption, kmsKeyId);
        return new CopyObjectResult(copiedObject.getModificationDate(), copiedObject.getMd5());
    }

    public Boolean doesBucketExist(String bucketName) {
        return this.getBucket(bucketName) != null;
    }

    public boolean deleteObject(String bucketName, String objectName) throws IOException {
        S3Object s3Object = this.getS3Object(bucketName, objectName);
        if (s3Object != null) {
            FileUtils.deleteDirectory((File)s3Object.getDataFile().getParentFile());
            return true;
        }
        return false;
    }

    public boolean deleteBucket(String bucketName) throws IOException {
        Bucket bucket = this.getBucket(bucketName);
        if (bucket != null) {
            FileUtils.deleteDirectory((File)bucket.getPath().toFile());
            return true;
        }
        return false;
    }

    public MultipartUpload prepareMultipartUpload(String bucketName, String fileName, String contentType, String uploadId, Owner owner, Owner initiator) {
        if (!Paths.get(this.rootFolder.getAbsolutePath(), bucketName, fileName, uploadId).toFile().mkdirs()) {
            throw new IllegalStateException("Directories for storing multipart uploads couldn't be created.");
        }
        MultipartUpload upload = new MultipartUpload(fileName, uploadId, owner, initiator, new Date());
        this.uploadIdToInfo.put(uploadId, new MultipartUploadInfo(upload, contentType));
        return upload;
    }

    public Collection<MultipartUpload> listMultipartUploads() {
        return this.uploadIdToInfo.values().stream().map(info -> info.upload).collect(Collectors.toList());
    }

    public void abortMultipartUpload(String bucketName, String fileName, String uploadId) {
        this.synchronizedUpload(uploadId, uploadInfo -> {
            try {
                File partFolder = Paths.get(this.rootFolder.getAbsolutePath(), bucketName, fileName, uploadId).toFile();
                FileUtils.deleteDirectory((File)partFolder);
                File entireFile = Paths.get(this.rootFolder.getAbsolutePath(), bucketName, fileName, DATA_FILE).toFile();
                FileUtils.deleteQuietly((File)entireFile);
                this.uploadIdToInfo.remove(uploadId);
                return null;
            }
            catch (IOException e) {
                throw new IllegalStateException("Could not delete multipart upload tmp data.", e);
            }
        });
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public String putPart(String bucketName, String fileName, String uploadId, String partNumber, InputStream inputStream, boolean useV4Signing) throws IOException {
        try (DigestInputStream digestingInputStream = new DigestInputStream(this.wrapStream(inputStream, useV4Signing), MessageDigest.getInstance("MD5"));){
            this.inputStreamToFile(digestingInputStream, Paths.get(this.rootFolder.getAbsolutePath(), bucketName, fileName, uploadId, partNumber + PART_SUFFIX));
            String string = new String(Hex.encodeHex((byte[])digestingInputStream.getMessageDigest().digest()));
            return string;
        }
        catch (NoSuchAlgorithmException e) {
            throw new IllegalStateException(e);
        }
    }

    public String completeMultipartUpload(String bucketName, String fileName, String uploadId) {
        return this.completeMultipartUpload(bucketName, fileName, uploadId, null, null);
    }

    public String completeMultipartUpload(String bucketName, String fileName, String uploadId, String encryption, String kmsKeyId) {
        return this.synchronizedUpload(uploadId, uploadInfo -> {
            S3Object s3Object = new S3Object();
            s3Object.setName(fileName);
            s3Object.setEncrypted(encryption != null || kmsKeyId != null);
            if (encryption != null) {
                s3Object.setKmsEncryption(encryption);
            }
            if (kmsKeyId != null) {
                s3Object.setKmsEncryptionKeyId(kmsKeyId);
            }
            File partFolder = Paths.get(this.rootFolder.getAbsolutePath(), bucketName, fileName, uploadId).toFile();
            File entireFile = Paths.get(this.rootFolder.getAbsolutePath(), bucketName, fileName, DATA_FILE).toFile();
            Object[] partNames = partFolder.list((dir, name) -> name.endsWith(PART_SUFFIX));
            Arrays.sort(partNames);
            try (DigestOutputStream targetStream = new DigestOutputStream(new FileOutputStream(entireFile), MessageDigest.getInstance("MD5"));){
                int size = 0;
                for (Object partName : partNames) {
                    size = (int)((long)size + Files.copy(Paths.get(partFolder.getAbsolutePath(), new String[]{partName}), targetStream));
                }
                FileUtils.deleteDirectory((File)partFolder);
                BasicFileAttributes attributes = Files.readAttributes(entireFile.toPath(), BasicFileAttributes.class, new LinkOption[0]);
                s3Object.setCreationDate(S3_OBJECT_DATE_FORMAT.format(new Date(attributes.creationTime().toMillis())));
                s3Object.setModificationDate(S3_OBJECT_DATE_FORMAT.format(new Date(attributes.lastModifiedTime().toMillis())));
                s3Object.setLastModified(attributes.lastModifiedTime().toMillis());
                s3Object.setMd5(new String(Hex.encodeHex((byte[])targetStream.getMessageDigest().digest())) + "-1");
                s3Object.setSize(Integer.toString(size));
                s3Object.setContentType(uploadInfo.contentType != null ? uploadInfo.contentType : DEFAULT_CONTENT_TYPE);
                this.uploadIdToInfo.remove(uploadId);
            }
            catch (IOException | NoSuchAlgorithmException e) {
                throw new IllegalStateException("Error finishing multipart upload", e);
            }
            try {
                this.objectMapper.writeValue(Paths.get(this.rootFolder.getAbsolutePath(), bucketName, fileName, META_FILE).toFile(), (Object)s3Object);
            }
            catch (IOException e) {
                throw new IllegalStateException("Could not write metadata-file", e);
            }
            return s3Object.getMd5();
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private <T> T synchronizedUpload(String uploadId, Function<MultipartUploadInfo, T> callback) {
        MultipartUploadInfo uploadInfo = this.uploadIdToInfo.get(uploadId);
        if (uploadInfo == null) {
            throw new IllegalArgumentException("Unknown upload " + uploadId);
        }
        MultipartUploadInfo multipartUploadInfo = uploadInfo;
        synchronized (multipartUploadInfo) {
            if (!this.uploadIdToInfo.containsKey(uploadId)) {
                throw new IllegalStateException("Upload " + uploadId + " was aborted or completed concurrently");
            }
            return callback.apply(uploadInfo);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private String digest(String salt, File dataFile) throws IOException {
        try (FileInputStream inputStream = new FileInputStream(dataFile);){
            String string = HashUtil.getDigest(salt, inputStream);
            return string;
        }
        catch (NoSuchAlgorithmException e) {
            LOG.error((Object)"Hash can not be calculated!", (Throwable)e);
            return null;
        }
    }

    public String copyPart(String bucket, String key, int from, int to, boolean useV4Signing, String partNumber, String destinationBucket, String destinationFilename, String uploadId) throws IOException {
        this.verifyMultipartUploadPreparation(destinationBucket, destinationFilename, uploadId);
        File targetPartFile = this.ensurePartFile(partNumber, destinationBucket, destinationFilename, uploadId);
        this.copyPart(bucket, key, from, to, useV4Signing, targetPartFile);
        return UUID.randomUUID().toString();
    }

    private void copyPart(String bucket, String key, int from, int to, boolean useV4Signing, File partFile) throws IOException {
        int len = to - from + 1;
        S3Object s3Object = this.resolveS3Object(bucket, key);
        try (InputStream sourceStream = this.wrapStream(FileUtils.openInputStream((File)s3Object.getDataFile()), useV4Signing);
             FileOutputStream targetStream = new FileOutputStream(partFile);){
            sourceStream.skip(from);
            IOUtils.copy((InputStream)new BoundedInputStream(sourceStream, (long)len), (OutputStream)targetStream);
        }
    }

    private File ensurePartFile(String partNumber, String destinationBucket, String destinationFilename, String uploadId) throws IOException {
        File partFile = Paths.get(this.rootFolder.getAbsolutePath(), destinationBucket, destinationFilename, uploadId, partNumber + PART_SUFFIX).toFile();
        if (!partFile.exists() && !partFile.createNewFile()) {
            throw new IllegalStateException("Could not create buffer file");
        }
        return partFile;
    }

    private void verifyMultipartUploadPreparation(String destinationBucket, String destinationFilename, String uploadId) {
        Path partsFolder = Paths.get(this.rootFolder.getAbsolutePath(), destinationBucket, destinationFilename, uploadId);
        if (!partsFolder.toFile().exists() || !partsFolder.toFile().isDirectory()) {
            throw new IllegalStateException("Missed preparing Multipart Request");
        }
    }

    private S3Object resolveS3Object(String bucket, String key) {
        S3Object s3Object = this.getS3Object(bucket, key);
        if (s3Object == null) {
            throw new IllegalStateException("Source Object not found");
        }
        return s3Object;
    }
}

