/*
 * Decompiled with CFR 0.152.
 */
package com.microsoft.azure.kusto.ingest;

import com.microsoft.azure.kusto.data.Client;
import com.microsoft.azure.kusto.data.KustoOperationResult;
import com.microsoft.azure.kusto.data.KustoResultSetTable;
import com.microsoft.azure.kusto.data.exceptions.DataClientException;
import com.microsoft.azure.kusto.data.exceptions.DataServiceException;
import com.microsoft.azure.kusto.ingest.exceptions.IngestionClientException;
import com.microsoft.azure.kusto.ingest.exceptions.IngestionServiceException;
import java.io.Closeable;
import java.lang.invoke.MethodHandles;
import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class ResourceManager
implements Closeable {
    private Map<ResourceType, IngestionResource> ingestionResources;
    private String identityToken;
    private final Client client;
    private final Logger log = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    private final Timer timer;
    private ReadWriteLock ingestionResourcesLock = new ReentrantReadWriteLock();
    private ReadWriteLock authTokenLock = new ReentrantReadWriteLock();
    private static final long REFRESH_INGESTION_RESOURCES_PERIOD = 3600000L;
    private static final long REFRESH_INGESTION_RESOURCES_PERIOD_ON_FAILURE = 900000L;
    private final Long defaultRefreshTime;
    private final Long refreshTimeOnFailure;
    public static final String SERVICE_TYPE_COLUMN_NAME = "ServiceType";

    ResourceManager(Client client, long defaultRefreshTime, long refreshTimeOnFailure) {
        this.defaultRefreshTime = defaultRefreshTime;
        this.refreshTimeOnFailure = refreshTimeOnFailure;
        this.client = client;
        this.timer = new Timer(true);
        this.init();
    }

    ResourceManager(Client client) {
        this(client, 3600000L, 900000L);
    }

    @Override
    public void close() {
        this.timer.cancel();
        this.timer.purge();
    }

    private void init() {
        this.ingestionResources = Collections.synchronizedMap(new EnumMap(ResourceType.class));
        class RefreshIngestionAuthTokenTask
        extends TimerTask {
            RefreshIngestionAuthTokenTask() {
            }

            @Override
            public void run() {
                try {
                    ResourceManager.this.refreshIngestionAuthToken();
                    ResourceManager.this.timer.schedule((TimerTask)new RefreshIngestionAuthTokenTask(), ResourceManager.this.defaultRefreshTime);
                }
                catch (Exception e) {
                    ResourceManager.this.log.error("Error in refreshIngestionAuthToken.", (Throwable)e);
                    ResourceManager.this.timer.schedule((TimerTask)new RefreshIngestionAuthTokenTask(), ResourceManager.this.refreshTimeOnFailure);
                }
            }
        }
        this.timer.schedule((TimerTask)new RefreshIngestionAuthTokenTask(), 0L);
        class RefreshIngestionResourcesTask
        extends TimerTask {
            RefreshIngestionResourcesTask() {
            }

            @Override
            public void run() {
                try {
                    ResourceManager.this.refreshIngestionResources();
                    ResourceManager.this.timer.schedule((TimerTask)new RefreshIngestionResourcesTask(), ResourceManager.this.defaultRefreshTime);
                }
                catch (Exception e) {
                    ResourceManager.this.log.error("Error in refreshIngestionResources.", (Throwable)e);
                    ResourceManager.this.timer.schedule((TimerTask)new RefreshIngestionResourcesTask(), ResourceManager.this.refreshTimeOnFailure);
                }
            }
        }
        this.timer.schedule((TimerTask)new RefreshIngestionResourcesTask(), 0L);
    }

    String getIngestionResource(ResourceType resourceType) throws IngestionServiceException, IngestionClientException {
        IngestionResource ingestionResource = this.ingestionResources.get((Object)resourceType);
        if (ingestionResource == null) {
            this.refreshIngestionResources();
            try {
                this.ingestionResourcesLock.readLock().lock();
                ingestionResource = this.ingestionResources.get((Object)resourceType);
                if (ingestionResource == null) {
                    throw new IngestionServiceException("Unable to get ingestion resources for this type: " + resourceType.getResourceTypeName());
                }
            }
            finally {
                this.ingestionResourcesLock.readLock().unlock();
            }
        }
        return ingestionResource.nextStorageUrl();
    }

    String getIdentityToken() throws IngestionServiceException, IngestionClientException {
        if (this.identityToken == null) {
            this.refreshIngestionAuthToken();
            try {
                this.authTokenLock.readLock().lock();
                if (this.identityToken == null) {
                    throw new IngestionServiceException("Unable to get Identity token");
                }
            }
            finally {
                this.authTokenLock.readLock().unlock();
            }
        }
        return this.identityToken;
    }

    private void addIngestionResource(String resourceTypeName, String storageUrl) {
        ResourceType resourceType = ResourceType.findByResourceTypeName(resourceTypeName);
        if (resourceType != null) {
            if (!this.ingestionResources.containsKey((Object)resourceType)) {
                this.ingestionResources.put(resourceType, new IngestionResource(resourceType));
            }
            this.ingestionResources.get((Object)resourceType).addStorageUrl(storageUrl);
        }
    }

    private void refreshIngestionResources() throws IngestionClientException, IngestionServiceException {
        if (this.ingestionResourcesLock.writeLock().tryLock()) {
            try {
                this.log.info("Refreshing Ingestion Resources");
                KustoOperationResult ingestionResourcesResults = this.client.execute(".show ingestion resources");
                this.ingestionResources = Collections.synchronizedMap(new EnumMap(ResourceType.class));
                if (ingestionResourcesResults != null && ingestionResourcesResults.hasNext()) {
                    KustoResultSetTable table = ingestionResourcesResults.next();
                    while (table.next()) {
                        String resourceTypeName = table.getString(0);
                        String storageUrl = table.getString(1);
                        this.addIngestionResource(resourceTypeName, storageUrl);
                    }
                }
            }
            catch (DataServiceException e) {
                throw new IngestionServiceException(e.getIngestionSource(), "Error refreshing IngestionResources", (Exception)((Object)e));
            }
            catch (DataClientException e) {
                throw new IngestionClientException(e.getIngestionSource(), "Error refreshing IngestionResources", (Exception)((Object)e));
            }
            finally {
                this.ingestionResourcesLock.writeLock().unlock();
            }
        }
    }

    private void refreshIngestionAuthToken() throws IngestionClientException, IngestionServiceException {
        if (this.authTokenLock.writeLock().tryLock()) {
            try {
                this.log.info("Refreshing Ingestion Auth Token");
                KustoOperationResult identityTokenResult = this.client.execute(".get kusto identity token");
                if (identityTokenResult != null && identityTokenResult.hasNext() && !identityTokenResult.getResultTables().isEmpty()) {
                    KustoResultSetTable resultTable = identityTokenResult.next();
                    resultTable.next();
                    this.identityToken = resultTable.getString(0);
                }
            }
            catch (DataServiceException e) {
                throw new IngestionServiceException(e.getIngestionSource(), "Error refreshing IngestionAuthToken", (Exception)((Object)e));
            }
            catch (DataClientException e) {
                throw new IngestionClientException(e.getIngestionSource(), "Error refreshing IngestionAuthToken", (Exception)((Object)e));
            }
            finally {
                this.authTokenLock.writeLock().unlock();
            }
        }
    }

    protected String retrieveServiceType() throws IngestionServiceException, IngestionClientException {
        this.log.info("Getting version to determine endpoint's ServiceType");
        try {
            KustoOperationResult versionResult = this.client.execute(".show version");
            if (versionResult != null && versionResult.hasNext() && !versionResult.getResultTables().isEmpty()) {
                KustoResultSetTable resultTable = versionResult.next();
                resultTable.next();
                return resultTable.getString(SERVICE_TYPE_COLUMN_NAME);
            }
        }
        catch (DataServiceException e) {
            throw new IngestionServiceException(e.getIngestionSource(), "Couldn't retrieve ServiceType because of a service exception executing '.show version'", (Exception)((Object)e));
        }
        catch (DataClientException e) {
            throw new IngestionClientException(e.getIngestionSource(), "Couldn't retrieve ServiceType because of a client exception executing '.show version'", (Exception)((Object)e));
        }
        throw new IngestionServiceException("Couldn't retrieve ServiceType because '.show version' didn't return any records");
    }

    private static class IngestionResource {
        ResourceType resourceType;
        int roundRobinIdx = 0;
        List<String> storageUrls;

        IngestionResource(ResourceType resourceType) {
            this.resourceType = resourceType;
            this.storageUrls = new ArrayList<String>();
        }

        void addStorageUrl(String storageUrl) {
            this.storageUrls.add(storageUrl);
        }

        String nextStorageUrl() {
            this.roundRobinIdx = (this.roundRobinIdx + 1) % this.storageUrls.size();
            return this.storageUrls.get(this.roundRobinIdx);
        }
    }

    public static enum ResourceType {
        SECURED_READY_FOR_AGGREGATION_QUEUE("SecuredReadyForAggregationQueue"),
        FAILED_INGESTIONS_QUEUE("FailedIngestionsQueue"),
        SUCCESSFUL_INGESTIONS_QUEUE("SuccessfulIngestionsQueue"),
        TEMP_STORAGE("TempStorage"),
        INGESTIONS_STATUS_TABLE("IngestionsStatusTable");

        private final String resourceTypeName;

        private ResourceType(String resourceTypeName) {
            this.resourceTypeName = resourceTypeName;
        }

        String getResourceTypeName() {
            return this.resourceTypeName;
        }

        public static ResourceType findByResourceTypeName(String resourceTypeName) {
            for (ResourceType resourceType : ResourceType.values()) {
                if (!resourceType.resourceTypeName.equalsIgnoreCase(resourceTypeName)) continue;
                return resourceType;
            }
            return null;
        }
    }
}

