package com.vaadin.uitest.ai;

import java.util.HashMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vaadin.uitest.ai.prompts.AiPrompts;
import com.vaadin.uitest.model.Framework;
import com.vaadin.uitest.model.TestFramework;

/**
 * A LLM service that generates a response based on a prompt. All
 * responsibilities related to the model usage have to be implemented in this
 * service. This could be APIKEY providing, parameter setting, prompt template
 * generation, etc.
 */
public interface LLMService {

    public static final String PARSER_ASSISTANT_QUESTION = "PARSER_ASSISTANT_QUESTION";

    public static final String PARSER_SYSTEM_MSG_LIT = "PARSER_SYSTEM_MSG_LIT";
    public static final String PARSER_SYSTEM_MSG_REACT = "PARSER_SYSTEM_MSG_REACT";
    static final String PARSER_SYSTEM_MSG_FLOW = "PARSER_SYSTEM_MSG_FLOW";

    static final String GENERATE_LOGIN_PLAYWRIGHT_REACT = "GENERATE_LOGIN_PLAYWRIGHT_REACT";
    public static final String GENERATE_LOGIN_PLAYWRIGHT_JAVA = "GENERATE_LOGIN_PLAYWRIGHT_JAVA";

    public static final String GENERATE_IMPORTS_PLAYWRIGHT_REACT = "GENERATE_IMPORTS_PLAYWRIGHT_REACT";
    public static final String GENERATE_IMPORTS_PLAYWRIGHT_JAVA = "GENERATE_IMPORTS_PLAYWRIGHT_JAVA";
    public static final String GENERATE_SNIPPETS = "GENERATE_SNIPPETS";
    public static final String GENERATE_IMPORTS = "GENERATE_IMPORTS";
    public static final String GENERATE_HEADING_PLAYWRIGHT_REACT = "GENERATE_HEADING_PLAYWRIGHT_REACT";
    public static final String GENERATE_HEADING_PLAYWRIGHT_JAVA = "GENERATE_HEADING_PLAYWRIGHT_JAVA";

    Logger LOGGER = LoggerFactory.getLogger(LLMService.class);

    class ServiceLocator {

        private static final HashMap<Class<?>, LLMService> SERVICES = new HashMap<>();

        public static <T extends LLMService> T createService(Class<T> clazz) {
            LLMService service = SERVICES.get(clazz);
            if (!clazz.isInstance(service)) {
                try {
                    service = clazz.getDeclaredConstructor().newInstance();
                    SERVICES.put(clazz, service);
                } catch (Exception e) {
                    throw new RuntimeException(e);
                }
            }
            return clazz.cast(service);
        }

        public static <T extends LLMService> void setService(Class<T> clazz,
                LLMService instance) {
            if (instance == null) {
                SERVICES.remove(clazz);
            } else {
                SERVICES.put(clazz, instance);
            }
        }

        public static <T extends LLMService> void reset(Class<T> clazz) {
            if (clazz == null) {
                SERVICES.clear();
            } else {
                setService(clazz, null);
            }
        }
    }

    /**
     * Generates a response based on the input prompt from the AI module.
     *
     * @param prompt
     *            the prompt to be used by the AI module
     * @param gherkin
     * @return the generated response from the AI module.
     */
    default String getGeneratedResponse(String prompt, Framework framework,
            String gherkin) {
        AiArguments aiArguments = new AiArguments();
        aiArguments.setPrompt(prompt);
        aiArguments.setModel(getModel());
        aiArguments.setFramework(framework);
        aiArguments.setGherkin(gherkin);
        return getGeneratedResponse(aiArguments);
    }

    default String getGeneratedResponse(AiArguments aiArguments) {
        double start = System.currentTimeMillis();

        LOGGER.info("request prompt: {} Bytes - {} Words, model: {}",
                aiArguments.getPrompt().length(),
                aiArguments.getPrompt().split("[^\\w]").length,
                aiArguments.getModel());

        if (Boolean.getBoolean("ai.debug")) {
            LOGGER.info("--- AI PROMPT:\n{}", aiArguments.getPrompt());
        }
        String aiResponse = requestAI(aiArguments);
        if (Boolean.getBoolean("ai.debug")) {
            LOGGER.info("--- AI RESPONSE:\n{}", aiResponse);
        }
        LOGGER.info("response: {} Bytes - {} Words, {} secs",
                aiResponse.length(), aiResponse.split("[^\\w]").length,
                (System.currentTimeMillis() - start) / 1000);
        return extractCode(aiResponse).trim();
    }

    String requestAI(AiArguments aiArguments);

    /**
     * ID of the model to use.
     *
     * @return model
     */
    default String getModel() {
        return System.getProperty("model", "gpt-4-1106-preview");
    }

    default AiPrompts prompts() {
        return new AiPrompts(this.getClass());
    }

    /**
     * What sampling temperature to use, between 0 and 2. Higher values like 0.8
     * will make the output more random, while lower values like 0.2 will make
     * it more focused and deterministic.
     *
     * @return temperature
     */
    default double getTemperature() {
        return Double.parseDouble(System.getProperty("temperature", "0"));
    }

    /**
     * The maximum number of tokens to generate in the completion.
     *
     * @return max tokens limit
     */
    default int getMaxTokens() {
        return Integer.parseInt(System.getProperty("maxTokens", "2000"));
    }

    /**
     * Timeout for AI module response in seconds
     *
     * @return timeout in seconds
     */
    static int getTimeout() {
        return Integer.parseInt(System.getProperty("timeout", "160"));
    }

    static String extractCode(String markdown) {
        String patternString = "```(?:java|html|javascript|typescript|gherkin)?(.*?)```";
        Pattern pattern = Pattern.compile(patternString, Pattern.DOTALL);
        Matcher matcher = pattern.matcher(markdown);
        if (matcher.find()) {
            return matcher.group(1);
        }
        return markdown;
    }

    default String getPromptTemplate(AiArguments aiArguments) {
        return null;
    }

    default String getChatAssistantSystemMessage(String name,
            Framework framework) {
        switch (framework) {
        case FLOW -> {
            return String.format(prompts().get(PARSER_SYSTEM_MSG_FLOW), name);
        }
        case LIT -> {
            return String.format(prompts().get(PARSER_SYSTEM_MSG_LIT), name);
        }
        case REACT -> {
            return String.format(prompts().get(PARSER_SYSTEM_MSG_REACT), name);
        }
        }
        return "";
    }

    default String getImports(TestFramework testFramework) {
        switch (testFramework) {
        case PLAYWRIGHT_JAVA -> prompts().get(GENERATE_IMPORTS_PLAYWRIGHT_JAVA);
        case PLAYWRIGHT_NODE ->
            prompts().get(GENERATE_IMPORTS_PLAYWRIGHT_REACT);
        default -> {
            return null;
        }
        }
        return null;
    }

}