package com.vaadin.uitest.generator;

import static com.vaadin.uitest.generator.utils.CopyUtils.copyResourceAndReplaceContent;
import static com.vaadin.uitest.generator.utils.CopyUtils.removeOldResources;
import static com.vaadin.uitest.generator.utils.MvnUtils.addMavenDependency;
import static com.vaadin.uitest.generator.utils.MvnUtils.addMavenExecPlugin;
import static com.vaadin.uitest.generator.utils.MvnUtils.addMavenFailsafePlugin;
import static com.vaadin.uitest.generator.utils.NpmUtils.addPackageProperty;

import java.io.File;
import java.io.IOException;
import java.util.List;
import java.util.stream.Collectors;

import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.exc.StreamReadException;
import com.fasterxml.jackson.databind.DatabindException;
import com.vaadin.uitest.ai.AiArguments;
import com.vaadin.uitest.ai.LLMService;
import com.vaadin.uitest.model.Framework;
import com.vaadin.uitest.model.TestFramework;
import com.vaadin.uitest.model.UiLogin;
import com.vaadin.uitest.model.UiModel;
import com.vaadin.uitest.model.UiRoute;
import com.vaadin.uitest.parser.Parser;

public interface Generator {

    static String PLAYWRIGHT_CONFIG = "playwright.config.ts";

    Logger logger = LoggerFactory.getLogger(Generator.class);

    default LLMService getService() {
        return null;
    }

    static UiModel parseJson(File json)
            throws IOException, StreamReadException, DatabindException {
        return Parser.objectMapper.readValue(json, UiModel.class);
    }

    default String generateTest(UiRoute route, UiLogin login,
            TestFramework testFramework) {
        if (route.getGherkin() == null) {
            throw new IllegalArgumentException(
                    " There is no Gherkin info in JSON for the view "
                            + route.getClassName());
        }

        double start = System.currentTimeMillis();

        Framework framework = Framework.getByValue(route.getFramework());
        if (testFramework == null) {
            testFramework = framework.getDefaultTestFramework();
        }

        AiArguments aiArguments = new AiArguments();
        aiArguments.setGherkin(route.getGherkin());
        aiArguments.setLogin(login);
        aiArguments.setFramework(framework);
        aiArguments.setViewUrl(route.getBaseUrl() + route.getRoute());
        aiArguments.setTestClassName(route.getTestClass());
        aiArguments.setImports(getService().getImports(testFramework));
        aiArguments.setTestFramework(testFramework);

        String prompt = getService().getPromptTemplate(aiArguments);
        logger.info(String.format(
                "Generating a test for the view %s (%s) Gherkin: %d Bytes - %d Words",
                route.getSimpleName(), route.getRoute(),
                route.getGherkin().length(),
                route.getGherkin().split("[^\\w]+").length));

        String response = getService().getGeneratedResponse(prompt, framework,
                route.getGherkin());
        logger.info(String.format("Response took: %s sec. %d Bytes - %d Words",
                (System.currentTimeMillis() - start) / 1000, response.length(),
                response.split("[^\\w]+").length));
        return response;
    }

    default File writeTestFile(UiRoute route, String source) throws Exception {
        File file = new File(route.getTestfile());
        if (route.getProjectRoot() != null) {
            file = new File(route.getProjectRoot(), route.getTestfile());
        }
        logger.info("Writing: {}", file);
        FileUtils.createParentDirectories(file);
        FileUtils.writeStringToFile(file, source, "utf-8");
        return file;
    }

    default File generateTest(UiRoute route, File testsDir, boolean keep,
            UiLogin login, TestFramework testFramework) throws Exception {

        route.computeTestName(testsDir, testFramework);

        if (new File(route.getTestfile()).exists() && keep) {
            logger.error("Test already exists for view: "
                    + route.getSimpleName() + " " + route.getTestfile());
            return null;
        }

        String response = generateTest(route, login, testFramework);

        return writeTestFile(route, response);
    }

    default boolean addTestDependencies(String projectDir, String testsDir,
            List<UiRoute> views, TestFramework testFramework) throws Exception {
        boolean changed = false;

        String pom = new File(projectDir, "pom.xml").getAbsolutePath();

        List<UiRoute> javaViews = views.stream()
                .filter(v -> "flow".equals(v.getFramework()))
                .collect(Collectors.toList());
        List<UiRoute> tsViews = views.stream()
                .filter(v -> ("react".equals(v.getFramework())
                        || "lite".equals(v.getFramework())))
                .collect(Collectors.toList());

        if (javaViews.size() > 0) {
            changed = addMavenFailsafePlugin(pom);
            changed = addMavenDependency(pom, "com.microsoft.playwright",
                    "playwright", "LATEST", "test") || changed;
            changed = addMavenDependency(pom, "org.junit.jupiter",
                    "junit-jupiter-api", "LATEST", "test") || changed;
            changed = addMavenDependency(pom, "org.junit.jupiter",
                    "junit-jupiter", "LATEST", "test") || changed;
            changed = addMavenDependency(pom, "org.junit.platform",
                    "junit-platform-engine", "LATEST", "test") || changed;
            // We are not using base class anymore
            // copySources("com/vaadin/uitest/common", framework, testsDir);
            changed = removeOldResources("com/vaadin/uitest/common", testsDir)
                    || changed;
        }
        if (tsViews.size() > 0) {
            changed = addMavenExecPlugin(pom) || changed;
            changed = addPackageProperty(projectDir, "scripts", "test",
                    "npx playwright install chromium && npx playwright test")
                    || changed;
            changed = addPackageProperty(projectDir, "devDependencies",
                    "@playwright/test", "latest") || changed;
            changed = addPackageProperty(projectDir, "devDependencies",
                    "@types/node", "latest") || changed;

            UiRoute r = tsViews.get(tsViews.size() - 1);
            String runServer = System.getProperty("os.name").toLowerCase()
                    .contains("windows") ? "mvnw.cmd" : "mvn";
            runServer = new File(projectDir, runServer).canExecute() ? runServer
                    : "mvn";
            runServer += "-B -ntp -Dspring-boot.run.jvmArguments='-Dvaadin.launch-browser=false'";

            String relativeTestsDir = new File(projectDir).toPath()
                    .relativize(new File(r.getTestsDir()).toPath()).toString()
                    .replace('\\', '/');

            changed = copyResourceAndReplaceContent(PLAYWRIGHT_CONFIG,
                    projectDir, "BASE_URL", r.getBaseUrl(),
                    "FRONTEND_TEST_FOLDER", relativeTestsDir, "START_SERVER",
                    runServer) || changed;
        }
        return changed;
    }

}
