package com.vaadin.uitest.generator.utils;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.util.Collections;
import java.util.Enumeration;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vaadin.uitest.model.TestFramework;

public class CopyUtils {

    private static final Logger LOGGER = LoggerFactory
            .getLogger(CopyUtils.class);

    public static boolean copyResourceAndReplaceContent(String resource,
            String target, String... args) throws IOException {
        ClassLoader classLoader = Thread.currentThread()
                .getContextClassLoader();
        InputStream stream = classLoader.getResourceAsStream(resource);
        if (stream == null) {
            return false;
        }
        String content = IOUtils.toString(stream, "UTF-8");
        for (int i = 0; i < args.length; i += 2) {
            content = content.replaceAll(args[i], args[i + 1]);
        }
        String action = "Created";
        File targetFile = new File(target, resource);
        if (targetFile.canRead()) {
            String current = FileUtils.readFileToString(targetFile, "UTF-8");
            if (current.equals(content)) {
                return false;
            }
            action = "Updated";
        }
        FileUtils.writeStringToFile(targetFile, content, "UTF-8");
        LOGGER.info("{} file: {}", action, targetFile);
        return true;
    }

    public static boolean copyResource(String resource, String target)
            throws IOException {
        ClassLoader classLoader = Thread.currentThread()
                .getContextClassLoader();
        InputStream stream = classLoader.getResourceAsStream(resource);
        if (stream != null) {
            String action = "Created";
            Path destination = Paths.get(target, resource);
            if (destination.toFile().exists()) {
                InputStream currentStream = new FileInputStream(
                        destination.toFile());
                if (IOUtils.contentEquals(stream, currentStream)) {
                    return false;
                }
                destination.toFile().delete();
                action = "Updated";
            }
            stream = classLoader.getResourceAsStream(resource);
            Files.createDirectories(destination.getParent());
            try (OutputStream outputStream = Files
                    .newOutputStream(destination)) {
                IOUtils.copy(stream, outputStream);
            }
            LOGGER.info("{} file: {}", action, destination);
            return true;
        }
        return false;
    }

    public static boolean removeOldResources(String resource, String target)
            throws IOException {
        File folder = new File(resource, target);
        if (folder.exists()) {
            FileUtils.deleteDirectory(folder);
            LOGGER.info(
                    "Remove resources in folder that are not used anymore:{}",
                    folder);
            return true;
        }
        return false;
    }

    public static void copySources(String classpathResourcePath,
            TestFramework testFramework, String target)
            throws IOException, URISyntaxException {
        ClassLoader classLoader = Thread.currentThread()
                .getContextClassLoader();
        Enumeration<URL> resources = classLoader
                .getResources(classpathResourcePath);
        Set<URL> urls = Collections.list(resources).stream()
                .collect(Collectors.toSet());

        for (URL url : urls) {
            Path walkPath;
            Path basePath;
            if ("file".equals(url.getProtocol())) {
                walkPath = Paths.get(url.toURI());
                basePath = Paths.get(walkPath.toString()
                        .replaceFirst(classpathResourcePath + "$", ""));
            } else if ("jar".equals(url.getProtocol())) {
                walkPath = FileSystems
                        .newFileSystem(url.toURI(), Collections.emptyMap())
                        .getPath(classpathResourcePath);
                basePath = null;
            } else {
                continue;
            }
            Files.walk(walkPath).filter(Files::isRegularFile).forEach(file -> {
                try {
                    String name = file.getFileName().toString();
                    // Do not copy Base files for other frameworks, and non
                    // .java files
                    if (name.startsWith("Base")
                            && !name.startsWith(
                                    getBaseTestClassName(testFramework))
                            || (!name.endsWith(".java")
                                    && !name.endsWith(".ts"))) {
                        return;
                    }
                    String action = "Created";
                    Path relativePath = basePath == null ? file
                            : basePath.relativize(file);

                    Path destination = Paths.get(target,
                            relativePath.toString());
                    if (destination.toFile().exists()) {
                        if (FileUtils.contentEquals(file.toFile(),
                                destination.toFile())) {
                            return;
                        }
                        destination.toFile().delete();
                        action = "Updated";
                    }

                    Files.createDirectories(destination.getParent());
                    Files.copy(file, destination,
                            StandardCopyOption.REPLACE_EXISTING);
                    LOGGER.info("{} file: {}", action, destination);
                } catch (IOException e) {
                    LOGGER.error(String.format("Failed create file: %s", file),
                            e);
                }
            });
        }
    }

    private static String getBaseTestClassName(TestFramework testFramework) {
        if (TestFramework.TEST_BENCH.equals(testFramework)) {
            return "BaseTestBench";
        }
        return "BasePlaywright";
    }
}