/*
 * Decompiled with CFR 0.152.
 */
package ratpack.session.clientside.internal;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import com.google.common.collect.Ordering;
import com.google.inject.Inject;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufAllocator;
import io.netty.buffer.ByteBufUtil;
import io.netty.buffer.Unpooled;
import io.netty.handler.codec.base64.Base64;
import io.netty.handler.codec.base64.Base64Dialect;
import io.netty.handler.codec.http.cookie.Cookie;
import io.netty.util.AsciiString;
import io.netty.util.CharsetUtil;
import java.nio.CharBuffer;
import java.nio.charset.Charset;
import java.util.Optional;
import javax.inject.Provider;
import ratpack.exec.Operation;
import ratpack.exec.Promise;
import ratpack.http.Request;
import ratpack.http.Response;
import ratpack.session.SessionCookieConfig;
import ratpack.session.SessionStore;
import ratpack.session.clientside.ClientSideSessionConfig;
import ratpack.session.clientside.Crypto;
import ratpack.session.clientside.Signer;

public class ClientSideSessionStore
implements SessionStore {
    private static final String SESSION_SEPARATOR = ":";
    private final Provider<Request> request;
    private final Provider<Response> response;
    private final Signer signer;
    private final Crypto crypto;
    private final ByteBufAllocator bufferAllocator;
    private final SessionCookieConfig cookieConfig;
    private final ClientSideSessionConfig config;
    private final CookieOrdering latCookieOrdering;
    private final CookieOrdering dataCookieOrdering;
    private final long expirySeconds;

    @Inject
    public ClientSideSessionStore(Provider<Request> request, Provider<Response> response, Signer signer, Crypto crypto, ByteBufAllocator bufferAllocator, SessionCookieConfig cookieConfig, ClientSideSessionConfig config) {
        this.request = request;
        this.response = response;
        this.signer = signer;
        this.crypto = crypto;
        this.bufferAllocator = bufferAllocator;
        this.cookieConfig = cookieConfig;
        this.config = config;
        this.expirySeconds = cookieConfig.getExpires() == null ? 0L : cookieConfig.getExpires().getSeconds();
        this.latCookieOrdering = new CookieOrdering(config.getLastAccessTimeCookieName());
        this.dataCookieOrdering = new CookieOrdering(config.getSessionCookieName());
    }

    @Override
    public Operation store(AsciiString sessionId, ByteBuf sessionData) {
        return Operation.of(() -> {
            int i;
            CookieStorage cookieStorage = this.getCookieStorage();
            int oldSessionCookiesCount = cookieStorage.data.size();
            String[] sessionCookiePartitions = this.serialize(sessionData);
            for (i = 0; i < sessionCookiePartitions.length; ++i) {
                this.setCookie(this.config.getSessionCookieName() + "_" + i, sessionCookiePartitions[i]);
            }
            for (i = sessionCookiePartitions.length; i < oldSessionCookiesCount; ++i) {
                this.invalidateCookie(this.config.getSessionCookieName() + "_" + i);
            }
            this.setLastAccessTime(cookieStorage);
        });
    }

    @Override
    public Promise<ByteBuf> load(AsciiString sessionId) {
        return Promise.sync(() -> {
            CookieStorage cookieStorage = this.getCookieStorage();
            if (!this.isValid(cookieStorage)) {
                return Unpooled.EMPTY_BUFFER;
            }
            this.setLastAccessTime(cookieStorage);
            return this.deserialize((ImmutableList<Cookie>)cookieStorage.data);
        });
    }

    @Override
    public Operation remove(AsciiString sessionId) {
        return Operation.of(() -> this.reset(this.getCookieStorage()));
    }

    private void reset(CookieStorage cookieStorage) {
        cookieStorage.lastAccessToken.forEach(this::invalidateCookie);
        cookieStorage.data.forEach(this::invalidateCookie);
        cookieStorage.clear();
    }

    @Override
    public Promise<Long> size() {
        return Promise.value((Object)-1L);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean isValid(CookieStorage cookieStorage) throws Exception {
        ByteBuf payload = null;
        try {
            long lastAccessTime;
            payload = this.deserialize((ImmutableList<Cookie>)cookieStorage.lastAccessToken);
            if (payload.readableBytes() == 0) {
                this.reset(cookieStorage);
                boolean bl = false;
                return bl;
            }
            try {
                lastAccessTime = payload.readLong();
            }
            catch (IndexOutOfBoundsException e) {
                lastAccessTime = payload.getLong(payload.readerIndex());
            }
            long currentTime = System.currentTimeMillis();
            long maxInactivityIntervalMillis = this.config.getMaxInactivityInterval().toMillis();
            if (currentTime - lastAccessTime > maxInactivityIntervalMillis) {
                this.reset(cookieStorage);
                boolean bl = false;
                return bl;
            }
        }
        finally {
            if (payload != null) {
                payload.release();
            }
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void setLastAccessTime(CookieStorage cookieStorage) throws Exception {
        ByteBuf data = null;
        try {
            int i;
            data = Unpooled.buffer();
            data.writeLong(System.currentTimeMillis());
            int oldCookiesCount = cookieStorage.lastAccessToken.size();
            String[] partitions = this.serialize(data);
            for (i = 0; i < partitions.length; ++i) {
                this.setCookie(this.config.getLastAccessTimeCookieName() + "_" + i, partitions[i]);
            }
            for (i = partitions.length; i < oldCookiesCount; ++i) {
                this.invalidateCookie(this.config.getLastAccessTimeCookieName() + "_" + i);
            }
        }
        finally {
            if (data != null) {
                data.release();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private String[] serialize(ByteBuf sessionData) throws Exception {
        if (sessionData == null || sessionData.readableBytes() == 0) {
            return new String[0];
        }
        ByteBuf encrypted = null;
        ByteBuf digest = null;
        try {
            encrypted = this.crypto.encrypt(sessionData, this.bufferAllocator);
            String encryptedBase64 = this.toBase64(encrypted);
            digest = this.signer.sign(encrypted.resetReaderIndex(), this.bufferAllocator);
            String digestBase64 = this.toBase64(digest);
            String digestedBase64 = encryptedBase64 + SESSION_SEPARATOR + digestBase64;
            if (digestedBase64.length() <= this.config.getMaxSessionCookieSize()) {
                String[] stringArray = new String[]{digestedBase64};
                return stringArray;
            }
            int count = (int)Math.ceil((double)digestedBase64.length() / (double)this.config.getMaxSessionCookieSize());
            String[] partitions = new String[count];
            for (int i = 0; i < count; ++i) {
                int from = i * this.config.getMaxSessionCookieSize();
                int to = Math.min(from + this.config.getMaxSessionCookieSize(), digestedBase64.length());
                partitions[i] = digestedBase64.substring(from, to);
            }
            String[] stringArray = partitions;
            return stringArray;
        }
        finally {
            if (encrypted != null) {
                encrypted.release();
            }
            if (digest != null) {
                digest.release();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private ByteBuf deserialize(ImmutableList<Cookie> sessionCookies) throws Exception {
        if (sessionCookies.isEmpty()) {
            return Unpooled.EMPTY_BUFFER;
        }
        StringBuilder sessionCookie = new StringBuilder();
        for (Cookie cookie : sessionCookies) {
            sessionCookie.append(cookie.value());
        }
        String[] parts = sessionCookie.toString().split(SESSION_SEPARATOR);
        if (parts.length != 2) {
            return Unpooled.buffer((int)0, (int)0);
        }
        ByteBuf payload = null;
        ByteBuf digest = null;
        ByteBuf expectedDigest = null;
        ByteBuf decryptedPayload = null;
        try {
            payload = this.fromBase64(this.bufferAllocator, parts[0]);
            digest = this.fromBase64(this.bufferAllocator, parts[1]);
            expectedDigest = this.signer.sign(payload, this.bufferAllocator);
            decryptedPayload = ByteBufUtil.equals((ByteBuf)digest, (ByteBuf)expectedDigest) ? this.crypto.decrypt(payload.resetReaderIndex(), this.bufferAllocator) : Unpooled.buffer((int)0, (int)0);
        }
        finally {
            if (payload != null) {
                payload.touch().release();
            }
            if (digest != null) {
                digest.release();
            }
            if (expectedDigest != null) {
                expectedDigest.release();
            }
        }
        return decryptedPayload.touch();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private String toBase64(ByteBuf byteBuf) {
        ByteBuf encoded = Base64.encode((ByteBuf)byteBuf, (boolean)false, (Base64Dialect)Base64Dialect.STANDARD);
        try {
            String string = encoded.toString(CharsetUtil.UTF_8);
            return string;
        }
        finally {
            encoded.release();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private ByteBuf fromBase64(ByteBufAllocator bufferAllocator, String string) {
        ByteBuf byteBuf = ByteBufUtil.encodeString((ByteBufAllocator)bufferAllocator, (CharBuffer)CharBuffer.wrap(string), (Charset)CharsetUtil.UTF_8);
        try {
            ByteBuf byteBuf2 = Base64.decode((ByteBuf)byteBuf, (Base64Dialect)Base64Dialect.STANDARD);
            return byteBuf2;
        }
        finally {
            byteBuf.release();
        }
    }

    private CookieStorage getCookieStorage() {
        CookieStorage cookieStorage;
        Request request = (Request)this.request.get();
        Optional cookieStorageOpt = request.maybeGet(CookieStorage.class);
        if (cookieStorageOpt.isPresent()) {
            cookieStorage = (CookieStorage)cookieStorageOpt.get();
        } else {
            cookieStorage = new CookieStorage(this.readCookies(this.latCookieOrdering, request), this.readCookies(this.dataCookieOrdering, request));
            request.add(CookieStorage.class, (Object)cookieStorage);
        }
        return cookieStorage;
    }

    private ImmutableList<Cookie> readCookies(CookieOrdering cookieOrdering, Request request) {
        Iterable iterable = Iterables.filter((Iterable)request.getCookies(), c -> c.name().startsWith(cookieOrdering.prefix));
        return cookieOrdering.immutableSortedCopy(iterable);
    }

    private void invalidateCookie(Cookie cookie) {
        this.invalidateCookie(cookie.name());
    }

    private void invalidateCookie(String name) {
        Cookie cookie = this.responseCookie(name);
        cookie.setValue("");
        cookie.setMaxAge(0L);
    }

    private void setCookie(String name, String value) {
        Cookie cookie = this.responseCookie(name);
        cookie.setValue(value);
        if (this.expirySeconds > 0L) {
            cookie.setMaxAge(this.expirySeconds);
        }
    }

    private Cookie responseCookie(String name) {
        Response r = (Response)this.response.get();
        return r.getCookies().stream().filter(c -> c.name().equals(name)).findFirst().orElseGet(() -> {
            Cookie createdCookie = r.cookie(name, "");
            if (this.cookieConfig.getPath() != null) {
                createdCookie.setPath(this.cookieConfig.getPath());
            }
            if (this.cookieConfig.getDomain() != null) {
                createdCookie.setDomain(this.cookieConfig.getDomain());
            }
            createdCookie.setHttpOnly(this.cookieConfig.isHttpOnly());
            createdCookie.setSecure(this.cookieConfig.isSecure());
            return createdCookie;
        });
    }

    private static class CookieStorage {
        private ImmutableList<Cookie> lastAccessToken;
        private ImmutableList<Cookie> data;

        public CookieStorage(ImmutableList<Cookie> lastAccessToken, ImmutableList<Cookie> data) {
            this.lastAccessToken = lastAccessToken;
            this.data = data;
        }

        void clear() {
            this.lastAccessToken = ImmutableList.of();
            this.data = ImmutableList.of();
        }
    }

    private static class CookieOrdering
    extends Ordering<Cookie> {
        private final int prefixLen;
        private final String prefix;

        public CookieOrdering(String prefix) {
            this.prefix = prefix;
            this.prefixLen = prefix.length() + 1;
        }

        public int compare(Cookie left, Cookie right) {
            Integer leftNum = Integer.valueOf(left.name().substring(this.prefixLen));
            Integer rightNum = Integer.valueOf(right.name().substring(this.prefixLen));
            return leftNum.compareTo(rightNum);
        }
    }
}

