/*
 * Decompiled with CFR 0.152.
 */
package io.rsocket.fragmentation;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufAllocator;
import io.netty.buffer.CompositeByteBuf;
import io.netty.buffer.Unpooled;
import io.netty.util.ReferenceCountUtil;
import io.netty.util.collection.IntObjectHashMap;
import io.netty.util.collection.IntObjectMap;
import io.rsocket.frame.FragmentationCodec;
import io.rsocket.frame.FrameHeaderCodec;
import io.rsocket.frame.FrameType;
import io.rsocket.frame.PayloadFrameCodec;
import io.rsocket.frame.RequestChannelFrameCodec;
import io.rsocket.frame.RequestFireAndForgetFrameCodec;
import io.rsocket.frame.RequestResponseFrameCodec;
import io.rsocket.frame.RequestStreamFrameCodec;
import java.util.concurrent.atomic.AtomicBoolean;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.Disposable;
import reactor.core.publisher.SynchronousSink;
import reactor.util.annotation.Nullable;

final class FrameReassembler
extends AtomicBoolean
implements Disposable {
    private static final long serialVersionUID = -4394598098863449055L;
    private static final Logger logger = LoggerFactory.getLogger(FrameReassembler.class);
    final IntObjectMap<ByteBuf> headers;
    final IntObjectMap<CompositeByteBuf> metadata;
    final IntObjectMap<CompositeByteBuf> data;
    final ByteBufAllocator allocator;
    final int maxInboundPayloadSize;

    public FrameReassembler(ByteBufAllocator allocator, int maxInboundPayloadSize) {
        this.allocator = allocator;
        this.maxInboundPayloadSize = maxInboundPayloadSize;
        this.headers = new IntObjectHashMap();
        this.metadata = new IntObjectHashMap();
        this.data = new IntObjectHashMap();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void dispose() {
        if (this.compareAndSet(false, true)) {
            FrameReassembler frameReassembler = this;
            synchronized (frameReassembler) {
                for (ByteBuf byteBuf : this.headers.values()) {
                    ReferenceCountUtil.safeRelease((Object)byteBuf);
                }
                this.headers.clear();
                for (ByteBuf byteBuf : this.metadata.values()) {
                    ReferenceCountUtil.safeRelease((Object)byteBuf);
                }
                this.metadata.clear();
                for (ByteBuf byteBuf : this.data.values()) {
                    ReferenceCountUtil.safeRelease((Object)byteBuf);
                }
                this.data.clear();
            }
        }
    }

    public boolean isDisposed() {
        return this.get();
    }

    @Nullable
    synchronized ByteBuf getHeader(int streamId) {
        return (ByteBuf)this.headers.get(streamId);
    }

    synchronized CompositeByteBuf getMetadata(int streamId) {
        CompositeByteBuf byteBuf = (CompositeByteBuf)this.metadata.get(streamId);
        if (byteBuf == null) {
            byteBuf = this.allocator.compositeBuffer();
            this.metadata.put(streamId, (Object)byteBuf);
        }
        return byteBuf;
    }

    synchronized int getMetadataSize(int streamId) {
        CompositeByteBuf byteBuf = (CompositeByteBuf)this.metadata.get(streamId);
        if (byteBuf == null) {
            return 0;
        }
        return byteBuf.readableBytes();
    }

    synchronized CompositeByteBuf getData(int streamId) {
        CompositeByteBuf byteBuf = (CompositeByteBuf)this.data.get(streamId);
        if (byteBuf == null) {
            byteBuf = this.allocator.compositeBuffer();
            this.data.put(streamId, (Object)byteBuf);
        }
        return byteBuf;
    }

    synchronized int getDataSize(int streamId) {
        CompositeByteBuf byteBuf = (CompositeByteBuf)this.data.get(streamId);
        if (byteBuf == null) {
            return 0;
        }
        return byteBuf.readableBytes();
    }

    @Nullable
    synchronized ByteBuf removeHeader(int streamId) {
        return (ByteBuf)this.headers.remove(streamId);
    }

    @Nullable
    synchronized CompositeByteBuf removeMetadata(int streamId) {
        return (CompositeByteBuf)this.metadata.remove(streamId);
    }

    @Nullable
    synchronized CompositeByteBuf removeData(int streamId) {
        return (CompositeByteBuf)this.data.remove(streamId);
    }

    synchronized void putHeader(int streamId, ByteBuf header) {
        this.headers.put(streamId, (Object)header);
    }

    void cancelAssemble(int streamId) {
        ByteBuf header = this.removeHeader(streamId);
        CompositeByteBuf metadata = this.removeMetadata(streamId);
        CompositeByteBuf data = this.removeData(streamId);
        if (header != null) {
            ReferenceCountUtil.safeRelease((Object)header);
        }
        if (metadata != null) {
            ReferenceCountUtil.safeRelease((Object)metadata);
        }
        if (data != null) {
            ReferenceCountUtil.safeRelease((Object)data);
        }
    }

    void handleNoFollowsFlag(ByteBuf frame, SynchronousSink<ByteBuf> sink, int streamId) {
        ByteBuf header = this.removeHeader(streamId);
        if (header != null) {
            int currentPayloadSize;
            int maxReassemblySize = this.maxInboundPayloadSize;
            if (maxReassemblySize != Integer.MAX_VALUE && (currentPayloadSize = this.getMetadataSize(streamId) + this.getDataSize(streamId)) + frame.readableBytes() - FrameHeaderCodec.size() > maxReassemblySize) {
                frame.release();
                throw new IllegalStateException("Reassembled payload went out of allowed size");
            }
            if (FrameHeaderCodec.hasMetadata(header)) {
                ByteBuf assembledFrame = this.assembleFrameWithMetadata(frame, streamId, header);
                sink.next((Object)assembledFrame);
            } else {
                ByteBuf data = this.assembleData(frame, streamId);
                ByteBuf assembledFrame = FragmentationCodec.encode(this.allocator, header, data);
                sink.next((Object)assembledFrame);
            }
            frame.release();
        } else {
            sink.next((Object)frame);
        }
    }

    void handleFollowsFlag(ByteBuf frame, int streamId, FrameType frameType) {
        ByteBuf data;
        int currentPayloadSize;
        int maxReassemblySize = this.maxInboundPayloadSize;
        if (maxReassemblySize != Integer.MAX_VALUE && (currentPayloadSize = this.getMetadataSize(streamId) + this.getDataSize(streamId)) + frame.readableBytes() - FrameHeaderCodec.size() > maxReassemblySize) {
            frame.release();
            throw new IllegalStateException("Reassembled payload went out of allowed size");
        }
        ByteBuf header = this.getHeader(streamId);
        if (header == null) {
            header = frame.copy(frame.readerIndex(), FrameHeaderCodec.size());
            if (frameType == FrameType.REQUEST_CHANNEL || frameType == FrameType.REQUEST_STREAM) {
                long i = RequestChannelFrameCodec.initialRequestN(frame);
                header.writeInt(i > Integer.MAX_VALUE ? Integer.MAX_VALUE : (int)i);
            }
            this.putHeader(streamId, header);
        }
        ByteBuf metadata = null;
        if (FrameHeaderCodec.hasMetadata(frame)) {
            switch (frameType) {
                case REQUEST_FNF: {
                    metadata = RequestFireAndForgetFrameCodec.metadata(frame);
                    break;
                }
                case REQUEST_STREAM: {
                    metadata = RequestStreamFrameCodec.metadata(frame);
                    break;
                }
                case REQUEST_RESPONSE: {
                    metadata = RequestResponseFrameCodec.metadata(frame);
                    break;
                }
                case REQUEST_CHANNEL: {
                    metadata = RequestChannelFrameCodec.metadata(frame);
                    break;
                }
                case PAYLOAD: 
                case NEXT: 
                case NEXT_COMPLETE: 
                case COMPLETE: {
                    metadata = PayloadFrameCodec.metadata(frame);
                    break;
                }
                default: {
                    throw new IllegalStateException("unsupported fragment type");
                }
            }
            if (metadata != null) {
                this.getMetadata(streamId).addComponents(true, new ByteBuf[]{metadata.retain()});
            }
        }
        switch (frameType) {
            case REQUEST_FNF: {
                data = RequestFireAndForgetFrameCodec.data(frame).retain();
                break;
            }
            case REQUEST_STREAM: {
                data = RequestStreamFrameCodec.data(frame).retain();
                break;
            }
            case REQUEST_RESPONSE: {
                data = RequestResponseFrameCodec.data(frame).retain();
                break;
            }
            case REQUEST_CHANNEL: {
                data = RequestChannelFrameCodec.data(frame).retain();
                break;
            }
            case PAYLOAD: 
            case NEXT: 
            case NEXT_COMPLETE: 
            case COMPLETE: {
                data = PayloadFrameCodec.data(frame).retain();
                break;
            }
            default: {
                frame.release();
                throw new IllegalStateException("unsupported fragment type");
            }
        }
        this.getData(streamId).addComponents(true, new ByteBuf[]{data});
        frame.release();
        if (metadata != null && metadata.readableBytes() == 0 && data.readableBytes() == 0) {
            throw new IllegalStateException("Empty frame.");
        }
    }

    void reassembleFrame(ByteBuf frame, SynchronousSink<ByteBuf> sink) {
        try {
            FrameType frameType = FrameHeaderCodec.frameType(frame);
            int streamId = FrameHeaderCodec.streamId(frame);
            switch (frameType) {
                case CANCEL: 
                case ERROR: {
                    this.cancelAssemble(streamId);
                }
            }
            if (!frameType.isFragmentable()) {
                sink.next((Object)frame);
                return;
            }
            boolean hasFollows = FrameHeaderCodec.hasFollows(frame);
            if (hasFollows) {
                this.handleFollowsFlag(frame, streamId, frameType);
            } else {
                this.handleNoFollowsFlag(frame, sink, streamId);
            }
        }
        catch (Throwable t) {
            logger.error("error reassemble frame", t);
            sink.error(t);
        }
    }

    private ByteBuf assembleFrameWithMetadata(ByteBuf frame, int streamId, ByteBuf header) {
        CompositeByteBuf cm = this.removeMetadata(streamId);
        ByteBuf decodedMetadata = PayloadFrameCodec.metadata(frame);
        Object metadata = decodedMetadata != null ? (cm != null ? cm.addComponents(true, new ByteBuf[]{decodedMetadata.retain()}) : PayloadFrameCodec.metadata(frame).retain()) : cm;
        ByteBuf data = this.assembleData(frame, streamId);
        return FragmentationCodec.encode(this.allocator, header, (ByteBuf)metadata, data);
    }

    private ByteBuf assembleData(ByteBuf frame, int streamId) {
        ByteBuf data;
        CompositeByteBuf cd = this.removeData(streamId);
        if (cd != null) {
            cd.addComponents(true, new ByteBuf[]{PayloadFrameCodec.data(frame).retain()});
            data = cd;
        } else {
            data = Unpooled.EMPTY_BUFFER;
        }
        return data;
    }
}

