/*
 * Decompiled with CFR 0.152.
 */
package net.finmath.equities.pricer;

import java.time.LocalDate;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import net.finmath.equities.marketdata.FlatYieldCurve;
import net.finmath.equities.models.EquityForwardStructure;
import net.finmath.equities.models.FlatVolatilitySurface;
import net.finmath.equities.models.VolatilitySurface;
import net.finmath.equities.pricer.AnalyticOptionValuation;
import net.finmath.equities.pricer.EquityValuationRequest;
import net.finmath.equities.pricer.EquityValuationResult;
import net.finmath.equities.pricer.OptionValuation;
import net.finmath.equities.products.EuropeanOption;
import net.finmath.equities.products.Option;
import net.finmath.rootfinder.BisectionSearch;
import net.finmath.rootfinder.SecantMethod;
import net.finmath.time.daycount.DayCountConvention;
import org.apache.commons.math3.linear.DecompositionSolver;
import org.apache.commons.math3.linear.LUDecomposition;
import org.apache.commons.math3.linear.MatrixUtils;
import org.apache.commons.math3.linear.RealMatrix;
import org.apache.commons.math3.linear.RealVector;

public class PdeOptionValuation
implements OptionValuation {
    private final int timeStepsPerYear;
    private final double spaceMinForwardMultiple;
    private final double spaceMaxForwardMultiple;
    private final int spaceNbOfSteps;
    private final double spaceStepSize;
    private final ArrayList<Double> spots;
    private final int spotIndex;
    private final DayCountConvention dayCounter;
    private final boolean isLvPricer;
    private final boolean includeDividendDatesInGrid;

    public PdeOptionValuation(double spaceMinForwardMultiple, double spaceMaxForwardMultiple, int spaceNbPoints, int timeStepsPerYear, DayCountConvention dcc, boolean isLvPricer, boolean includeDividendDatesInGrid) {
        assert (spaceMinForwardMultiple < 1.0) : "min multiple of forward must be below 1.0";
        assert (spaceMaxForwardMultiple > 1.0) : "max multiple of forward must be below 1.0";
        this.timeStepsPerYear = timeStepsPerYear;
        this.dayCounter = dcc;
        this.isLvPricer = isLvPricer;
        this.includeDividendDatesInGrid = includeDividendDatesInGrid;
        double tmpSpaceStepSize = (spaceMaxForwardMultiple - spaceMinForwardMultiple) / (double)spaceNbPoints;
        int tmpSpaceNbPoints = spaceNbPoints;
        ArrayList<Double> tmpSpots = new ArrayList<Double>();
        for (int i = 0; i < tmpSpaceNbPoints; ++i) {
            tmpSpots.add(spaceMinForwardMultiple + tmpSpaceStepSize * (double)i);
        }
        int lowerBound = Math.abs(Collections.binarySearch(tmpSpots, 1.0)) - 2;
        if ((Double)tmpSpots.get(lowerBound) != 1.0) {
            tmpSpaceStepSize += (1.0 - (Double)tmpSpots.get(lowerBound)) / (double)lowerBound;
            tmpSpots = new ArrayList();
            tmpSpaceNbPoints = 0;
            double tmpSpot = 0.0;
            while (tmpSpot < spaceMaxForwardMultiple) {
                tmpSpot = spaceMinForwardMultiple + tmpSpaceStepSize * (double)tmpSpaceNbPoints;
                tmpSpots.add(tmpSpot);
                ++tmpSpaceNbPoints;
            }
        }
        this.spaceMinForwardMultiple = spaceMinForwardMultiple;
        this.spaceMaxForwardMultiple = (Double)tmpSpots.get(tmpSpots.size() - 1);
        this.spaceNbOfSteps = tmpSpaceNbPoints;
        this.spots = tmpSpots;
        this.spaceStepSize = tmpSpaceStepSize;
        this.spotIndex = lowerBound;
    }

    @Override
    public EquityValuationResult calculate(EquityValuationRequest request, EquityForwardStructure forwardStructure, FlatYieldCurve discountCurve, VolatilitySurface volaSurface) {
        HashMap<EquityValuationRequest.CalculationRequestType, Double> results = new HashMap<EquityValuationRequest.CalculationRequestType, Double>();
        if (request.getCalcsRequested().isEmpty()) {
            return new EquityValuationResult(request, results);
        }
        double price = 0.0;
        if (request.getCalcsRequested().contains((Object)EquityValuationRequest.CalculationRequestType.EqDelta) || request.getCalcsRequested().contains((Object)EquityValuationRequest.CalculationRequestType.EqGamma)) {
            double[] spotSensis = this.getPdeSensis(request.getOption(), forwardStructure, discountCurve, volaSurface);
            price = spotSensis[0];
            if (request.getCalcsRequested().contains((Object)EquityValuationRequest.CalculationRequestType.EqDelta)) {
                results.put(EquityValuationRequest.CalculationRequestType.EqDelta, spotSensis[1]);
            }
            if (request.getCalcsRequested().contains((Object)EquityValuationRequest.CalculationRequestType.EqGamma)) {
                results.put(EquityValuationRequest.CalculationRequestType.EqGamma, spotSensis[2]);
            }
        } else {
            price = this.getPrice(request.getOption(), forwardStructure, discountCurve, volaSurface);
        }
        if (request.getCalcsRequested().contains((Object)EquityValuationRequest.CalculationRequestType.Price)) {
            results.put(EquityValuationRequest.CalculationRequestType.Price, price);
        }
        if (request.getCalcsRequested().contains((Object)EquityValuationRequest.CalculationRequestType.EqVega)) {
            double volShift = 1.0E-4;
            double priceShifted = this.getPrice(request.getOption(), forwardStructure, discountCurve, volaSurface.getShiftedSurface(1.0E-4));
            results.put(EquityValuationRequest.CalculationRequestType.EqVega, (priceShifted - price) / 1.0E-4);
        }
        return new EquityValuationResult(request, results);
    }

    public double getPrice(Option option, EquityForwardStructure forwardStructure, FlatYieldCurve discountCurve, VolatilitySurface volSurface) {
        return this.evolvePde(option, forwardStructure, discountCurve, volSurface, false)[0];
    }

    public double[] getPdeSensis(Option option, EquityForwardStructure forwardStructure, FlatYieldCurve discountCurve, VolatilitySurface volSurface) {
        return this.evolvePde(option, forwardStructure, discountCurve, volSurface, true);
    }

    public double getVega(Option option, EquityForwardStructure forwardStructure, FlatYieldCurve discountCurve, VolatilitySurface volSurface, double basePrice, double volShift) {
        double shiftedPrice = this.getPrice(option, forwardStructure, discountCurve, volSurface.getShiftedSurface(volShift));
        return (shiftedPrice - basePrice) / volShift;
    }

    public double getTheta(Option option, EquityForwardStructure forwardStructure, FlatYieldCurve discountCurve, VolatilitySurface volSurface, double basePrice) {
        LocalDate valDate = forwardStructure.getValuationDate();
        LocalDate thetaDate = valDate.plusDays(1L);
        double thetaSpot = forwardStructure.getForward(thetaDate);
        EquityForwardStructure shiftedFwdStructure = forwardStructure.cloneWithNewSpot(thetaSpot).cloneWithNewDate(thetaDate);
        double shiftedPrice = this.getPrice(option, shiftedFwdStructure, discountCurve, volSurface);
        return (shiftedPrice - basePrice) / this.dayCounter.getDaycountFraction(valDate, thetaDate);
    }

    private double[] evolvePde(Option option, EquityForwardStructure forwardStructure, FlatYieldCurve discountCurve, VolatilitySurface volSurface, boolean calculateSensis) {
        LocalDate valDate = forwardStructure.getValuationDate();
        LocalDate expiryDate = option.getExpiryDate();
        double expiryTime = this.dayCounter.getDaycountFraction(valDate, expiryDate);
        assert (!forwardStructure.getValuationDate().isAfter(expiryDate)) : "Valuation date must not be after option expiry";
        double impliedVol = volSurface.getVolatility(option.getStrike(), expiryDate, forwardStructure);
        double forward = forwardStructure.getForward(expiryDate);
        double fdf = forwardStructure.getFutureDividendFactor(expiryDate);
        RealMatrix idMatrix = MatrixUtils.createRealIdentityMatrix((int)this.spaceNbOfSteps);
        RealMatrix tridiagMatrix = MatrixUtils.createRealMatrix((int)this.spaceNbOfSteps, (int)this.spaceNbOfSteps);
        double spaceStepSq = this.spaceStepSize * this.spaceStepSize;
        for (int i = 0; i < this.spaceNbOfSteps; ++i) {
            for (int j = 0; j < this.spaceNbOfSteps; ++j) {
                if (i == j) {
                    tridiagMatrix.setEntry(i, j, Math.pow(this.spots.get(i), 2.0) / spaceStepSq);
                    continue;
                }
                if (i == j - 1 || i == j + 1) {
                    tridiagMatrix.setEntry(i, j, -0.5 * Math.pow(this.spots.get(i), 2.0) / spaceStepSq);
                    continue;
                }
                tridiagMatrix.setEntry(i, j, 0.0);
            }
        }
        RealVector prices = MatrixUtils.createRealVector((double[])new double[this.spaceNbOfSteps]);
        for (int i = 0; i < this.spaceNbOfSteps; ++i) {
            prices.setEntry(i, option.getPayoff((forward - fdf) * this.spots.get(i) + fdf));
        }
        ArrayList<LocalDate> diviDates = forwardStructure.getDividendStream().getDividendDates();
        ArrayList<Double> anchorTimes = new ArrayList<Double>();
        anchorTimes.add(0.0);
        if (this.includeDividendDatesInGrid) {
            for (LocalDate date : diviDates) {
                if (!date.isAfter(valDate) || !date.isBefore(expiryDate)) continue;
                anchorTimes.add(this.dayCounter.getDaycountFraction(valDate, date));
            }
        }
        anchorTimes.add(expiryTime);
        anchorTimes.sort(Comparator.comparing(pt -> pt));
        double lastAtmPrice = 0.0;
        double dt = 0.0;
        for (int a = anchorTimes.size() - 1; a > 0; --a) {
            int i;
            double timeStepSize;
            int timeNbOfSteps;
            double timeInterval = (Double)anchorTimes.get(a) - (Double)anchorTimes.get(a - 1);
            if (this.timeStepsPerYear == 0) {
                timeNbOfSteps = (int)Math.ceil(2.0 * impliedVol * Math.pow(timeInterval, 1.5) / this.spaceStepSize);
                timeStepSize = timeInterval / (double)timeNbOfSteps;
            } else {
                timeNbOfSteps = (int)Math.floor(timeInterval * (double)this.timeStepsPerYear);
                timeStepSize = timeInterval / (double)timeNbOfSteps;
            }
            ArrayList<Double> times = new ArrayList<Double>();
            for (i = 0; i <= 4; ++i) {
                times.add((Double)anchorTimes.get(a) - (double)i * 0.25 * timeStepSize);
            }
            for (i = timeNbOfSteps - 2; i >= 0; --i) {
                times.add((Double)anchorTimes.get(a - 1) + (double)i * timeStepSize);
            }
            for (i = 1; i < times.size(); ++i) {
                RealMatrix explicitMatrix;
                RealMatrix implicitMatrix;
                lastAtmPrice = prices.getEntry(this.spotIndex);
                dt = (Double)times.get(i - 1) - (Double)times.get(i);
                double theta = 0.5;
                if (i <= 4) {
                    theta = 1.0;
                }
                double theta1 = 1.0 - theta;
                double volSq = impliedVol * impliedVol;
                if (this.isLvPricer) {
                    implicitMatrix = tridiagMatrix.scalarMultiply(theta * dt);
                    explicitMatrix = tridiagMatrix.scalarMultiply(-theta1 * dt);
                    double[] localVol = new double[this.spaceNbOfSteps];
                    for (int s = 0; s < this.spaceNbOfSteps; ++s) {
                        double lv = volSurface.getLocalVolatility(Math.log(this.spots.get(s)), (Double)times.get(i - 1), forwardStructure, this.spaceStepSize, dt);
                        localVol[s] = lv * lv;
                    }
                    RealMatrix volaMatrix = MatrixUtils.createRealDiagonalMatrix((double[])localVol);
                    implicitMatrix = volaMatrix.multiply(implicitMatrix);
                    explicitMatrix = volaMatrix.multiply(explicitMatrix);
                } else {
                    implicitMatrix = tridiagMatrix.scalarMultiply(theta * dt * volSq);
                    explicitMatrix = tridiagMatrix.scalarMultiply(-theta1 * dt * volSq);
                }
                implicitMatrix = idMatrix.add(implicitMatrix);
                explicitMatrix = idMatrix.add(explicitMatrix);
                if (option.isAmericanOption()) {
                    double penaltyFactor = 1.0 / Math.min(timeStepSize * timeStepSize, this.spaceStepSize * this.spaceStepSize);
                    forward = forwardStructure.getForward((Double)times.get(i));
                    fdf = forwardStructure.getFutureDividendFactor((Double)times.get(i));
                    double discountFactor = discountCurve.getForwardDiscountFactor((Double)times.get(i), expiryTime);
                    RealVector payoffs = MatrixUtils.createRealVector((double[])new double[this.spaceNbOfSteps]);
                    RealMatrix penaltyMatrix = MatrixUtils.createRealMatrix((int)this.spaceNbOfSteps, (int)this.spaceNbOfSteps);
                    for (int j = 1; j < this.spaceNbOfSteps - 1; ++j) {
                        double payoff = option.getPayoff((forward - fdf) * this.spots.get(j) + fdf) / discountFactor;
                        payoffs.setEntry(j, payoff);
                        penaltyMatrix.setEntry(j, j, prices.getEntry(j) < payoff ? penaltyFactor : 0.0);
                    }
                    RealVector b = explicitMatrix.operate(prices);
                    RealVector oldPrices = prices.copy();
                    RealMatrix oldPenaltyMatrix = penaltyMatrix.copy();
                    double tol = 1.0 / penaltyFactor;
                    int iterations = 0;
                    while (true) {
                        assert (iterations++ < 100) : "Penalty algorithm for american exercise did not converge in 100 steps";
                        RealVector c = b.add(penaltyMatrix.operate(payoffs));
                        RealMatrix A = implicitMatrix.add(penaltyMatrix);
                        DecompositionSolver solver = new LUDecomposition(A).getSolver();
                        prices = solver.solve(c);
                        for (int j = 1; j < this.spaceNbOfSteps - 1; ++j) {
                            penaltyMatrix.setEntry(j, j, prices.getEntry(j) < payoffs.getEntry(j) ? penaltyFactor : 0.0);
                        }
                        if (!penaltyMatrix.equals(oldPenaltyMatrix) && !(prices.subtract(oldPrices).getLInfNorm() / Math.max(oldPrices.getLInfNorm(), 1.0) < tol)) {
                            oldPrices = prices.copy();
                            continue;
                        }
                        break;
                    }
                } else {
                    prices = explicitMatrix.operate(prices);
                    DecompositionSolver solver = new LUDecomposition(implicitMatrix).getSolver();
                    prices = solver.solve(prices);
                }
                prices.setEntry(0, option.getPayoff((forward - fdf) * this.spaceMinForwardMultiple + fdf));
                prices.setEntry(this.spaceNbOfSteps - 1, option.getPayoff((forward - fdf) * this.spaceMaxForwardMultiple + fdf));
            }
        }
        double discountFactor = discountCurve.getDiscountFactor(expiryDate);
        double price = discountFactor * prices.getEntry(this.spotIndex);
        if (calculateSensis) {
            double dFdX = forwardStructure.getDividendAdjustedStrike(forwardStructure.getForward(expiryDate), expiryDate);
            double dFdS = forwardStructure.getGrowthDiscountFactor(valDate, expiryDate);
            double delta = discountFactor * 0.5 * (prices.getEntry(this.spotIndex + 1) - prices.getEntry(this.spotIndex - 1)) / this.spaceStepSize * dFdS / dFdX;
            double gamma = discountFactor * (prices.getEntry(this.spotIndex + 1) + prices.getEntry(this.spotIndex - 1) - 2.0 * prices.getEntry(this.spotIndex)) / spaceStepSq * dFdS * dFdS / dFdX / dFdX;
            double theta = (discountFactor * lastAtmPrice - price) / dt;
            return new double[]{price, delta, gamma, theta};
        }
        return new double[]{price, Double.NaN, Double.NaN, Double.NaN};
    }

    public double getImpliedVolatility(Option option, EquityForwardStructure forwardStructure, FlatYieldCurve discountCurve, double price) {
        double initialGuess = 0.25;
        double forward = forwardStructure.getForward(option.getExpiryDate());
        if (option.isAmericanOption() && option.getPayoff(forward) > 0.0) {
            BisectionSearch bisectionSolver = new BisectionSearch(1.0E-5, 1.0);
            for (int i = 0; i < 3; ++i) {
                double currentVol = bisectionSolver.getNextPoint();
                double currentPrice = this.getPrice(option, forwardStructure, discountCurve, new FlatVolatilitySurface(currentVol));
                bisectionSolver.setValue(currentPrice - price);
            }
            initialGuess = bisectionSolver.getBestPoint();
        } else {
            AnalyticOptionValuation anaPricer = new AnalyticOptionValuation(this.dayCounter);
            Option testOption = option.isAmericanOption() ? new EuropeanOption(option.getExpiryDate(), option.getStrike(), option.isCallOption()) : option;
            initialGuess = anaPricer.getImpliedVolatility(testOption, forwardStructure, discountCurve, price);
        }
        SecantMethod solver = new SecantMethod(initialGuess, initialGuess * 1.01);
        while (solver.getAccuracy() / price > 0.001 && !solver.isDone()) {
            double currentVol = solver.getNextPoint();
            double currentPrice = this.getPrice(option, forwardStructure, discountCurve, new FlatVolatilitySurface(currentVol));
            solver.setValue(currentPrice - price);
        }
        return Math.abs(solver.getBestPoint());
    }
}

