package net.objectlab.qalab.m2.util;

import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.TransformerException;

import javax.xml.transform.stream.StreamSource;
import javax.xml.transform.stream.StreamResult;

import java.io.File;
import java.io.InputStream;

import java.util.Map;
import java.util.HashMap;
import java.util.Iterator;

/**
 * A simple XML Transformer that is supplied with an XML file, an XSLT
 * file and an output file.
 * @author <a href="http://www.davesag.com">Dave Sag</a>.
 */
public class XmlTransformer {

    /** the XMl source file. */
    private final Source theXML;

    /** the XSL source file. */
    private final Source theXSLT;

    /** the output file. */
    private final StreamResult theOutput;

    /** the map of parameters. */
    private transient Map theParameters;

    /**
     * Constructor that takes the xml and xslt input stream and output file.
     * @param anXml The file containing well formed XML to be transformed.
     * @param anXslt The file containing the XSLT that tranforms the XML.
     * @param anOutput The file to write the outout to.
     */
    public XmlTransformer(final InputStream anXml,
            final InputStream anXslt, final File anOutput) {
        assert anXml != null : "The supplied XML file was null.";
        assert anXslt != null : "The supplied XSLT file was null.";
        assert anOutput != null : "The supplied output file was null.";

        anOutput.getParentFile().mkdirs();

        theXML = new StreamSource(anXml);
        theXSLT = new StreamSource(anXslt);
        theOutput = new StreamResult(anOutput);
    }

    /**
     * Add an optional XSLT parameter.
     * @param aKey The key. a two-part string, the namespace URI enclosed in
     * curly braces ({}), followed by the local name. If the name has a
     * null URL, the String only contain the local name.
     * @param aValue The value.
     */
    public final void addParameter(final String aKey, final String aValue) {
        if (theParameters == null) {
            theParameters = new HashMap();
        }
        theParameters.put(aKey, aValue);
    }

    /**
     * Performs the XSLT transformation.
     * @throws TransformerException if anything went wrong.
     */
    public final void transform() throws TransformerException {

        final TransformerFactory tf = TransformerFactory.newInstance();
        final Transformer t = tf.newTransformer(theXSLT);

        // apply any parameters.
        if (theParameters != null && !theParameters.isEmpty()) {
            for (final Iterator i = theParameters.keySet().iterator();
                    i.hasNext();) {
                final String key = (String) i.next();
                t.setParameter(key, theParameters.get(key));
            }
        }

        t.transform(theXML, theOutput);
    }
}
