package org.openliberty.wsc;

import java.net.URL;

import javax.crypto.Mac;
import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;

import net.shibboleth.utilities.java.support.xml.XMLParserException;

import org.apache.commons.codec.binary.Hex;
import org.apache.log4j.Logger;
import org.apache.xml.security.utils.Base64;
import org.openliberty.xmltooling.sasl.Data;
import org.openliberty.xmltooling.sasl.DataBuilder;
import org.openliberty.xmltooling.sasl.SASLRequest;
import org.openliberty.xmltooling.sasl.SASLRequestBuilder;
import org.openliberty.xmltooling.sasl.SASLResponse;
import org.openliberty.xmltooling.soap.soap11.HeaderIDWSF;
import org.openliberty.xmltooling.utility_2_0.Status;
import org.openliberty.xmltooling.wsa.Address;
import org.openliberty.xmltooling.wsa.EndpointReference;
import org.openliberty.xmltooling.wsa.MessageID;
import org.openliberty.xmltooling.wsa.RelatesTo;
import org.opensaml.core.xml.XMLObject;
import org.opensaml.core.xml.io.UnmarshallingException;
import org.opensaml.saml.saml2.core.RequestedAuthnContext;
import org.opensaml.soap.soap11.Envelope;

/**
 * This class bootstraps into the WSF environment authenticating with an Authentication Service (AS)
 * to retrieve a Discovery Service (DS) {@link EndpointReference}, utilizing SASL.
 * 
 * @author curtis
 * @author asa
 *
 */
public class AuthenticationService extends BaseServiceClient
{

    private static Logger log = Logger.getLogger(AuthenticationService.class);

    /**
     * Indicates whether the service client will attempt to down grade in
     * the event that the server returns an AUTH MECHANISM that was not requested
     *
     */
    boolean promiscuousMode = false;

    /**
     * Request Values
     */
    private URL serviceURL = null;

    /**
     * Response Values
     */
    
    private ResponseCode lastResponseCode;
    /**
     * Used for the RelatesTo header when there is a CONTINUE
     * 
     */
    private String lastMessageId;
    
    /**
     * {@inheritDoc}
     */
    public AuthenticationService(DiscoveryService discoveryService, EndpointReference initialEndpointReference)
    {
        super(discoveryService, initialEndpointReference);
    }
    
    /**
     * Creates an Authentication Service Client from the specified EndpointReference
     * 
     * @param epr
     * @return
     */
    public static AuthenticationService serviceForEndpointReference(DiscoveryService discoveryService, EndpointReference epr)
    {
        AuthenticationService service = null;

        try 
        {
            service =  new AuthenticationService(discoveryService, epr);
            service.setServiceURL(new URL(epr.getAddress().getValue()));
        }
        catch (Exception e) 
        {
            e.printStackTrace();
        }

        return service;
    }    


    /**
     * Sets the Authentication Service URL
     * 
     * @param serviceURL
     */
    public void setServiceURL(URL serviceURL)
    {
        this.serviceURL = serviceURL;
    }


    /**
     * Returns the last response code logged by this client
     * 
     * @return
     */
    public ResponseCode getLastResponseCode()
    {
        return lastResponseCode;
    }


    public boolean isPromiscuousMode()
    {
        return promiscuousMode;
    }


    public void setPromiscuousMode(boolean promiscuousMode)
    {
        this.promiscuousMode = promiscuousMode;
    }

    /**
     * This wraps method calls 
     * 
     * @param username
     * @param password
     * @param mechanism
     * @return
     * @throws WSCException 
     */
    public EndpointReference authenticate(String username, String password, AuthMechanism mechanism) throws WSCException
    {
        return authenticate(username, password, mechanism, null);
    }

    /**
     * 
     * 
     * @param username
     * @param password
     * @param mechanism
     * @param requestedAuthnContext
     * @return
     * @throws WSCException
     */
    public EndpointReference authenticate(String username, String password, AuthMechanism mechanism, RequestedAuthnContext requestedAuthnContext) throws WSCException
    {
        lastMessageId = null;
        lastResponseCode = null;
        
        if(AuthMechanism.PLAIN == mechanism)
        {
            return authenticatePLAIN(username, password, requestedAuthnContext);
        }
        else if(AuthMechanism.CRAM_MD5 == mechanism)
        {
            return authenticateCRAM_MD5(username, password, requestedAuthnContext);
        }

        throw new WSCException("");
    }






    /**
     * authenticate
     * 
     * Perform the (multi-step) authentication
     * 
     * @return EPR or null on error
     */
    public EndpointReference authenticatePLAIN(String username, String password, RequestedAuthnContext requestedAuthnContext) throws WSCException
    {
        if(username==null || password==null) throw new WSCException(WSCExceptionType.ILLEGAL_ARGUMENTS.shortDesc+"Both username and password are required.");

        boolean isDebug = log.isDebugEnabled();        

        if(isDebug) log.debug("AS: authenticatePLAIN(username "+username+", password "+password+")");

        SASLRequest request = new SASLRequestBuilder().buildObject();
        request.setMechanism(AuthMechanism.PLAIN.code);      
        request.setRequestedAuthnContext(requestedAuthnContext);
        
        // Optional AdvisoryAuthnID attribute
//      if (authID != null)
//      {          
//      request.setAdvisoryAuthnID(authID);
//      }

        /**
         *  NOTE: in the SASL PLAN RFC 4616 (http://www.ietf.org/rfc/rfc4616.txt)
         *  there is a authzid which represents the identity that the client may
         *  act on the behalf of (kind of like a target id).  This would be encoded 
         *  along with the authcid and password as shown below.
         *  
         */
        Data data = new DataBuilder().buildObject();
        String base64AuthToken = Base64.encode(('\0'+username+'\0'+password).getBytes());
        data.setValue(base64AuthToken);
        request.setData(data);

        /**
         * This is the service invocation.
         */
        SASLResponse response = invokeSASLRequest(serviceURL, request);


        ResponseCode responseCode = ResponseCode.getResponseCode(response);

        if(isDebug) log.debug("     STATUS CODE: "+responseCode.codeValue);

        EndpointReference epr = null;

        if(ResponseCode.OK==responseCode)
        {
            lastResponseCode = responseCode;
            epr = response.getEndpointReference();
            if(isDebug) log.debug("     Authentication completed successfully.");
        }
        else if(ResponseCode.ABORT==responseCode)
        {
            lastResponseCode = responseCode;
            if(isDebug) log.debug("     Authentication aborted.");
        }
        else if(ResponseCode.CONTINUE==responseCode)
        {
            lastResponseCode = responseCode;
            response.getStatus();
            throw new WSCException(WSCExceptionType.CONTINUE_NOT_SUPPORTED_IN_AUTHENTICATE_CONTEXT.shortDesc);
        }

        return epr;
    }

    // TODO: test to see what happens when a mechanism is specified that does not exist on the server

    /**
     * CRAM-MD5 is a two stage authentication procedure.  First the Server
     * is told of the intention to use CRAM-MD5, the server responds with
     * a challenge.  The challenge is the key used in the MD5 response
     * <p>
     * http://www.ietf.org/internet-drafts/draft-ietf-sasl-crammd5-09.txt
     * </p>
     * @param username
     * @param password
     * @return
     * @throws WSCException
     */
    public EndpointReference authenticateCRAM_MD5(String username, String password, RequestedAuthnContext requestedAuthnContext) throws WSCException
    {                
        if(username==null || password==null) throw new WSCException(WSCExceptionType.ILLEGAL_ARGUMENTS.shortDesc+"Both username and password are required.");

        boolean isDebug = log.isDebugEnabled();

        SASLRequestBuilder requestBuilder = new SASLRequestBuilder();

        // STAGE 1: Make a Query to the AS to get the CRAM-MD5 challenge
        String challenge = null;

        SASLRequest stage1Request = requestBuilder.buildObject();
        stage1Request.setMechanism(AuthMechanism.CRAM_MD5.code); 
        SASLResponse stage1Response = invokeSASLRequest(serviceURL, stage1Request);
        
        AuthMechanism serverMechanism = AuthMechanism.findAuthMechanism(stage1Response.getServerMechanism());

        // bail or downgrade if the server returns an AUTH METHOD that is not CRM_MD5
        if(AuthMechanism.CRAM_MD5 != serverMechanism)
        {
            if(promiscuousMode)
            {
                return authenticatePLAIN(username, password, requestedAuthnContext);
            }
            else
            {
                throw new WSCException(WSCExceptionType.UNEXPECTED_SERVER_AUTH_MECHANISM.shortDesc);
            }
        }

        // the only acceptable response code is CONTINUE
        if(ResponseCode.CONTINUE != lastResponseCode)
        {
            throw new WSCException(WSCExceptionType.UNEXPECTED_STATUS_CODE.shortDesc);
        }                

        // the data must not be empty
        if(null==stage1Response.getData())
        {
            throw new WSCException(WSCExceptionType.NO_DATA_IN_SASL_RESPONSE.shortDesc);
        }

        challenge = stage1Response.getData().getValue();

        // The challenge must not be empty
        if(null==challenge)
        {
            throw new WSCException(WSCExceptionType.CRAM_MD5_CHALLENGE_IS_EMPTY.shortDesc);
        }

        StringBuffer buff = new StringBuffer();

        try
        {
            //http://www.ietf.org/internet-drafts/draft-ietf-sasl-crammd5-09.txt
            //
            //The mechanism starts with the server issuing a <challenge>.  The data
            //contained in the challenge contains a string of random data.
            //
            //The client makes note of the data and then responds with a <response>
            //consisting of the <username>, a space, and a <digest>.  The digest is
            //computed by applying the keyed MD5 algorithm from [RFC2104] where the
            //key is a shared secret and the digested text is the <challenge>
            //(including angle-brackets).  The client MUST NOT interpret or attempt
            //to validate the contents of the challenge in any way.
            //
            //This shared secret is a string known only to the client and server.
            //The digest parameter itself is a 16-octet value which is sent in a
            //restricted hexadecimal format (see the <digest> production in
            //Section 3).
            //
            //When the server receives this client response, it verifies the digest
            //provided.  Since the user name may contain the space character, the
            //server MUST ensure the right-most space character is recognised as
            //the token separating the user name from the digest.  If the digest is
            //correct, the server should consider the client authenticated.
            
            buff.append(username).append(' ');

            // Instantiate secret key for HMAC-MD5
            SecretKey key = new SecretKeySpec(challenge.getBytes(), "HmacMD5");

            
            Mac mac = Mac.getInstance(key.getAlgorithm());
            mac.init(key);
            
            byte[] digest = mac.doFinal(password.getBytes());
            
            
            buff.append(Hex.encodeHex(digest));
        }
        catch(Exception e)
        {
            e.printStackTrace();
        }
        
        
        System.out.println(buff);
        
        // STAGE 2: Send the CRAM-MD5 credentials

        // add the encoded username/password string to a Data element
        Data data = new DataBuilder().buildObject();

        data.setValue(Base64.encode( buff.toString().getBytes() ));
        
        SASLRequest stage2Request = requestBuilder.buildObject();
        stage2Request.setMechanism(AuthMechanism.CRAM_MD5.code);          
        stage2Request.setData(data);
        // stage2Request.setAdvisoryAuthnID(username);
        stage2Request.setRequestedAuthnContext(requestedAuthnContext);
        
        SASLResponse stage2Response = invokeSASLRequest(serviceURL, stage2Request);

        if(isDebug) log.debug("     STATUS CODE: "+lastResponseCode.codeValue);

        EndpointReference epr = null;

        if(ResponseCode.OK==lastResponseCode)
        {
            epr = stage2Response.getEndpointReference();            
            if(isDebug) 
            {
                log.debug("     Authentication completed successfully.");
            }
        }
        else if(ResponseCode.ABORT==lastResponseCode)
        {
            if(isDebug) log.debug("     Authentication aborted.");
        }
        else if(ResponseCode.CONTINUE==lastResponseCode)
        {
            throw new WSCException(WSCExceptionType.CONTINUE_NOT_SUPPORTED_IN_AUTHENTICATE_CONTEXT.shortDesc);
        }

        return epr;
    }





    /**
     * Invokes a SASLRequest, returning the SASLResponse
     * 
     * @param addressUrl
     * @param request 
     * @return the SASLResponse
     * @throws WSCException 
     */
    private SASLResponse invokeSASLRequest(URL addressURL, SASLRequest request) throws WSCException
    {
        boolean isDebug = log.isDebugEnabled();
        /**
         * Create a basic EPR with an address to support the instantiation of a WSFMessage
         */
        Address address = new Address();
        address.setValue(addressURL.toString());
        EndpointReference epr = new EndpointReference();
        epr.setAddress(address);

        /**
         * Instantiate the WSFMessage that will be used for Auth
         */
        WSFMessage message = null;
        
        
        try
        {
            message = WSFMessage.createWSFMessage(this, "urn:liberty:sa:2006-08:SASLRequest");
        }
        catch (XMLParserException e1)
        {
            e1.printStackTrace();
            return null;
        }
        catch (UnmarshallingException e1)
        {
            e1.printStackTrace();
            return null;            
        }
        
        
        message.getRequestEnvelope().getBody().getUnknownXMLObjects().add(request);

        if(ResponseCode.CONTINUE==lastResponseCode && null!=lastMessageId)
        {
            RelatesTo relatesTo = new RelatesTo();
            relatesTo.setValue(lastMessageId);
            message.addWSUIdAttribute(relatesTo, "relHdr");
            message.addSOAP11Attributes(relatesTo, true);
            
            ((HeaderIDWSF)message.getRequestEnvelope().getHeader()).setRelatesTo(relatesTo);
        }
        
        if(isDebug) log.debug("SASL REQUEST\n"+WSFMessage.prettyPrintRequestMessage(message));

        try
        {
            message.invoke();
            if(isDebug) log.debug("SASL RESPONSE\n"+WSFMessage.prettyPrintResponseMessage(message));
        }
        catch (Exception e)
        {
            e.printStackTrace();
            throw new WSCException(WSCExceptionType.AUTHENTICATION_SERVICE_INVOCATION_FAILURE.shortDesc);
        } 

        // Get The SOAP Envelope
        Envelope envelope = message.getResponseEnvelope();

        SASLResponse response = null;

        for (XMLObject object : envelope.getBody().getUnknownXMLObjects()) 
        {
            if (object instanceof SASLResponse)
            {
                response = (SASLResponse)object;
                break;
            }
        }

        if(null==response)
        {
            throw new WSCException(WSCExceptionType.UNRECOGNIZED_RESPONSE.shortDesc);
        }

        MessageID messageId = ((HeaderIDWSF)envelope.getHeader()).getMessageID();
        if(null==messageId || null==messageId.getValue()) 
        {
            throw new WSCException(WSCExceptionType.AUTHENTICATION_SERVICE_FAILURE.shortDesc+" No <MessageID> Element in SOAP Response Header");
        }
        if(isDebug) log.debug("     SASL RESPONSE MessageID " + messageId.getValue());

        lastMessageId = messageId.getValue();
        lastResponseCode = ResponseCode.getResponseCode(response);

        return response;
    }


    /**
     * An enumeration that describes the various exceptions that the Authentication
     * Service Client may throw.
     * 
     * @author asa
     *
     */
    public enum WSCExceptionType
    {
        /**
         * 
         */
        ILLEGAL_ARGUMENTS("ILLEGAL_ARGUMENTS: "),
        /**
         * The server returned a serverMechanism in the SASLResponse that does not match the one requested
         */
        UNEXPECTED_SERVER_AUTH_MECHANISM("UNEXPECTED_SERVER_AUTH_MECHANISM: The server returned a serverMechanism in the SASLResponse that does not match the one requested"),
        /**
         * The server returned a CRAM-MD5 response with no challenge in the &lt;Data&gt; element in the SASLResponse
         */        
        CRAM_MD5_CHALLENGE_IS_EMPTY("CRAM_MD5_CHALLENGE_IS_EMPTY: The SASLResponse contained a <Data> element that should have contained a CRAM-MD5 Challenge, but did not"),
        /**
         * This exception is thrown when it is expected that there will be a &lt;Data&gt; element in the SASLResponse
         */
        NO_DATA_IN_SASL_RESPONSE("NO_DATA_IN_SASL_RESPONSE: There is no <Data> element in the SASLResponse"),
        /**
         * This exception is thrown when the Authentication Service response for a SASLRequest did not contain a SASLResponse
         */
        UNRECOGNIZED_RESPONSE("UNRECOGNIZED_RESPONSE: The Authentication Service response for a SASLRequest did not contain a SASLResponse"),
        /**
         * When the service client receives a status code in a context where it is
         * not expected, this Exception will be thrown 
         */
        UNEXPECTED_STATUS_CODE("UNEXPECTED_STATUS_CODE: "),        
        /**
         * This is a general failure exception thrown when the WSFMessage.invoke(sign) 
         * fails.
         */
        AUTHENTICATION_SERVICE_INVOCATION_FAILURE("AUTHENTICATION_SERVICE_INVOCATION_FAILURE: Failed to invoke the service call as specified"),
        /**
         * An Auth Service Failure Exception is thrown when the Authentication Service does
         * not return required elements in its SOAP response
         */
        AUTHENTICATION_SERVICE_FAILURE("AUTHENTICATION_SERVICE_FAILURE: "),
        /**
         * While authentication, continue is currently not supported. Continue is supported 
         * when making the initial query for matching security mechanisms.
         */
        CONTINUE_NOT_SUPPORTED_IN_AUTHENTICATE_CONTEXT("CONTINUE_NOT_SUPPORTED_IN_AUTHENTICATE_CONTEXT: AuthenticationService Client Exception")
        ;

        String shortDesc;

        WSCExceptionType(String shortDesc)
        {
            this.shortDesc = shortDesc;
        }

    }

    /**
     * This enumeration lists the AUTH mechanisms currently supported by
     * the client library
     * 
     * @author asa
     *
     */
    public enum AuthMechanism
    {
        CRAM_MD5("CRAM-MD5"),
        PLAIN("PLAIN")
        ;

        public String code;

        private AuthMechanism(String code)
        {
            this.code = code;
        }

        public static AuthMechanism findAuthMechanism(String mechanism) throws WSCException
        {
            mechanism = mechanism.toUpperCase();
            for(AuthMechanism authMechanism : AuthMechanism.values())
            {
                if(authMechanism.code.equals(mechanism)) return authMechanism;
            }

            throw new WSCException("Authentication Service Client Failure: unsupported authorization mechanism \'"+mechanism+"\'");    
        }

    }


    /**
     * This enum models the ResponseCodes that are supported by the WSC ClientLib Authentication Service 
     * Client at this point.
     * 
     * @author asa
     *
     */
    public enum ResponseCode
    {
        CONTINUE("continue"), ABORT("abort"), OK("ok");

        public String codeValue;

        private ResponseCode(String codeValue)
        {
            this.codeValue = codeValue;
        }

        /**
         * Returns a matching ResponseCode enum, or throws a WSCException
         * if the status is null or not supported.
         * 
         * @param SASLResponse
         * @return the matching ResponseCode enum
         * @throws WSCException
         */
        public static ResponseCode getResponseCode(SASLResponse response) throws WSCException
        {
            Status status = response.getStatus();
            if(null!=status) return getResponseCode(status.getCode());

            throw new WSCException("Authentication Service Failure: No <Status> Element in SASL Response");            
        }

        /**
         * Returns a matching ResponseCode enum, or throws a WSCException
         * if the codeValue is not supported.
         * 
         * @param codeValue
         * @return
         * @throws WSCException
         */
        public static ResponseCode getResponseCode(String codeValue) throws WSCException
        {
            codeValue = codeValue.toLowerCase();
            for(ResponseCode responseCode : ResponseCode.values())
            {
                if(responseCode.codeValue.equals(codeValue)) return responseCode;
            }

            throw new WSCException("Authentication Service Client Failure: unsupported status returned from AS \'"+codeValue+"\'");    
        }


    }


}
