package org.openliberty.wsc;

import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;
import org.joda.time.DateTime;
import org.openliberty.xmltooling.disco.Option;
import org.openliberty.xmltooling.disco.Options;
import org.openliberty.xmltooling.wsa.EndpointReference;
import org.openliberty.xmltooling.wsa.Metadata;

/**
 * All service clients extend this class
 * 
 * @author asa
 *
 */
public abstract class BaseServiceClient
{
    private static Logger log = Logger.getLogger(BaseServiceClient.class);
    
    /**
     * Holds a reference to the discovery service that provided the epr. This is 
     * used when a 
     * 
     */
    private DiscoveryService discoveryService;
    
    /**
     * The EndpointReferences that are used to invoke the service.
     */
    private List<EndpointReference> serviceEndpointReferenceStore;

    /**
     * Indicates whether outgoing messages should be signed.
     */
    private boolean signingOutgoingMessages = false;
    
    /**
     * The base constructor requires a DiscoverService parameter.  If not null, the
     * DS reference is used when attempting to get a new sech mech if there is a
     * CredentialsContext header specifying appropriate sech mechs
     * 
     * @param discoveryService
     */
    public BaseServiceClient(DiscoveryService discoveryService, EndpointReference initialEndpointReference)
    {
        
        if(null!=initialEndpointReference) setServiceEndpointReference(initialEndpointReference);
        this.discoveryService = discoveryService;
    }
    
    /**
     * This method clears out the epr chain and sets it to the specified epr
     * 
     * @param serviceEndPointReference
     */
    public void replaceServiceEndpointReference(EndpointReference serviceEndPointReference)
    {
        serviceEndpointReferenceStore = null;
        setServiceEndpointReference(serviceEndPointReference);
    }
    
    /**
     * Set the ID-WSF 2.0 Endpoint Reference for the Service that will be invoked.
     * 
     * @param serviceEndPointReference
     */
    public void setServiceEndpointReference(EndpointReference serviceEndPointReference)
    {
                
        DateTime notOnOrAfter = serviceEndPointReference.getNotOnOrAfter();
        
        // CASE 1: The new EndpointReference has no expire time.  Make it the only entry in the store
        if(null==notOnOrAfter)
        {
            serviceEndpointReferenceStore = new ArrayList<EndpointReference>();
            serviceEndpointReferenceStore.add(serviceEndPointReference);
        }
        // CASE 2: Look through and compare the serviceEndpointReference with the old ones, putting it
        // in the proper order in the list based on its notOnOrAfter value
        else 
        {

            // This method goes through the list and finds the first EPR where the
            // expiration date is greater.  It then truncates the list there, and adds the new
            // EPR.
            List<EndpointReference> eprs = getServiceEndpointReferenceStore();
            for(int i=0;i<eprs.size(); i++)
            {
                EndpointReference epr = eprs.get(i);

                DateTime eprNotOnOrAfter = epr.getNotOnOrAfter();

                if(null==eprNotOnOrAfter || -1==eprNotOnOrAfter.compareTo(notOnOrAfter)) 
                {
                    if(i==0)
                    {
                        serviceEndpointReferenceStore = new ArrayList<EndpointReference>();
                    }
                    else
                    {
                        serviceEndpointReferenceStore = eprs.subList(0, i-1);                    
                    }
                    break;
                }
            }

            eprs.add(serviceEndPointReference);
        }
    }
    
    /**
     * Get the ID-WSF 2.0 EndpointReference for the Service that will be invoked.
     * 
     * @return
     */
    public EndpointReference getServiceEndpointReference()
    {
        if(null!=serviceEndpointReferenceStore)
        {
            int size = serviceEndpointReferenceStore.size();
            if(size>0)
            {
                return serviceEndpointReferenceStore.get(size-1);
            }
        }
        return null;        
    }

    /**
     * Returns the full list of stored EPRs
     * 
     * @return
     */
    public List<EndpointReference> getServiceEndpointReferenceStore()
    {
        if(null==serviceEndpointReferenceStore) serviceEndpointReferenceStore = new ArrayList<EndpointReference>();
        return serviceEndpointReferenceStore;        
    }


    /**
     * Indicates whether or not outgoing messages will be signed.
     * 
     * @return
     */
    public boolean isSigningOutgoingMessages()
    {
        return signingOutgoingMessages;
    }

    /**
     * Sets whether outgoing messages should be signed.
     * 
     * @param signingOutgoingMessages
     */
    public void setSigningOutgoingMessages(boolean signingOutgoingMessages)
    {
        this.signingOutgoingMessages = signingOutgoingMessages;
    }

    /**
     * Test to see whether the Service has registered support for the 
     * specified Service
     * 
     * @param optionUri
     * @return 
     */
    public boolean serviceExplicitlySupportsOption(String optionUri)
    {
        Metadata metadata = getServiceEndpointReference().getMetadata();        

        Options optionsElement = metadata.getOptions();        
        if(null!=optionsElement)
        {
            List<Option> options = optionsElement.getOptions();
            if(options.size()>0)
            {
                for(Option option : options)
                {
                    if(option.getValue().equals(optionUri)) 
                    {
                        log.debug("Option Found");
                        return true;
                    }
                }
                log.debug("Option not found");
            }
            else 
            {
                log.debug("This service has explicitly advertised that there are no available Options");
            }
        }
        else 
        {
            log.debug("No Options element in Metadata");
        }

        return false;
    }
    
    /**
     * Returns the DS Client that was used to get an EPR to the service
     * extending BaseServiceClient
     * 
     * @return
     */
    public DiscoveryService getDiscoveryService()
    {
        return discoveryService;
    }
    
    /**
     * Used to explicitly set the DS
     * 
     * @param discoveryService
     */
    public void setDiscoveryService(DiscoveryService discoveryService)
    {
        this.discoveryService = discoveryService;
    }
    

    
    
}
