package org.openliberty.wsc;


import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.List;

import net.shibboleth.utilities.java.support.xml.XMLParserException;

import org.apache.log4j.Logger;
import org.openliberty.xmltooling.idsis.dap.DAPData;
import org.openliberty.xmltooling.idsis.dap.DAPItemData;
import org.openliberty.xmltooling.idsis.dap.DAPModify;
import org.openliberty.xmltooling.idsis.dap.DAPModifyResponse;
import org.openliberty.xmltooling.idsis.dap.DAPQuery;
import org.openliberty.xmltooling.idsis.dap.DAPQueryResponse;
import org.openliberty.xmltooling.soapbinding.ProcessingContext;
import org.openliberty.xmltooling.wsa.EndpointReference;
import org.opensaml.core.xml.XMLObject;
import org.opensaml.core.xml.io.UnmarshallingException;
import org.opensaml.core.xml.util.XMLObjectChildrenList;
import org.opensaml.soap.soap11.Body;

/**
 * This is a Service Client for an ID-SIS-DAP Service.  This class provides
 * fairly simple access to ID-SIS-DAP, and is based entirely on the xmltooling
 * classes in org.openliberty.xmltooling.idsis.dap which have a full implementation
 * of all possible options over the protocol.
 * <p>
 * NOTE: This is an incomplete service client tested with SYMLABS.
 * 
 * @author asa
 *
 */
public class DirectoryAccessProtocolService extends BaseServiceClient
{
    protected static Logger log = Logger.getLogger(DirectoryAccessProtocolService.class);

    /**
     * This is the URN representing the ID-SIS-DAP Service
     */
    public static final String SERVICE_URN = DiscoveryService.WSFServiceType.DIRECTORY_ACCESS_PROTOCOL_SERVICE.getUrn();

    /**
     * Sorting is OPTIONAL (typically, the back end would have to implement [RFC2891]). If sorting is not
     * supported for objectType "entry," discovery option keyword urn:liberty:dst:noSorting MUST be registered.
     */
    public static final String OPTION_NO_SORTING = "urn:liberty:dst:noSorting";

    /**
     * Pagination of results is OPTIONAL (typically, the backend would have to implement [RFC2696]
     * or [LDAPVLV]). If pagination is not supported for objectType "entry," discovery option keyword
     * urn:liberty:dst:noPagination MUST be registered.
     */
    public static final String OPTION_NO_PAGINATION = "urn:liberty:dst:noPagination";

    /**
     * Support static sets in pagination is OPTIONAL (the backend probably would have to support [LDAPVLV]).
     * If static sets are not supported, discovery option keyword urn:liberty:dst:noStatic MUST be
     * registered.
     */
    public static final String OPTION_NO_STATIC = "urn:liberty:dst:noStatic";

    /**
     * Supporting objectType "_Subscription" is OPTIONAL (typically, the backend would have
     * to support [LDAPPSearch]). If subscription is not supported, the discovery option keyword
     * urn:liberty:dst:noSubscribe MUST be registered
     */
    public static final String OPTION_NO_SUBSCRIBE = "urn:liberty:dst:noSubscribe";

    /**
     * The simulation method using the ProcessingContext header with value "urn:liberty:sb:2003-08:ProcessingContext:Simulate," 
     * as specified in [LibertySOAPBinding], MUST be supported. If simulated 
     * operation succeeds, similar actual operation SHOULD have a high probability of succeeding within next 30 minutes.
     * This feature allows a WSC to test whether a modification is plausible prior to invoking the user interface to query data
     * from the Principal, thus avoiding principal-supplied non-actionable data unnecessarily.
     * 
     */
    public static final String SIMULATION_URN = ProcessingContext.Defined.SIMULATE.getUri();

    /**
     * {@inheritDoc}
     */
    public DirectoryAccessProtocolService(DiscoveryService discoveryService, EndpointReference initialEndpointReference)
    {
        super(discoveryService, initialEndpointReference);
    }
    
    /**
     * Factory method that creates and initializes a DirectoryAccessProtocolService.
     * 
     * @param epr
     * @return an instantiated DirectoryAccessProtocolService based on the EndpointReference that is passed in.
     */
    public static DirectoryAccessProtocolService serviceForEndpointReference(DiscoveryService discoveryService, EndpointReference epr)
    {
        DirectoryAccessProtocolService service =  new DirectoryAccessProtocolService(discoveryService, epr);
        return service;
    }    
    
    /**
     * If sorting is not supported for objectType "entry" this will return true
     * 
     * @return
     */
    public boolean serviceDoesNotSupportSorting()
    {
        return serviceExplicitlySupportsOption(OPTION_NO_SORTING);
    }
    
    /**
     * If pagination is not supported for objectType "entry" this method 
     * returns true.
     * 
     * @return
     */
    public boolean serviceDoesNotSupportPagination()
    {
        return serviceExplicitlySupportsOption(OPTION_NO_PAGINATION);
    }
    
    /**
     * This method returns true if the server does not 
     * support static sets in pagination.
     * 
     * @return
     */
    public boolean serviceDoesNotSupportStatic()
    {
        return serviceExplicitlySupportsOption(OPTION_NO_STATIC);
    }
    
    /**
     * returns true id subscribe is not supported by the server
     * 
     * @return
     */
    public boolean serviceDoesNotSupportSubscribe()
    {
        return serviceExplicitlySupportsOption(OPTION_NO_SUBSCRIBE);
    }
    
    
    
    /**
     * Takes a hash of attribute pairs and sends them in a {@link DAPModify} request,  the
     * results returned in a similar hash.
     * 
     * @param attributes
     * @param isSimulation
     * @return
     */
    public Hashtable<String, String> modifyEntityAttributes(Hashtable<String,String> attributes, boolean isSimulation)
    {
        
        StringBuffer sbuff = new StringBuffer();
        
        for(Enumeration<String> e = attributes.keys() ; e.hasMoreElements() ;) 
        {
            if(sbuff.length()>0) sbuff.append("\n");
            String key = e.nextElement();
            sbuff.append(key).append(": ").append(attributes.get(key));
        }
       
        DAPModify dapModify = new DAPModify(sbuff.toString());
        
        DAPModifyResponse response = modify(dapModify, isSimulation);
        
        if(log.isDebugEnabled())
        {
            if(null!=response.getStatus())
            {
                log.debug("Status Code: "+response.getStatus().getCode());
            }
            else
            {
                log.debug("Status Code: DAPQueryResponse has no <Status> element!");
            }
        }
        
        Hashtable<String, String> responseAttributes = new Hashtable<String, String>();
        
        if(null!=response)
        {
            // Should only be one, but it is possible that there are many
            XMLObjectChildrenList<DAPItemData> dataList = response.getItemDataList();
            for(DAPItemData data: dataList)
            {
                data.getLDIF();
                
                //attributes.putAll(data.getLDIFDataAsHashtable());
            }            
        }
        
        return responseAttributes;
        
    }
    
    
    /**
     * Takes a list of attributes and returns a Hashtable of
     * attribute key:value pairs.
     * 
     * @param attributeNames
     * @param isSimulation
     * @return
     */
    public Hashtable<String, String> getEntityAttributes(List<String> attributeNames, boolean isSimulation)
    {
        Hashtable<String, String> attributes = new Hashtable<String, String>();
        
        StringBuffer attributeNamesBuff = new StringBuffer();

        // Create a comma seperated String for the query
        for(String attributeName : attributeNames)
        {               
            if(attributeName.length()>0)
            {
                if(attributeNamesBuff.length()>0) attributeNamesBuff.append(",");
                attributeNamesBuff.append(attributeName);
            }
        }
        
        DAPQuery query = DAPQuery.entityQueryWithSelectAttributes(attributeNamesBuff.toString(), null);
       
        DAPQueryResponse response = invokeQuery( query, isSimulation);
        
        if(log.isDebugEnabled())
        {
            if(null!=response.getStatus())
            {
                log.debug("Status Code: "+response.getStatus().getCode());
            }
            else
            {
                log.debug("Status Code: DAPQueryResponse has no <Status> element!");
            }
        }
        
        if(null!=response)
        {
            // Should only be one, but it is possible that there are many
            XMLObjectChildrenList<DAPData> dataList = response.getDAPDatas();
            for(DAPData data: dataList)
            {
                attributes.putAll(data.getLDIFDataAsHashtable());
            }            
        }
        
        return attributes;
    }
    
    
    /**
     * Invokes a single DAPQUery
     * 
     * @param query
     * @param isSimulation
     * @return
     */
    public DAPQueryResponse invokeQuery(DAPQuery query, boolean isSimulation)
    {
        ArrayList<DAPQuery> queries = new ArrayList<DAPQuery>();
        queries.add(query);
        List<DAPQueryResponse> responses = invokeQueries(queries, isSimulation);
        if(responses.size()==1) return responses.get(0);
        else return null;
    }
    
    /**
     * Send a list of DAPQuery objects and 
     * 
     * @param queries
     * @return
     */
    public List<DAPQueryResponse> invokeQueries(List<DAPQuery> queries, boolean isSimulation)
    {      
        
        List<DAPQueryResponse> queryResponses = new ArrayList<DAPQueryResponse>();
        
        Body responseBody = this.invokeRequest(queries, RequestType.QUERY, isSimulation);
        
        
        if(null!=responseBody)
        {                      
            // Now return results
            for(XMLObject obj : responseBody.getUnknownXMLObjects())
            {
                if(obj instanceof DAPQueryResponse)
                {
                    queryResponses.add((DAPQueryResponse)obj);
                }
            }
        }
        
        return queryResponses;
    }
    
    
    public DAPModifyResponse modify(DAPModify request, boolean isSimulation)
    {
        ArrayList<DAPModify> requests = new ArrayList<DAPModify>();
        requests.add(request);
        List<DAPModifyResponse> responses = modify(requests, isSimulation);
        if(responses.size()==1) return responses.get(0);
        else return null;
    }
    
    public List<DAPModifyResponse> modify(List<DAPModify> modifications, boolean isSimulation)
    {      
        
        List<DAPModifyResponse> responses = new ArrayList<DAPModifyResponse>();

        
//        // Build the request Body
//        Body requestBody = (Body) new BodyBuilder().buildObject();  
//
//        // Add the Queries to the Body of the WSFMessage        
//        requestBody.getUnknownXMLObjects().addAll(modifications);
        
        
        Body responseBody = this.invokeRequest(modifications, RequestType.MODIFY, isSimulation);
                
        if(null!=responseBody)
        {                      
            // Now return results
            for(XMLObject obj : responseBody.getUnknownXMLObjects())
            {
                if(obj instanceof DAPModifyResponse)
                {
                    responses.add((DAPModifyResponse)obj);
                }
            }
        }
        
        return responses;
    }
    
    
    

    // QUERY -- building multiple Query Items and Multiple Queries
    // DAPQueryItem:
    //  - entry.dn
    //  - subscriptionId

    // CREATE -- build multiple "Create" elements

    // MODIFY -- multiple Modify and MofifyItems  -- multiple ModifyItems may fail independently  -- notChangedSince supported

    // DELETE -- build multiple "Delete" elements 

    // NOTIFY


    /**
     * Subscription Methods: only if DISCO OPTION_NO_SUBSCRIBE is not set
     */

    // QUERY - multiple Subscription elements in Query

    // Querying existing subscriptions SHOULD be supported.

    // CREATE - multiple Subscription elements in Create

    // MODIFY - multiple Subscription elements in Modify

    // Subscription.starts attribute MUST be supported
    // Subscription.expires attribute MUST be supported  - ommission means that the expires is when the credentials expire - whichever comes first

    /**
     * If subscription is established by specifying a specific list of attributes, then the notification is triggered only if one of
     * the specified attributes changes. If no specific attributes are specified, then change to any attribute in the entry triggers
     * the notification.
     */

    // Notifications SHOULD be acknowledged.






    /**
     * This method provides support for multiple QueryItem elements
     * <p>
     * As specified in [LibertyDST], a minimally compliant ID-SIS-PP implementation MUST support multiple QueryItem elements. 
     * 
     * @param queryItems
     * @return
     */
    @SuppressWarnings("unchecked")
    private Body invokeRequest(List bodyObjects, RequestType requestType, boolean isSimulation)
    {        

        /**
         * Create a Message
         */
        WSFMessage queryMessage = null;
        
        try
        {
            queryMessage = WSFMessage.createWSFMessage(this, SERVICE_URN+requestType.getActionSuffix());
        }
        catch (XMLParserException e1)
        {
            e1.printStackTrace();
            return null;
        }
        catch (UnmarshallingException e1)
        {
            e1.printStackTrace();
            return null;
        }

        
        
        
        /**
         * Create and add the ProcessingContext simulation header if this is a simulation
         * 
         */
        if(isSimulation)
        {
            ProcessingContext processingContext = new ProcessingContext();
            processingContext.setValue(SIMULATION_URN);
            queryMessage.setHeaderProcessingContext(processingContext);
        }
        
        /**
         * Add the body content
         */
        queryMessage.getRequestEnvelope().getBody().getUnknownXMLObjects().addAll(bodyObjects);

        if(log.isDebugEnabled())
        {
            log.debug(WSFMessage.prettyPrintRequestMessage(queryMessage));
        }
        
        try
        {
            
            // invoke the request
            queryMessage.invoke();    
            
            if(log.isDebugEnabled())
            {
                log.debug(WSFMessage.prettyPrintResponseMessage(queryMessage));
            }
            
            // return the message body
            return queryMessage.getResponseEnvelope().getBody();
          
        } 
        catch (Exception e)
        {
            
            log.error("Exception while attempting to invoke a WSFMessage", e);
            
            if(log.isDebugEnabled())
            {
                log.debug(WSFMessage.prettyPrintResponseMessage(queryMessage));
            }
        }
        
        return null;
    }

    
    

    
    /**
     * Enumeration of the 5 request types possib le within
     * ID-SIS-DAP 
     * 
     * @author asa
     *
     */
    public enum RequestType
    {
        QUERY(":Query"),
        MODIFY(":Modify"),
        CREATE(":Create"),
        DELETE(":Delete"),
        NOTIFY(":Notify");

        private String actionSuffix;
        
        private RequestType(String actionSuffix)
        {
            this.actionSuffix = actionSuffix;
        }
        
        public String getActionSuffix()
        {
            return this.actionSuffix;
        }
        
    }



}

/*
 * SAMPLE LDIF DATA 
 *
# root
dn: dc=example,dc=com
objectClass: organization
objectClass: dcObject
o: Example, Inc.
dc: example

# people
dn: ou=People,dc=example,dc=com
objectClass: organizationalUnit
ou: People

# company-wide address book
dn: ou=Address Book,dc=example,dc=com
objectClass: organizationalUnit
ou: Address Book

# entry in company-wide address book
dn: cn=Rosco P. Coltrane,ou=Address Book,dc=example,dc=com
objectClass: inetOrgPerson
cn: Rosco P. Coltrane
sn: Coltrane
mail: rpc@example.com

# address book administrator
dn: cn=Admin,ou=People,dc=example,dc=com
objectClass: inetOrgPerson
cn: Admin
sn: Admin
userPassword: admin

# person
dn: cn=Flash Gordon,ou=People,dc=example,dc=com
objectClass: inetOrgPerson
cn: Flash Gordon
sn: Gordon
mail: fg@example.com
userPassword: secret

# person's personal address book
dn: ou=Address Book,cn=Flash Gordon,ou=People,dc=example,dc=com
objectClass: organizationalUnit
ou: Address Book

# another person
dn: cn=Clark Kent,ou=People,dc=example,dc=com
objectClass: inetOrgPerson
cn: Clark Kent
sn: Kent
mail: ck@example.com
userPassword: secret

# another person's personal address book
dn: ou=Address Book,cn=Clark Kent,ou=People,dc=example,dc=com
objectClass: organizationalUnit
ou: Address Book

# entry in another person's personal address book
dn: cn=Mickey Mouse,ou=Address Book,cn=Clark Kent,ou=People,dc=example,dc=com
objectClass: inetOrgPerson
cn: Mickey Mouse
sn: Mouse
mail: mickey.mouse@example.com

# one more person, yet without personal address book
dn: cn=Donald Duck,ou=People,dc=example,dc=com
objectClass: inetOrgPerson
cn: Donald Duck
sn: Duck
mail: dd@example.com
userPassword: secret


*/