package org.openliberty.wsc;

import java.util.ArrayList;
import java.util.List;

import net.shibboleth.utilities.java.support.xml.XMLParserException;

import org.apache.log4j.Logger;
import org.openliberty.xmltooling.disco.DiscoQuery;
import org.openliberty.xmltooling.disco.DiscoQueryBuilder;
import org.openliberty.xmltooling.disco.DiscoQueryResponse;
import org.openliberty.xmltooling.disco.ProviderID;
import org.openliberty.xmltooling.disco.RequestedService;
import org.openliberty.xmltooling.disco.RequestedServiceBuilder;
import org.openliberty.xmltooling.disco.SecurityMechID;
import org.openliberty.xmltooling.disco.ServiceType;
import org.openliberty.xmltooling.soap.soap11.BodyBuilder;
import org.openliberty.xmltooling.soapbinding.Framework;
import org.openliberty.xmltooling.wsa.EndpointReference;
import org.opensaml.core.xml.XMLObject;
import org.opensaml.core.xml.io.UnmarshallingException;
import org.opensaml.soap.soap11.Body;
import org.opensaml.soap.soap11.Envelope;


/**
 * This client models communication with an ID-WSF 2.0 Discovery Service and requires a
 * Disco EPR in order to operate.
 * 
 * @author asa
 *
 */
public class DiscoveryService extends BaseServiceClient
{
    protected static Logger log = Logger.getLogger(DiscoveryService.class);

    
    /**
     * Personal Profile Service URN
     * <p>
     * NOTE: The Personal Profile Service is based on DST 1.1.  For use in this library it was brought up to DST 2.1, which is 
     * now a non-conformant version of PP.
     * <p>
     * There are two URNs in use for PP presently.  This library uses the latest one, but has been tested with 
     * an older URN.  If the older URN is used, then the configuration in the pp-config.xml must be adjusted 
     * appropriately for the XML Tooling to work poroperly.
     * <ul>
     * 	<li>urn:liberty:id-sis-pp:2005-05 - matches Conor Cahill's OSS Server</li>
     *  <li>urn:liberty:id-sis-pp:2003-08 - matches SYMLABS Personal Profile Service</li>
     * </ul>
     * 
     */
    private static final String IDWSF_20_PERSONAL_PROFILE_SERVICE_URN = "urn:liberty:id-sis-pp:2005-05";    
    
    /**
     * People Service URN
     */
    private static final String IDWSF_20_PEOPLE_SERVICE_URN = "urn:liberty:ps:2006-08";
    
    /**
     * Authentication Service URN
     */
    private static final String IDWSF_20_AUTHENTICATION_SERVICE_URN= "urn:liberty:sa:2006-08";
    
    /**
     * Discovery Service URN
     */
    private static final String IDWSF_20_DISCOVERY_SERVICE_URN = "urn:liberty:disco:2006-08";
    
    /**
     * ID-SIS-DAP Directory Access Protocol Service URN
     */
    private static final String IDWSF_20_ID_SIS_DAP_SERVICE_URN = "urn:liberty:id-sis-dap:2006-02:dst-2.1";

    //private static final String IDWSF_20_INTERACTION_SERVICE = "urn:liberty:is:2003-08";    
    //private static final String MEDIA_SERVICE = "urn:x-demo:me:2006-01";         // Conor Cahill's Media Service
    //private static final String PROVISIONING_SERVICE = "urn:x-demo:pv:2006-01";  // Conor Cahill's Provisioning Service

    /**
     * The urn indicating the default DS Action
     */
    public static final String DISCO_QUERY = "urn:liberty:disco:2006-08:Query";

    /** 
     * stores requested services for discovery
     */
    private List<RequestedService> requestedServices;

    /**
     * The discovery service EPR is not necessarily supplied by a Discovery Service, 
     * therefore a constructor with only an EPR is provided.
     * 
     * @param initialEndpointReference
     */
    public DiscoveryService(EndpointReference initialEndpointReference)
    {
        super(null, initialEndpointReference);
    }

    /**
     * {@inheritDoc}
     */
    public DiscoveryService(DiscoveryService discoveryService, EndpointReference initialEndpointReference)
    {
        super(discoveryService, initialEndpointReference);
    }


    /**
     * Factory method that creates a RequestedService from a list of service types
     * 
     * @param serviceTypes
     * @return
     */
    public static RequestedService requestedServiceForServiceTypes(String[] serviceTypes)    
    {
        RequestedService requestedService = null;

        if(null!=serviceTypes && serviceTypes.length>0)
        {        
            requestedService = baseRequestedService();
            addServiceTypesToRequestedService(serviceTypes, requestedService);
        }

        return requestedService;
    }


    /**
     * Utility Method to add ServiceType objects to an existing RequestedService
     * 
     * @param requestedService
     * @param providerIDs
     */
    public static void addServiceTypesToRequestedService(String[] serviceTypes, RequestedService requestedService)
    {        
        // build and add all service types specified
        for(int t=0; t<serviceTypes.length; t++)
        {                
            ServiceType serviceType = new ServiceType();       
            serviceType.setValue(serviceTypes[t]);
            requestedService.getServiceTypes().add(serviceType);
        }
    }

    /**
     *  Factory method that creates a RequestedService and adds the specified providerIDs
     *  
     * @param providerIDs
     * @return
     */
    public static RequestedService requestedServiceForProviderIDs(String[] providerIDs)    
    {
        RequestedService requestedService = null;

        if(null!=providerIDs && providerIDs.length>0)
        {
            requestedService = baseRequestedService();
            addProviderIDsToRequestedService(providerIDs, requestedService);
        }

        return requestedService;
    }

    /**
     * Utility Method to add ProviderID objects to an existing RequestedService
     * 
     * @param requestedService
     * @param providerIDs
     */
    public static void addProviderIDsToRequestedService(String[] providerIDs, RequestedService requestedService)
    {
        for(int t=0; t<providerIDs.length; t++)
        {                
            ProviderID providerID = new ProviderID();       
            providerID.setValue(providerIDs[t]);
            requestedService.getProviderIDs().add(providerID);
        }
    }


    /**
     *  Factory method that creates a RequestedService and adds the specified SecurityMechIDs
     *  
     * @param providerIDs
     * @return
     */
    public static RequestedService requestedServiceForSecurityMechIDs(String[] securityMechIDs)    
    {
        RequestedService requestedService = null;

        if(null!=securityMechIDs && securityMechIDs.length>0)
        {            
            requestedService = baseRequestedService();
            addSecurityMechIDsToRequestedService(securityMechIDs, requestedService);
        }

        return requestedService;
    }

    /**
     * Utility Method to add SecurityMechID objects to an existing RequestedService
     * 
     * @param requestedService
     * @param securityMechIDs
     */
    public static void addSecurityMechIDsToRequestedService(String[] securityMechIDs, RequestedService requestedService)
    {
        for(int t=0; t<securityMechIDs.length; t++)
        {
            SecurityMechID securityMechID = new SecurityMechID();
            securityMechID.setValue(securityMechIDs[t]);
            requestedService.getSecurityMechIDs().add(securityMechID);
        }
    }

    /**
     * Creates a base RequestedService, used by the other Factory methods
     * <p>
     * At the moment only Version 2.0 is supported and therefore it is enforced in this method which
     * specifies the 2.0 framework.
     * 
     * @return 
     */
    public static RequestedService baseRequestedService()
    {
        RequestedServiceBuilder requestedServiceBuilder = new RequestedServiceBuilder();        
        RequestedService requestedService = requestedServiceBuilder.buildObject();

        // VERSION 2.0 IS CURRENTLY THE ONLY SUPPORTED FRAMEWORK
        Framework framework = new Framework();        
        framework.setVersion(Framework.VERSION_2_0);
        requestedService.getFrameworks().add(framework);

        return requestedService;      
    }

    /**
     * Allows the ClientLib User to add a requested service to the up-coming request
     * 
     * @param requestedService
     */
    public void addARequestedService(RequestedService requestedService)
    {
        if(null==requestedServices) requestedServices = new ArrayList<RequestedService>();
        requestedServices.add(requestedService);
    }


    public void clearRequestedServices()
    {
        requestedServices = new ArrayList<RequestedService>();
    }

    /**
     * This method creates a disco request from the specified EndpointReference generating
     * a ServiceRequest for each ServiceType specified
     * 
     * @param epr
     * @param serviceTypes
     * @return
     */
    public List<EndpointReference> invoke()
    {

        List<EndpointReference> discoveredEPRs = null;   

        // initialize the WSFMessage using the Disco Bootstrap EPR
        WSFMessage wsfMessage = null;

        try
        {
            wsfMessage = WSFMessage.createWSFMessage(this, DISCO_QUERY);


            // default action is query
            // wsfMessage.setHeaderAction(DISCO_QUERY);

            // now build the body of the disco request
            BodyBuilder bodyBuilder = new BodyBuilder();
            Body body = bodyBuilder.buildObject();		

            DiscoQueryBuilder discoQueryBuilder = new DiscoQueryBuilder();
            DiscoQuery discoQuery = discoQueryBuilder.buildObject();		

            // If there are no requested services specified, then add the default
            if(requestedServices==null)
            {
                requestedServices = new ArrayList<RequestedService>();
                requestedServices.add(DiscoveryService.baseRequestedService());
            }

            // Add in the requested services
            for(RequestedService requestedService : requestedServices)
            {
                discoQuery.getRequestedServices().add(requestedService);
            }

            body.getUnknownXMLObjects().add(discoQuery);		
            wsfMessage.getRequestEnvelope().setBody(body);

            if(log.isDebugEnabled())
            {
                log.debug("Disco Request: "+WSFMessage.prettyPrintRequestMessage(wsfMessage));
            }

            try
            {
                wsfMessage.invoke();
                discoveredEPRs = DiscoveryService.eprsFromDiscoResponse(wsfMessage);
            } 
            catch (Exception e)
            {
                e.printStackTrace();
            }

            if(log.isDebugEnabled())
            {
                log.debug("Disco Response: "+WSFMessage.prettyPrintResponseMessage(wsfMessage));
            }

        }
        catch (XMLParserException e1)
        {
            e1.printStackTrace();
        }
        catch (UnmarshallingException e1)
        {
            e1.printStackTrace();
        }

        /**
         * Clear out the requested services to insure a fresh start. This prevents the exception: <br/>
         * <pre>
         *      java.lang.IllegalArgumentException: {urn:liberty:disco:2006-08}RequestedService is already the child of another XMLObject and may not be inserted in to this list
         * </pre>
         * Another way would be to clone the requested services when adding to the WSFMessage
         */
        this.clearRequestedServices();

        return discoveredEPRs;
    }



    /**
     * Utility method to extract the EndpointReferences from the WSFMessage response.
     * 
     * @param message
     * @return a List of EndpointReference objects, or null if there are none
     */
    public static List<EndpointReference> eprsFromDiscoResponse(WSFMessage message)
    {

        Envelope discoResponse = message.getResponseEnvelope();
        if(null!=discoResponse)
        {
            Body body = discoResponse.getBody();

            List<XMLObject> kids = body.getOrderedChildren();            
            for(XMLObject kid : kids)
            {
                if(kid instanceof DiscoQueryResponse)
                {
                    return ((DiscoQueryResponse)kid).getEndpointReferences();
                }
            }

        }
        return null;
    }


    /**
     * Convenience method that returns a Service Client that is supported
     * and enumerated by the ClientLibrary, from this discovery service
     * 
     * @param serviceType
     * @param endpointReference
     * @return
     */
    public BaseServiceClient serviceClientForTypeAndEndpointReference(WSFServiceType serviceType, EndpointReference endpointReference)
    {
        return DiscoveryService.serviceClientForTypeAndEndpointReference(this, serviceType, endpointReference);
    }


    /**
     * Convenience method that returns a Service Client that is supported
     * and enumerated by the ClientLibrary
     * 
     * @param discoveryService
     * @param serviceType
     * @param endpointReference
     * @return
     */
    public static BaseServiceClient serviceClientForTypeAndEndpointReference(DiscoveryService discoveryService, WSFServiceType serviceType, EndpointReference endpointReference)
    {

        if(serviceType==WSFServiceType.AUTHENTICATION_SERVICE)
        {
            return new AuthenticationService(discoveryService, endpointReference);
        }    
        else if(serviceType==WSFServiceType.DISCOVERY_SERVICE)
        {
            return new DiscoveryService(discoveryService, endpointReference);
        }
        else if(serviceType==WSFServiceType.DIRECTORY_ACCESS_PROTOCOL_SERVICE)
        {
            return new DirectoryAccessProtocolService(discoveryService, endpointReference);
        }
        else if(serviceType==WSFServiceType.PERSONAL_PROFILE_SERVICE)
        {
            return new PersonalProfileService(discoveryService, endpointReference);
        }                
        else if(serviceType==WSFServiceType.PEOPLE_SERVICE)
        {
            return new PeopleService(discoveryService, endpointReference);
        }

        return null;
    }



    /**
     * An enumeration of services currently supported by the client library
     * 
     * @author asa
     *
     */
    public enum WSFServiceType
    {
        AUTHENTICATION_SERVICE(IDWSF_20_AUTHENTICATION_SERVICE_URN),
        DISCOVERY_SERVICE(IDWSF_20_DISCOVERY_SERVICE_URN),
        PERSONAL_PROFILE_SERVICE(IDWSF_20_PERSONAL_PROFILE_SERVICE_URN),
        PEOPLE_SERVICE(IDWSF_20_PEOPLE_SERVICE_URN),
        DIRECTORY_ACCESS_PROTOCOL_SERVICE(IDWSF_20_ID_SIS_DAP_SERVICE_URN),
        ;

        private String urn;

        private WSFServiceType(String urn)
        {
            this.urn = urn;
        }

        public static WSFServiceType serviceForUrn(String urn)
        {
            if(null!=urn)
            {
                for (WSFServiceType service : values())
                {
                    if(service.getUrn().equals(urn)) return service;
                }
            }
            return null;
        }

        public String getUrn()
        {
            return urn;
        }

    }



}
