package org.openliberty.wsc;

import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;
import org.opensaml.core.config.InitializationException;
import org.opensaml.core.config.InitializationService;
import org.opensaml.core.xml.config.XMLConfigurationException;
import org.opensaml.core.xml.config.XMLConfigurator;

/**
 * This class is used to bootstrap the openLiberty WSC Client Library with the default configurations that ship with the library.
 * It is also used to store basic settings for an Authentication server, SSL parameters, and trusted certificates.
 * 
 * @author asa
 */
public class OpenLibertyBootstrap 
{

    private static Logger log = Logger.getLogger(OpenLibertyBootstrap.class);

    public static final String AUTHETICATION_SERVICE_URL_KEY = "org.openliberty.wsc.OpenLibertyBootstrap.authenticationServiceUrl";
    
    /**
     * ProviderID to be used in the Sender header of WSF SOAP messages
     */
    private static String providerIDUri;
    
    /**
     * Default URL for a starting Authentication Service to enable bootstrapping the 
     * ID-WSF environment
     */
    private static String authenticationServiceUrl;
    
    /**
     * Default username for default AS
     */
    private static String authUsername;
    
    /**
     * Default password for default AS
     */
    private static String authPassword;
    
    private static AuthenticationService.AuthMechanism authMechanism;
    
    /**
     * These certificates are loaded as Trust material into the SSLClient when
     * a request is about to be made.
     */
    private static List<X509Certificate> trustedCertificates = new ArrayList<X509Certificate>();
    
    /**
     * This is the path (resource or system path) to the 
     * default PKS that the ClientLib will use for
     * signing.
     */
    private static String defaultSigningPKSPath;
        
    /**
     * Alias for the signing PKS private key 
     */
    private static String defaultSigningPKSAlias;
    
    /**
     * Password for the signing PKS private key 
     */
    private static String defaultSigningPKSPassword;
    
    /**
     * This is the path (resource or system path) to the 
     * default PKS that the ClientLib will use for
     * ClientTLS.
     */
    private static String defaultClientTLSPKSPath;
    
    /**
     * Alias for the ClientTLS PKS private key
     */
    private static String defaultClientTLSPKSAlias;
    
    /**
     * Password for the ClientTLS PKS private key
     */    
    private static String defaultClientTLSPKSPassword;

    /**
     * SSLClient setting for SSLUtilities: If false, host name is not 
     * verified for SSL transactions.  In production this should always
     * be set to true.  The default value is true.
     */
    private static boolean checkHostName = true;

    /**
     * SSLClient setting for SSLUtilities: If false the expiration 
     * date is ignored on certs used for TLS. The default value is true.
     */
    private static boolean checkExpiry = true;
    
    
    /**
     *  List of default XMLTooling configuration files.  This
     *  list is used to initialize all of the openLiberty
     *  and OpenSAML XML Tooling libraries.
     */
    private static String[] xmlToolingConfigs =
    { 
            "/sasl-xmltooling-config.xml",
            "/wsa-config.xml",
            "/disco-config.xml",
            "/lu-config.xml",
            "/id-sis-dap-config.xml",
            "/ps-config.xml",
            "/pp-config.xml",
            "/soap-binding-xmltooling-config.xml",
			"/openliberty-soap11-config.xml",
            "/sec-config.xml",
            "/subs-config.xml"
            // "/ims-config.xml"
    };
    
    /**
     * protected constructor
     */    
    protected OpenLibertyBootstrap() { }
    
    /**
     * Initializes the OpenLiberty library, loading default configurations.
     * 
     * @throws ConfigurationException thrown if there is a problem initializing the OpenSAML library
     */
    public static synchronized void bootstrap() throws XMLConfigurationException 
    {
        initializeOpenSAML();
    	initializeXMLTooling(xmlToolingConfigs);
        initializeClientLibProperties();
        // SSLUtilities.initializeSSLProtocol();        
    }
    
    private static void initializeOpenSAML() {
        try {
            InitializationService.initialize();
        } catch (InitializationException e) {
            log.error("Error initializing OpenSAML", e);
        }
    }

    /**
     * Initializes the OpenLiberty library, loading default configurations, and providing a mechanism
     * for a user to specify additional configs
     * 
     * @throws ConfigurationException thrown if there is a problem initializing the OpenSAML library
     */
    public static synchronized void bootstrap(String[] additionalXmlToolingConfigs) throws XMLConfigurationException 
    {
        String[] xmlToolingConfigsCombined = null; 
        if(null!=additionalXmlToolingConfigs)
        {
            xmlToolingConfigsCombined = new String[additionalXmlToolingConfigs.length+xmlToolingConfigs.length];
            int i=0;
            for(int j=0;j<xmlToolingConfigs.length;j++,i++)
            {
                xmlToolingConfigsCombined[i] = xmlToolingConfigs[j];
            }
            for(int j=0;j<additionalXmlToolingConfigs.length;j++,i++)
            {
                xmlToolingConfigsCombined[i] = additionalXmlToolingConfigs[j];
            }
        }
        else
        {
            xmlToolingConfigsCombined = xmlToolingConfigs;
        }
        
        initializeXMLTooling(xmlToolingConfigsCombined);
        initializeClientLibProperties();
        
        // SSLUtilities.initializeSSLProtocol();        
    }

    
    /**
     * Initializes the XMLTooling library with a default set of object providers.
     * 
     * @param providerConfigs list of provider configuration files located on the classpath
     * 
     * @throws XMLConfigurationException thrown if there is a problem loading the configuration files
     */
    protected static void initializeXMLTooling(String[] providerConfigs) throws XMLConfigurationException 
    {
        XMLConfigurator configurator = new XMLConfigurator();

        for (String config : providerConfigs) 
        {
        	log.info("Loading XMLTooling configuration " + config);
        	//configurator.load(Thread.currentThread().getContextClassLoader().getResourceAsStream(config));
        	//TODO
        	configurator.load(configurator.getClass().getResourceAsStream(config));
        }
    }
    
    /**
     * This static initializer loads all of the base settings for the
     * ClientLib
     * <p>
     * TODO: Load the base configuration from a settings file
     */
    protected static void initializeClientLibProperties()
    {
        
        // SP SENDER
        providerIDUri = "https://z-sp.openliberty.org";
        
        // AUTH
        authenticationServiceUrl = "";
        authUsername = "";
        authPassword = "";
        authMechanism = AuthenticationService.AuthMechanism.PLAIN;

        // SIGNING
        defaultSigningPKSPath = "/certs/z-sp-sign.jks";
        defaultSigningPKSAlias = "tomcat";
        defaultSigningPKSPassword = "changeit";

        // CLIENT TLS
        defaultClientTLSPKSPath = "/certs/z-sp-ssl.jks";
        defaultClientTLSPKSAlias ="tomcat";
        defaultClientTLSPKSPassword ="changeit";
    }

    public static String getAuthenticationServiceUrl()
    {
        return authenticationServiceUrl;
    }

    public static String getAuthPassword()
    {
        return authPassword;
    }

    public static String getAuthUsername()
    {
        return authUsername;
    }

    public static String getProviderIDUri()
    {
        return providerIDUri;
    }
    
    public static AuthenticationService.AuthMechanism getAuthMechanism()
    {
        return authMechanism;
    }

    public static String getDefaultClientTLSPKSAlias()
    {
        return defaultClientTLSPKSAlias;
    }

    public static String getDefaultClientTLSPKSPassword()
    {
        return defaultClientTLSPKSPassword;
    }

    public static String getDefaultClientTLSPKSPath()
    {
        return defaultClientTLSPKSPath;
    }

    public static String getDefaultSigningPKSAlias()
    {
        return defaultSigningPKSAlias;
    }

    public static String getDefaultSigningPKSPassword()
    {
        return defaultSigningPKSPassword;
    }

    public static String getDefaultSigningPKSPath()
    {
        return defaultSigningPKSPath;
    }
    
    public static List<X509Certificate> getTrustedCertificates()
    {
        return trustedCertificates;
    }
    
    public static boolean isCheckHostName()
    {
        return checkHostName;
    }

    public static boolean isCheckExpiry()
    {
        return checkExpiry;
    }
    
    public static void setAuthenticationServiceDefaults(String authenticationServiceUrl, String authUsername, String authPassword)
    {
        setAuthenticationServiceUrl(authenticationServiceUrl);
        setAuthUsername(authUsername);
        setAuthPassword(authPassword);
    }
        
    public static void setAuthenticationServiceUrl(String authenticationServiceUrl)
    {
        OpenLibertyBootstrap.authenticationServiceUrl = authenticationServiceUrl;
    }

    public static void setAuthPassword(String authPassword)
    {
        OpenLibertyBootstrap.authPassword = authPassword;
    }

    public static void setAuthUsername(String authUsername)
    {
        OpenLibertyBootstrap.authUsername = authUsername;
    }

    public static void setProviderIDUri(String providerIDUri)
    {
        OpenLibertyBootstrap.providerIDUri = providerIDUri;
    }

    public static void setAuthMechanism(AuthenticationService.AuthMechanism authMechanism)
    {
        OpenLibertyBootstrap.authMechanism = authMechanism;
    }

    public static void setDefaultClientTLSPKSAlias(String defaultClientTLSPKSAlias)
    {
        OpenLibertyBootstrap.defaultClientTLSPKSAlias = defaultClientTLSPKSAlias;
    }

    public static void setDefaultClientTLSPKSPassword(String defaultClientTLSPKSPassword)
    {
        OpenLibertyBootstrap.defaultClientTLSPKSPassword = defaultClientTLSPKSPassword;
    }

    public static void setDefaultClientTLSPKSPath(String defaultClientTLSPKSPath)
    {
        OpenLibertyBootstrap.defaultClientTLSPKSPath = defaultClientTLSPKSPath;
    }

    public static void setDefaultSigningPKSAlias(String defaultSigningPKSAlias)
    {
        OpenLibertyBootstrap.defaultSigningPKSAlias = defaultSigningPKSAlias;
    }

    public static void setDefaultSigningPKSPassword(String defaultSigningPKSPassword)
    {
        OpenLibertyBootstrap.defaultSigningPKSPassword = defaultSigningPKSPassword;
    }

    public static void setDefaultSigningPKSPath(String defaultSigningPKSPath)
    {
        OpenLibertyBootstrap.defaultSigningPKSPath = defaultSigningPKSPath;
    }

    public static void setCheckHostName(boolean checkHostName)
    {
        OpenLibertyBootstrap.checkHostName = checkHostName;
    }

    public static void setCheckExpiry(boolean checkExpiry)
    {
        OpenLibertyBootstrap.checkExpiry = checkExpiry;
    }


    
}
