package org.openliberty.wsc;

import java.util.ArrayList;
import java.util.List;

import net.shibboleth.utilities.java.support.xml.XMLParserException;

import org.apache.log4j.Logger;
import org.openliberty.xmltooling.ps.DisplayName;
import org.openliberty.xmltooling.ps.ObjectID;
import org.openliberty.xmltooling.ps.PSObject;
import org.openliberty.xmltooling.ps.PSObject.NodeType;
import org.openliberty.xmltooling.ps.TargetObjectID;
import org.openliberty.xmltooling.ps.request.AddCollectionRequest;
import org.openliberty.xmltooling.ps.request.AddEntityRequest;
import org.openliberty.xmltooling.ps.request.AddKnownEntityRequest;
import org.openliberty.xmltooling.ps.request.AddToCollectionRequest;
import org.openliberty.xmltooling.ps.request.CreatePSObject;
import org.openliberty.xmltooling.ps.request.Filter;
import org.openliberty.xmltooling.ps.request.GetObjectInfoRequest;
import org.openliberty.xmltooling.ps.request.ListMembersRequest;
import org.openliberty.xmltooling.ps.request.PStoSPRedirectURL;
import org.openliberty.xmltooling.ps.request.QueryObjectsRequest;
import org.openliberty.xmltooling.ps.request.RemoveCollectionRequest;
import org.openliberty.xmltooling.ps.request.RemoveEntityRequest;
import org.openliberty.xmltooling.ps.request.RemoveFromCollectionRequest;
import org.openliberty.xmltooling.ps.request.RequestAbstractType;
import org.openliberty.xmltooling.ps.request.ResolveIdentifierRequest;
import org.openliberty.xmltooling.ps.request.ResolveInput;
import org.openliberty.xmltooling.ps.request.SetObjectInfoRequest;
import org.openliberty.xmltooling.ps.request.Subscription;
import org.openliberty.xmltooling.ps.request.TestMembershipRequest;
import org.openliberty.xmltooling.ps.response.AddCollectionResponse;
import org.openliberty.xmltooling.ps.response.AddEntityResponse;
import org.openliberty.xmltooling.ps.response.AddKnownEntityResponse;
import org.openliberty.xmltooling.ps.response.AddToCollectionResponse;
import org.openliberty.xmltooling.ps.response.GetObjectInfoResponse;
import org.openliberty.xmltooling.ps.response.ListMembersResponse;
import org.openliberty.xmltooling.ps.response.QueryObjectsResponse;
import org.openliberty.xmltooling.ps.response.RemoveCollectionResponse;
import org.openliberty.xmltooling.ps.response.RemoveEntityResponse;
import org.openliberty.xmltooling.ps.response.RemoveFromCollectionResponse;
import org.openliberty.xmltooling.ps.response.ResolveIdentifierResponse;
import org.openliberty.xmltooling.ps.response.ResolveOutput;
import org.openliberty.xmltooling.ps.response.ResponseAbstractType;
import org.openliberty.xmltooling.ps.response.SetObjectInfoResponse;
import org.openliberty.xmltooling.ps.response.TestMembershipResponse;
import org.openliberty.xmltooling.security.Token;
import org.openliberty.xmltooling.security.TokenPolicy;
import org.openliberty.xmltooling.soap.soap11.BodyBuilder;
import org.openliberty.xmltooling.utility_2_0.Status;
import org.openliberty.xmltooling.wsa.EndpointReference;
import org.opensaml.core.xml.XMLObject;
import org.opensaml.core.xml.io.UnmarshallingException;
import org.opensaml.soap.soap11.Body;

/**
 * This implementation of a People Service Client has been tested against a People Service
 * defined inside of Conor Cahill's OSS ID-WSF Server.  This OSS Server does not support
 * subscriptions. 
 * 
 * @author asa
 *
 */
public class PeopleService extends BaseServiceClient
{
	protected static Logger log = Logger.getLogger(PeopleService.class);

	/**
	 * People Service URN
	 */
	public static final String SERVICE_URN = "urn:liberty:ps:2006-08";

	/**
	 * Enumeration that is used during a urn:liberty:ps:2006-08:ListMembersRequest action
	 * to indicate the structure (and with 'entities' to some degree the content) of the 
	 * response. 
	 * 
	 * @author asa
	 *
	 */
	public enum ListStructure 
	{ 
		children, tree, entities;
	}

	/**
	 * {@inheritDoc}
	 */
	public PeopleService(DiscoveryService discoveryService, EndpointReference initialEndpointReference)
	{
		super(discoveryService, initialEndpointReference);
	}

	/**
	 * Factory method that creates and initializes a PeopleService.
	 * 
	 * @param epr
	 * @return an instantiated PeopleService based on the EndpointReference that is passed in.
	 */
	public static PeopleService peopleServiceForEndpointReference(DiscoveryService discoveryService, EndpointReference epr)
	{
		PeopleService ps =  new PeopleService(discoveryService, epr);
		return ps;
	}

	/**
	 * (urn:liberty:ps:2006-08:AddEntity*) - The ClientLib will provide a mechanism for adding entities to the PS. The PS may optionally 
	 * return an URL and saml artifact that are used by the "invited" entity when responding to the "invitation"
	 * <p>
	 * <li>DisplayName will be required</li>
	 * <p>
	 * MUST include an &lt;Object&gt; element
	 * 
	 * @param entity the entity that the user would like to add
	 * @param optionalSubscription a notification subscription
	 * @return a PSEntity containing the ObjectID specified by the PeopleService, a flag will be set if the PS successfully adds, but does not return the object
	 */
	public AddEntityResponse addEntity(PSObject entity)
	{
		return addEntity(entity, null, null, null, false);
	}


	/**
	 * (urn:liberty:ps:2006-08:AddEntity*) - The ClientLib will provide a mechanism for adding entities to the PS. The PS may optionally 
	 * return an URL and saml artifact that are used by the "invited" entity when responding to the "invitation"
	 * <p>
	 * <li>DisplayName will be required</li>
	 * <li>Subscription will be supported, with TokenPolicy for requirements of the new identity's subscription</li>
	 * <p>
	 * MUST include an &lt;Object&gt; element
	 * 
	 * <pre>   
	 * &lt;!-- Declaration of AddEntityRequest element --&gt;
	 *     &lt;xs:element name="AddEntityRequest" type="AddEntityRequestType"/&gt;
	 *     &lt;!-- Definition of AddEntityRequestType --&gt;
	 *     &lt;xs:complexType name="AddEntityRequestType"&gt;
	 *         &lt;xs:complexContent&gt;
	 *             &lt;xs:extension base="RequestAbstractType"&gt;
	 *                 &lt;xs:sequence&gt;
	 *                     &lt;xs:element ref="Object"/&gt;
	 *                     &lt;xs:element ref="PStoSPRedirectURL" minOccurs="0"/&gt;
	 *                     &lt;xs:element ref="CreatePSObject" minOccurs="0"/&gt;
	 *                     &lt;xs:element ref="Subscription" minOccurs="0"/&gt;
	 *                     &lt;xs:element ref="sec:TokenPolicy" minOccurs="0"/&gt;
	 *                 &lt;/xs:sequence&gt;
	 *             &lt;/xs:extension&gt;
	 *         &lt;/xs:complexContent&gt;
	 *     &lt;/xs:complexType&gt;
	 * 
	 *     &lt;!-- Declaration of AddEntityResponse element --&gt;
	 *     &lt;xs:element name="AddEntityResponse" type="AddEntityResponseType"/&gt;
	 *     &lt;!-- Definition of AddEntityResponseType --&gt;
	 *     &lt;xs:complexType name="AddEntityResponseType"&gt;
	 *         &lt;xs:complexContent&gt;
	 *             &lt;xs:extension base="ResponseAbstractType"&gt;
	 *                 &lt;xs:sequence&gt;
	 *                     &lt;xs:element ref="Object" minOccurs="0"/&gt;
	 *                     &lt;xs:element ref="SPtoPSRedirectURL" minOccurs="0" maxOccurs="1"/&gt;
	 *                     &lt;xs:element ref="QueryString" minOccurs="0" maxOccurs="1"/&gt;
	 *                 &lt;/xs:sequence&gt;
	 *             &lt;/xs:extension&gt;
	 *         &lt;/xs:complexContent&gt;
	 *     &lt;/xs:complexType&gt;
	 * 
	 * </pre>
	 * 
	 * @param entity the entity that the user would like to add
	 * @param optionalSubscription a notification subscription
	 * @return a PSEntity containing the ObjectID specified by the PeopleService, a flag will be set if the PS successfully adds, but does not return the object
	 */
	public AddEntityResponse addEntity(PSObject entity, Subscription subscription, TokenPolicy policy, PStoSPRedirectURL pStoSPRedirectURL, boolean createReciprocalPSObject)
	{		

		if(entity.getNodeType()!=PSObject.NodeType.ENTITY)
		{
			if(log.isDebugEnabled()) log.debug("addEntity cannot be used to add a collection.  The PSObject provided must have a NodeType of ENTITY");
			return null;
		}

		// Make sure that the entity satisfies the MUSTs from the specification    	
		List<DisplayName> displayNames = entity.getDisplayNames();    	
		if(displayNames.size()<1 )
		{
			if(log.isDebugEnabled()) log.debug("Cannot add an entity that does not contain at least one display name");
			return null;
		}
		else
		{
			for(DisplayName displayName : displayNames)
			{
				String value = displayName.getValue();
				if(value==null || value.trim().length()==0)
				{
					if(log.isDebugEnabled()) log.debug("DisplayName requires a value, this entity cannot be created while containing an empty DisplayName value");
					return null;
				}
				String locale = displayName.getLocale();
				if(locale==null || locale.trim().length()==0)
				{
					if(log.isDebugEnabled()) log.debug("DisplayName requires a locale, this entity cannot be created while containing an unspecified locale");    				
					return null;
				}
			}
		}


		//
		// The WSC SHOULD NOT include a <sec:TokenPolicy> element unless also including a <Subscription> element
		//
		if(null!=policy && null==subscription)
		{
			return null;
		}

		// TODO: Any testing needed to insure proper PStoSPRedirectURL and sec:TokenPolicy

		// Build the request
		AddEntityRequest request = new AddEntityRequest();

		// add the entity to the request
		request.setObject(entity);        

		// now add the subscription if there is one
		if(null!=subscription)
		{
			request.setSubscription(subscription);
		}

		// now add the PStoSPRedirectURL if there is one
		if(null!=pStoSPRedirectURL)
		{
			request.setPStoSPRedirectURL(pStoSPRedirectURL);
		}

		// now add the CreatePSObject if it is requested
		if(createReciprocalPSObject)
		{
			CreatePSObject createPSObject = new CreatePSObject();
			request.setCreatePSObject(createPSObject);
		}

		// now add the TokenPolicy if there is one
		if(null!=policy)
		{
			request.setTokenPolicy(policy);
		}

		return (AddEntityResponse) invoke(request);
	}


	/**
	 * (urn:liberty:ps:2006-08:AddKnownEntity*) - The WSC sends an identity token to the PS to attempt the addition of a known entity.
	 * <p> 
	 * @param entity the entity that the user would like to add
	 * @param token used to convey an identity token for the target user Object being created.
	 * 
	 * @return an AddKnownEntityResponse
	 */
	public AddKnownEntityResponse addKnownEntity(PSObject entity, Token token)
	{
		return addKnownEntity(entity, token, null, null, null, false);
	}


	/**
	 * (urn:liberty:ps:2006-08:AddKnownEntity*) - The WSC sends an identity token to the PS to attempt the addition of a known entity.
	 * <p>
	 * <li>Subscription will be supported, with TokenPolicy for requirements of the new identity's subscription</li>
	 * 
	 * @param entity the entity that the user would like to add
	 * @param token used to convey an identity token for the target user Object being created.
	 * @param subscription 
	 * @param policy
	 * @param pStoSPRedirectURL
	 * @param createReciprocalPSObject if true, indicates that the WSC wants an object created representing the inviting user at the invited user's PS.
	 * 
	 * @return an AddKnownEntityResponse
	 */
	public AddKnownEntityResponse addKnownEntity(PSObject entity, Token token, Subscription subscription, TokenPolicy policy, PStoSPRedirectURL pStoSPRedirectURL, boolean createReciprocalPSObject)
	{		
		if(entity.getNodeType()!=PSObject.NodeType.ENTITY)
		{
			if(log.isDebugEnabled()) log.debug("addKnownEntity cannot be used to add a collection.  The PSObject provided must have a NodeType of ENTITY");
			return null;
		}

		// Make sure that the entity satisfies the MUSTs from the specification    	
		List<DisplayName> displayNames = entity.getDisplayNames();    	
		if(displayNames.size()<1 )
		{
			if(log.isDebugEnabled()) log.debug("Cannot add an entity that does not contain at least one display name");
			return null;
		}
		else
		{
			for(DisplayName displayName : displayNames)
			{
				String value = displayName.getValue();
				if(value==null || value.trim().length()==0)
				{
					if(log.isDebugEnabled()) log.debug("DisplayName requires a value, this entity cannot be created while containing an empty DisplayName value");
					return null;
				}
				String locale = displayName.getLocale();
				if(locale==null || locale.trim().length()==0)
				{
					if(log.isDebugEnabled()) log.debug("DisplayName requires a locale, this entity cannot be created while containing an unspecified locale");    				
					return null;
				}
			}
		}

		//
		//The WSC SHOULD NOT include a <sec:TokenPolicy> element unless also including a <Subscription> element
		//
		if(null!=policy && null==subscription)
		{
			return null;
		}

		// TODO: Any testing needed to insure proper sec:TokenPolicy

		// Build the request
		AddKnownEntityRequest request = new AddKnownEntityRequest(entity, token);

		// now add the subscription if there is one
		if(null!=subscription)
		{
			request.setSubscription(subscription);
		}

		// now add the CreatePSObject if it is requested
		if(createReciprocalPSObject)
		{
			CreatePSObject createPSObject = new CreatePSObject();
			request.setCreatePSObject(createPSObject);
		}

		// now add the TokenPolicy if there is one
		if(null!=policy)
		{
			request.setTokenPolicy(policy);
		}


		return (AddKnownEntityResponse) invoke(request);
	}


	/**
	 * This method takes a list of PSObjects and asks the PS to remove them
	 * 
	 * <pre>
	 * 
	 *     &lt;!-- Declaration of RemoveEntityRequest element --&gt;
	 *     &lt;xs:element name="RemoveEntityRequest" type="RemoveEntityRequestType"/&gt;
	 *     &lt;!-- Definition of RemoveEntityRequestType --&gt;
	 *     &lt;xs:complexType name="RemoveEntityRequestType"&gt;
	 *         &lt;xs:complexContent&gt;
	 *             &lt;xs:extension base="RequestAbstractType"&gt;
	 *                 &lt;xs:sequence&gt;
	 *                     &lt;xs:element ref="TargetObjectID" maxOccurs="unbounded"/&gt;
	 *                 &lt;/xs:sequence&gt;
	 *             &lt;/xs:extension&gt;
	 *         &lt;/xs:complexContent&gt;
	 *     &lt;/xs:complexType&gt;
	 * 
	 *     &lt;!-- Declaration of RemoveEntityResponse element --&gt;
	 *     &lt;xs:element name="RemoveEntityResponse" type="ResponseAbstractType"/&gt;
	 * </pre>
	 * @param objectRefs
	 * @return
	 */
	public RemoveEntityResponse removeEntities(List<PSObject> entities)
	{
		if(null==entities) return null;

		List<TargetObjectID> targetIDs = new ArrayList<TargetObjectID>(entities.size());

		for(PSObject anEntity : entities)
		{
			if(anEntity.getNodeType()==NodeType.ENTITY)
			{
				TargetObjectID targetID = anEntity.createTargetObjectID();
				if(null!=targetID) targetIDs.add(targetID);				
				else if(log.isDebugEnabled()) log.debug("PSObject \'"+anEntity.getName()+"\' has no objectId and therefore will not be removed using removeEntities.");

			}
			else
			{
				if(log.isDebugEnabled()) log.debug("PSObject \'"+anEntity.getName()+"\' cannot be removed using removeEntities, since it is a collection");
			}
		}

		return removeEntitiesWithTargetObjectIds(targetIDs);
	}

	
	/**
	 * Removes a list of entities from the People Service.
	 * <p>
	 * This method takes a list of PS Entity identifiers as strings, creates
	 * a list of TargetObjectID objects, and calls removeEntitiesWithTargetObjectIds
	 * which attempts to remove the list of entities from the People Service. 
	 * 
	 * @param entityIds
	 * @return
	 */
	public RemoveEntityResponse removeEntitiesWithIds(List<String> entityIds)
	{
		if(null==entityIds) return null;

		List<TargetObjectID> targetIDs = new ArrayList<TargetObjectID>(entityIds.size());

		for(String anEntityId : entityIds)
		{
			TargetObjectID targetID = new TargetObjectID(anEntityId);// anEntity.createTargetObjectID();
			if(null!=targetID) targetIDs.add(targetID);
		}
		return removeEntitiesWithTargetObjectIds(targetIDs);
	}
	

	/**
	 * A convenience method that removes a single PS Entity referenced by the 
	 * supplied TargetObjectID from the People Service.
	 * 
	 * @param id
	 * @return
	 */
	public RemoveEntityResponse removeEntityWithTargetObjectID(TargetObjectID id)
	{
		List<TargetObjectID> targetObjectIDs = new ArrayList<TargetObjectID>();
		targetObjectIDs.add(id);
		return removeEntitiesWithTargetObjectIds(targetObjectIDs);
	}


	/**
	 * This method makes a request to remove PS Entities from a People Service based on the supplied
	 * TargetObjectIDs.
	 * <p>
	 * This method assumes that the TargetObjectIDs are all associated with PS Entity objects.  The PS will
	 * fail the removal request if any of the TargetObjectIDs reference PS Collections.
	 * 
	 * @param targetObjectIDs
	 * @return
	 */
	public RemoveEntityResponse removeEntitiesWithTargetObjectIds(List<TargetObjectID> targetObjectIDs)
	{		
		RemoveEntityRequest request = new RemoveEntityRequest();
		request.getTargetObjectIDs().addAll(targetObjectIDs);

		return (RemoveEntityResponse) invoke(request);
	}

	/**
	 * (urn:liberty:ps:2006-08:TestMembership*) 
	 * Poses the question 'Is user X a member of group Y?' to the PS
	 * 
	 * @param identityToken required - identifies the entity being tested for membership
	 * @param collection - identifies the collection to query about membership
	 * @param subscription
	 * @return
	 */
	public TestMembershipResponse testMembership(Token identityToken, TargetObjectID collection, Subscription subscription) 
	{
		if(null==identityToken)
		{
			if(log.isDebugEnabled()) log.debug("Null values for the following parameters are not valid when testing membership in a collection: identityToken ("+identityToken+").");
			return null;
		}

		TestMembershipRequest request = new TestMembershipRequest(identityToken);
		if(null!=collection) request.setTargetObjectID(collection);
		if(null!=subscription) request.setSubscription(subscription);

		return (TestMembershipResponse) invoke(request);
	}


	/**
	 * (urn:liberty:ps:2006-08:AddCollection*) - The ClientLib provides a mechanism for adding collections to the PS.
	 * <p>
	 * <li>At least one DisplayName is required in the collection</li>
	 * <li>Subscription is optional</li>
	 * </p>
	 * 
	 * @param collection the collection that the user would like to add
	 * @param subscription a notification subscription
	 * 
	 * @return a PSCollection containing the ObjectID generated by the PeopleService, or null upon failure
	 */
	public AddCollectionResponse addCollection(PSObject collection, Subscription subscription)
	{		

		if(!collection.isCollection())			
		{
			if(log.isDebugEnabled()) log.debug("The PSObject specified in addCollection must be a collection");
			return null;
		}

		// Make sure that the entity satisfies the MUSTs from the specification    	
		List<DisplayName> displayNames = collection.getDisplayNames();    	
		if(displayNames.size()<1 )
		{
			if(log.isDebugEnabled()) log.debug("Cannot add an collection that does not contain at least one display name");
			return null;
		}
		else
		{
			for(DisplayName displayName : displayNames)
			{
				String value = displayName.getValue();
				if(value==null || value.trim().length()==0)
				{
					if(log.isDebugEnabled()) log.debug("DisplayName requires a value, this collection cannot be created while containing an empty DisplayName value");
					return null;
				}
				String locale = displayName.getLocale();
				if(locale==null || locale.trim().length()==0)
				{
					if(log.isDebugEnabled()) log.debug("DisplayName requires a locale, this collection cannot be created while containing an unspecified locale");    				
					return null;
				}
			}
		}

		AddCollectionRequest request = new AddCollectionRequest();

		// add the collection to the request
		request.setObject(collection);        

		// now add the subscription if there is one
		if(null!=subscription)
		{
			request.setSubscription(subscription);
		}

		return (AddCollectionResponse) invoke(request);
	}


	/**
	 * This method takes a list of PSCollections and asks the PS to remove them
	 * 
	 * @param objectRefs
	 * @return
	 */
	public RemoveCollectionResponse removeCollections(List<PSObject> collections)
	{
		if(null==collections) 
		{
			if(log.isDebugEnabled()) log.debug("collections must contain Collection PSObjects");
			return null;
		}

		List<TargetObjectID> targetIDs = new ArrayList<TargetObjectID>(collections.size());

		for(PSObject aCollection : collections)
		{
			if(aCollection.isCollection())			
			{
				TargetObjectID targetID = aCollection.createTargetObjectID();
				if(null!=targetID) targetIDs.add(targetID);
				else if(log.isDebugEnabled()) log.debug("The PSObject "+aCollection.getName()+" has no object id and cannot be removed");
			}
			else
			{
				if(log.isDebugEnabled()) log.debug("The PSObject "+aCollection.getName()+" is not a collection");
			}
		}	
		return removeCollectionsForTargetIDs(targetIDs);		
	}
	

	/**
	 * This convenience method takes a single TargetObjectID that should reference a collection
	 * and asks the PS to remove it.
	 * 
	 * @param id
	 * @return
	 */
	public RemoveCollectionResponse removeCollectionForTargetObjectID(TargetObjectID id)
	{
		List<TargetObjectID> targetCollectionIDs = new ArrayList<TargetObjectID>();
		targetCollectionIDs.add(id);
		return removeCollectionsForTargetIDs(targetCollectionIDs);
	}

	/**
	 * This method takes a list of TargetObjectIDs and asks the PS to remove their 
	 * associated Collections.  
	 * <p>
	 * Assumes that the TargeObjectID objects specified point to collections, if not, the PS
	 * will return a failure.
	 * 
	 * @param targetIDs
	 * @return
	 */
	public RemoveCollectionResponse removeCollectionsForTargetIDs(List<TargetObjectID> targetIDs)
	{
		RemoveCollectionRequest request = new RemoveCollectionRequest();
		request.getTargetObjectIDs().addAll(targetIDs);


		return (RemoveCollectionResponse) invoke(request);
	}


	/**
	 * This method requests that the PS add the objects referenced by the supplied object IDs to the collection specified by the
	 * collection ID
	 * 
	 * @param objectIDs
	 * @param collectionID
	 * @param subscription
	 */
	public AddToCollectionResponse addObjectsToCollection(List<ObjectID> objectIDs, TargetObjectID collectionID, Subscription subscription)
	{
		if(null==collectionID) 
		{
			if(log.isDebugEnabled()) log.debug("The ObjectID of the target collection is required");
			return null;
		}

		if(null==objectIDs || objectIDs.size()==0)
		{
			if(log.isDebugEnabled()) log.debug("1 to N ObjectID objects must be specified");
			return null;
		}

		AddToCollectionRequest request = new AddToCollectionRequest(collectionID, objectIDs);

		if(null!=subscription)
		{
			request.setSubscription(subscription);
		}

		return (AddToCollectionResponse) invoke(request);
	}


	/**
	 * This method requests that the PS remove the objects referenced by the supplied object IDs from the collection specified by the
	 * collection ID
	 * 
	 * @param objectIDs
	 * @param collectionID
	 * @param subscription
	 */
	public RemoveFromCollectionResponse removeObjectsFromCollection(List<ObjectID> objectIDs, TargetObjectID collectionID, Subscription subscription)
	{

		// MUST specify, as a value of the <TargetObjectID>, an ObjectID of the Object that has 
		// urn:liberty:ps:collection as a value of NodeType attribute. 
		if(null==collectionID) 
		{
			if(log.isDebugEnabled()) log.debug("The ObjectID of the target collection is required");
			return null;
		}

		if(null==objectIDs || objectIDs.size()==0)
		{
			if(log.isDebugEnabled()) log.debug("1 to N ObjectID objects must be specified");
			return null;
		}

		RemoveFromCollectionRequest request = new RemoveFromCollectionRequest(collectionID, objectIDs);

		if(null!=subscription)
		{
			request.setSubscription(subscription);
		}

		return (RemoveFromCollectionResponse) invoke(request);
	}	
	

	/**
	 * The PS Client will provide a facility for a ClientLIb user to list the members of a collection. If no collection is indicated, then the root collection is assumed.
	 * urn:liberty:ps:2006-08:ListMembers
	 * <p>
	 * structure can be "children" "tree" or "entities"
	 * <li>children indicates only the direct child entities and collections</li>
	 * <li>tree indicates return the full tree structure</li>
	 * <li>entities indicates a flat view of the full tree, with no collections</li> 
	 * <p>
	 * count and offset are used to paginate the response tree/list 
	 * 
	 * @param collectionID if not specified root is assumed
	 * @param structure indicates the structure of the results returned
	 * @param count non-negative value indicating how many objects to return max 
	 * @param offset non-negative value indicating where in the full results set to begin returning objects
	 * @param subscription
	 * @return
	 */
	public ListMembersResponse listCollectionMembers(TargetObjectID collectionID, ListStructure structure, Integer count, Integer offset, Subscription subscription)
	{

		/**
		 * The WSC SHOULD NOT include a <Subscription> element in a <ListMembersRequest> message if the 
		 * Offset attribute has any value other than "0."  
		 * <p>
		 * no failure here, but a debugging message
		 */
		if(offset!=0 && subscription!=null)			
		{
			if(log.isDebugEnabled()) log.debug("The WSC SHOULD NOT include a <Subscription> element in a <ListMembersRequest> message if the Offset attribute has any value other than \"0.\"");
		}

		ListMembersRequest request = new ListMembersRequest();

		request.setStructured(structure.name());
		request.setCount(new Integer(count));
		request.setOffset(new Integer(offset));

		if(null!=collectionID)
		{
			request.setTargetObjectID(collectionID);
		}

		if(null!=subscription)
		{
			request.setSubscription(subscription);
		}

		return (ListMembersResponse) invoke(request);
	}


	/**
	 * (urn:liberty:ps:2006-08:GetObjectInfo*) a method for retrieving the object info for a specific object. 
	 * 
	 *<pre>
	 *     &lt;!-- Declaration of GetObjectInfoRequest element --&gt;
	 *     &lt;xs:element name="GetObjectInfoRequest" type="GetObjectInfoRequestType"/&gt;
	 *     &lt;!-- Definition of GetObjectInfoRequestType --&gt;
	 *     &lt;xs:complexType name="GetObjectInfoRequestType"&gt;
	 *         &lt;xs:complexContent&gt;
	 *             &lt;xs:extension base="RequestAbstractType"&gt;
	 *                 &lt;xs:sequence&gt;
	 *                     &lt;xs:element ref="TargetObjectID" minOccurs="0"/&gt;
	 *                     &lt;xs:element ref="Subscription" minOccurs="0"/&gt;
	 *                 &lt;/xs:sequence&gt;
	 *             &lt;/xs:extension&gt;
	 *         &lt;/xs:complexContent&gt;
	 *     &lt;/xs:complexType&gt;
	 * </pre>
	 * 
	 * RESPONSE:
	 * <pre>
	 * &lt;!-- Declaration of GetObjectInfoResponse element --&gt;
	 *     &lt;xs:element name="GetObjectInfoResponse" type="GetObjectInfoResponseType"/&gt;
	 *     &lt;!-- Definition of GetObjectInfoResponseType --&gt;
	 *     &lt;xs:complexType name="GetObjectInfoResponseType"&gt;
	 *         &lt;xs:complexContent&gt;
	 *             &lt;xs:extension base="ResponseAbstractType"&gt;
	 *                 &lt;xs:sequence&gt;
	 *                     &lt;xs:element ref="Object" minOccurs="0"/&gt;
	 *                 &lt;/xs:sequence&gt;
	 *             &lt;/xs:extension&gt;
	 *         &lt;/xs:complexContent&gt;
	 *     &lt;/xs:complexType&gt;
	 * </pre>
	 * 
	 * @param targetObjectID 
	 * @param optionalSubscription
	 * @return
	 */
	public PSObject getObjectInfo(TargetObjectID targetObjectID, Subscription subscription)
	{
		// make sure that the targetObjectId has been specified
		if(null!=targetObjectID)
		{
			String id = targetObjectID.getValue();
			if(null==id || id.length()==0)
			{
				if(log.isDebugEnabled()) log.debug("targetObjectID is required for A GetObjectInfo request");
				return null;
			}
		}
		else
		{
			if(log.isDebugEnabled()) log.debug("targetObjectID is required for A GetObjectInfo request");
			return null;
		}

		GetObjectInfoRequest request = new GetObjectInfoRequest(targetObjectID); 

		if(null!=subscription)
		{
			request.setSubscription(subscription);
		}

		GetObjectInfoResponse response = (GetObjectInfoResponse) invoke(request);
		if(null!=response)
		{
			Status status = response.getStatus();
			if(log.isDebugEnabled()) log.debug("GetObjectInfoResponse Status: "+status.getCode());
			if(status.isOK())
			{
				return response.getObject();
			}
		}
		
		return null;
	}

	/**
	 * (urn:liberty:ps:2006-08:SetObjectInfo*) updating the object info for a list of existing objects (1...n)
	 * 
	 * @param objects
	 * @param subscription
	 * @return
	 */
	public SetObjectInfoResponse updateObjects(List<PSObject> objects, Subscription subscription)
	{
		SetObjectInfoRequest request = new SetObjectInfoRequest(objects);

		if(null!=subscription)
		{
			request.setSubscription(subscription);
		}

		// Invoke the request		
		return (SetObjectInfoResponse)invoke(request);
	}

	
	/**
	 * This method simply invokes updateObjects() and is included for naming consistency with the spec
	 * 
	 * @param objectRef
	 * @param optionalSubscription
	 * @return
	 */
	public SetObjectInfoResponse setObjectInfo(List<PSObject> objects, Subscription subscription)
	{
		return updateObjects(objects, subscription);
	}


	/**
	 * urn:liberty:ps:2006-08:ResolveIdentifier*
	 * Method for retrieving stored identity tokens from an object identifier list 1...n. Each object 
	 * identifier may specify the requirements of the identity token through the use of sec:TokenPolicy.
	 * 
	 * @param inputs
	 * @return
	 */
	public List<ResolveOutput> resolveIdentifiers(List<ResolveInput> inputs)
	{
		ResolveIdentifierRequest request = new ResolveIdentifierRequest();
		request.getResolveInputs().addAll(inputs);

		// Invoke the request
		ResolveIdentifierResponse response = (ResolveIdentifierResponse)invoke(request);

		return response!=null?response.getResolveOutputs():null;
	}
	

	/**
	 * (urn:liberty:ps:2006-08:QueryObjects*) The PS Client will provide a mechanism for creating an xpath base query to the PS, which will return 0...n Objects in a flat list (no tree).
	 * The query is constructed using XPath syntax, always beginning with //Object 
	 */
	public List<PSObject> queryObjects(String pathExpression, int count, int offset, Subscription subscription)
	{
		Filter filter = new Filter(pathExpression);		

		if(!filter.getPathExpression().equals(pathExpression)) 
		{
			if(log.isDebugEnabled()) log.debug("pathExpression \""+pathExpression+ "\" appears to be invalid.");
			return null;
		}

		QueryObjectsRequest request = new QueryObjectsRequest(filter);

		if(null!=subscription)
		{
			request.setSubscription(subscription);
		}

		if(count<0) count = 0;
		request.setCount(new Integer(count));

		if(offset<0) offset = 0;		
		request.setOffset(new Integer(offset));

		QueryObjectsResponse response = (QueryObjectsResponse)invoke(request);

		return null!=response?response.getObjects():null;
	}
	

	/**
	 * The basic invocation which involves the creation of an ID-WSF message and construction of a request 
	 * body remains the same from request to request.
	 * 
	 * @param request
	 * @return
	 */
	private ResponseAbstractType invoke(RequestAbstractType request)
	{
		WSFMessage message = null;

		try
		{
			// create the message
			message = WSFMessage.createWSFMessage(this, request.type().actionURI());

			// Build the request Body, adding the request to the SOAP call
			Body requestBody = new BodyBuilder().buildObject();  
			requestBody.getUnknownXMLObjects().add(request);
			message.getRequestEnvelope().setBody(requestBody);

			try
			{
				// invoke the message
				message.invoke();

				// Get the BasePeopleServiceResponse
				Body responseBody = message.getResponseEnvelope().getBody();            

				// Process the response, looking for a match with the class listed in the request type enumeration
				for(XMLObject object : responseBody.getUnknownXMLObjects())
				{
					// if the response is right, return it
					if(request.type().responseClass().isInstance(object))
					{
						return (ResponseAbstractType) object;
					}
				}            
				if(log.isDebugEnabled()) log.debug("No "+request.type().responseClass().getName()+" element found in the body of the response Envelope");


			} 
			catch (Exception e)
			{
				if(log.isDebugEnabled()) log.error("Exception while attempting to invoke a WSFMessage", e);
			}

		}
		catch (XMLParserException e1)
		{
			e1.printStackTrace();
		}
		catch (UnmarshallingException e1)
		{
			e1.printStackTrace();
		}

		return null;
	}



}

