package org.openliberty.wsc;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;

import net.shibboleth.utilities.java.support.xml.XMLParserException;

import org.apache.commons.lang.StringEscapeUtils;
import org.apache.log4j.Logger;
import org.openliberty.xmltooling.Konstantz;
import org.openliberty.xmltooling.dst2_1.ref.Data;
import org.openliberty.xmltooling.dst2_1.ref.DefaultBuilder;
import org.openliberty.xmltooling.dst2_1.ref.Modify;
import org.openliberty.xmltooling.dst2_1.ref.ModifyItem;
import org.openliberty.xmltooling.dst2_1.ref.ModifyResponse;
import org.openliberty.xmltooling.dst2_1.ref.NewData;
import org.openliberty.xmltooling.dst2_1.ref.Query;
import org.openliberty.xmltooling.dst2_1.ref.QueryItem;
import org.openliberty.xmltooling.dst2_1.ref.QueryResponse;
import org.openliberty.xmltooling.dst2_1.ref.Select;
import org.openliberty.xmltooling.pp.AddressCard;
import org.openliberty.xmltooling.pp.AltID;
import org.openliberty.xmltooling.pp.MsgContact;
import org.openliberty.xmltooling.pp.MsgTechnology;
import org.openliberty.xmltooling.pp.dst2_1.ct.AddrType;
import org.openliberty.xmltooling.pp.dst2_1.ct.IDType;
import org.openliberty.xmltooling.pp.dst2_1.ct.MsgMethod;
import org.openliberty.xmltooling.pp.dst2_1.ct.MsgType;
import org.openliberty.xmltooling.soap.soap11.BodyBuilder;
import org.openliberty.xmltooling.utility_2_0.Status;
import org.openliberty.xmltooling.wsa.EndpointReference;
import org.opensaml.core.xml.XMLObject;
import org.opensaml.core.xml.io.UnmarshallingException;
import org.opensaml.soap.soap11.Body;

/**
 * The Liberty ID-SIS Personal Profile (ID-SIS-PP) defines a web service. It offers profile information regarding a 
 * Principal. ID-SIS-PP is an instance of a data-oriented identity web service and is characterized by the ability to query 
 * and update attribute data and incorporates mechanisms from other specifications for access control and conveying 
 * data validation information and usage directives. 
 * <p>
 * This Service Client models PP v1.1 using DST 2.1 and utilizing the DST 2.1 Reference implementation.  Even though DST 2.1
 * defines a methodology for Delete and Create, PP only uses Query and Modify. 
 * <p>
 * Currently Supported:
 * <ul>
 *  <li> <b>ServiceType</b>                 urn:liberty:id-sis-pp:2005-05</li>
 *  <li> <b>Discovery Options</b>           query, but no impact on invocation</li>
 *  <li> <b>Data Schema</b>                 100%</li>
 *  <li> <b>SelectType Element</b></li>
 *  <li> <b>Query Language</b>              XPath Subset.  Not enforced, unless specified</li>
 *  <li> <b>Multiple QueryItem</b></li>
 *  <li> <b>Support Modification</b></li>
 *  <li> <b>Multiple Modification</b>       Supported, but in DST2.1 called "ModifyItem"
 *  <li> <b>changedSince</b>                indirect support through DST
 *  <li> <b>notChangedSince</b>             indirect support through DST
 *  <li> <b>includeCommmonAttributes</b>    indirect support through DST
 *  <li> <b>Data Extension Support</b>      through DST and through not enforcing specific service XPath strings
 *  
 * </ul>
 * Currently Not Supported:
 * <ul>
 *  <li> <b>Multiple Query</b>  listed as "MAY"
 *  <li> <b>Multiple Modify</b>   listed as "MAY"
 * </ul>
 *
 *
 * 
 * <p>
 * @author asa
 *
 */
public class PersonalProfileService extends BaseServiceClient
{

    private static Logger log = Logger.getLogger(PersonalProfileService.class);

    /**
     * This is the URN representing the Personal Profile Service
     */
    public static final String SERVICE_URN = DiscoveryService.WSFServiceType.PERSONAL_PROFILE_SERVICE.getUrn();


    /**
     * This is a URN that may be passed as an option from the Discovery 
     * Service indicating that the PP supports multiple ModifyItem objects in single Modify request
     */    
    public static final String OPTION_MULTIPLE_MODIFY_ITEM = "urn:liberty:dst:multipleModification";

    /**
     * This is a URN that may be passed as an option from the Discovery 
     * Service indicating that the PP supports utilizing the Extensions and xs:anyAttributes
     * specified in the DST
     */    
    public static final String OPTION_SUPPORTS_EXTENSIONS = "urn:liberty:dst:can:extend";

    /**
     * This is a URN that may be passed as an option from the Discovery 
     * Service indicating that the full XPath language is supported for making
     * Queries and Modifications.
     */
    public static final String OPTION_SUPPORTS_FULL_XPATH = "urn:liberty:dst fullXPath";

    /**
     * The suffix that is combined with the SERVICE_URN to indicate in the Action element of the SOAP header
     * that the request is a Personal Profile Query.  
     */
    public static final String DST_QUERY_SUFFIX = ":dst-2.1:Query";

    /**
     * The suffix that is combined with the SERVICE_URN to indicate in the Action element of the SOAP header
     * that the request is a Personal Profile Modification.  
     */
    public static final String DST_MODIFY_SUFFIX = ":dst-2.1:Modify";

    /**
     * {@inheritDoc}
     */
    public PersonalProfileService(DiscoveryService discoveryService, EndpointReference initialEndpointReference)
    {
        super(discoveryService, initialEndpointReference);
    }

    /**
     * Factory method that creates and initializes a PersonalProfileService.
     * 
     * @param epr
     * @return an instantiated PersonalProfileService based on the EndpointReference that is passed in.
     */
    public static PersonalProfileService personalProfileServiceForEndpointReference(DiscoveryService discoveryService, EndpointReference epr)
    {
        PersonalProfileService pp =  new PersonalProfileService(discoveryService, epr);
        return pp;
    }

    /**
     * This indicates whether Full XPath Expressions can be used for any modify or query invocations
     * <p>
     * The Select string holds an XPATH expression. An ID-SIS-PP implementation MAY support full XPATH expressions 
     * [XPATH] as a Select expression. If it does support full XPATH expressions, it MAY advertise the discovery option 
     * keyword "urn:liberty:dst:fullXPath" 
     * <p>
     * If an implementation supports full XPATH for querying, then it MUST also support full XPATH for modifies. 
     * 
     * @return
     */
    public boolean serviceExplicitlySupportsFullXPath()
    {
        return serviceExplicitlySupportsOption(OPTION_SUPPORTS_FULL_XPATH);
    }

    /**
     * ID-SIS-PP elements that have enumerated values use URIs as values ("values" may be referred to as "enumerators"). 
     * Each element�s description details the authority for adopting new official enumeration values. See [LibertyReg] for 
     * more information. 
     * <p>
     * All containers and elements defined in the ID-SIS-PP schema have an Extension element which permits arbitrary 
     * schema extension. An implementation MAY support schema extension, but is not required to do so. If an 
     * implementation does support schema extension, then it MAY register the corresponding discovery option keyword 
     * "urn:liberty:dst:can:extend" 
     * <p>
     * If you plan to extend this PP Client you will need to adhere to the approach designated by the OpenSAML Java XML-Tooling
     * libraries.  Any extended elements will need to extend XMLObject
     * 
     * @return
     */
    public boolean serviceExplicitlySupportsExtensions()
    {
        return serviceExplicitlySupportsOption(OPTION_SUPPORTS_EXTENSIONS);
    }

    /**
     * Checks whether multiple modification elements (ModifyItem) may be submitted in one Modify request
     * <p>
     * A minimally compliant implementation is not required to support multiple Modification elements. The 
     * Modify operation functions as described in [LibertyDST]. The Modify operation has the additional relaxation that 
     * a minimally compliant ID-SIS-PP implementation MAY refuse a Modify request with multiple Modification 
     * elements, provided all processing rules specified in [LibertyDST] are followed regarding failure to support multiple 
     * Modification elements. 
     * <p>
     * Implementations SHOULD support multiple Modification elements when feasible. If an implementation supports 
     * multiple Modification elements it MAY register the discovery option keyword "urn:liberty:dst:multipleModification." 
     * 
     * @return
     */
    public boolean serviceExplicitlySupportsMultipleModifyItems()
    {
        return serviceExplicitlySupportsOption(OPTION_MULTIPLE_MODIFY_ITEM);
    }

    /**
     * Test to see whether the Personal Profile Service has registered support for a specific 
     * data query option with the Discovery Service. 
     * 
     * @param option
     * @return
     */
    public boolean serviceExplicitlySupportsDataQueryOption(DataQueryOption option)
    {
        return serviceExplicitlySupportsOption(option.keyword);
    }

    /**
     * Test to see whether the Personal Profile Service has registered support for a specific 
     * data modify option with the Discovery Service. 
     * 
     * @param option
     * @return
     */    
    public boolean serviceExplicitlySupportsDataModifyOption(DataModifyOption option)
    {
        return serviceExplicitlySupportsOption(option.keyword);
    }


    /**
     * This method returns a ModifyItem with the data necessary to update the portion of the 
     * Personal Profile indicated by the xPathModifySelector with the provided modifiedObject
     * <p>
     * The following xPathModifySelectors MUST be supported by the PP:
     * <ul>
     * <li>/pp:PP </li>
     * <li>/pp:PP/pp:InformalName </li>
     * <li>/pp:PP/pp:LInformalName </li>
     * <li>/pp:PP/pp:CommonName </li>
     * <li>/pp:PP/pp:LegalIdentity </li>
     * <li>/pp:PP/pp:EmploymentIdentity </li>
     * <li>/pp:PP/pp:AddressCard </li>
     * <li>/pp:PP/pp:MsgContact </li>
     * <li>/pp:PP/pp:Facade </li>
     * <li>/pp:PP/pp:Demographics </li>
     * <li>/pp:PP/pp:SignKey </li>
     * <li>/pp:PP/pp:EncryptKey </li>
     * <li>/pp:PP/pp:EmergencyContact </li>
     * <li>/pp:PP/pp:LEmergencyContact </li>
     * </ul>
     * This set of slashed paths defines the minimal granularity of updates that MUST be supported. Updates to the containers 
     * listed above SHOULD be atomic when feasible. 
     * 
     * @param xPathModifySelector
     * @param modifiedObject
     * @return
     */
    public ModifyItem modifyItemForXPathAndObject(String xPathModifySelector, XMLObject modifiedObject)
    {

        // TODO: Some checking that the xPathString matches the modifiedObject

        // TODO: Check to make sure that the xPathModifyString is truly proper XPath 

        // Get a Builder for the DST Reference Items
        DefaultBuilder builder = new DefaultBuilder();

        // Build the ModifyItem (which is a "Modification" in DST2.0 and DST1.1)
        ModifyItem modifyItem =  new ModifyItem(Konstantz.PP_NS, ModifyItem.LOCAL_NAME_DST1_1, Konstantz.PP_PREFIX); //(ModifyItem)builder.buildObjectForClass(Konstantz.PP_NS, ModifyItem.LOCAL_NAME_DST1_1, Konstantz.PP_PREFIX, ModifyItem.class);
        modifyItem.getModifyItemAttributes().setOverrideAllowed(true, modifyItem);

        // Build the Select element
        Select select = (Select) builder.buildObject(Konstantz.PP_NS, Select.LOCAL_NAME, Konstantz.PP_PREFIX);        
        select.setValue(xPathModifySelector);

        // Build the NewData Object
        NewData newData = (NewData) builder.buildObject(Konstantz.PP_NS, NewData.LOCAL_NAME, Konstantz.PP_PREFIX);
        newData.getUnknownXMLObjects().add(modifiedObject);

        // Add the Select and the NewData
        modifyItem.setSelect(select);
        modifyItem.setNewData(newData);

        return modifyItem;

    }



    /**
     * This method attempts to update the portion of the Personal Profile indicated
     * by the xPathModifySelector with the provided modifiedObject
     * <p>
     * The following xPathModifySelectors MUST be supported by the PP:
     * <ul>
     * <li>/pp:PP </li>
     * <li>/pp:PP/pp:InformalName </li>
     * <li>/pp:PP/pp:LInformalName </li>
     * <li>/pp:PP/pp:CommonName </li>
     * <li>/pp:PP/pp:LegalIdentity </li>
     * <li>/pp:PP/pp:EmploymentIdentity </li>
     * <li>/pp:PP/pp:AddressCard </li>
     * <li>/pp:PP/pp:MsgContact </li>
     * <li>/pp:PP/pp:Facade </li>
     * <li>/pp:PP/pp:Demographics </li>
     * <li>/pp:PP/pp:SignKey </li>
     * <li>/pp:PP/pp:EncryptKey </li>
     * <li>/pp:PP/pp:EmergencyContact </li>
     * <li>/pp:PP/pp:LEmergencyContact </li>
     * </ul>
     * This set of slashed paths defines the minimal granularity of updates that MUST be supported. Updates to the containers 
     * listed above SHOULD be atomic when feasible. 
     * 
     * @param xPathModifySelector
     * @param modifiedObject
     * @return
     */
    public boolean invokeModify(String xPathModifySelector, XMLObject modifiedObject)
    {
        ModifyItem modifyItem = modifyItemForXPathAndObject(xPathModifySelector, modifiedObject);
        modifyItem.getModifyItemAttributes().setOverrideAllowed(true, modifyItem);

        List<ModifyItem> modifyItems = new ArrayList<ModifyItem>();
        modifyItems.add(modifyItem);
        ModifyResponse modifyResponse = invokeModifyForModifyItems(modifyItems);
        if(null!=modifyResponse)
        {
            Status status = modifyResponse.getStatus();
            if(status.isOK()) 
            {
                // ModifyResponse/Status@code==OK
                if(log.isDebugEnabled()) log.debug("Update of \""+xPathModifySelector+"\" succeeded");               
                return true;
            }
            else
            {
                // ModifyResponse/Status@code==Failed
            	if(log.isDebugEnabled()) log.debug("Update of \""+xPathModifySelector+"\" failed");
                return false;
            }
        }

        // failed to invoke the Modify
        if(log.isDebugEnabled()) log.debug("Personal Profile Service failed to be invoked for Modify");
        return false;
    }



    /**
     * This method invoked a Modify request to the Profile Service. PP v1.1 is based on DST1.1 but here we use DST2.1.  
     * <p>
     * A minimally compliant implementation is not required to support multiple Modification elements. The 
     * Modify operation functions as described in [LibertyDST]. The Modify operation has the additional 
     * relaxation that a minimally compliant ID-SIS-PP implementation MAY refuse a Modify request with multiple
     * Modification elements, provided all processing rules specified in [LibertyDST] are followed regarding 
     * failure to support multiple Modification elements.
     * <p>
     * Although the DST 2.1 reference implementation provides for an additional  
     * Query to be placed in ResultQuery elements inside of the Modify element,
     * this is not provided for in PP v1.1, and is therefore not implemented here.
     * 
     * @param modifyItems
     * @return
     */
    public ModifyResponse invokeModifyForModifyItems(List<ModifyItem> modifyItems)
    {        

        // TODO: honor whether or not the service provides for multiple ModifyItems.  Possibly send one at a time and 
        // compile the results together if the service does not.  This would be discovered with the discovery option
        // urn:liberty:dst:multipleModification advertised by DISCO 

        /**
         * Build the Modify
         */
        // Get a Builder for the DST Reference Items
        DefaultBuilder builder = new DefaultBuilder();

        // Build the Modify
        Modify modify = (Modify) builder.buildObject(Konstantz.PP_NS, Modify.LOCAL_NAME, Konstantz.PP_PREFIX);

        // Add the ModifyItems to the Modify
        modify.getModifyItems().addAll(modifyItems);

        /**
         * Now build the WSFMessage and add the Modify element to the body of the request
         */
        // Create a Message
        WSFMessage message = null;

        try
        {
            message = WSFMessage.createWSFMessage(this, SERVICE_URN+DST_MODIFY_SUFFIX);

            // Build the request Body
            Body body = (Body) new BodyBuilder().buildObject();  

            // Add the Query to the Body of the WSFMessage        
            body.getUnknownXMLObjects().add(modify);
            message.getRequestEnvelope().setBody(body);


            try
            {
                message.invoke();           
                Body responseBody = message.getResponseEnvelope().getBody();            
                for(XMLObject object : responseBody.getUnknownXMLObjects())
                {
                    if(object instanceof ModifyResponse)
                    {
                        return (ModifyResponse) object;
                    }
                }            
                if(log.isDebugEnabled()) log.debug("No ModifyResponse element found in the body of the response Envelope");

            } 
            catch (Exception e)
            {
            	if(log.isDebugEnabled()) log.error("Exception while attempting to invoke a WSFMessage", e);
            }
        }
        catch (XMLParserException e1)
        {
            e1.printStackTrace();
        }
        catch (UnmarshallingException e1)
        {
            e1.printStackTrace();
        }
        return null;
    }


    /**
     * Selection of /pp:PP/pp:LegalIdentity/pp:AltID by an exact match against the AltID's IDType.
     * 
     * @param iDType Type enumerated in IDType
     * @return
     */
    public AltID altIdForIDType(IDType.Type iDType)
    {
        return altIdForIDTypeURI(iDType.uri());
    }


    /**
     * Selection of /pp:PP/pp:LegalIdentity/pp:AltID by an exact match against the AltID's IDType element content.
     * This method allows for the user to specify IDTypes that are not enumerated in the IDType class
     * 
     * @param iDTypeURI
     * @return
     */
    public AltID altIdForIDTypeURI(String iDTypeURI)
    {
        QueryItem queryItem = queryItemForAltIdByIDType(iDTypeURI, null);

        QueryResponse queryResponse = invokeQueryForQueryItem(queryItem);
        if(null!=queryResponse)
        {
            // Check the Status, and if it is OK, continue processing
            Status status = queryResponse.getStatus();
            if(status.isFailed())
            {            
                List<Data> datas = queryResponse.getDatas();
                // since we made a single query there should be a single Data
                if(datas.size()>0)
                {
                    for(XMLObject object : datas.get(0).getUnknownXMLObjects())
                    {
                        if(object instanceof AltID) return (AltID) object;
                    }       
                    if(log.isDebugEnabled()) log.debug("No AltID element returned for the IDType query: "+iDTypeURI);
                }
                else
                {
                	if(log.isDebugEnabled()) log.debug("No Data Elements returned for the IDType query: "+iDTypeURI);
                }
            }
            else
            {
            	if(log.isDebugEnabled()) log.debug("Status: Failed for the IDType query: "+iDTypeURI);
            }
        }
        else
        {
        	if(log.isDebugEnabled()) log.debug("No QueryResponse returned. Service invocation failed.");
        }

        return null;
    }



    /**
     * Selection of a MsgContact by its id XML attribute.
     *  
     * @param id the id that will match a MsgContact
     * @return
     */
    public MsgContact msgContactForID(String id)
    {        
        QueryItem queryItem = queryItemForMsgContactID(id, null);        
        List<MsgContact> msgContacts = msgContactsForQueryItem(queryItem);
        if(msgContacts.size()>0) return msgContacts.get(0);
        else return null;
    }


    /**
     * Selection of MsgContact by an exact match on the contents of a leaf element for the following leaf elements: 
     * <ul>
     * <li>pp:MsgTechnology </li>
     * <li>pp:MsgMethod </li>
     * <li>pp:MsgType </li>
     * </ul>
     * MsgTechnology, MsgMethod, and MsgType can be tested in isolation or simultaneously combined with an AND operator.
     * 
     * @param technology an enumerated technology from the MsgTechnology Object
     * @param method an enumerated method from the MsgMethod Object
     * @param type an enumerated type from the MsgType Object
     * 
     * @return a list of MsgContact objects
     */
    public List<MsgContact> msgContactsForSpecifics(MsgTechnology.Technology technology, MsgMethod.Method method, MsgType.Type type)    
    {
        QueryItem queryItem = queryItemForMsgContactSpecifics(technology, method, type, null);
        return msgContactsForQueryItem(queryItem);
    }

    /**
     * Selection of MsgContact by an exact match on the contents of a leaf element for the following leaf elements: 
     * <ul>
     * <li>pp:MsgTechnology </li>
     * <li>pp:MsgMethod </li>
     * <li>pp:MsgType </li>
     * </ul>
     * MsgTechnology, MsgMethod, and MsgType can be tested in isolation or simultaneously combined with an AND operator.
     * 
     * @param technologyUri 
     * @param methodUri
     * @param typeUri
     * 
     * @return a list of MsgContact objects
     */
    public List<MsgContact> msgContactsForSpecifics(String technologyUri, String methodUri, String typeUri)    
    {        
        QueryItem queryItem = queryItemForMsgContactSpecifics(technologyUri, methodUri, typeUri, null);
        return msgContactsForQueryItem(queryItem);
    }


    /**
     * Selection of MsgContact by an exact match on the contents of a leaf element for the following leaf elements: 
     * <ol>
     * <li> pp:Nick </li>
     * <li> pp:LNick </li>
     * </ol>
     *  
     * @param nickname
     * @param localScriptVersion  if true indicates that the query should be for pp:LNick. false indicates pp:Nick
     * @return
     */
    public List<MsgContact> msgContactsForNickName(String nickname, boolean localScriptVersion)
    {
        QueryItem queryItem = queryItemForMsgContactNickName(nickname, localScriptVersion, null);
        return msgContactsForQueryItem(queryItem);
    }


    private List<MsgContact> msgContactsForQueryItem(QueryItem queryItem)
    {
        List<MsgContact> msgContacts = new LinkedList<MsgContact>();

        // TODO: Check that the AddrType specified is supported by this PP service.  Should be in the DISCO response

        QueryResponse queryResponse = invokeQueryForQueryItem(queryItem);
        if(null!=queryResponse)
        {
            // Check the Status, and if it is OK, continue processing
            Status status = queryResponse.getStatus();
            if(status.isOK())
            {            
                List<Data> datas = queryResponse.getDatas();
                // since we made a single query there should be a single Data
                if(datas.size()>0)
                {
                    Data data = datas.get(0);                
                    List<XMLObject> objects = data.getUnknownXMLObjects();
                    for(XMLObject object : objects)
                    {
                        if(object instanceof MsgContact) msgContacts.add((MsgContact) object);             
                    }
                    if(log.isDebugEnabled() && msgContacts.size()==0)
                    {
                        log.debug("No MsgContact elements returned for the query: "+queryItem.getSelect().getValue());
                    }
                }
                else
                {
                	if(log.isDebugEnabled()) log.debug("No Data Elements returned for the query: "+queryItem.getSelect().getValue());
                }
            }
            else
            {
            	if(log.isDebugEnabled()) log.debug("Status: Failed for the query: "+queryItem.getSelect().getValue());
            }
        }
        else
        {
        	if(log.isDebugEnabled()) log.debug("No QueryResponse returned. Service invocation failed.");
        }

        return msgContacts;
    }



    /**
     * Selection by the id XML attribute of AddressCard. 
     *  
     * @param ID
     * @return
     */
    public AddressCard addressCardForID(String id)
    {

        QueryItem queryItem = queryItemForAddressCardID(id, null);

        QueryResponse queryResponse = invokeQueryForQueryItem(queryItem);
        if(null!=queryResponse)
        {
            // Check the Status, and if it is OK, continue processing
            Status status = queryResponse.getStatus();
            if(status.isOK())
            {            
                List<Data> datas = queryResponse.getDatas();
                // since we made a single query there should be a single Data
                if(datas.size()>0)
                {
                    for(XMLObject object : datas.get(0).getUnknownXMLObjects())
                    {
                        if(object instanceof AddressCard) return (AddressCard) object;
                    }       
                    if(log.isDebugEnabled()) log.debug("No AddressCard element returned for the id: "+id);
                }
                else
                {
                	if(log.isDebugEnabled()) log.debug("No Data Elements returned for the AddressCard id query: "+id);
                }
            }
            else
            {
            	if(log.isDebugEnabled()) log.debug("Status: Failed for the AddressCard id query: "+id);
            }
        }
        else
        {
        	if(log.isDebugEnabled()) log.debug("No QueryResponse returned. Service invocation failed.");
        }

        return null;
    }



    /**
     *  Selection of AddressCard by an exact match on the contents of a leaf element for the following leaf elements: 
     *  <ol>
     *  <li> pp:AddrType </li>
     *  </ol>
     *  <p>
     *  Example:
     *  <pre>
     *  &lt;QueryItem itemID="home"&gt; 
     *      &lt;Select&gt; 
     *          /pp:PP/pp:AddressCard
     *              [pp:AddressType="urn:liberty:id-sis-pp:addrType:home"] 
     *      &lt;/Select&gt; 
     *  &lt;/QueryItem&gt; 
     *  </pre>
     *  Only one of the tests needs to be supported in any one slashed path. Such bracketed expression may also appear 
     *  within (i.e., in the middle of) a slashed path. 
     *  <p>
     * @param addressType
     * @return
     */
    public List<AddressCard> addressCardsForAddressType(AddrType.Type addressType)
    {
        return addressCardsForQueryString(addressType.queryString());
    }


    /**
     * Selection of AddressCard by an exact match on the contents of a leaf element for the following leaf elements: 
     * <ol>
     * <li> pp:Nick </li>
     * <li> pp:LNick </li>
     * </ol>
     *  
     * @param nickname
     * @param localScriptVersion
     * @return
     */
    public List<AddressCard> addressCardsForNickName(String nickname, boolean localScriptVersion)
    {
        StringBuffer queryBuffer = new StringBuffer("/pp:PP/pp:AddressCard [pp:");
        queryBuffer.append( localScriptVersion ? "LNick=\"" : "Nick=\"" );
        queryBuffer.append(StringEscapeUtils.escapeXml(nickname)).append("\"]");
        return addressCardsForQueryString(queryBuffer.toString());
    }


    private List<AddressCard> addressCardsForQueryString(String queryString)
    {        
        List<AddressCard> addressCards = new LinkedList<AddressCard>();

        // TODO: Check that the AddrType specified is supported by this PP service.  Should be in the DISCO response

        QueryResponse queryResponse = invokeQuery(queryString);
        if(null!=queryResponse)
        {
            // Check the Status, and if it is OK, continue processing
            Status status = queryResponse.getStatus();
            if(status.isOK())
            {            
                List<Data> datas = queryResponse.getDatas();
                // since we made a single query there should be a single Data
                if(datas.size()>0)
                {
                    Data data = datas.get(0);                
                    List<XMLObject> objects = data.getUnknownXMLObjects();
                    for(XMLObject object : objects)
                    {
                        if(object instanceof AddressCard) addressCards.add((AddressCard) object);             
                    }
                    if(log.isDebugEnabled() && addressCards.size()==0)
                    {
                        log.debug("No AddressCard elements returned for the query: "+queryString);
                    }
                }
                else
                {
                	if(log.isDebugEnabled()) log.debug("No Data Elements returned for the query: "+queryString);
                }
            }
            else
            {
            	if(log.isDebugEnabled()) log.debug("Status: Failed for the query: "+queryString);
            }
        }
        else
        {
        	if(log.isDebugEnabled()) log.debug("No QueryResponse returned. Service invocation failed.");
        }

        return addressCards;
    }




    /**
     * Takes an id (unique) and builds an MsgContact QueryItem
     * 
     * @param id the id that will match a MsgContact
     * @param itemID an id that the Data response will reference to indicate correlation with the QueryItem
     * @return
     */
    public QueryItem queryItemForMsgContactID(String id, String itemID)
    {
        StringBuffer queryBuffer = new StringBuffer("/pp:PP/pp:MsgContact [@id=\"");
        queryBuffer.append(id).append("\"]");

        return queryItemForString(queryBuffer.toString(), itemID);
    }

    /**
     * Creates a QueryItem that is for a MsgContact based on a Nickname. Supports Nickname and 
     * nickname in local script
     *  
     * @param nickname
     * @param localScriptVersion if true indicates that the query should be for pp:LNick.  false indicates pp:Nick
     * @param itemID  an id that the Data response will reference to indicate correlation with the QueryItem
     * @return
     */
    public QueryItem queryItemForMsgContactNickName(String nickname, boolean localScriptVersion, String itemID)
    {
        StringBuffer queryBuffer = new StringBuffer("/pp:PP/pp:MsgContact [pp:");
        queryBuffer.append( localScriptVersion ? "LNick=\"" : "Nick=\"" );
        queryBuffer.append(StringEscapeUtils.escapeXml(nickname)).append("\"]");

        return  queryItemForString(queryBuffer.toString(), itemID);
    }


    /**
     * Generation of a QueryItem to query for MsgContact by an exact match on the contents of a leaf element for the following leaf elements: 
     * <ul>
     * <li>pp:MsgTechnology </li>
     * <li>pp:MsgMethod </li>
     * <li>pp:MsgType </li>
     * </ul>
     * MsgTechnology, MsgMethod, and MsgType can be tested in isolation or simultaneously combined with an AND operator.
     * 
     * @param technologyUri 
     * @param methodUri
     * @param typeUri
     * @param itemID an id that the Data response will reference to indicate correlation with the QueryItem
     * 
     * @return a list of MsgContact objects
     */
    public QueryItem queryItemForMsgContactSpecifics(String technologyUri, String methodUri, String typeUri, String itemID)    
    {
        StringBuffer queryBuffer = new StringBuffer("");

        if(null!=technologyUri)
        {
            queryBuffer.append("pp:MsgTechnology=\"").append(StringEscapeUtils.escapeXml(technologyUri)).append("\"");
        }

        if(null!=methodUri)
        {
            if(queryBuffer.length()>0) queryBuffer.append(" AND "); 
            queryBuffer.append("pp:MsgMethod=\"").append(StringEscapeUtils.escapeXml(methodUri)).append("\"");
        }

        if(null!=typeUri)
        {
            if(queryBuffer.length()>0) queryBuffer.append(" AND "); 
            queryBuffer.append("pp:MsgType=\"").append(StringEscapeUtils.escapeXml(typeUri)).append("\"");
        }


        if(queryBuffer.length()==0) 
        {
            return null;
        }
        else
        {
            queryBuffer.insert(0, "/pp:PP/pp:MsgContact [").append("]");
            return queryItemForString(queryBuffer.toString(), itemID);
        }
    }


    /**
     * Generation of a QueryItem to query for MsgContact by an exact match on the contents of a leaf element for the following leaf elements: 
     * <ul>
     * <li>pp:MsgTechnology </li>
     * <li>pp:MsgMethod </li>
     * <li>pp:MsgType </li>
     * </ul>
     * MsgTechnology, MsgMethod, and MsgType can be tested in isolation or simultaneously combined with an AND operator.
     * 
     * @param technology an enumerated technology from the MsgTechnology Object
     * @param method an enumerated method from the MsgMethod Object
     * @param type an enumerated type from the MsgType Object
     * @param itemID an id that the Data response will reference to indicate correlation with the QueryItem
     * 
     * @return a list of MsgContact objects
     */
    public QueryItem queryItemForMsgContactSpecifics(MsgTechnology.Technology technology, MsgMethod.Method method, MsgType.Type type, String itemID)    
    {
        String technologyUri=null, methodUri=null, typeUri=null;

        if(null!=technology) 
        {
            technologyUri = technology.uri();
        }

        if(null!=method) 
        {
            methodUri = method.uri();
        }

        if(null!=type) 
        {
            typeUri = type.uri();
        }

        return queryItemForMsgContactSpecifics(technologyUri, methodUri, typeUri, itemID);
    }



    /**
     * Takes an id (unique) and builds an AddressCard QueryItem
     * 
     * @param id the id that will match an AddressCard
     * @param itemID an id that the Data response will reference to indicate correlation with the QueryItem
     * @return
     */
    public QueryItem queryItemForAddressCardID(String id, String itemID)
    {
        StringBuffer queryBuffer = new StringBuffer("/pp:PP/pp:AddressCard [@id=\"");
        queryBuffer.append(id).append("\"]");

        return queryItemForString(queryBuffer.toString(), itemID);
    }


    /**
     * Returns a QueryItem for the specified XPath String, if the string compiles.  itemID
     * is optional, but is used to connect the QueryItem with the resultant Data element's
     * itemIDRef.
     * <p>
     * NOTE: Should this be used to prepare every QueryItem ?
     * 
     * @param xpathQueryString
     * @param itemID the id of the QueryItem so that the Data element may be matched by itemIDRef
     * @return
     */
    public QueryItem queryItemForXPathQuery(String xpathQueryString, String itemID)
    {
        XPathFactory factory = XPathFactory.newInstance();
        XPath xpath = factory.newXPath();
        try
        {
            // make sure this is a valid XPath Query
            xpath.compile(xpathQueryString);

            // create the QueryItem
            return queryItemForString(xpathQueryString, itemID);
        }
        catch (XPathExpressionException e)
        {
            log.error(e.toString(), e);
        }

        return null;
    }

    /**
     * Returns a QueryItem for an AltID based on that AltIDs IDType element content
     * 
     * @param iDType
     * @param itemID the id of the QueryItem so that the Data element may be matched by itemIDRef
     * @return
     */
    public QueryItem queryItemForAltIdByIDType(String iDType, String itemID)
    {
        StringBuffer queryBuffer = new StringBuffer("/pp:PP/pp:LegalIdentity/pp:AltID [pp:IDType=\"");
        queryBuffer.append(iDType).append("\"]");
        return queryItemForString(queryBuffer.toString(), itemID);
    }


    /**
     * Accepts any string and places it inside of the select
     * 
     * @param properlyFormedQueryString
     * @return
     */
    public QueryItem queryItemForString(String properlyFormedQueryString, String itemID)
    {
        // Get a Builder for the DST Reference Items
        DefaultBuilder builder = new DefaultBuilder();

        // Build the QueryItem
        QueryItem queryItem = (QueryItem) builder.buildObject(Konstantz.PP_NS, QueryItem.LOCAL_NAME, Konstantz.PP_PREFIX);

        // set the itemID attribute of the QueryItem if it is specified
        if(null!=itemID) queryItem.setItemID(itemID);

        // Build the Select element
        Select select = (Select) builder.buildObject(Konstantz.PP_NS, Select.LOCAL_NAME, Konstantz.PP_PREFIX);        
        select.setValue(properlyFormedQueryString);

        // Add the Select element to the QueryItem
        queryItem.setSelect(select);

        return queryItem;
    }




    /**
     * slash-separated path to any depth. The path is always anchored at the document root and may not contain wild 
     * cards or empty nodes. ID-SIS-PP may be extended; the current complete set of all possible slashed paths is as 
     * follows: 
     * <p>
     * /pp:PP<br />
     * /pp:PP/pp:InformalName<br />
     * /pp:PP/pp:LInformalName<br />
     * /pp:PP/pp:CommonName<br />
     * /pp:PP/pp:CommonName/pp:CN<br />
     * /pp:PP/pp:CommonName/pp:LCN<br />
     * /pp:PP/pp:CommonName/pp:AltCN<br />
     * /pp:PP/pp:CommonName/pp:LAltCN<br />
     * /pp:PP/pp:CommonName/pp:AnalyzedName<br />
     * /pp:PP/pp:CommonName/pp:AnalyzedName/pp:PersonalTitle<br />
     * /pp:PP/pp:CommonName/pp:AnalyzedName/pp:LPersonalTitle<br />
     * /pp:PP/pp:CommonName/pp:AnalyzedName/pp:FN<br />
     * /pp:PP/pp:CommonName/pp:AnalyzedName/pp:LFN<br />
     * /pp:PP/pp:CommonName/pp:AnalyzedName/pp:SN<br />
     * /pp:PP/pp:CommonName/pp:AnalyzedName/pp:LSN<br />
     * /pp:PP/pp:CommonName/pp:AnalyzedName/pp:MN<br />
     * /pp:PP/pp:CommonName/pp:AnalyzedName/pp:LMN<br />
     * /pp:PP/pp:LegalIdentity<br />
     * /pp:PP/pp:LegalIdentity/pp:LegalName<br />
     * /pp:PP/pp:LegalIdentity/pp:LLegalName<br />
     * /pp:PP/pp:LegalIdentity/pp:AnalyzedName<br />
     * /pp:PP/pp:LegalIdentity/pp:AnalyzedName/pp:PersonalTitle<br />
     * /pp:PP/pp:LegalIdentity/pp:AnalyzedName/pp:LPersonalTitle<br />
     * /pp:PP/pp:LegalIdentity/pp:AnalyzedName/pp:FN<br />
     * /pp:PP/pp:LegalIdentity/pp:AnalyzedName/pp:LFN<br />
     * /pp:PP/pp:LegalIdentity/pp:AnalyzedName/pp:SN<br />
     * /pp:PP/pp:LegalIdentity/pp:AnalyzedName/pp:LSN<br />
     * /pp:PP/pp:LegalIdentity/pp:AnalyzedName/pp:MN<br />
     * /pp:PP/pp:LegalIdentity/pp:AnalyzedName/pp:LMN<br />
     * /pp:PP/pp:LegalIdentity/pp:VAT<br />
     * /pp:PP/pp:LegalIdentity/pp:VAT/pp:IDValue<br />
     * /pp:PP/pp:LegalIdentity/pp:VAT/pp:IDType<br />
     * /pp:PP/pp:LegalIdentity/pp:AltID<br />
     * /pp:PP/pp:LegalIdentity/pp:AltID/pp:IDValue<br />
     * /pp:PP/pp:LegalIdentity/pp:AltID/pp:IDType<br />
     * /pp:PP/pp:LegalIdentity/pp:DOB<br />
     * /pp:PP/pp:LegalIdentity/pp:Gender<br />
     * /pp:PP/pp:LegalIdentity/pp:MaritalStatus<br />
     * /pp:PP/pp:EmploymentIdentity<br />
     * /pp:PP/pp:EmploymentIdentity/pp:JobTitle<br />
     * /pp:PP/pp:EmploymentIdentity/pp:LJobTitle<br />
     * /pp:PP/pp:EmploymentIdentity/pp:O<br />
     * /pp:PP/pp:EmploymentIdentity/pp:LO<br />
     * /pp:PP/pp:EmploymentIdentity/pp:AltO<br />
     * /pp:PP/pp:EmploymentIdentity/pp:AltLO<br />
     * /pp:PP/pp:AddressCard<br />
     * /pp:PP/pp:AddressCard/pp:AddrType<br />
     * /pp:PP/pp:AddressCard/pp:Address<br />
     * /pp:PP/pp:AddressCard/pp:Address/pp:PostalAddress<br />
     * /pp:PP/pp:AddressCard/pp:Address/pp:LPostalAddress<br />
     * /pp:PP/pp:AddressCard/pp:Address/pp:PostalCode<br />
     * /pp:PP/pp:AddressCard/pp:Address/pp:L<br />
     * /pp:PP/pp:AddressCard/pp:Address/pp:LL<br />
     * /pp:PP/pp:AddressCard/pp:Address/pp:St<br />
     * /pp:PP/pp:AddressCard/pp:Address/pp:LSt<br />
     * /pp:PP/pp:AddressCard/pp:Address/pp:C<br />
     * /pp:PP/pp:AddressCard/pp:Nick<br />
     * /pp:PP/pp:AddressCard/pp:LNick<br />
     * /pp:PP/pp:AddressCard/pp:LComment<br />
     * /pp:PP/pp:MsgContact<br />
     * /pp:PP/pp:MsgContact/pp:Nick<br />
     * /pp:PP/pp:MsgContact/pp:LNick<br />
     * /pp:PP/pp:MsgContact/pp:LComment<br />
     * /pp:PP/pp:MsgContact/pp:MsgType<br />
     * /pp:PP/pp:MsgContact/pp:MsgMethod<br />
     * /pp:PP/pp:MsgContact/pp:MsgTechnology<br />
     * /pp:PP/pp:MsgContact/pp:MsgProvider<br />
     * /pp:PP/pp:MsgContact/pp:MsgAccount<br />
     * /pp:PP/pp:MsgContact/pp:MsgSubaccount<br />
     * /pp:PP/pp:Facade<br />
     * /pp:PP/pp:Facade/pp:MugShot<br />
     * /pp:PP/pp:Facade/pp:WebSite<br />
     * /pp:PP/pp:Facade/pp:NamePronounced<br />
     * /pp:PP/pp:Facade/pp:GreetSound<br />
     * /pp:PP/pp:Facade/pp:GreetMeSound<br />
     * /pp:PP/pp:Demographics<br />
     * /pp:PP/pp:Demographics/pp:DisplayLanguage<br />
     * /pp:PP/pp:Demographics/pp:Language<br />
     * /pp:PP/pp:Demographics/pp:Birthday<br />
     * /pp:PP/pp:Demographics/pp:Age<br />
     * /pp:PP/pp:Demographics/pp:TimeZone<br />
     * /pp:PP/pp:SignKey<br />
     * /pp:PP/pp:EncryptKey<br />
     * /pp:PP/pp:EmergencyContact<br />
     * /pp:PP/pp:LEmergencyContact<br />
     * 
     * @param xpathQueryString
     * @return
     */
    public List<XMLObject> profileObjectsForXPathQuery(String xpathQueryString)
    {
        List<XMLObject> results = new LinkedList<XMLObject>();

        QueryItem queryItem = queryItemForXPathQuery(xpathQueryString, null);        
        if(null!=queryItem)
        {
            QueryResponse queryResponse = invokeQueryForQueryItem(queryItem);
            if(null!=queryResponse)
            {
                log.debug("QueryResponse Found");

                // Check the Status, and if it is OK, continue processing
                Status status = queryResponse.getStatus();
                if(status.isOK())
                {            
                    List<Data> datas = queryResponse.getDatas();
                    // since we made a single query there should be a single Data
                    if(datas.size()>0)
                    {
                        Data data = datas.get(0);                

                        results.addAll(data.getUnknownXMLObjects());

                        if(log.isDebugEnabled() && results.size()==0)
                        {
                            log.debug("No elements returned inside of Data for the query: "+xpathQueryString);
                        }
                    }
                    else
                    {
                    	if(log.isDebugEnabled()) log.debug("No Data Elements returned for the query: "+xpathQueryString);
                    }
                }
                else
                {
                	if(log.isDebugEnabled()) log.debug("Status: Failed for the query: "+xpathQueryString);
                }
            }
            else
            {
            	if(log.isDebugEnabled()) log.debug("No QueryResponse returned. Service invocation failed.");
            }
        }
        else
        {
        	if(log.isDebugEnabled()) log.debug("A QueryItem could not be created from the specified query: "+xpathQueryString);
        }

        return results;
    }


    /**
     * 
     * 
     * TODO: Add ApplicationEPR SOAP Header to the request
     *
     *<pre>
     *&lt;Query&gt;
     *  &lt;QueryItem itemID="djkfgjkdf"&gt;
     *      &lt;Select&gt;/hp:HP/hp:AddressCard&lt;/Select &gt;
     *  &lt;/QueryItem&gt;
     *  &lt;Subscription includeData="Yes" subscriptionID="tr578k-kydg4b" notifyToRef="#123"&gt;
     *      &lt;RefItem itemIDRef="djkfgjkdf"/&gt;
     *  &lt;/Subscription&gt;
     * &lt;/Query&gt;
     *</pre>
     *
     *
     * @param addressCard
     * @return
     *
    public String subscribeToAddressCard(AddressCard addressCard, URI notificationURI)
    {

        String notificationTo = null;
        try
        {
            notificationTo = notificationURI.toURL().toString();
        }
        catch (MalformedURLException e)
        {
            e.printStackTrace();
            return null;
        }
        
        // Create the subscription Query
        String queryItemID = "query-"+UUID.randomUUID().toString();        
        QueryItem queryItem = queryItemForAddressCardID(addressCard.attributes().getId(), queryItemID);   
        queryItem.getItemID();

        // Create the RefItem that will reference the subscription
        RefItem refItem = new RefItem();
        refItem.setItemRefID(queryItemID);
        
        // Build the ApplicationEPR Header
        ApplicationEPR epr = new ApplicationEPR();        
        String applicationEPRId = UUID.randomUUID().toString();
        epr.setId(applicationEPRId);
        // epr.getUnknownAttributes().put(XMLHelper.constructQName(Konstantz.WSU_NS, "Id", Konstantz.WSU_PREFIX), applicationEPRId);
        Address address = new Address();
        address.setValue(notificationTo);
        epr.setAddress(address);
        
        // Create the subscription
        Subscription subscription = new Subscription();
        // subscription expires one year from now
        subscription.setExpires(((new DateTime()).plusDays(365)));
        subscription.setId(UUID.randomUUID().toString());
        subscription.setSubscriptionID(UUID.randomUUID().toString());
        subscription.setNotifyToRef( applicationEPRId );
        
        // Add the RefItem to the subscription
        subscription.getRefItems().add(refItem);
        

        
        // build the request
        
        return subscription.getSubscriptionID();
    }
    */

    
    
    /**
     * Using the DST 2.1 Reference implementation classes.
     * <p>
     * (From DST v2.1 specification document 2317 - 2321) When SOAP action names are need, they SHOULD be formed by appending to service type one of the Request names, 
     * i.e., Create, Delete, Query, Modify, etc. 
     * For Example:
     * <pre>
     * urn:liberty:id-sis-dap:2005-10:dst-2.1:Query 
     * </pre>
     * <p>
     * In the case of the Personal Profile Service the Query Action would look like this:
     * <pre>
     * urn:liberty:id-sis-pp:2005-05:dst-2.1:Query 
     * </pre>
     * 
     * 
     * @param queryItemSelect an XPath query string, not enforced
     */
    public QueryResponse invokeQuery(String queryItemSelect)
    {
    	if(log.isDebugEnabled()) log.debug(queryItemSelect);

        // Get a Builder for the DST Reference Items
        DefaultBuilder builder = new DefaultBuilder();

        // Build the QueryItem
        QueryItem queryItem = (QueryItem) builder.buildObject(Konstantz.PP_NS, QueryItem.LOCAL_NAME, Konstantz.PP_PREFIX);

        // Build the Select element
        Select select = (Select) builder.buildObject(Konstantz.PP_NS, Select.LOCAL_NAME, Konstantz.PP_PREFIX);        
        select.setValue(queryItemSelect);

        // Add the Select element to the QueryItem
        queryItem.setSelect(select);

        return invokeQueryForQueryItem(queryItem);        
    }

    /**
     * Invokes a PP Query for a single QueryItem
     * 
     * @param queryItem
     * @return
     */
    public QueryResponse invokeQueryForQueryItem(QueryItem queryItem)
    {  
        List<QueryItem> queryItems = new ArrayList<QueryItem>();
        queryItems.add(queryItem);
        
        return invokeQueryForQueryItems(queryItems); 
    }

    /**
     * This method provides support for multiple QueryItem elements
     * <p>
     * As specified in [LibertyDST], a minimally compliant ID-SIS-PP implementation MUST support multiple QueryItem elements. 
     * 
     * @param queryItems
     * @return
     */
    public QueryResponse invokeQueryForQueryItems(List<QueryItem> queryItems)
    {        

        /**
         * Build the Query
         */
        // Get a Builder for the DST Reference Items
        DefaultBuilder builder = new DefaultBuilder();

        // Build the Query
        Query query = (Query) builder.buildObject(Konstantz.PP_NS, Query.LOCAL_NAME, Konstantz.PP_PREFIX);

        // Add the QueryItem to the Query
        query.getQueryItems().addAll(queryItems);
        
        /**
         * Now build the WSFMessage and add the query to the body
         */
        // Create a Message
        WSFMessage queryMessage = null;

        try
        {
            queryMessage = WSFMessage.createWSFMessage(this, SERVICE_URN+DST_QUERY_SUFFIX);            

            // ADDING AN APP_EPR FOR USE WITH SUBSCRIPTION
//            if(null!=applicationEPR)
//            {
//                queryMessage.addWSUIdAttribute(applicationEPR, "appEprHdr");
//                ((HeaderIDWSF)queryMessage.getRequestEnvelope().getHeader()).setApplicationEPR(applicationEPR);
//            }
            
            // Build the request Body
            Body body = (Body) new BodyBuilder().buildObject();  

            // Add the Query to the Body of the WSFMessage        
            body.getUnknownXMLObjects().add(query);
            queryMessage.getRequestEnvelope().setBody(body);

            try
            {
                queryMessage.invoke();           
                Body responseBody = queryMessage.getResponseEnvelope().getBody();            
                for(XMLObject object : responseBody.getUnknownXMLObjects())
                {
                    if(object instanceof QueryResponse)
                    {
                        return (QueryResponse) object;
                    }
                }      
                if(log.isDebugEnabled()) log.debug("No QueryResponse element found in the body of the response Envelope");

            } 
            catch (Exception e)
            {
            	if(log.isDebugEnabled()) log.error("Exception while attempting to invoke a WSFMessage", e);
            }
        }
        catch (XMLParserException e1)
        {
            e1.printStackTrace();
        }
        catch (UnmarshallingException e1)
        {
            e1.printStackTrace();
        }


        return null;
    }


    /**
     * This enumeration models Table 3. Data Availability Discovery Option Keywords from section
     * 2.1.1 of the liberty-idsis-pp-v1.1 specification document.
     * <p>
     * The keywords that express data availability extract selected components from the profile as if an XPATH expression 
     * were applied. An implementation is not required to use XPATH if the results are equivalent. Presence of the keyword 
     * implies that the corresponding data can be obtained, if queried. However, the data may not be available due to 
     * permissions or race conditions between data removal and updates to the discovery service.  
     * 
     * @author asa
     *
     */
    public enum DataQueryOption
    {

        /**
         * Has some ID-SIS-PP data
         */
        PP("urn:liberty:id-sis-pp", new String[]{"/pp:PP"}, "Has some ID-SIS-PP data"),
        /**
         * Has some address card data corresponding to the domicile
         */
        DOMOCILE("urn:liberty:id-sis-pp:domicile", new String[]{"/pp:PP/pp:AddressCard [pp:AddrType=\"urn:liberty:id-sis-pp:addrType:domicile\"]"}, "Has some address card data corresponding to the domicile"),
        /**
         * Has some address card data corresponding to the home address
         */
        HOME("urn:liberty:id-sis-pp:home", new String[]{"/pp:PP/pp:AddressCard [pp:AddrType=\"urn:liberty:id-sis-pp:addrType:home\"]"}, "Has some address card data corresponding to the home address"),
        /**
         * Has some address card or messaging contact data corresponding to the office address
         */
        WORK("urn:liberty:id-sis-pp:work", new String[]{"/pp:PP/pp:AddressCard [pp:AddrType=\"urn:liberty:id-sis-pp:addrType:work\"]","/pp:PP/pp:MsgContact [pp:MsgType=\"urn:liberty:id-sis-pp:msgType:work\"]"}, "Has some address card or messaging contact data corresponding to the office address "),
        /**
         * Has some messaging contact data corresponding to personal contact
         */
        PERSONAL("urn:liberty:id-sis-pp:personal", new String[]{"/pp:PP/pp:MsgContact [pp:MsgType=\"urn:liberty:id-sis-pp:msgType:personal\"]"}, "Has some messaging contact data corresponding to personal contact "),
        /**
         * Has some messaging contact data for mobile contact
         */
        MOBILE("urn:liberty:id-sis-pp:mobile", new String[]{"/pp:PP/pp:MsgContact [pp:MsgType=\"urn:liberty:id-sis-pp:msgType:mobile\"]"}, "Has some messaging contact data for mobile contact "),
        /**
         * Has some messaging contact or address data for vacation contact
         */
        VACATION("urn:liberty:id-sis-pp:vacation", new String[]{"/pp:PP/pp:MsgContact [pp:MsgType=\"urn:liberty:id-sis-pp:msgType:vacation\"]", "/pp:PP/pp:AddressCard [pp:AddrType=\"urn:liberty:id-sis-pp:addrType:vacation\"]"}, "Has some messaging contact or address data for vacation contact "),
        /**
         * Has some address card data
         */
        ADDRESS("urn:liberty:id-sis-pp:address", new String[]{"/pp:PP/pp:AddressCard"}, "Has some address card data "),
        /**
         * Has some common name data
         */
        COMMON_NAME("urn:liberty:id-sis-pp:cn", new String[]{"/pp:PP/pp:CommonName"}, "Has some common name data "),
        /**
         * Has informal name
         */
        INFORMAL_NAME("urn:liberty:id-sis-pp:informalName", new String[]{"/pp:PP/pp:InformalName", "/pp:PP/pp:LInformalName"}, "Has informal name "),
        /**
         * Has some legal identity data
         */
        LEGAL_IDENTITY("urn:liberty:id-sis-pp:legal", new String[]{"/pp:PP/pp:LegalIdentity"}, "Has some legal identity data "),
        /**
         * Has some employment identity data
         */
        EMPLOYMENT("urn:liberty:id-sis-pp:employment", new String[]{"/pp:PP/pp:EmploymentIdentity"}, "Has some employment identity data "),
        /**
         * Has some facade data
         */
        FACADE("urn:liberty:id-sis-pp:facade", new String[]{"/pp:PP/pp:Facade"}, "Has some facade data "),
        /**
         * Has either or both keys
         */
        KEYS("urn:liberty:id-sis-pp:keys", new String[]{"/pp:PP/pp:SignKey", "/pp:PP/pp:EncryptKey"}, "Has either or both keys "),
        /**
         * Has some demographics data
         */
        DEMOGRAPHICS("urn:liberty:id-sis-pp:demographics", new String[]{"/pp:PP/pp:Demographics"}, "Has some demographics data "),
        /**
         * Has some emergency contact data
         */
        EMERGENCY("urn:liberty:id-sis-pp:emergency", new String[]{"/pp:PP/pp:EmergencyContact", "/pp:PP/pp:AddressCard [pp:AddrType=\"urn:liberty:id-sis-pp:addrType:emergency\"]"}, "Has some emergency contact data ");

        private String keyword;
        private String[] equivalentXPaths;
        private String meaning;

        DataQueryOption(String keyword, String[] equivalentXPaths, String meaning)
        {
            this.keyword = keyword;
            this.equivalentXPaths = equivalentXPaths;
            this.meaning = meaning;   
        }

        /**
         * The keyword uri for this DataQueryOption.
         * 
         * @return
         */
        public String keyword() { return keyword; }
        
        /**
         * XPath equivalents that may be used for creating queries
         * 
         * @return
         */
        public String[] equivalentXPaths() { return equivalentXPaths; }
        
        /**
         * The english meaning of this DataQueryOption
         * 
         * @return
         */
        public String meaning() { return meaning; }

        /**
         * returns a DataQueryOption matching the supplied keyword
         * 
         * @param keyword
         * @return a DataQueryOption or null if there is no match
         */
        public DataQueryOption dataAvailabilityOptionForKeyword(String keyword)
        {
            if(null==keyword || keyword.trim().length()==0 || 0!=keyword.indexOf("urn:liberty:id-sis-pp")) return null;
            else
            {
                for(DataQueryOption c : DataQueryOption.values())
                {
                    if(c.keyword.equals(keyword)) return c;
                }
            }
            return null;
        }

    }



    /**
     * This enumeration models Table 4. Data Update Discovery Option Keywords from section
     * 2.1.2 of the liberty-idsis-pp-v1.1 specification document.
     * <p>
     * The data update discovery option keywords express the willingness and ability of the AP to store some data 
     * corresponding to the given XPATH expression. These keywords do not imply that the AP currently has any data 
     * regarding the containers referenced by the keyword. 
     * 
     * @author asa
     *
     */
    public enum DataModifyOption
    {
        /**
         * Can store some ID-SIS-PP data
         */
        PP("urn:liberty:id-sis-pp:can", new String[]{"/pp:PP"}, "Can store some ID-SIS-PP data"),
        /**
         * Can store some address card data corresponding to the domicile
         */
        DOMOCILE("urn:liberty:id-sis-pp:can:domicile", new String[]{"/pp:PP/pp:AddressCard [pp:AddrType=\"urn:liberty:id-sis-pp:addrType:domicile\"]"}, "Can store some address card data corresponding to the domicile"),
        /**
         * Can store some address card data corresponding to the home address
         */
        HOME("urn:liberty:id-sis-pp:can:home", new String[]{"/pp:PP/pp:AddressCard [pp:AddrType=\"urn:liberty:id-sis-pp:addrType:home\"]"}, "Can store some address card data corresponding to the home address"),
        /**
         * Can store some address card or messaging contact data corresponding to the office address
         */
        WORK("urn:liberty:id-sis-pp:can:work", new String[]{"/pp:PP/pp:AddressCard [pp:AddrType=\"urn:liberty:id-sis-pp:addrType:work\"]","/pp:PP/pp:MsgContact [pp:MsgType=\"urn:liberty:id-sis-pp:msgType:work\"]"}, "Can store some address card or messaging contact data corresponding to the office address "),
        /**
         * Can store some messaging contact data corresponding to personal contact
         */
        PERSONAL("urn:liberty:id-sis-pp:can:personal", new String[]{"/pp:PP/pp:MsgContact [pp:MsgType=\"urn:liberty:id-sis-pp:msgType:personal\"]"}, "Can store some messaging contact data corresponding to personal contact "),
        /**
         * Can store some messaging contact data for mobile contact
         */
        MOBILE("urn:liberty:id-sis-pp:can:mobile", new String[]{"/pp:PP/pp:MsgContact [pp:MsgType=\"urn:liberty:id-sis-pp:msgType:mobile\"]"}, "Can store some messaging contact data for mobile contact "),
        /**
         * Can store some messaging contact or address data for vacation contact
         */
        VACATION("urn:liberty:id-sis-pp:can:vacation", new String[]{"/pp:PP/pp:MsgContact [pp:MsgType=\"urn:liberty:id-sis-pp:msgType:vacation\"]", "/pp:PP/pp:AddressCard [pp:AddrType=\"urn:liberty:id-sis-pp:addrType:vacation\"]"}, "Can store some messaging contact or address data for vacation contact "),
        /**
         * Can store some address card data
         */
        ADDRESS("urn:liberty:id-sis-pp:can:address", new String[]{"/pp:PP/pp:AddressCard"}, "Can store some address card data "),
        /**
         * Can store some common name data
         */
        COMMON_NAME("urn:liberty:id-sis-pp:can:cn", new String[]{"/pp:PP/pp:CommonName"}, "Can store some common name data "),
        /**
         * Can store informal name
         */
        INFORMAL_NAME("urn:liberty:id-sis-pp:can:informalName", new String[]{"/pp:PP/pp:InformalName", "/pp:PP/pp:LInformalName"}, "Can store informal name "),
        /**
         * Can store some legal identity data
         */
        LEGAL_IDENTITY("urn:liberty:id-sis-pp:can:legal", new String[]{"/pp:PP/pp:LegalIdentity"}, "Can store some legal identity data "),
        /**
         * Can store some employment identity data
         */
        EMPLOYMENT("urn:liberty:id-sis-pp:can:employment", new String[]{"/pp:PP/pp:EmploymentIdentity"}, "Can store some employment identity data "),
        /**
         * Can store some facade data
         */
        FACADE("urn:liberty:id-sis-pp:can:facade", new String[]{"/pp:PP/pp:Facade"}, "Can store some facade data "),
        /**
         * Can store either or both keys
         */
        KEYS("urn:liberty:id-sis-pp:can:keys", new String[]{"/pp:PP/pp:SignKey", "/pp:PP/pp:EncryptKey"}, "Can store either or both keys "),
        /**
         * Can store some demographics data
         */
        DEMOGRAPHICS("urn:liberty:id-sis-pp:can:demographics", new String[]{"/pp:PP/pp:Demographics"}, "Can store some demographics data "),
        /**
         * Can store some emergency contact data
         */
        EMERGENCY("urn:liberty:id-sis-pp:can:emergency", new String[]{"/pp:PP/pp:EmergencyContact", "/pp:PP/pp:AddressCard [pp:AddrType=\"urn:liberty:id-sis-pp:addrType:emergency\"]"}, "Can store some emergency contact data ");

        private String keyword;
        private String[] equivalentXPaths;
        private String meaning;

        DataModifyOption(String keyword, String[] equivalentXPaths, String meaning)
        {
            this.keyword = keyword;
            this.equivalentXPaths = equivalentXPaths;
            this.meaning = meaning;   
        }

        /**
         * The keyword uri for this DataModifyOption.
         * 
         * @return
         */        
        public String keyword() { return keyword; }
        
        /**
         * XPath equivalents that may be used for creating modify queries
         * 
         * @return
         */
        public String[] equivalentXPaths() { return equivalentXPaths; }
        
        /**
         * The english meaning of this DataModifyOption
         * 
         * @return
         */
        public String meaning() { return meaning; }

        /**
         * returns a DataModifyOption matching the supplied keyword
         * 
         * @param keyword
         * @return a DataModifyOption or null if there is no match
         */
        public DataModifyOption dataUpdateOptionForKeyword(String keyword)
        {
            if(null==keyword || keyword.trim().length()==0 || 0!=keyword.indexOf("urn:liberty:id-sis-pp:can")) return null;
            else
            {
                for(DataModifyOption c : DataModifyOption.values())
                {
                    if(c.keyword.equals(keyword)) return c;
                }
            }
            return null;
        }

    }


}



