package org.openliberty.wsc;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.List;

import org.openliberty.wsc.DiscoveryService.WSFServiceType;
import org.openliberty.xmltooling.Konstantz.WSFSecurityMechanism;
import org.openliberty.xmltooling.disco.RequestedService;
import org.openliberty.xmltooling.wsa.Address;
import org.openliberty.xmltooling.wsa.EndpointReference;
import org.openliberty.xmltooling.wsa.Metadata;

/**
 * This class contains a collection of convenience methods designed to make common WSC operations easier. It consists entirely 
 * of static methods and is never meant to be instantiated. 
 * 
 * @author asa
 *
 */
public class WSCUtilities
{

    /**
     * This method takes a discovery EndpointReference and a WSFServiceType and returns an instantiated service client.
     * <p>
     * The process is:
     * <ol>
     * <li>Instantiate, and then send a request to the DiscoveryService for the service type specified</li>
     * <li>Instantiate a service with the first </li>
     * </ol>
     * 
     * @param discoEPR the endpoint reference for the discovery service that will be used to discover the specified service
     * @param serviceType an enumerated reference to a WSF service that the client library supports
     * @return
     */
    public static BaseServiceClient clientForDiscoveryEPR(EndpointReference discoEPR, DiscoveryService.WSFServiceType serviceType) throws WSCException
    {
        // throw NPEs if necessary
        if(null==discoEPR) throw new NullPointerException("discoEPR may not be null");
        if(null==serviceType) throw new NullPointerException("serviceType may not be null");

        // instantiate the DS
        DiscoveryService discoveryService = new DiscoveryService(null, discoEPR);

        // build and add the RequestedService element
        RequestedService requestedService = DiscoveryService.baseRequestedService();        
        DiscoveryService.addServiceTypesToRequestedService(new String[] { serviceType.getUrn() }, requestedService);             
        discoveryService.addARequestedService(requestedService);

        // invoke discovery service
        List<EndpointReference> eprs = discoveryService.invoke();

        if(null!=eprs && eprs.size()>0)
        {
            // Get the first reference
            EndpointReference endpointReference = eprs.get(0);

            // Get the metadata
            Metadata metadata = endpointReference.getMetadata();
            if(null==metadata)
            {
                throw new WSCException("The EndpointReference contains not Metadata element.  While this is not illegal, there is no way for the Utility to determine whether or not there is a valid Service as requested available in the ID-WSF2.0 environment.");
            }
            else 
            {
                // make sure that the metadata references the correct urn
                if(metadata.referencesServiceTypeURN( serviceType.getUrn() ))
                {
                    // instantiate a service client
                    return DiscoveryService.serviceClientForTypeAndEndpointReference(discoveryService, serviceType, endpointReference);
                }                                
            }

        }

        // if nothing was found, bail!
        throw WSCNoServiceAvailableException.exceptionForRequestedService(requestedService);
    }


    public static BaseServiceClient clientForDiscoveryEPR(EndpointReference discoEPR, DiscoveryService.WSFServiceType serviceType, WSFSecurityMechanism sechMech) throws WSCException
    {
        // throw NPEs if necessary
        if(null==discoEPR) throw new NullPointerException("discoEPR may not be null");
        if(null==serviceType) throw new NullPointerException("serviceType may not be null");

        // instantiate the DS
        DiscoveryService discoveryService = new DiscoveryService(null, discoEPR);

        // build and add the RequestedService element
        RequestedService requestedService = DiscoveryService.baseRequestedService();        
        DiscoveryService.addServiceTypesToRequestedService(new String[] { serviceType.getUrn() }, requestedService);             
        if(null!=sechMech) DiscoveryService.addSecurityMechIDsToRequestedService(new String[] { sechMech.getUri() }, requestedService);        
        discoveryService.addARequestedService(requestedService);

        // invoke discovery service
        List<EndpointReference> eprs = discoveryService.invoke();

        if(null!=eprs && eprs.size()>0)
        {
            // Get the first reference
            EndpointReference endpointReference = eprs.get(0);

            // Get the metadata
            Metadata metadata = endpointReference.getMetadata();
            if(null==metadata)
            {
                throw new WSCException("The EndpointReference contains not Metadata element.  While this is not illegal, there is no way for the Utility to determine whether or not there is a valid Service as requested available in the ID-WSF2.0 environment.");
            }
            else 
            {
                // make sure that the metadata references the correct urn
                if(metadata.referencesServiceTypeURN( serviceType.getUrn() ))
                {
                    // instantiate a service client
                    return DiscoveryService.serviceClientForTypeAndEndpointReference(discoveryService, serviceType, endpointReference);
                }                                
            }

        }

        // if nothing was found, bail!
        throw WSCNoServiceAvailableException.exceptionForRequestedService(requestedService);
    }
    

    /**
     * Attempts to retrieve a Discovery Service Endpoint Reference from the default 
     * Authentication Service.
     * <p>
     * This convenience method assumes that there are values set in OpenLibertyBootstrap 
     * for authenticationServiceUrl, authUsername, and authPassword.
     * 
     * @param authMechanism
     * @return 
     * @throws WSCException 
     * @throws NullPointerException if authMechanism is null
     */
    public static EndpointReference boostrapDiscoveryEPRFromDefaultAS(AuthenticationService.AuthMechanism authMechanism) throws WSCException
    {
        if(null==authMechanism) throw new NullPointerException("authMechanism is a required parameter and must not be null");

        URL authServiceURL = null;

        try
        {
            authServiceURL = new URL(OpenLibertyBootstrap.getAuthenticationServiceUrl());
        }
        catch (MalformedURLException e1)
        {
            throw new WSCException("Default AS url \'"+OpenLibertyBootstrap.getAuthenticationServiceUrl()+"\' is malformed");
        }

        Address address = new Address();
        address.setValue(authServiceURL.toString());

        EndpointReference authEpr = new EndpointReference();
        authEpr.setAddress(address);

        // Create new authentication service object
        AuthenticationService authService = AuthenticationService.serviceForEndpointReference(null, authEpr);
        
        // Attempt to authenticate using the specified auth mechanism
        return authService.authenticate(OpenLibertyBootstrap.getAuthUsername(), OpenLibertyBootstrap.getAuthPassword(), authMechanism);

    }  

    /**
     * Returns a DS EPR by attempting the suggested AuthMechanism using the 
     * username/password supplied and the default Authentication service settings
     * that are loaded by OpenLibertyBootstrap
     * 
     * @param authMechanism
     * @param username
     * @param password
     * @return
     * @throws WSCException
     */
    public static EndpointReference boostrapDiscoveryEPRFromDefaultAS(AuthenticationService.AuthMechanism authMechanism, String username, String password) throws WSCException
    {
        if(null==authMechanism) throw new NullPointerException("authMechanism is a required parameter and must not be null");

        URL authServiceURL = null;

        try
        {
            authServiceURL = new URL(OpenLibertyBootstrap.getAuthenticationServiceUrl());
        }
        catch (MalformedURLException e1)
        {
            throw new WSCException("Default AS url \'"+OpenLibertyBootstrap.getAuthenticationServiceUrl()+"\' is malformed");
        }

        Address address = new Address();
        address.setValue(authServiceURL.toString());

        EndpointReference authEpr = new EndpointReference();
        authEpr.setAddress(address);

        // Create new authentication service object
        AuthenticationService authService = AuthenticationService.serviceForEndpointReference(null, authEpr);
        
        // Attempt to authenticate using the specified auth mechanism
        return authService.authenticate(username, password, authMechanism);

    }  

    
       
    
    
    /**
     * Retrieves a list of {@link EndpointReference} objects from the specified 
     * Discovery Service using the supplied parameters.
     * 
     * @param discoService
     * @param type
     * @param mechanism
     * @return
     */
    public static List<EndpointReference> queryDiscoveryServiceForServiceEPRs(DiscoveryService discoService, WSFServiceType type, WSFSecurityMechanism mechanism)
    {
        String[] serviceTypes = type!=null ? new String[] { type.getUrn() } : null;
        String[] securityMechanisms = mechanism!=null ? new String[] { mechanism.getUri() } : null;

        return queryDiscoveryServiceForServiceEPRs(discoService, serviceTypes, null, securityMechanisms);
    }
    
    /**
     * Retrieves a list of {@link EndpointReference} objects from the specified 
     * Discovery Service using the supplied parameters.
     * 
     * @param discoService
     * @param serviceTypes
     * @param providerIDs
     * @param securityMechIDs
     * @return
     */
    public static List<EndpointReference> queryDiscoveryServiceForServiceEPRs(DiscoveryService discoService, String[] serviceTypes, String[] providerIDs, String[] securityMechIDs)
    {
        
        RequestedService requestedService = null;

        if(null!=serviceTypes)
        {            
            requestedService = DiscoveryService.requestedServiceForServiceTypes(serviceTypes);
        } 

        if(null!=providerIDs)
        {            
            if(null==requestedService) requestedService = DiscoveryService.requestedServiceForProviderIDs(providerIDs);
            else DiscoveryService.addProviderIDsToRequestedService(providerIDs, requestedService);
        } 

        if(null!=securityMechIDs)
        {
            if(null==requestedService) requestedService = DiscoveryService.requestedServiceForSecurityMechIDs(securityMechIDs);
            else DiscoveryService.addSecurityMechIDsToRequestedService(securityMechIDs, requestedService);            
        }

        if(null!=requestedService) discoService.addARequestedService(requestedService);        
        
        return discoService.invoke();

    }
    
    
    
}
