package org.openliberty.wsc;

import java.io.IOException;
import java.io.StringReader;
import java.security.GeneralSecurityException;
import java.util.ArrayList;
import java.util.List;

import net.shibboleth.utilities.java.support.component.ComponentInitializationException;
import net.shibboleth.utilities.java.support.xml.BasicParserPool;
import net.shibboleth.utilities.java.support.xml.QNameSupport;
import net.shibboleth.utilities.java.support.xml.SerializeSupport;
import net.shibboleth.utilities.java.support.xml.XMLParserException;

import org.joda.time.DateTime;
import org.openliberty.xmltooling.Konstantz;
import org.openliberty.xmltooling.OpenLibertyHelpers;
import org.openliberty.xmltooling.disco.SecurityContext;
import org.openliberty.xmltooling.security.Token;
import org.openliberty.xmltooling.soap.soap11.HeaderBuilder;
import org.openliberty.xmltooling.soap.soap11.HeaderIDWSF;
import org.openliberty.xmltooling.soapbinding.EndpointUpdate;
import org.openliberty.xmltooling.soapbinding.Framework;
import org.openliberty.xmltooling.soapbinding.ProcessingContext;
import org.openliberty.xmltooling.soapbinding.RedirectRequest;
import org.openliberty.xmltooling.soapbinding.SecurityMechID;
import org.openliberty.xmltooling.soapbinding.Sender;
import org.openliberty.xmltooling.utility_2_0.Status;
import org.openliberty.xmltooling.wsa.Action;
import org.openliberty.xmltooling.wsa.Address;
import org.openliberty.xmltooling.wsa.CredentialsContext;
import org.openliberty.xmltooling.wsa.EndpointReference;
import org.openliberty.xmltooling.wsa.MessageID;
import org.openliberty.xmltooling.wsa.Metadata;
import org.openliberty.xmltooling.wsa.ReferenceParameters;
import org.openliberty.xmltooling.wsa.RelatesTo;
import org.openliberty.xmltooling.wsa.To;
import org.openliberty.xmltooling.wsse.Security;
import org.openliberty.xmltooling.wsse.SecurityBuilder;
import org.openliberty.xmltooling.wsu.Created;
import org.openliberty.xmltooling.wsu.CreatedBuilder;
import org.openliberty.xmltooling.wsu.Timestamp;
import org.openliberty.xmltooling.wsu.TimestampBuilder;
import org.opensaml.core.xml.AttributeExtensibleXMLObject;
import org.opensaml.core.xml.XMLObject;
import org.opensaml.core.xml.XMLObjectBuilderFactory;
import org.opensaml.core.xml.XMLRuntimeException;
import org.opensaml.core.xml.config.XMLObjectProviderRegistrySupport;
import org.opensaml.core.xml.io.Marshaller;
import org.opensaml.core.xml.io.MarshallerFactory;
import org.opensaml.core.xml.io.MarshallingException;
import org.opensaml.core.xml.io.Unmarshaller;
import org.opensaml.core.xml.io.UnmarshallerFactory;
import org.opensaml.core.xml.io.UnmarshallingException;
import org.opensaml.core.xml.util.AttributeMap;
import org.opensaml.core.xml.util.XMLObjectChildrenList;
import org.opensaml.core.xml.util.XMLObjectSupport;
import org.opensaml.saml.saml2.core.Assertion;
import org.opensaml.saml.saml2.core.EncryptedAssertion;
import org.opensaml.soap.soap11.Detail;
import org.opensaml.soap.soap11.Envelope;
import org.opensaml.soap.soap11.Fault;
import org.opensaml.soap.util.SOAPConstants;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

/**
 * WSFMessage models a round trip request/response ID-WSF SOAP transaction.
 * 
 * @author curtis
 * @author asa
 *
 */
public class WSFMessage 
{
    protected static org.slf4j.Logger log = LoggerFactory.getLogger(WSFMessage.class);

	/** Parser manager used to parse XML. */
	protected static BasicParserPool parser;

	/** XMLObject marshaller factory. */
	protected static MarshallerFactory marshallerFactory;

	/** XMLObject marshaller factory. */
	protected static UnmarshallerFactory unmarshallerFactory;


	/** XMLObject builder factory. */
	protected static XMLObjectBuilderFactory builderFactory;

	/**
	 * This is the Service Client that is invoking a particular WSFMessage.  It is 
	 * referenced here so that we can do EPR updates.
	 *  
	 */
	private BaseServiceClient serviceClient;

	/** SOAP Request */
	private Envelope envelope;

	/**
	 * Response String
	 */
	private String response;

	/** SOAP Response */
	private Envelope responseEnvelope;


	/**
	 * This is the action that the message is intended to perform
	 */
	private String actionString;

	/**
	 * Holds the reference IDs of the elements to be included in any signature
	 */
	private List<String> signatureIds;    

	/**
	 * If the SOAP Response contains a Fault with a RedirectRequest element, it is stored here.
	 */
	private RedirectRequest redirectRequest = null;

	/**
	 * When an EndpointUpdate header is received and processed, this is set to true 
	 */
	private boolean endpointUpdated = false;

	/**
	 * If true, only the SOAP body is printed during a logger debug session
	 */
	private static boolean debugJustSOAPBody = false;

	/**
	 * This method adds an id to an AttributeExtensibleXMLObject and 
	 * then places that Id in signatureIds for signing 
	 * <p>
	 * http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-utility-1.0.xsd
	 * <pre>
	 *  &lt;xsd:attribute name="Id" type="xsd:ID"&gt;
	 *      &lt;xsd:annotation&gt;
	 *          &lt;xsd:documentation&gt;
	 *              This global attribute supports annotating arbitrary elements with an ID.
	 *           &lt;/xsd:documentation&gt;
	 *      &lt;/xsd:annotation&gt;
	 *  &lt;/xsd:attribute&gt;
	 *  </pre>
	 */
	public void addWSUIdAttribute(AttributeExtensibleXMLObject obj, String id)
	{
		this.getSignatureIds().add(id);
		obj.getUnknownAttributes().put(QNameSupport.constructQName(Konstantz.WSU_NS, "Id", Konstantz.WSU_PREFIX), id);
	}


	public void addSOAP11Attributes(AttributeExtensibleXMLObject obj, boolean mustUnderstand)
	{
		AttributeMap atts = obj.getUnknownAttributes();
		atts.put(QNameSupport.constructQName(SOAPConstants.SOAP11_NS, "mustUnderstand", SOAPConstants.SOAP11_PREFIX), (mustUnderstand?"1":"0"));
		atts.put(QNameSupport.constructQName(SOAPConstants.SOAP11_NS, "actor", SOAPConstants.SOAP11_PREFIX), Konstantz.SOAP_ACTOR);
	}

	private WSFMessage(BaseServiceClient serviceClient)
	{
		this.serviceClient = serviceClient;
	}

	/**
	 * Factory method to create a WSFMessage from an EPR.
	 * 
	 * @return
	 * @throws UnmarshallingException 
	 * @throws XMLParserException 
	 */
	public static WSFMessage createWSFMessage(BaseServiceClient serviceClient, String actionString) throws XMLParserException, UnmarshallingException
	{
		WSFMessage message = new WSFMessage(serviceClient);
		message.actionString = actionString;
		message.initializeRequestEnvelope();        
		return message;
	}

	public static void setDebugJustSOAPBody(boolean _debugJustSOAPBody)
	{
		debugJustSOAPBody = _debugJustSOAPBody;
	}

	public static boolean getDebugJustSOAPBody()
	{
		return debugJustSOAPBody;
	}

	/**
	 * The WSFMessage class contains a global Envelope that is used for making a request,
	 * this method initializes the global request envelope including adding the security
	 * token from the message EPR
	 * 
	 * @throws XMLParserException
	 * @throws UnmarshallingException
	 */
	private void initializeRequestEnvelope() throws XMLParserException, UnmarshallingException
	{

		EndpointReference epr = serviceClient.getServiceEndpointReference();

		// Get the the value of the intended destination of the message.
		String destinationUrl = epr.getAddress().getValue();

		Envelope envelope = null;

		envelope = WSFMessage.buildSOAPEnvelope();
		// add wsu:id to 
		this.addWSUIdAttribute(envelope.getBody(), "body");

		HeaderIDWSF header = new HeaderBuilder().buildObject();         

		// Framework
		Framework fw = new Framework();
		fw.setVersion("2.0");
		this.addSOAP11Attributes(fw, true);
		this.addWSUIdAttribute(fw, "fwkHdr");
		header.setFramework(fw);        

		// To
		To to = new To();      
		to.setValue(destinationUrl);
		//message.addSOAP11Attributes(to, true);
		this.addWSUIdAttribute(to, "toHdr");
		header.setTo(to);

		// Sender
		String providerID = OpenLibertyBootstrap.getProviderIDUri();
		if(null!=providerID && providerID.trim().length()>0)
		{
			Sender sender = new Sender();
			sender.setProviderID(providerID);
			this.addWSUIdAttribute(sender, "sndrHdr");
			header.setSender(sender);
		}

		// Action
		if(null!=actionString)
		{
			Action action = new Action();
			action.setValue(actionString);
			//message.addSOAP11Attributes(action, true);
			this.addWSUIdAttribute(action, "actHdr");
			header.setAction(action); 
		}

		// MessageID
		MessageID messageID = new MessageID();    
		// messageID.setValue("uuid:"+UUID.randomUUID().toString());
		//message.addSOAP11Attributes(messageID, true);
		this.addWSUIdAttribute(messageID, "midHdr");
		header.setMessageID(messageID);

		// Security
		Created created = new CreatedBuilder().buildObject();
		created.setValue(OpenLibertyHelpers.stringForDateTime(new DateTime()));  //2007-08-22T18:41:19.505-04:00
		Timestamp ts = new TimestampBuilder().buildObject();   
		ts.setCreated(created);
		//message.addSOAP11Attributes(ts, true);

		this.addWSUIdAttribute(ts, "tsSecHdr");
		Security security = new SecurityBuilder().buildObject();

		this.addSOAP11Attributes(security, true);
		security.getUnknownXMLObjects().add(ts);
		header.setSecurity(security);


		// TODO: more intelligent approach to selecting security context and token.  Loop through contexts perhaps?

		// Are there sec tokens?
		Metadata metaData = epr.getMetadata();

		if(null!=metaData)
		{
			XMLObjectChildrenList<SecurityContext> securityContexts = metaData.getSecurityContexts();	

			// if there are contexts, choose the first
			if(securityContexts.size()>0)
			{              
				SecurityContext securityContext = securityContexts.get(0);         

				// if there are tokens in the context, choose the first
				if(null!=securityContext)
				{
					if(null!=securityContext.getUnknownXMLObjects())
					{
						// {urn:liberty:security:2006-08}Token
						for(XMLObject obj : securityContext.getUnknownXMLObjects())
						{
							if(log.isDebugEnabled()) log.debug(obj.getElementQName().toString());
						}
					}


					if(null!=securityContext.getTokens() && securityContext.getTokens().size()>0)
					{
						// Clone the token
						//Token token = (Token)OpenLibertyHelpers.cloneXMLObject(securityContext.getTokens().get(0));                   
						Token token = securityContext.getTokens().get(0);

						EncryptedAssertion encryptedAssertion = token.getEncryptedAssertion();

						if(null!=encryptedAssertion)
						{            
							/**
							 * Set the parent to null to prevent the following exception when reusing the same bearer token assertion:
							 * 
							 * java.lang.IllegalArgumentException: {urn:oasis:names:tc:SAML:2.0:assertion}EncryptedAssertion is already the child of another XMLObject and may not be inserted in to this list
							 *     at org.opensaml.xml.util.XMLObjectChildrenList.setParent(XMLObjectChildrenList.java:202)
							 * 
							 * 
							 * NOTE: The EPR.metaData().securityContexts().get(0).getTokens().get(0).getEncryptedAssertion() reference chain will remain in place even once the
							 * parent has been set to null.  The trick here is to remove the reference from the last "Security" header which was used in a separate outgoing 
							 * SOAP message.
							 * 
							 */                    
							//                          XMLObject parent = encryptedAssertion.getParent();
							//                          if(null!=parent && parent instanceof Security)
							//                          {
							//                          ((Security)parent).getUnknownXMLObjects().remove(encryptedAssertion);                     
							//                          }
							//                          encryptedAssertion.setParent(null);

							// add to sec header
							// Security security = header.getSecurity();
							security.getUnknownXMLObjects().add(encryptedAssertion);
						}
						else 
						{
							Assertion saml2Assertion = (Assertion)OpenLibertyHelpers.cloneXMLObject( token.getAssertion() );

							if(null!=saml2Assertion)
							{

								// Add the ID of the saml assertion to the signature id list
								this.getSignatureIds().add( saml2Assertion.getID() );

								/**
								 * Set the parent to null to prevent the following exception when reusing the same EPR as the source
								 * for the bearer token assertion:
								 * 
								 * java.lang.IllegalArgumentException: {urn:oasis:names:tc:SAML:2.0:assertion}EncryptedAssertion is already the child of another XMLObject and may not be inserted in to this list
								 *     at org.opensaml.xml.util.XMLObjectChildrenList.setParent(XMLObjectChildrenList.java:202)
								 * 
								 * NOTE: The EPR.metaData().securityContexts().get(0).getTokens().get(0).getAssertion() reference chain will remain in place even once the
								 * parent has been set to null.  The trick here is to remove the reference from the last "Security" header which was used in a separate outgoing 
								 * SOAP message.
								 * 
								 * 2008.12.17 - A.H. Updated so that a clone of the Assertion is created if and only if the parent is not null and is 
								 * a Security element
								 * 
								 */

								//                                XMLObject parent = saml2Assertion.getParent();
								//                                if(null!=parent && parent instanceof Security)
								//                                {
								//                                	EndpointReference tmp_epr = (EndpointReference)OpenLibertyHelpers.cloneXMLObject(serviceClient.getServiceEndpointReference());
								//                                	Metadata tmp_metaData = epr.getMetadata();
								//                                	XMLObjectChildrenList<SecurityContext> tmp_securityContexts = metaData.getSecurityContexts();
								//                                	Token tmp_token = (Token)securityContext.getTokens().get(0);
								//                                    saml2Assertion = tmp_token.getAssertion();;
								//                                    //((Security)parent).getUnknownXMLObjects().remove(saml2Assertion);     
								//                                }
								//                                //saml2Assertion.setParent(null);

								saml2Assertion.releaseDOM();
								saml2Assertion.releaseChildrenDOM(true);                                
								security.getUnknownXMLObjects().add(saml2Assertion);                                
							}
							else
							{
								// add raw content to security header
								security.setValue(token.getValue());
							}                
						}
					}
				}
			}
		}

		// set the Header of the envelope
		envelope.setHeader(header);

		// set the WSFMessage SOAP Envelope
		this.setRequestEnvelope(envelope);

	}


	/**
	 * Holds the reference ids of the SOAP Body element (the element itself), the security token associated
	 * with the signature, and all headers in the message that have been defined in the Liberty SOAP Bindings specification,
	 * including both required and optional header blocks.
	 * 
	 * @return
	 */
	public List<String> getSignatureIds()
	{
		if(null==signatureIds)
		{
			signatureIds = new ArrayList<String>();
		}
		return signatureIds;
	}


	protected BaseServiceClient getServiceClient()
	{
		return serviceClient;
	}

	protected void setServiceClient(BaseServiceClient serviceClient)
	{
		this.serviceClient = serviceClient;
	}

	/**
	 * Adds this ProcessingContext to the header or replaces an 
	 * existing ProcessingContext with this one
	 * 
	 * @param processingContext
	 */
	public void setHeaderProcessingContext(ProcessingContext processingContext)
	{
		HeaderIDWSF header = (HeaderIDWSF)envelope.getHeader();

		// remove any existing ProcessingContext header
		for(XMLObject obj: header.getUnknownXMLObjects())
		{
			if(obj instanceof ProcessingContext)
			{
				header.getUnknownXMLObjects().remove(obj);
				break;
			}
		}

		// add the new ProcessingContext if it is not null
		if(null!=processingContext)
		{
			this.addWSUIdAttribute(processingContext, "pcHdr");
			this.addSOAP11Attributes(processingContext, true);
			header.getUnknownXMLObjects().add(processingContext);                 
		}

	}


	/**
	 * Invoke the WSFMessage sending the SOAP requestEvelope and placing the response in the responseEnvelope.
	 * Signing is done based on the message level setting of "signed"
	 * 
	 * @throws IOException
	 * @throws MarshallingException
	 * @throws XMLParserException
	 * @throws UnmarshallingException
	 * @throws WSCException 
	 * @throws GeneralSecurityException 
	 */
	public void invoke() throws IOException, MarshallingException, XMLParserException, UnmarshallingException, WSCException, GeneralSecurityException
	{
		//boolean sign = serviceClient.isSigningOutgoingMessages();

		/**
		 * If invoke is being called after a redirect request has been made, then
		 * a new request Envelope is created and the RelatesTo header is set to the
		 * last response message id
		 */
		handleRedirectRequestState();

		/**
		 * If invoke is being called as a result of an EndpointUpdate header being
		 * passed back in the response, this will create a new request Envelope.
		 * 
		 */
		handleEndpointUpdatedState();
		HeaderIDWSF header = (HeaderIDWSF)envelope.getHeader();
		String to = header.getTo().getValue();

		Konstantz.WSFSecurityMechanism securityMechanism = Konstantz.WSFSecurityMechanism.ID_WSF20_TLS_BEARER;

		if(null!=serviceClient.getServiceEndpointReference().getMetadata())
		{
			org.openliberty.xmltooling.disco.SecurityMechID securityMechID = serviceClient.getServiceEndpointReference().getMetadata().getSecurityContexts().get(0).getSecurityMechIDs().get(0); //.getValue();
			securityMechanism = Konstantz.WSFSecurityMechanism.findSecurityMechanismForURI(securityMechID.getValue());
		}

		Element outgoingMessage = null;

		// If the Security mechanism id SAMLv2 then sign the message                
		if(securityMechanism==Konstantz.WSFSecurityMechanism.ID_WSF20_CLIENT_TLS_SAML2 || securityMechanism==Konstantz.WSFSecurityMechanism.ID_WSF20_TLS_SAML2 || securityMechanism==Konstantz.WSFSecurityMechanism.ID_WSF20_NULL_SAML2)
		{
			WSFMessageSigner messageSigner = new WSFMessageSigner();            
			try
			{
				outgoingMessage = messageSigner.sign(this);
				if(log.isDebugEnabled())
				{
					log.debug("\n\n#### SIGNED REQUEST MESSAGE ####################################################################");                    
					log.debug(SerializeSupport.prettyPrintXML(outgoingMessage));
					log.debug("#### END SIGNED REQUEST MESSAGE ################################################################\n\n");
				}
			}
			catch (Exception e) 
			{
				e.printStackTrace();
			}

		}
		else 
		{
			// Marshall the XMLObject Envelope into a DOM object
			Marshaller em = XMLObjectSupport.getMarshaller(Envelope.DEFAULT_ELEMENT_NAME);
			outgoingMessage = em.marshall(envelope);

			if(log.isDebugEnabled())
			{
				if(debugJustSOAPBody)
				{
					NodeList nl = outgoingMessage.getChildNodes();
					log.debug(" REQUEST\n"+SerializeSupport.prettyPrintXML(nl.item(1))+"\n");  
				}
				else
				{
					log.debug(" REQUEST\n"+SerializeSupport.prettyPrintXML(outgoingMessage)+"\n");    
				}
			}

		}

		response = null;

		/**
		 * If the Sech Mech contains ClientTLS then configure mutual TLS
		 */
		if(securityMechanism==Konstantz.WSFSecurityMechanism.ID_WSF20_CLIENT_TLS_SAML2)
		{            
			response = SSLUtilities.postSOAPMessage(to, outgoingMessage, true);            
		}        
		/**
		 * For Peer saml2 the HoK Assertion references the certificate that should be used 
		 * the client TLS, there is no signing
		 */
		else if(securityMechanism==Konstantz.WSFSecurityMechanism.ID_WSF20_CLIENT_TLS_PEER_SAML2)
		{
			// TODO: Need to indicate a cert to use for transport on the fly
			response = SSLUtilities.postSOAPMessage(to, outgoingMessage, true);
		}
		else if(securityMechanism==Konstantz.WSFSecurityMechanism.ID_WSF20_NULL_NULL)
		{
			response = SSLUtilities.postSOAPMessageNOTLS(to, outgoingMessage);
		}
		else
		{
			response = SSLUtilities.postSOAPMessage(to, outgoingMessage, false);
		}

		processResponse(response);
	}


	/**
	 * This method will take care of all of the responsibilities required of a WSC when receiving
	 * a response from a WSP.
	 * <p>
	 * e.g. Fault:<br />
	 * <pre>
	 *         &lt;se:Fault xmlns:sb-ext="urn:liberty:sb:2004-04"&gt;
	 *             &lt;faultcode&gt;se:Client&lt;/faultcode&gt;
	 *             &lt;faultstring&gt;soap fault&lt;/faultstring&gt;
	 *              &lt;detail namespace="urn:liberty:sb:2003-08"&gt;
	 *                &lt;lu:Status xmlns:lu="urn:liberty:util:2006-08" code="IDStarMsgNotUnderstood"/&gt;
	 *             &lt;/detail&gt;
	 *          &lt;/se:Fault&gt;
	 * </pre>
	 * 
	 * @param responseString
	 * @throws XMLParserException
	 * @throws UnmarshallingException
	 * @throws WSCException 
	 * @throws MarshallingException 
	 * @throws IOException 
	 * @throws GeneralSecurityException 
	 */
	private void processResponse(String responseString) throws XMLParserException, UnmarshallingException, WSCException, IOException, MarshallingException, GeneralSecurityException
	{

		// Parse response string into DOM document
		Document responseDoc = parser.parse(new StringReader(responseString));

		if(log.isDebugEnabled())
		{
			if(debugJustSOAPBody)
			{
				NodeList nl = responseDoc.getDocumentElement().getChildNodes();
				log.debug(" RESPONSE\n"+SerializeSupport.prettyPrintXML(nl.item(1))+"\n");  
			}
			else
			{
				log.debug(" RESPONSE\n"+SerializeSupport.prettyPrintXML(responseDoc.getDocumentElement())+"\n");    
			}			
		}

		// Unmarshall the Response Envelope 
		Element envelopeElem = responseDoc.getDocumentElement();
		Unmarshaller unmarshaller = XMLObjectSupport.getUnmarshaller(envelopeElem);
		responseEnvelope = (Envelope) unmarshaller.unmarshall(envelopeElem);

		/**
		 * Check for a SOAP Fault, If the fault exists, bubble up as an exception
		 */
		List<XMLObject> faultList = responseEnvelope.getBody().getUnknownXMLObjects(Fault.TYPE_NAME);
		if(faultList.size()>0)
		{
			Fault fault = (Fault) faultList.get(0);            
			//          FaultCode faultCode =  fault.getCode();
			//          FaultString faultString = fault.getMessage();
			Detail detail = fault.getDetail();
			if(detail!=null)
			{

				List<XMLObject> redirectRequestList = detail.getUnknownXMLObjects(RedirectRequest.LOCAL_Q_NAME);
				if(redirectRequestList.size()>0)
				{
					redirectRequest = (RedirectRequest)redirectRequestList.get(0);
					// no more processing required, return
					return;
				}

				Status status = null;                
				List<XMLObject> statusList = detail.getUnknownXMLObjects(Status.LOCAL_Q_NAME);
				if(statusList.size()>0)
				{
					status = (Status)statusList.get(0);

					String statusCode = status.getCode();
					if(null!=statusCode)
					{
						if(statusCode.equals(Konstantz.Status.ENDPOINT_UPDATED.getCode()))
						{
							if( processEndpointUpdated() )
							{
								// invoke again
								invoke();
								return;
							}
						}
						else if(statusCode.equals(Konstantz.Status.INAPPROPRIATE_CREDENTIALS.getCode()))
						{
							if( handleInappropriateCredentials() ) 
							{
								// invoke again
								invoke();
								return;
							}
						}
					}
				}
			}
			// TODO: determine how this is best handled
			//throw new WSCException("SOAP Fault. "+OpenLibertyHelpers.prettyPrintXMLObject(fault));
		}
	}


	/**
	 * Handles the InappropriateCredentials Fault.
	 * <p>
	 * In the case of a Fault with a Status code "InappropriateCredentials" this method
	 * examines the CredentialsContext header and attempts to renegotiate with 
	 * the DS for a more appropriate SecurityMechanism or reauthenticate with 
	 * the AS in the case of a RequestedAuthnContext element being present in
	 * the CredentialsContext header.
	 * 
	 * @return
	 */
	private boolean handleInappropriateCredentials()
	{
		HeaderIDWSF header = (HeaderIDWSF)responseEnvelope.getHeader();
		CredentialsContext credentialsContext = header.getCredentialsContext();
		if(null!=credentialsContext)
		{
			List<SecurityMechID> securityMechIds = credentialsContext.getSecurityMechIDs();
			if(null!=securityMechIds && securityMechIds.size()>0)
			{
				String[] sechMechs = new String[securityMechIds.size()];            
				for(int i=0;i<sechMechs.length; i++)
				{
					sechMechs[i] = securityMechIds.get(i).getValue();
				}

				List<EndpointReference> eprs = WSCUtilities.queryDiscoveryServiceForServiceEPRs(
						serviceClient.getDiscoveryService(), 
						new String[] { serviceClient.getServiceEndpointReference().getMetadata().getServiceTypes().get(0).getValue() },
						new String[] { serviceClient.getServiceEndpointReference().getMetadata().getProviderID().getValue() },
						sechMechs                                
				);

				if(null!=eprs && eprs.size()<0)
				{
					serviceClient.replaceServiceEndpointReference(eprs.get(0));         
					return true;                
				}
			}
			else if(null!=credentialsContext.getRequestedAuthnContext())
			{
				if(null!=OpenLibertyBootstrap.getAuthenticationServiceUrl())
				{

					// get the AUTH SERVICE EPR
					List<EndpointReference> authEPRs = WSCUtilities.queryDiscoveryServiceForServiceEPRs(
							serviceClient.getDiscoveryService(), 
							new String[] { DiscoveryService.WSFServiceType.AUTHENTICATION_SERVICE.getUrn() },
							null,
							null
					);

					if(null!=authEPRs && authEPRs.size()>0)
					{

						// Create new authentication service object
						AuthenticationService authService = (AuthenticationService) serviceClient.getDiscoveryService().serviceClientForTypeAndEndpointReference(DiscoveryService.WSFServiceType.AUTHENTICATION_SERVICE, authEPRs.get(0));

						// Attempt to authenticate using the specified auth mechanism
						try
						{
							// get the discovery EPR
							EndpointReference discoEPR = authService.authenticate(OpenLibertyBootstrap.getAuthUsername(), OpenLibertyBootstrap.getAuthPassword(), OpenLibertyBootstrap.getAuthMechanism(), credentialsContext.getRequestedAuthnContext());

							// get the discovery service
							DiscoveryService ds = new DiscoveryService(discoEPR);
							// get an 

							List<EndpointReference> eprs = WSCUtilities.queryDiscoveryServiceForServiceEPRs(
									ds, 
									new String[] { serviceClient.getServiceEndpointReference().getMetadata().getServiceTypes().get(0).getValue() },
									null,
									null
							);

							serviceClient.replaceServiceEndpointReference(eprs.get(0));
							serviceClient.setDiscoveryService(ds);
							return true;

						}
						catch (WSCException e)
						{
							e.printStackTrace();
						}
					}
				}
			}
		}

		return false;
	}


	/**
	 * This method handles the creation of a new {@link EndpointReference} from an {@link EndpointUpdate} element.
	 * <p>
	 * <b>For a COMPLETE Update:</b>
	 * <p>
	 * If updateType is not present or has the value urn:liberty:sb:2006-08:EndpointUpdate:Complete, the
	 * &lt;wsa:EndpointUpdate&gt; is a completely specified endpoint reference.
	 * <p>
	 * <b>For a PARTIAL Update:</b>
	 * <p>
	 * 1. Take the &lt;wsa:Address&gt; from the &lt;wsa:EndpointUpdate&gt;. If the value is urn:liberty:sb:2006-08:EndpointUpdate:NoChange, 
	 * then take the &lt;wsa:Address&gt; from the original endpoint reference.
	 * <p>
	 * 2. Take the &lt;wsa:ReferenceParameters&gt; from the &lt;wsa:EndpointUpdate&gt;, if present. Then, if
	 * &lt;wsa:ReferenceParameters&gt; is present in the orginal endpoint reference, take each direct child from
	 * that element that does not match an element already taken from the update (comparing the namespace qualified
	 * names of the elements).
	 * <p>
	 * 3. Take the &lt;wsa:Metadata&gt; from the &lt;wsa:EndpointUpdate&gt;, if present. Then, if &lt;wsa:Metadata&gt; is
	 * present in the orginal endpoint reference, take each direct child from that element that does not match an element
	 * already taken from the update (comparing the namespace qualified names of the elements).
	 * <p>
	 * 4. Take any extension elements from the &lt;wsa:EndpointUpdate&gt;, if present. Then, if any extension elements are
	 * present in the orginal endpoint reference, take each one that does not match     * @return
	 */
	private boolean processEndpointUpdated()
	{
		HeaderIDWSF header = (HeaderIDWSF)responseEnvelope.getHeader();
		EndpointUpdate endpointUpdate = header.getEndpointUpdate();
		if(null!=endpointUpdate)
		{
			/**
			 * PARTIAL UPDATE
			 */
			if(endpointUpdate.getUpdateType()==EndpointUpdate.UpdateType.PARTIAL)
			{

				EndpointReference originalEPR = serviceClient.getServiceEndpointReference();     

				// Make a clone of the originalEPR so that we can steal from it, keeping the 
				// original EPR intact.
				try
				{
					Marshaller m = marshallerFactory.getMarshaller(originalEPR);
					Element originalEPRElement = m.marshall(originalEPR);            
					originalEPRElement = (Element)originalEPRElement.cloneNode(true);
					Unmarshaller u = unmarshallerFactory.getUnmarshaller(originalEPRElement);
					originalEPR = (EndpointReference)u.unmarshall(originalEPRElement);
				}
				catch (UnmarshallingException e)
				{
					e.printStackTrace();
				}         
				catch (MarshallingException e)
				{
					e.printStackTrace();
				}   


				EndpointReference newEPR = new EndpointReference();
				newEPR.setNotOnOrAfter(endpointUpdate.getNotOnOrAfter());

				/**
				 * 1. Take the &lt;wsa:Address&gt; from the &lt;wsa:EndpointUpdate&gt;. If the value is urn:liberty:sb:2006-08:EndpointUpdate:NoChange, 
				 * then take the &lt;wsa:Address&gt; from the original endpoint reference.
				 */
				if(endpointUpdate.isNoAddressChange())
				{
					Address address = new Address();
					address.setValue(originalEPR.getAddress().getValue());
					newEPR.setAddress(address);
				}
				else
				{
					Address address = new Address();
					address.setValue(endpointUpdate.getAddress().getValue());
					newEPR.setAddress(address);
				}


				/**
				 * 2. Take the &lt;wsa:ReferenceParameters&gt; from the &lt;wsa:EndpointUpdate&gt;, if present. Then, if
				 * &lt;wsa:ReferenceParameters&gt; is present in the orginal endpoint reference, take each direct child from
				 * that element that does not match an element already taken from the update (comparing the namespace qualified
				 * names of the elements).
				 */
				ReferenceParameters updatedReferenceParameters = endpointUpdate.getReferenceParameters();
				if(null!=updatedReferenceParameters)
				{
					updatedReferenceParameters.setParent(newEPR);
					endpointUpdate.setReferenceParameters(null);
					updatedReferenceParameters.setParent(newEPR);
					newEPR.setReferenceParameters(updatedReferenceParameters);
				}

				ReferenceParameters originalReferenceParameters = originalEPR.getReferenceParameters();
				if(null!=originalReferenceParameters)
				{
					//                  // rinse, dry, and then add water to create a copy of the original ReferenceParameters
					//                  Marshaller m = marshallerFactory.getMarshaller(originalReferenceParameters);
					//                  try
					//                  {
					//                  Element ele = m.marshall(originalReferenceParameters);                        
					//                  Unmarshaller u = unmarshallerFactory.getUnmarshaller(ele);
					//                  originalReferenceParameters = (ReferenceParameters)u.unmarshall(ele);
					//                  } 
					//                  catch (UnmarshallingException e)
					//                  {
					//                  e.printStackTrace();
					//                  }         
					//                  catch (MarshallingException e)
					//                  {
					//                  e.printStackTrace();
					//                  }    


					// Simple case, there were no updated reference parameters
					ReferenceParameters newEPRReferenceParameters = newEPR.getReferenceParameters();
					if(null==newEPRReferenceParameters)
					{
						newEPR.setReferenceParameters(originalReferenceParameters);   
					}
					// Complex case, there are new reference parameters and original reference parameters, must go through every top level 
					// element
					else
					{                            
						for(XMLObject obj : originalReferenceParameters.getOrderedChildren())
						{
							// If the obj is not found in newReferenceParameters, then add it
							List<XMLObject> parameters = newEPRReferenceParameters.getUnknownXMLObjects(obj.getElementQName());
							if(parameters.size()==0)
							{
								newEPRReferenceParameters.getUnknownXMLObjects().add(obj);
							}
						}
					}
				}  

				/**
				 * 3. Take the &lt;wsa:Metadata&gt; from the &lt;wsa:EndpointUpdate&gt;, if present. Then, if &lt;wsa:Metadata&gt; is
				 * present in the orginal endpoint reference, take each direct child from that element that does not match an element
				 * already taken from the update (comparing the namespace qualified names of the elements).
				 */
				Metadata updatedMetadata = endpointUpdate.getMetadata();
				if(null!=updatedMetadata)
				{
					updatedMetadata.setParent(newEPR);
					endpointUpdate.setMetadata(null);
					updatedMetadata.setParent(newEPR);
					newEPR.setMetadata(updatedMetadata);
				}

				Metadata originalMetadata = originalEPR.getMetadata();
				if(null!=originalMetadata)
				{
					//                  // rinse, dry, and then add water to create a copy of the original metadata
					//                  Marshaller m = marshallerFactory.getMarshaller(originalMetadata);
					//                  try
					//                  {
					//                  Element ele = m.marshall(originalMetadata);
					//                  Unmarshaller u = unmarshallerFactory.getUnmarshaller(ele);
					//                  originalMetadata = (Metadata)u.unmarshall(ele);
					//                  } 
					//                  catch (UnmarshallingException e)
					//                  {
					//                  e.printStackTrace();
					//                  }         
					//                  catch (MarshallingException e)
					//                  {
					//                  e.printStackTrace();
					//                  }    


					// Simple case, there were no updated reference parameters
					Metadata newEPRMetadata = newEPR.getMetadata();
					if(null==newEPRMetadata)
					{
						newEPR.setMetadata(originalMetadata);
					}
					// Complex case, there is metadata and original metadata, must go through every top level 
					// element
					else
					{
						for(XMLObject obj : originalMetadata.getOrderedChildren())
						{
							// If the obj is not found in newReferenceParameters, then add it
							List<XMLObject> entries = newEPRMetadata.getUnknownXMLObjects(obj.getElementQName());
							if(entries.size()==0)
							{
								newEPRMetadata.getUnknownXMLObjects().add(obj);
							}
						}
					}
				}        


				/**
				 * 4. Take any extension elements from the &lt;wsa:EndpointUpdate&gt;, if present. Then, if any extension elements are
				 * present in the orginal endpoint reference, take each one that does not match
				 */
				List<XMLObject> updatedExtensions = endpointUpdate.getUnknownXMLObjects();
				if(null!=updatedExtensions)
				{
					// remove all from the endpointUpdate
					endpointUpdate.getUnknownXMLObjects().removeAll(updatedExtensions);

					// set the new parent and add to 
					for(XMLObject obj : updatedExtensions)
					{
						obj.setParent(newEPR);
						newEPR.getUnknownXMLObjects().add(obj);
					}                        
				}

				List<XMLObject> originalExtensions = originalEPR.getUnknownXMLObjects();               
				if(null!=originalExtensions)
				{   
					//                  List<XMLObject> originalExtensionsCopy = new ArrayList<XMLObject>();

					//                  for(XMLObject obj : originalExtensions)
					//                  {
					//                  // rinse, dry, and then add water to create copies of each 
					//                  Marshaller m = marshallerFactory.getMarshaller(obj);
					//                  try
					//                  {
					//                  Element ele = m.marshall(obj);
					//                  Unmarshaller u = unmarshallerFactory.getUnmarshaller(ele);
					//                  originalExtensionsCopy.add(u.unmarshall(ele));
					//                  } 
					//                  catch (UnmarshallingException e)
					//                  {
					//                  e.printStackTrace();
					//                  }         
					//                  catch (MarshallingException e)
					//                  {
					//                  e.printStackTrace();
					//                  }                            
					//                  }

					// Simple case, there were no updated reference parameters
					if(null==updatedExtensions)
					{
						newEPR.getUnknownXMLObjects().addAll(originalExtensions);   
					}
					// Complex case
					else
					{   

						for(XMLObject obj : originalExtensions)
						{
							// If the obj is not found in newReferenceParameters, then add it
							List<XMLObject> parameters = newEPR.getUnknownXMLObjects(obj.getElementQName());
							if(parameters.size()==0)
							{
								newEPR.getUnknownXMLObjects().add(obj);
							}
						}
					}
				}  

				// Set the newly created EPR to the current EPR in the serviceClient
				serviceClient.setServiceEndpointReference(newEPR);                
				return true;
			}
			/**
			 * FULL UPDATE
			 * If updateType is not present or has the value urn:liberty:sb:2006-08:EndpointUpdate:Complete, the
			 * &lt;wsa:EndpointUpdate&gt; is a completely specified endpoint reference.
			 */
			else 
			{                
				serviceClient.setServiceEndpointReference(endpointUpdate);
				return true;
			}

		}

		return false;
	}


	/**
	 * After an EndpointUpdate, the original message is invoked again, with a 
	 * different EPR
	 * @throws UnmarshallingException 
	 * @throws XMLParserException 
	 *
	 */
	private void handleEndpointUpdatedState() throws XMLParserException, UnmarshallingException
	{
		if(endpointUpdated)
		{

			initializeRequestEnvelope();

			endpointUpdated = false;
		}
	}


	/**
	 * If the WSC resends its request it MUST set the value of the wsa:RelatesTo SOAP Header to the same value of the
	 * wsa:MessageID SOAP Header of the SOAP Fault that carried the &lt;RedirectRequest&gt; element.
	 * <p>
	 * (From Section 7 [LibertySOAPBinding] document http://www.projectliberty.org/specs )
	 * @throws UnmarshallingException 
	 * @throws XMLParserException 
	 * 
	 */
	private void handleRedirectRequestState() throws XMLParserException, UnmarshallingException
	{
		if(this.hasRedirectRequestBeenIssued())
		{

			initializeRequestEnvelope();

			HeaderIDWSF header = (HeaderIDWSF)envelope.getHeader();

			// Add a RelatesTo Header
			RelatesTo relatesTo = new RelatesTo();
			relatesTo.setValue(((HeaderIDWSF)responseEnvelope.getHeader()).getMessageID().getValue());
			this.addSOAP11Attributes(relatesTo, true);
			this.addWSUIdAttribute(relatesTo, "relHdr");
			header.setRelatesTo(relatesTo);

			// nullify the redirectRequest, as it has supposedly been taken care of
			redirectRequest = null;
		}
	}



	/**
	 * Prints a nice looking output of the request message
	 * 
	 * @param message
	 */
	public static String prettyPrintRequestMessage(WSFMessage message)
	{
		return OpenLibertyHelpers.prettyPrintXMLObject(message.getRequestEnvelope());
	}

	/**
	 * Prints a nice looking output of the response message
	 * 
	 * @param message
	 */
	public static String prettyPrintResponseMessage(WSFMessage message)
	{
		return OpenLibertyHelpers.prettyPrintXMLObject(message.getResponseEnvelope());      
	}



	/**
	 * This method creates the Basic SOAP Envelope
	 * <p>
	 * Read in a SOAP Envelope and unmarshall it
	 * Currently being built by reading in SOAP template file (is)
	 * This may ultimately be built from scratch to improve performance
	 * 
	 * @param is
	 * @return
	 * @throws XMLParserException
	 * @throws UnmarshallingException
	 */
	public static Envelope buildSOAPEnvelope() throws XMLParserException, UnmarshallingException 
	{
		// Construct SOAP element for Envelope using a global template, which looks like this
		Document soapDoc = parser.parse(new StringReader(Konstantz.EMPTY_SOAP_TEMPLATE));

		// Get the DOM element for the document within the SOAP doc
		Element envelopeElem = soapDoc.getDocumentElement();

		// Create envelope java object from DOM element (unmarshall envelope)
		Unmarshaller unmarshaller = XMLObjectSupport.getUnmarshaller(envelopeElem);

		return (Envelope) unmarshaller.unmarshall(envelopeElem);
	}


	public Envelope getRequestEnvelope()
	{
		return envelope;
	}

	public void setRequestEnvelope(Envelope envelope)
	{
		this.envelope = envelope;
	}

	public Envelope getResponseEnvelope()
	{
		return responseEnvelope;
	}

	public void setResponseEnvelope(Envelope envelope)
	{
		this.responseEnvelope = envelope;
	}

	public String getResponseString()
	{
		return response;
	}



	/**
	 * The RedirectRequest element instructs the WSC to redirect the user to the WSP. It is an indication of the WSP that
	 * it cannot service a request made by the WSC before it obtains some more information from the user. If the SOAP 
	 * response contains a Fault with the &lt;RedirectRequest&gt; element, this will return true.
	 * 
	 */
	public boolean hasRedirectRequestBeenIssued()
	{
		return null!=redirectRequest;
	}

	/**
	 * This boolean indicates that there is an updated endpoint reference that needs to be processed
	 * by the service client.
	 * 
	 * @return
	 */
	public boolean hasUpdatedEndpointReference()
	{
		return endpointUpdated;
	}


	static 
	{
		// TODO: what is the cost of instantiating the BasicParserPool
		parser = new BasicParserPool();
		parser.setNamespaceAware(true);
		try {
            parser.initialize();
        } catch (ComponentInitializationException e) {
            throw new XMLRuntimeException("Could not initialize BasicParserPool", e);
        }
		
		// TODO: what is the cost of the factory calls?
		marshallerFactory = XMLObjectProviderRegistrySupport.getMarshallerFactory();
		unmarshallerFactory = XMLObjectProviderRegistrySupport.getUnmarshallerFactory();
		builderFactory = XMLObjectProviderRegistrySupport.getBuilderFactory();		
	}






}
