package org.openliberty.wsc;

import java.io.ByteArrayInputStream;
import java.security.KeyStore;
import java.security.PrivateKey;
import java.security.Provider;
import java.security.PublicKey;
import java.security.cert.Certificate;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Vector;

import javax.xml.crypto.dsig.CanonicalizationMethod;
import javax.xml.crypto.dsig.DigestMethod;
import javax.xml.crypto.dsig.Reference;
import javax.xml.crypto.dsig.SignatureMethod;
import javax.xml.crypto.dsig.SignedInfo;
import javax.xml.crypto.dsig.Transform;
import javax.xml.crypto.dsig.XMLSignature;
import javax.xml.crypto.dsig.XMLSignatureFactory;
import javax.xml.crypto.dsig.dom.DOMSignContext;
import javax.xml.crypto.dsig.keyinfo.KeyInfo;
import javax.xml.crypto.dsig.keyinfo.KeyInfoFactory;
import javax.xml.crypto.dsig.keyinfo.KeyValue;
import javax.xml.crypto.dsig.spec.C14NMethodParameterSpec;
import javax.xml.crypto.dsig.spec.ExcC14NParameterSpec;
import javax.xml.crypto.dsig.spec.TransformParameterSpec;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import net.shibboleth.utilities.java.support.xml.SerializeSupport;

import org.openliberty.xmltooling.Konstantz;
import org.opensaml.core.xml.io.Marshaller;
import org.opensaml.core.xml.io.MarshallingException;
import org.opensaml.core.xml.util.XMLObjectSupport;
import org.opensaml.soap.soap11.Envelope;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

/**
 * This class is used to sign an ID-* message.  
 * 
 * <pre>
543 6.3.2. Message Integrity rules for senders and receivers

544 This section only applies if SOAP message security is used for a message bound to SOAP (i.e., is a "SOAP-bound-ID-*
545 message") according to the Liberty SOAP Binding (v2.0) [LibertySOAPBinding].

546 In this case the sender MUST create a single <ds:Signature> contained in the <wsse:Security> header and this
547 signature MUST reference all of the message components required to be signed.

548 In particular, this signature MUST reference the SOAP Body element (the element itself), the security token associated
549 with the signature, and all headers in the message that have been defined in the Liberty SOAP Bindings specification,
550 including both required and optional header blocks [LibertySOAPBinding].

551 An example security token is a <saml2:Assertion> element conveyed in the <wsse:Security> header.

552 The wsu:Timestamp header in the wsse:Security header block, the wsa:MessageID, wsa:RelatesTo, sb:Framework,
553 sb:Sender and sb:InvocationIdentity header blocks are examples of header elements that would be referenced in a
554 signature.

555 Note that care must be taken when constructing elements contained in Reference Parameters in Endpoint References,
556 as these will be promoted to SOAP header blocks. Effort should be taken to avoid conflicting or duplicate id attributes,
557 for example by using techniques to generate ids where it is highly likely that they are unique.
558 If the message is signed the sender MUST include the resultant XML signature in a <ds:Signature> element as a
559 child of the <wsse:Security> header.

560 The <ds:Signature> element MUST refer to the subject confirmation key with a <ds:KeyInfo> element.

561 The <ds:KeyInfo> element MUST include a <wsse:SecurityTokenReference> element so that the subject
562 confirmation key can be located within the <wsse:Security> header. The inclusion of the reference SHOULD
563 adhere to the guidance specified in section 3.4.2 of [wss-saml11] (section 3.3.2 of [wss-saml]).
 * </pre>
 * 
 * @author asa
 *
 */
public class WSFMessageSigner 
{

    
    /**
     * Takes the requestEnvelope, signs the elements that have been specified in WSFMessage.getSignatureIds(),
     * and returns a signed root element
     * <p>
     * TODO: This method needs to be reworked so that it adds the proper IDs to the header and body elements before signing
     * this would take the burden from the creators of the message and also make it easier to insure that IDs are unique.  It 
     * would also have the effect of keeping the message sizes smaller, and faster if a UUID is used to identify elements that 
     * are referenced by the signature. 
     * </p>
     * @param message
     * @return
     * @throws Exception
     */
    public Element sign(WSFMessage message) throws Exception 
    { 

        // First, create a DOM XMLSignatureFactory that will be used to
        // generate the XMLSignature and marshal it to DOM.
        String providerName = System.getProperty("jsr105Provider", "org.jcp.xml.dsig.internal.dom.XMLDSigRI");
        XMLSignatureFactory xmlSignatureFactory = XMLSignatureFactory.getInstance("DOM", (Provider) Class.forName(providerName).newInstance());

        // Create a Reference to an external URI that will be digested
        // using the SHA1 digest algorithm
        DigestMethod digestMethod = xmlSignatureFactory.newDigestMethod(DigestMethod.SHA1, null);

        List<Transform> transforms = new Vector<Transform>(2);
        transforms.add(xmlSignatureFactory.newTransform("http://www.w3.org/2000/09/xmldsig#enveloped-signature",(TransformParameterSpec)null));

        List<String> prefixlist = new Vector<String>(1);
        prefixlist.add("xsd");
        transforms.add(xmlSignatureFactory.newTransform("http://www.w3.org/2001/10/xml-exc-c14n#",new ExcC14NParameterSpec(prefixlist)));


        List<Reference> references = new ArrayList<Reference>();

        /**
         * Create the Reference info for each of the elements 
         * that are to be signed.
         * 
         */
        for(String id : message.getSignatureIds())
        {
            Reference ref = xmlSignatureFactory.newReference("#"+id, digestMethod, transforms, null, null);
            references.add(ref);
        }

        CanonicalizationMethod canonicalizationMethod = xmlSignatureFactory.newCanonicalizationMethod(CanonicalizationMethod.EXCLUSIVE, (C14NMethodParameterSpec) null);
        //CanonicalizationMethod cm = fac.newCanonicalizationMethod(CanonicalizationMethod.INCLUSIVE_WITH_COMMENTS, (C14NMethodParameterSpec) null);

        SignatureMethod signatureMethod = xmlSignatureFactory.newSignatureMethod(SignatureMethod.RSA_SHA1, null);

        // Create the SignedInfo
        SignedInfo signedInfo = xmlSignatureFactory.newSignedInfo(canonicalizationMethod, signatureMethod, references);
        
        // Extract the private key & certificate from the key store
        KeyStore keyStore = KeyStore.getInstance("JKS");

        // FileInputStream fis = new FileInputStream(OpenLibertyBootstrap.getDefaultSigningPKSPath()); 

        keyStore.load(WSFMessageSigner.class.getResourceAsStream(OpenLibertyBootstrap.getDefaultSigningPKSPath()), OpenLibertyBootstrap.getDefaultSigningPKSPassword().toCharArray());
        
        // grab the private key for signing the XML   
        PrivateKey privateKey = (PrivateKey) keyStore.getKey(OpenLibertyBootstrap.getDefaultSigningPKSAlias(), OpenLibertyBootstrap.getDefaultSigningPKSPassword().toCharArray());
        
        // extract the public certificate for the KeyInfo
        Certificate c = keyStore.getCertificate(OpenLibertyBootstrap.getDefaultSigningPKSAlias());
        PublicKey p = c.getPublicKey();
        KeyInfoFactory keyInfoFactory = xmlSignatureFactory.getKeyInfoFactory();
        KeyValue kv = keyInfoFactory.newKeyValue(p);
        KeyInfo ki = keyInfoFactory.newKeyInfo(Collections.singletonList(kv));
        
        XMLSignature signature = xmlSignatureFactory.newXMLSignature(signedInfo, ki);
        

        // Create the Document that will hold the resulting XMLSignature
        DocumentBuilderFactory documentBuilderFactory = DocumentBuilderFactory.newInstance();
        documentBuilderFactory.setNamespaceAware(true); // must be set
        DocumentBuilder documentBuilder = documentBuilderFactory.newDocumentBuilder();

        Document doc = null; 
        Element securityHeader = null;
        Element timestamp = null;

        try
        {
            Envelope envelope = message.getRequestEnvelope();
            Marshaller envelopeMarshaller = XMLObjectSupport.getMarshaller(Envelope.DEFAULT_ELEMENT_NAME);
            Element envelopeElement = envelopeMarshaller.marshall(envelope);        
                        
            Document messageDoc = envelopeElement.getOwnerDocument();             
            
            String docAsString = SerializeSupport.nodeToString(messageDoc.getFirstChild());
            doc = documentBuilder.parse(new ByteArrayInputStream(docAsString.getBytes()));
                       
            // Get the Security Header, where the signature will be placed
            NodeList nodeList = doc.getElementsByTagNameNS(Konstantz.WSSE_NS, "Security");
            securityHeader = (Element)nodeList.item(0);
            
            nodeList = securityHeader.getElementsByTagNameNS(Konstantz.WSU_NS, "Timestamp");
            timestamp = (Element)nodeList.item(0);
            
        } 
        catch (MarshallingException e)
        {
            e.printStackTrace();
        }      
        

        // Now create the context and sign the document.  The document is pulled based on the
        // assertion_node and subject_node which indicate where the signature will be placed
        // The assertion_node is the Parent node to the signature element and the subject_node
        // is the next Sibling of the signature element.
        DOMSignContext signContext = new DOMSignContext(privateKey, securityHeader, timestamp); 
        signContext.putNamespacePrefix("http://www.w3.org/2000/09/xmldsig#", "ds");
        signContext.putNamespacePrefix("http://www.w3.org/2001/10/xml-exc-c14n#", "ec");

        
        // Marshal, generate (and sign) the detached XMLSignature. The DOM
        // Document will contain the XML Signature if this method returns
        // successfully.
        // HIERARCHY_REQUEST_ERR: Raised if this node is of a type that does not allow children of the type of the newChild  node, or if the node to insert is one of this node's ancestors.
        signature.sign(signContext);
                
        return doc.getDocumentElement();
        
    }




}




