package org.openliberty.xmltooling;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.util.Scanner;

import javax.xml.namespace.QName;

import net.shibboleth.utilities.java.support.primitive.StringSupport;
import net.shibboleth.utilities.java.support.xml.BasicParserPool;
import net.shibboleth.utilities.java.support.xml.SerializeSupport;
import net.shibboleth.utilities.java.support.xml.XMLParserException;

import org.apache.commons.lang.math.NumberUtils;
import org.apache.log4j.Logger;
import org.joda.time.DateTime;
import org.joda.time.chrono.ISOChronology;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.opensaml.core.xml.AbstractXMLObject;
import org.opensaml.core.xml.XMLObject;
import org.opensaml.core.xml.XMLObjectBuilder;
import org.opensaml.core.xml.XMLObjectBuilderFactory;
import org.opensaml.core.xml.config.XMLObjectProviderRegistrySupport;
import org.opensaml.core.xml.io.Marshaller;
import org.opensaml.core.xml.io.MarshallerFactory;
import org.opensaml.core.xml.io.MarshallingException;
import org.opensaml.core.xml.io.Unmarshaller;
import org.opensaml.core.xml.io.UnmarshallerFactory;
import org.opensaml.core.xml.io.UnmarshallingException;
import org.opensaml.core.xml.util.XMLObjectSupport;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.google.common.base.Objects;

/**
 * A group of methods that make it easier to build the tooling classes
 * by modeling some string to object and visa-versa, providing some
 * DOM clearing by modeling special cases of prepareForAssigment, 
 * cloning objects from the DOM to prevent exceptions that occur when 
 * you attempt to re-use opensaml modeled tooling objects, and 
 * some basic XML utilities.
 * 
 * @author asa
 *
 */
public class OpenLibertyHelpers
{
    private static Logger log = Logger.getLogger(OpenLibertyHelpers.class);

    public static String FALSE_STR = "0";
    public static String TRUE_STR = "1";
    
    private static String defaultDateFormat = "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'";  
    private static DateTimeFormatter defaultDateFormatter;


    /** XMLObject marshaller factory. */
    protected static MarshallerFactory marshallerFactory;

    /** XMLObject marshaller factory. */
    protected static UnmarshallerFactory unmarshallerFactory;

    /** Parser pool */
    protected static BasicParserPool parserPool;

    /** XMLObject builder factory */
    protected static XMLObjectBuilderFactory builderFactory;

    public static XMLObject createXMLObjectForQName(QName qname)
    {
        XMLObject object = null;
        
        XMLObjectBuilder<?> builder = XMLObjectSupport.getBuilder(qname);
        
        if (builder == null) 
        {
        	log.error("Unable to retrieve builder for object QName " + qname);
        }
        else
        {
            object = builder.buildObject(qname.getNamespaceURI(), qname.getLocalPart(), qname.getPrefix());
        }
        
        return object;
    }


    /**
     * Attempt to convert a DateTime object into a String
     * 
     * @param dateTime
     * @return
     */
    public static String stringForDateTime(DateTime dateTime)
    {
        if(null!=dateTime)
        {
            return getDefaultDateFormatter().print(dateTime);  
        }
        else 
        {
            return "";
        }
    }

    /**
     * Attempt to convert a String into a DateTime object
     * 
     * @param dateTimeString
     * @return
     */
    public static DateTime dateTimeForString(String dateTimeString)
    {
        if(dateTimeString!=null)
        {            
            try
            {
                return getDefaultDateFormatter().parseDateTime(dateTimeString);
            }
            catch(IllegalArgumentException e)
            {
                log.error(e);
            }
        }

        return null;
    }

    /**
     * Safely create an Integer from a String
     * 
     * @param integerString
     * @return
     */
    public static Integer integerFromString(String integerString)
    {
        if(NumberUtils.isNumber(integerString))
        {
            return NumberUtils.createInteger(integerString);
        }

        return null;
    }

    /**
     * Safely create a positive Integer from a String
     * 
     * @param integerString
     * @return
     */
    public static Integer positiveIntegerFromString(String integerString, Integer defaultValue)
    {
        if(NumberUtils.isNumber(integerString))
        {
            Integer result = NumberUtils.createInteger(integerString);
            if(result>-1) return result;            
        }

        return defaultValue;
    }

    /**
     * Safely create a Boolean from a String
     * 
     * @param booleanString
     * @return
     */
    public static Boolean booleanFromString(String booleanString)
    {
        if(null!=booleanString)
        {
        	booleanString = booleanString.toLowerCase();
            if(booleanString.equals(FALSE_STR) || booleanString.equals("false")) return Boolean.FALSE;
            else if(booleanString.equals(TRUE_STR) || booleanString.equals("true")) return Boolean.TRUE;
        }

        return null;
    }

    public static String stringFromBoolean(Boolean b, String defaultString)
    {
        if(null==b) return defaultString;
        else return (b) ? TRUE_STR : FALSE_STR;
    }


    /**
     * Use this utility method to reset the DOM if required based on a 
     * change to the objects (element value or attribute value)
     * 
     * @param oldValue
     * @param newValue
     * @param abstractXMLObject
     * @return
     */
    public static String prepareForAssignment(String oldStringValue, String newStringValue, AbstractXMLObject abstractXMLObject) 
    {
        String newString = StringSupport.trimOrNull(newStringValue);

        if (!Objects.equal(oldStringValue, newString)) 
        {
            abstractXMLObject.releaseThisandParentDOM();
        }

        return newString;
    }


    /**
     * Compares two objects.  Use this utility method for comparing enums or other objects
     * 
     * @param oldValue
     * @param newValue
     * @param abstractXMLObject
     * @return
     */
    public static <T>T prepareForAssignment(T oldValue, T newValue, AbstractXMLObject abstractXMLObject) 
    {        
        if (!Objects.equal(oldValue, newValue)) 
        {
            abstractXMLObject.releaseThisandParentDOM();
        }
        return newValue;
    }


    /**
     * Compares DateTime objects looking for change, using the specified DateTimeFormatter.
     * If no DateTimeFormatter is specified, the test is done on the DateTime objects themselves.
     * 
     * 
     * @param oldValue 
     * @param newValue 
     * @param formatter optional, used to compare the eventual xml output based on formatting
     * @param abstractXMLObject required and must be Non null
     * @return
     */
    public static DateTime prepareForAssignment(DateTime oldValue, DateTime newValue, DateTimeFormatter formatter, AbstractXMLObject abstractXMLObject)
    {
        if(null!=oldValue && null!=newValue)
        {
            if(null!=formatter)
            {
                if(!formatter.print(oldValue).equals(formatter.print(newValue)))
                {
                    abstractXMLObject.releaseThisandParentDOM();
                }
            }
            else if(0!=oldValue.compareTo(newValue))
            {
                abstractXMLObject.releaseThisandParentDOM();
            }
        }
        // basically testing whether both are == null
        else if(oldValue!=newValue)
        {
            abstractXMLObject.releaseThisandParentDOM();
        }

        return newValue;
    }

    /**
     * Compares Integer values, releasing the DOM when oldValue and newValue
     * are not equivalent
     * 
     * @param oldValue
     * @param newValue
     * @param abstractXMLObject
     * @return
     */
    public static Integer prepareForAssignment(Integer oldValue, Integer newValue, AbstractXMLObject abstractXMLObject)
    {
        // test for inequality and for null
        if(((null!=oldValue && null!=newValue) && newValue.intValue()!=oldValue.intValue()) || oldValue!=newValue)
        {
            abstractXMLObject.releaseThisandParentDOM();
        }

        return newValue;
    }


    /**
     * Returns a joda DateTimeFormatter for parsing and printing a DateTime
     * 
     * @return 
     */
    public static DateTimeFormatter getDefaultDateFormatter() 
    {
        if(null==defaultDateFormatter) 
        {
            DateTimeFormatter formatter = DateTimeFormat.forPattern(defaultDateFormat);
            defaultDateFormatter = formatter.withChronology(ISOChronology.getInstanceUTC());
        }
        return defaultDateFormatter;
    }

    /**
     * Clone an XMLObject
     * 
     * @param xmlObject
     * @return
     */
    public static XMLObject cloneXMLObject(XMLObject xmlObject)
    {
        XMLObject clonedXMLObject = null;

        if(null!=xmlObject)
        {
            try
            {
                Marshaller marshaller = marshallerFactory.getMarshaller(xmlObject);
                Element ele = marshaller.marshall(xmlObject);
                ele = (Element)ele.cloneNode(true);
                Unmarshaller u = unmarshallerFactory.getUnmarshaller(ele);
                clonedXMLObject = u.unmarshall(ele);
            }
            catch (UnmarshallingException e)
            {
                e.printStackTrace();
            }
            catch (MarshallingException e)
            {
                e.printStackTrace();
            }
        }

        return clonedXMLObject;
    }


    /**
     * Takes an XMLObject as a parameter, marshalls it, and returns a 
     * pretty printed version of the generated XML.
     * 
     * @param xmlObject
     * @return
     */
    public static String prettyPrintXMLObject(XMLObject xmlObject)
    {
        if(null!=xmlObject)
        {
            Marshaller m = marshallerFactory.getMarshaller(xmlObject);
            try
            {
                Element ele = m.marshall(xmlObject);
                return SerializeSupport.prettyPrintXML(ele);
            } 
            catch (MarshallingException e)
            {
                e.printStackTrace();
            }            
        }
        return "";
    }


    /**
     * Takes a list of objects and allows the user to specify a
     * selection by number.  This is only meant for use in a 
     * console/testing environment
     * 
     * @param list
     * @param name
     * @return
     */
    public static <E>E selectFromAList(E[] list, String name)
    {
        Scanner in = new Scanner(System.in);

        System.out.println("\n\n"+name+"\n"); 

        int i=0;

        for(E obj : list)
        { 
            System.out.println(" ("+(i++)+") "+obj.toString()); 
        } 

        System.out.println(" ("+(i++)+") to select none\n"); 
        System.out.print(" $ "); 

        // read in the selection
        int w = in.nextInt(); 
        // in.close();

        if(w<0 || w>list.length-1)
        {
            return null;
        }
        else
        {
            return list[w];
        }
    }






    public static XMLObject unmarshallElementFromString(String inputString) 
    {
        try 
        {

            InputStream is = new ByteArrayInputStream(inputString.getBytes());
            if(null==is)
            {               
                log.debug("Failed to create a stream from: "+inputString);
                return null;
            }
            Document doc = parserPool.parse(is);
            Element ele = doc.getDocumentElement();
            Unmarshaller unmarshaller = XMLObjectSupport.getUnmarshaller(ele);
            if(unmarshaller == null) 
            {
                log.debug("Unable to retrieve unmarshaller by DOM Element");
                return null;
            }

            return unmarshaller.unmarshall(ele);
        }
        catch (XMLParserException e)
        {
            e.printStackTrace();
        }
        catch (UnmarshallingException e)
        {
            e.printStackTrace();
        }

        return null;
    }


    /**
     * Unmarshalls an element file into its XMLObject.
     * 
     * @return the XMLObject from the file
     */
    public static XMLObject unmarshallElementFromFile(File elementFile) 
    {
        try 
        {

            InputStream is = new FileInputStream(elementFile);
            if(null==is)
            {               
                log.debug("Input Stream is null. Unable to load "+elementFile.getPath());
                return null;
            }

            Document doc = parserPool.parse(is);
            Element ele = doc.getDocumentElement();
            Unmarshaller unmarshaller = XMLObjectSupport.getUnmarshaller(ele);
            if(unmarshaller == null) 
            {
                log.debug("Unable to retrieve unmarshaller by DOM Element");
                return null;
            }

            return unmarshaller.unmarshall(ele);
        } 
        catch (XMLParserException e) 
        {
            System.err.println("Unable to parse element file " + elementFile.getPath());
        } 
        catch (UnmarshallingException e) 
        {
            System.err.println("Unmarshalling failed when parsing element file " + elementFile.getPath() + ": " + e);
        }
        catch (FileNotFoundException e)
        {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }

        return null;
    }



    /**
     * Unmarshalls an element file into its XMLObject.
     * 
     * @return the XMLObject from the file
     */
    protected static XMLObject unmarshallElementFromFile(String elementFileName) 
    {
        try 
        {

            InputStream is = OpenLibertyHelpers.class.getResourceAsStream(elementFileName);
            if(null==is)
            {               
                log.debug("Input Stream is null. Unable to load "+elementFileName);
                return null;
            }

            Document doc = parserPool.parse(is);
            Element ele = doc.getDocumentElement();
            Unmarshaller unmarshaller = XMLObjectSupport.getUnmarshaller(ele);
            if(unmarshaller == null) 
            {
                log.debug("Unable to retrieve unmarshaller by DOM Element");
                return null;
            }

            return unmarshaller.unmarshall(ele);


        } 
        catch (XMLParserException e) 
        {
            System.err.println("Unable to parse element file " + elementFileName);
        } 
        catch (UnmarshallingException e) 
        {
            System.err.println("Unmarshalling failed when parsing element file " + elementFileName + ": " + e);
        }

        return null;
    }


    static 
    {
        try 
        {
            parserPool = new BasicParserPool();
            parserPool.setNamespaceAware(true);
            builderFactory = XMLObjectProviderRegistrySupport.getBuilderFactory();
            marshallerFactory = XMLObjectProviderRegistrySupport.getMarshallerFactory();
            unmarshallerFactory = XMLObjectProviderRegistrySupport.getUnmarshallerFactory();
        } 
        catch (Exception e) 
        {
            System.err.println("Can not initialize GeneralUtilities" + e);
            e.printStackTrace();
        }
    }




}
