package org.openliberty.xmltooling.dst2_1;

import org.joda.time.DateTime;
import org.openliberty.xmltooling.OpenLibertyHelpers;
import org.opensaml.core.xml.AbstractXMLObject;
import org.w3c.dom.Attr;
import org.w3c.dom.Element;

public class ModifyItemAttributeGroup extends SelectQualifAttributes
{

    public static final String NOT_CHANGED_SINCE = "notChangedSince";
    public static final String OVERRIDE_ALLOWED = "overrideAllowed";
    public static final String ID = "id";
    public static final String ITEM_ID = "itemID";

    private DateTime notChangedSince;
    private Boolean overrideAllowed;
    private String id;
    private String itemID;


    /**
     * This is a convenience method used by any Marshaller that is marshalling 
     * an object that has ModifyItemAttributeGroup
     *
     */
    @Override
	public void marshallAttributes(Element domElement)
    {

        // Marshall the SelectQualifAttributes
        super.marshallAttributes(domElement);

        if(notChangedSince!=null)
        {
            domElement.setAttributeNS(null, NOT_CHANGED_SINCE, OpenLibertyHelpers.stringForDateTime(notChangedSince));
        }

        if(overrideAllowed!=null)
        {
            domElement.setAttributeNS(null, OVERRIDE_ALLOWED, OpenLibertyHelpers.stringFromBoolean(overrideAllowed, OpenLibertyHelpers.FALSE_STR));
        }

        if(id!=null) 
        {
            domElement.setAttributeNS(null, ID, id);
            domElement.setIdAttributeNS(null, ID, true);
        }

        if(itemID!=null) 
        {
            domElement.setAttributeNS(null, ITEM_ID, itemID);
        }

    }

    /**
     * This method assists the unmarshaller, adding an XML Attribute to 
     * the ModifyItemAttributeGroup object
     *     
     * @param attribute the attribute to set
     * 
     * @return whether an attribute was found and set
     */
    @Override
	public boolean processAttribute(Attr attribute, AbstractXMLObject abstractXMLObject)
    {

        if(!super.processAttribute(attribute, abstractXMLObject))
        {
            if(attribute.getLocalName().equals(ID)) 
            {
                this.setId(attribute.getValue(), abstractXMLObject);
                attribute.getOwnerElement().setIdAttributeNode(attribute, true);
            }
            else if(attribute.getLocalName().equals(NOT_CHANGED_SINCE))
            {            
                this.setNotChangedSince(OpenLibertyHelpers.dateTimeForString(attribute.getValue()), abstractXMLObject);
            }
            else if(attribute.getLocalName().equals(OVERRIDE_ALLOWED)) 
            {
                this.setOverrideAllowed(OpenLibertyHelpers.booleanFromString(attribute.getValue()), abstractXMLObject);
            }
            else if(attribute.getLocalName().equals(ITEM_ID))
            {            
                this.setItemID(attribute.getValue(), abstractXMLObject);
            }
            // Attribute Not Found
            else 
            {
                return false;
            }
        }
        
        // Attribute Was Found
        return true;

    }






    public void setNotChangedSince(DateTime notChangedSince, AbstractXMLObject abstractXMLObject)
    {
        this.notChangedSince = OpenLibertyHelpers.prepareForAssignment(this.notChangedSince, notChangedSince, OpenLibertyHelpers.getDefaultDateFormatter(), abstractXMLObject);     
    }

    public DateTime getNotChangedSince()
    {
        return notChangedSince;
    }

    public void setOverrideAllowed(Boolean overrideAllowed, AbstractXMLObject abstractXMLObject)
    {
        this.overrideAllowed = OpenLibertyHelpers.prepareForAssignment(this.overrideAllowed, overrideAllowed, abstractXMLObject);
    }

    public Boolean getOverrideAllowed()
    {
        return this.overrideAllowed;
    }

    public void setId(String id, AbstractXMLObject abstractXMLObject)
    {
        this.id = OpenLibertyHelpers.prepareForAssignment(this.id, id, abstractXMLObject);
    }

    public String getId()
    {
        return this.id;
    }

    public void setItemID(String itemID, AbstractXMLObject abstractXMLObject)
    {
        this.itemID = OpenLibertyHelpers.prepareForAssignment(this.itemID, itemID, abstractXMLObject);
    }

    public String getItemID()
    {
        return this.itemID;
    }

}
/*
  <xs:attributeGroup name="ModifyItemAttributeGroup">
    <xs:attributeGroup ref="dst:selectQualif"/>
    <xs:attribute name="notChangedSince" use="optional" type="xs:dateTime"/>
    <xs:attribute name="overrideAllowed" use="optional" type="xs:boolean" default="0"/>
    <xs:attribute name="id" use="optional" type="xs:ID"/>
    <xs:attribute ref="lu:itemID" use="optional"/>
  </xs:attributeGroup>
 */