package org.openliberty.xmltooling.idsis.dap;

import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Map.Entry;
import java.util.UUID;

import javax.xml.namespace.QName;

import net.shibboleth.utilities.java.support.xml.AttributeSupport;
import net.shibboleth.utilities.java.support.xml.QNameSupport;

import org.openliberty.xmltooling.Konstantz;
import org.openliberty.xmltooling.dst2_1.RequestType;
import org.openliberty.xmltooling.utility_2_0.Extension;
import org.opensaml.core.xml.AbstractXMLObjectBuilder;
import org.opensaml.core.xml.XMLObject;
import org.opensaml.core.xml.config.XMLObjectProviderRegistrySupport;
import org.opensaml.core.xml.io.AbstractXMLObjectMarshaller;
import org.opensaml.core.xml.io.AbstractXMLObjectUnmarshaller;
import org.opensaml.core.xml.io.MarshallingException;
import org.opensaml.core.xml.io.UnmarshallingException;
import org.opensaml.core.xml.util.XMLObjectChildrenList;
import org.w3c.dom.Attr;
import org.w3c.dom.Element;

/**
 * 
 * 
 * <pre>
 * &lt;xs:complexType name="CreateType"&gt;
 *  &lt;xs:complexContent&gt;
 *   &lt;xs:extension base="dst:RequestType"&gt;
 *     &lt;xs:sequence&gt;
 *      &lt;xs:element ref="dap:Subscription"    minOccurs="0" maxOccurs="unbounded"/&gt;
 *      &lt;xs:element ref="dap:CreateItem"  minOccurs="1" maxOccurs="unbounded"/&gt;
 *      &lt;xs:element ref="dap:ResultQuery" minOccurs="0" maxOccurs="unbounded"/&gt;
 *     &lt;/xs:sequence&gt;
 *   &lt;/xs:extension&gt;
 *  &lt;/xs:complexContent&gt;
 * &lt;/xs:complexType&gt;
 * </pre>
 * 
 * @author asa
 *
 */
public class DAPCreate extends RequestType
{

    public static String LOCAL_NAME = "Create";

    // Elements
    private XMLObjectChildrenList<DAPSubscription> subscriptions;
    private XMLObjectChildrenList<DAPCreateItem> createItems;
    private XMLObjectChildrenList<DAPResultQuery> resultQueries;

    /**
     * Constructor that insures the minimum requirement of 1 CreateItem
     * element
     * 
     * @param createItem
     */
    public DAPCreate(DAPCreateItem createItem)
    {
        super(Konstantz.DAP_NS, LOCAL_NAME, Konstantz.DAP_PREFIX);

        createItems = new XMLObjectChildrenList<DAPCreateItem>(this);
        createItems.add(createItem);   
    }

    /**
     * Constructor that takes an LDIF string meant to be
     * added to the LDAP directory and builds the appropriate
     * and simplest element construction to request the add 
     * through ID-DAP:
     * 
     * 
     * @param ldifCreateString
     */
    public DAPCreate(String ldifCreateString, String itemId)
    {

        super(Konstantz.DAP_NS, LOCAL_NAME, Konstantz.DAP_PREFIX);

        if(null==itemId)
        {
            itemId = UUID.randomUUID().toString();            
        }
        
        setItemID(itemId);
        
        DAPCreateItem createItem = new DAPCreateItem();
        createItem.getCreateItemAttributeGroup().setItemID(UUID.randomUUID().toString(), createItem);

        LDIF ldif = new LDIF();
        ldif.setValue(ldifCreateString);        

        DAPNewData newData =  new DAPNewData();        
        newData.setLDIF(ldif);

        createItem.setNewData(newData);

        createItems = new XMLObjectChildrenList<DAPCreateItem>(this);
        createItems.add(createItem);   
    }

    /**
     * Constructor that takes an LDIF string meant to be
     * added to the LDAP directory and builds the appropriate
     * and simplest element construction to request the add 
     * through ID-DAP.  It also takes a NameSpace string.  This 
     * was built because symlabs uses a slightly different NS
     * than is defined in the spec for their DST 2.1 ID-SIS-DAP
     * results.
     * 
     * @param ldifCreateString
     * @param NS
     *
    public DAPCreate(String ldifCreateString, String NS, String itemId)
    {
        super(NS, LOCAL_NAME, Konstantz.DAP_PREFIX);

        setItemID(UUID.randomUUID().toString());

        DAPCreateItem createItem = new DAPCreateItem(NS, DAPCreateItem.LOCAL_NAME, Konstantz.DAP_PREFIX);
        createItem.getCreateItemAttributeGroup().setItemID(UUID.randomUUID().toString(), createItem);

        LDIF ldif = new LDIF(NS, LDIF.LOCAL_NAME, Konstantz.DAP_PREFIX);
        ldif.setValue(ldifCreateString);

        DAPNewData newData =  new DAPNewData(NS, DAPNewData.LOCAL_NAME, Konstantz.DAP_PREFIX);
        newData.setLDIF(ldif);

        createItem.setNewData(newData);

        createItems = new XMLObjectChildrenList<DAPCreateItem>(this);
        createItems.add(createItem);   
    }
     */


    protected DAPCreate(String namespaceURI, String elementLocalName, String namespacePrefix)
    {
        super(namespaceURI, elementLocalName, namespacePrefix);
        createItems = new XMLObjectChildrenList<DAPCreateItem>(this);
    }

    public XMLObjectChildrenList<DAPSubscription> getDAPSubscriptions()
    {
        if(null==subscriptions) subscriptions = new XMLObjectChildrenList<DAPSubscription>(this);
        return subscriptions;
    }

    public XMLObjectChildrenList<DAPCreateItem> getDAPCreateItems()
    {
        return createItems;
    }

    public XMLObjectChildrenList<DAPResultQuery> getDAPResultQueries()
    {
        if(null==resultQueries) resultQueries = new XMLObjectChildrenList<DAPResultQuery>(this);
        return resultQueries;
    }

    @Override
	public List<XMLObject> getOrderedChildren()
    {        
        List<XMLObject> children = new LinkedList<XMLObject>();

        // Add the super kids
        children.addAll(super.getOrderedChildren()); 

        // now add the local kids
        if(null!=subscriptions) children.addAll(subscriptions); 
        if(null!=createItems) children.addAll(createItems); 
        if(null!=resultQueries) children.addAll(resultQueries); 

        return Collections.unmodifiableList(children);
    }



    /**
     * Marshall attributes into the DOM
     * 
     * @author asa
     *
     */
    public static class Marshaller extends AbstractXMLObjectMarshaller
    {

        @Override
        protected void marshallAttributes(XMLObject xmlObject, Element domElement) throws MarshallingException
        {
            DAPCreate obj = (DAPCreate)xmlObject;

            if(obj.getItemID() != null) 
            {
                domElement.setAttributeNS(null, RequestType.ATT_ITEM_ID, obj.getItemID());
            } 

            // marshall the unknown attributes
            Attr attr;
            for(Entry<QName, String> entry: obj.getUnknownAttributes().entrySet())
            {
                attr = AttributeSupport.constructAttribute(domElement.getOwnerDocument(), entry.getKey());
                attr.setValue(entry.getValue());
                domElement.setAttributeNodeNS(attr);
                if (XMLObjectProviderRegistrySupport.isIDAttribute(entry.getKey()) || obj.getUnknownAttributes().isIDAttribute(entry.getKey())) 
                {
                    attr.getOwnerElement().setIdAttributeNode(attr, true);
                }
            }

        }


        @Override
        protected void marshallElementContent(XMLObject xmlObject, Element domElement) throws MarshallingException 
        {
            // no element content
        }



    }




    /**
     * Internal class that handles unmarshalling from the DOM
     * 
     * @author asa
     *
     */
    public static class Unmarshaller extends AbstractXMLObjectUnmarshaller
    {       

        @Override
        protected void processChildElement(XMLObject parentXMLObject, XMLObject childXMLObject) throws UnmarshallingException
        {            
            DAPCreate obj = (DAPCreate) parentXMLObject;

            if(childXMLObject instanceof DAPResultQuery)
            {
                obj.getDAPResultQueries().add((DAPResultQuery)childXMLObject);                   
            }
            else if(childXMLObject instanceof DAPCreateItem)
            {
                obj.getDAPCreateItems().add((DAPCreateItem)childXMLObject);                   
            }
            else if(childXMLObject instanceof DAPSubscription)
            {
                obj.getDAPSubscriptions().add((DAPSubscription)childXMLObject);                   
            }
            else if(childXMLObject instanceof Extension)
            {
                obj.getExtensions().add((Extension)childXMLObject);                   
            }
        }

        @Override
        protected void processAttribute(XMLObject xmlObject, Attr attribute) throws UnmarshallingException
        {
            DAPCreate obj = (DAPCreate) xmlObject;

            if (attribute.getLocalName().equals(RequestType.ATT_ITEM_ID)) 
            {
                obj.setItemID(attribute.getValue());
            } 
            // xs:anyAttribute
            else 
            {
                QName attribQName = QNameSupport.getNodeQName(attribute);
                if (attribute.isId()) 
                {
                    obj.getUnknownAttributes().registerID(attribQName);
                }
                obj.getUnknownAttributes().put(attribQName, attribute.getValue());
            }                           
        }

        @Override
        protected void processElementContent(XMLObject xmlObject, String elementContent)
        {
            // no element content            
        }

    }




    /**
     * Internal class for the builder to simplify the class structure
     * 
     * @author asa
     *
     */
    public static class Builder extends AbstractXMLObjectBuilder<DAPCreate> 
    {

        @Override
		public DAPCreate buildObject(String namespaceURI, String localName, String namespacePrefix) 
        {
            return new DAPCreate(namespaceURI, localName, namespacePrefix);
        }

    }



}
