package org.openliberty.xmltooling.idsis.dap;

import java.util.Hashtable;

import net.shibboleth.utilities.java.support.xml.ElementSupport;

import org.apache.log4j.Logger;
import org.openliberty.xmltooling.Konstantz;
import org.openliberty.xmltooling.dst2_1.PaginationResponseAttributeGroup;
import org.opensaml.core.xml.AbstractXMLObjectBuilder;
import org.opensaml.core.xml.XMLObject;
import org.opensaml.core.xml.io.MarshallingException;
import org.opensaml.core.xml.io.UnmarshallingException;
import org.w3c.dom.Attr;
import org.w3c.dom.Element;


/**
 * DAPData represents a Data element in the DAP QueryResponse. This object
 * is {@link DAPItemData} with pagination.
 * <p>
 * <pre>
 *   &lt;xs:element name="Data" type="dap:DataType"/&gt;
 *   &lt;xs:complexType name="DataType"&gt;
 *     &lt;xs:complexContent&gt;
 *       &lt;xs:extension base="dap:ItemDataType"&gt;
 *         &lt;xs:attributeGroup ref="dst:PaginationResponseAttributeGroup"/&gt;
 *       &lt;/xs:extension&gt;
 *     &lt;/xs:complexContent&gt;
 *   &lt;/xs:complexType&gt;
 *  </pre>
 *  </p>
 *  <p>
 *  In practice the element content of the Data element may contain the LDIF 
 *  string data.  So a convenience method getLDIFDataString() will extract 
 *  the LDIF data forst checking for the LDIF element and then for the Subscription 
 *  element, and then passing &lt;Data&gt;s element content
 *  </p>
 *  
 *  
 * @author asa
 *
 */
public class DAPData extends DAPItemData
{
    private static final Logger log = Logger.getLogger(DAPData.class);

    // Attributes
    private PaginationResponseAttributeGroup paginationResponseAttributeGroup;

    // Element Content
    private String value;

    public DAPData()
    {
        super(Konstantz.DAP_NS, LOCAL_NAME, Konstantz.DAP_PREFIX);
        paginationResponseAttributeGroup = new PaginationResponseAttributeGroup();
        if(log.isDebugEnabled()) log.debug("new DAPData()");
    }

    public DAPData(String namespaceURI, String elementLocalName, String namespacePrefix)
    {
        super(namespaceURI, elementLocalName, namespacePrefix);
        paginationResponseAttributeGroup = new PaginationResponseAttributeGroup();
        if(log.isDebugEnabled()) log.debug("new DAPData("+namespaceURI+", "+elementLocalName+", "+namespacePrefix+")");
    }

    /**
     * Extracts the LDIF data forst checking for the LDIF element and then for the Subscription 
     * element, and then passing &lt;Data&gt;s element content
     * 
     * @return
     */
    public String getLDIFDataString()
    {
        if(null==getSubscription())
        {
            if(null!=getLDIF())
            {
                return getLDIF().getValue();
            }
            else 
            {
                return value;
            }
        }
        return null;
    }

    public Hashtable<String, String> getLDIFDataAsHashtable()
    {
        Hashtable<String, String> hashedData = new Hashtable<String, String>();

        String data = getLDIFDataString();
        if(null!=data)
        {
            String[] attributes = data.split("\\n");
            for(String attribute: attributes)
            {
                int colon = attribute.indexOf(':');
                if(colon>1)
                {
                    hashedData.put(attribute.substring(0,colon), (attribute.length()>colon?attribute.substring(colon+1).trim():""));
                }
            }
        }
        return hashedData;
    }

    /**
     * Data is specified in the ID-SIS-DAP as a complextype and 
     * as such has no element content.  This however is not true 
     * in practice, ergo, {@link DAPData} handles the element 
     * content.
     * 
     * @param value
     */
    public void setValue(String value)
    {
        this.value = prepareForAssignment(this.value, value);
    }

    public String getValue()
    {
        return value;
    }

    public PaginationResponseAttributeGroup getPaginationResponseAttributes()
    {
        return paginationResponseAttributeGroup;
    }

    /**
     * Internal marshaller
     * 
     * @author asa
     *
     */
    public static class Marshaller extends DAPItemData.Marshaller
    {
        @Override
        protected void marshallAttributes(XMLObject xmlObject, Element domElement) throws MarshallingException
        {
            DAPData obj = (DAPData)xmlObject;
            obj.getPaginationResponseAttributes().marshallAttributes(domElement);
            super.marshallAttributes(xmlObject, domElement);
        }

        @Override
        protected void marshallElementContent(XMLObject xmlObject, Element domElement) throws MarshallingException
        {
            DAPData obj = (DAPData) xmlObject;
            ElementSupport.appendTextContent(domElement, obj.getValue());
        }

    }

    /**
     * Internal unmarshaller
     * 
     * @author asa
     *
     */
    public static class Unmarshaller extends DAPItemData.Unmarshaller
    {
        @Override
        protected void processAttribute(XMLObject xmlObject, Attr attribute) throws UnmarshallingException
        {
            DAPData obj = (DAPData) xmlObject;
            if(!obj.getPaginationResponseAttributes().processAttribute(attribute, obj))
            {
                super.processAttribute(xmlObject, attribute);
            }
        }  

        @Override
        protected void processElementContent(XMLObject xmlObject, String elementContent)
        {
            DAPData obj = (DAPData) xmlObject;
            obj.setValue(elementContent);
        }
    }    

    /**
     * Internal Class that builds a DAPData object. 
     * 
     * @author asa
     *
     */
    public static class Builder extends AbstractXMLObjectBuilder<DAPData> 
    {

        public DAPData buildObject()
        {
            return new DAPData(Konstantz.DAP_NS, LOCAL_NAME, Konstantz.DAP_PREFIX);
        }

        @Override
		public DAPData buildObject(String namespaceURI, String localName, String namespacePrefix) 
        {
            return new DAPData(namespaceURI, localName, namespacePrefix);
        }

    }


}
