package org.openliberty.xmltooling.idsis.dap;

import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.UUID;

import org.openliberty.xmltooling.Konstantz;
import org.openliberty.xmltooling.dst2_1.RequestType;
import org.opensaml.core.xml.AbstractXMLObjectBuilder;
import org.opensaml.core.xml.XMLObject;
import org.opensaml.core.xml.io.UnmarshallingException;
import org.opensaml.core.xml.util.XMLObjectChildrenList;

/**
 * This class is based on the DST 2.1 Reference implementation Query
 * but adds Subscriptions 
 * <pre>
 *   &lt;xs:complexType name="QueryType"&gt;
 *     &lt;xs:complexContent&gt;
 *       &lt;xs:extension base="dst:RequestType"&gt;
 *         &lt;xs:sequence&gt;
 *           &lt;xs:element ref="dap:TestItem"  minOccurs="0" maxOccurs="unbounded"/&gt;
 *           &lt;xs:element ref="dap:QueryItem" minOccurs="0" maxOccurs="unbounded"/&gt;
 *           &lt;xs:element ref="dap:Subscription"  minOccurs="0" maxOccurs="unbounded"/&gt;
 *         &lt;/xs:sequence&gt;
 *       &lt;/xs:extension&gt;
 *     &lt;/xs:complexContent&gt;
 *   &lt;/xs:complexType&gt;
 * </pre>
 * @author asa
 *
 */
public class DAPQuery extends RequestType
{
    
    public static String LOCAL_NAME = "Query";
        
    // Elements
    private XMLObjectChildrenList<DAPTestItem> testItems;
    private XMLObjectChildrenList<DAPQueryItem> queryItems;
    private XMLObjectChildrenList<DAPSubscription> subscriptions;
    
    
    public DAPQuery()
    {
        super(Konstantz.DAP_NS, LOCAL_NAME, Konstantz.DAP_PREFIX);
    }
    
    
    
    /**
     * Convenience method to generate a simple entity attribute Select.
     * <p>
     * In the ID-SIS-DAP world the DN is usually gleaned from the identity
     * information provided by the WSF layer.  Therefore it is likely that 
     * a query based on an attribute will be created with the DN implied.
     * </p>
     * <p>
     * This method will generate a Query that looks like this:
     * <pre>
     * 
     * </pre>
     * 
     * @param selectAttributes
     * @param itemId
     * @return
     */
    public static DAPQuery entityQueryWithSelectAttributes(String selectAttributes, String itemId)
    {
        if(null==itemId)
        {
            itemId = UUID.randomUUID().toString();            
        }        
        
        // Create the Select
        DAPSelect select = new DAPSelect();
        select.setAttributes(selectAttributes);
        
        // Create the QueryItem
        DAPQueryItem queryItem = new DAPQueryItem();        
        queryItem.setObjectType(DAPQueryItem.ObjectType.ENTRY);
        queryItem.setDAPSelect(select);
        
        // Create the Query
        DAPQuery query = new DAPQuery(Konstantz.DAP_NS, LOCAL_NAME, Konstantz.DAP_PREFIX);
        query.setItemID(itemId);
        query.getDAPQueryItems().add(queryItem);
        
        return query;
    }
    
    
    public DAPQuery(String namespaceURI, String elementLocalName, String namespacePrefix)
    {
        super(namespaceURI, elementLocalName, namespacePrefix);
    }

    public XMLObjectChildrenList<DAPQueryItem> getDAPQueryItems()
    {
        if(null==queryItems)
        {
            queryItems = new XMLObjectChildrenList<DAPQueryItem>(this);
        }
        return queryItems;
    }
    
    public XMLObjectChildrenList<DAPTestItem> getDAPTestItems()
    {
        if(null==testItems)
        {
            testItems = new XMLObjectChildrenList<DAPTestItem>(this);
        }
        return testItems;
    }
    
    public XMLObjectChildrenList<DAPSubscription> getDAPSubscriptions()
    {
        if(null==subscriptions) subscriptions = new XMLObjectChildrenList<DAPSubscription>(this);
        return subscriptions;
    }

    @Override
	public List<XMLObject> getOrderedChildren()
    {        
        List<XMLObject> children = new LinkedList<XMLObject>();
        
        // Add the super kids
        children.addAll(super.getOrderedChildren()); 

        // now add the local kids
        if(null!=testItems) children.addAll(testItems); 
        if(null!=queryItems) children.addAll(queryItems); 
        if(null!=subscriptions) children.addAll(subscriptions); 
        
        return Collections.unmodifiableList(children);
    }
    
   
    
    public static class Marshaller extends RequestType.Marshaller
    {        
    }
    

    /**
     * Internal class that handles unmarshalling specific to DAPQuery
     * 
     * @author asa
     *
     */
    public static class Unmarshaller extends RequestType.Unmarshaller
    {       

        @Override
        protected void processChildElement(XMLObject parentXMLObject, XMLObject childXMLObject) throws UnmarshallingException
        {            
            DAPQuery obj = (DAPQuery) parentXMLObject;

            if(childXMLObject instanceof DAPTestItem)
            {
                obj.getDAPTestItems().add((DAPTestItem)childXMLObject);                   
            }
            else if(childXMLObject instanceof DAPQueryItem)
            {
                obj.getDAPQueryItems().add((DAPQueryItem)childXMLObject);                   
            }
            else if(childXMLObject instanceof DAPSubscription)
            {
                obj.getDAPSubscriptions().add((DAPSubscription)childXMLObject);                   
            }
            else
            {
                super.processChildElement(parentXMLObject, childXMLObject);
            } 
        }

    }

    /**
     * Internal class for the builder to simplify the class structure
     * 
     * @author asa
     *
     */
    public static class Builder extends AbstractXMLObjectBuilder<DAPQuery> 
    {
        @Override
		public DAPQuery buildObject(String namespaceURI, String localName, String namespacePrefix) 
        {
            return new DAPQuery(namespaceURI, localName, namespacePrefix);
        }
    }

}
