package org.openliberty.xmltooling.idsis.dap;


import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

import org.openliberty.xmltooling.Konstantz;
import org.openliberty.xmltooling.OpenLibertyHelpers;
import org.openliberty.xmltooling.dst2_1.ref.SelectMarshaller;
import org.opensaml.core.xml.AbstractXMLObjectBuilder;
import org.opensaml.core.xml.XMLObject;
import org.opensaml.core.xml.io.AbstractXMLObjectMarshaller;
import org.opensaml.core.xml.io.AbstractXMLObjectUnmarshaller;
import org.opensaml.core.xml.io.MarshallingException;
import org.opensaml.core.xml.io.UnmarshallingException;
import org.opensaml.xmlsec.signature.AbstractSignableXMLObject;
import org.w3c.dom.Attr;
import org.w3c.dom.Element;

/**
 * The ID-DAP Select differs from the DST 2.1 Reference Implementation in that
 * it contains the elements "dn" and "filter" and a set of attributes.
 * <p>
 * Both the "dn" element and the "filter" element are optional.  If "dn"
 * is omitted the WSP will figure out the dn (distinguished name) from 
 * the identity information provided.
 * </p>
 * <p>
 * If &lt;Select&gt; is used in &lt;DeleteItem&gt; or &lt;ModifyItem&gt; 
 * relating to objectType "entry," &lt;Filter&gt; element 
 * MUST NOT be specified and NONE of the XML attributes MUST be specified.
 * </p>
 * <p>
 * unmarshaller: DAPSelect$DAPSelectUnmarshaller {@link DAPSelectUnmarshaller}<br />
 * marshaller:  DAPSelect$DAPSelectMarshaller {@link DAPSelectMarshaller}<br />
 * builder: DAPSelect$DAPSelectBuilder {@link DAPSelectBuilder}<br /> 
 * </p>
 * 
 * @author asa
 *
 */
public class DAPSelect extends AbstractSignableXMLObject
{
    
    public static String LOCAL_NAME = "Select";
    
    // Elements
    private DN dn;  
    private Filter filter;  
        
    // Attributes
    private Integer sizelimit;
    private Integer timelimit;
    private Boolean typesonly;
    private Integer scope;
    private Integer derefaliases;
    private String attributes;
    
    // Attribute names
    public static final String ATT_SCOPE = "scope";
    public static final String ATT_SIZELIMIT = "sizelimit";
    public static final String ATT_TIMELIMIT = "timelimit";
    public static final String ATT_ATTRIBUTES = "attributes";
    public static final String ATT_TYPESONLY = "typesonly";
    public static final String ATT_DEREFALIASES= "derefaliases";
    

    public DAPSelect()
    {
        super(Konstantz.DAP_NS, LOCAL_NAME, Konstantz.DAP_PREFIX);
    }
    
    public DAPSelect(String namespaceURI, String elementLocalName, String namespacePrefix)
    {
        super(namespaceURI, elementLocalName, namespacePrefix);
    }
    
    
    // Attributes
    public void setTypesonly(Boolean typesonly)
    {
        this.typesonly = prepareForAssignment(this.typesonly, typesonly);
    }
    
    public Boolean getTypesonly()
    {
        return this.typesonly;
    }
     
    public void setSizelimit(Integer sizelimit)
    {
        this.sizelimit = prepareForAssignment(this.sizelimit, sizelimit);
    }

    public Integer getSizeLimit()
    {
        return sizelimit;
    }

    public void setTimelimit(Integer timelimit)
    {
        this.timelimit = prepareForAssignment(this.timelimit, timelimit);
    }

    public Integer getTimelimit()
    {
        return timelimit;
    }
    
    /**
     * If objectType is "entry" The attributes single-valued XML attribute is converted to an AttributeDescriptionList by 
     * splitting the string on comma characters and forming an AttributeDescription from each 
     * resulting substring.
     * 
     * @param attributes
     */
    public void setAttributes(String attributes)
    {
        this.attributes = prepareForAssignment(this.attributes, attributes);
    }
    
    public String getAttributes()
    {
        return this.attributes;
    }
    
    
    /**
     * optional attribute, default of 0
     * 
     * @param scope
     */
    public void setScope(Integer scope)
    {
        this.scope = prepareForAssignment(this.scope, scope);
    }

    public Integer getScope()
    {
        return scope;
    }

    
    public void setDerefaliases(Integer derefaliases)
    {
        this.derefaliases = prepareForAssignment(this.derefaliases, derefaliases);
    }

    public Integer getDerefaliases()
    {
        return derefaliases;
    }

    // Elements
    
    public DN getDn()
    {
        return dn;
    }

    public void setDn(DN dn)
    {
        this.dn = prepareForAssignment(this.dn, dn);
    } 
    
    /**
     * Convenience method for setting the DN
     * 
     * @param dnString
     */
    public void setDn(String dnString)
    {
        DN dn = new DN();
        dn.setValue(dnString);
        this.dn = prepareForAssignment(this.dn, dn);
    }
    
    public Filter getFilter()
    {
        return filter;
    }

    public void setFilter(Filter filter)
    {
        this.filter = prepareForAssignment(this.filter, filter);
    }
    
    /**
     * Convenience method for setting the Filter
     * 
     * @param dnString
     */
    public void setFilter(String filterString)
    {
        Filter filter = new Filter();
        filter.setValue(filterString);
        this.filter = prepareForAssignment(this.filter, filter);
    }
    
    
    public List<XMLObject> getOrderedChildren()
    {
        List<XMLObject> children = new LinkedList<XMLObject>();
        
        if(null!=filter) children.add(filter); 
        if(null!=dn) children.add(dn); 
        
        return Collections.unmodifiableList(children);
    }
    
    
    
    
    /**
     * This is the java-xmltooling Builder class for DAPSelect
     * 
     * @author asa
     *
     */
    public static class Builder extends AbstractXMLObjectBuilder<DAPSelect> 
    {
        
        public DAPSelect buildObject()
        {
            return new DAPSelect();
        }
               
        @Override
		public DAPSelect buildObject(String namespaceURI, String localName, String namespacePrefix) 
        {
            return new DAPSelect(namespaceURI, localName, namespacePrefix);
        }
        
    }
    
    
    /**
     * Internal Marshaller class primarily to marshall attributes.  The
     * string content is handled by {@link SelectMarshaller}
     * 
     * @author asa
     *
     */
    public static class Marshaller extends AbstractXMLObjectMarshaller
    {
        @Override
        protected void marshallAttributes(XMLObject xmlObject, Element domElement) throws MarshallingException
        {
            DAPSelect obj = (DAPSelect)xmlObject;

            if(obj.getScope() != null) 
            {
                domElement.setAttributeNS(null, DAPSelect.ATT_SCOPE, obj.getScope().toString());
            }

            if(obj.getSizeLimit() != null) 
            {
                domElement.setAttributeNS(null, DAPSelect.ATT_SIZELIMIT, obj.getSizeLimit().toString());
            }

            if(obj.getTimelimit() != null) 
            {
                domElement.setAttributeNS(null, DAPSelect.ATT_TIMELIMIT, obj.getTimelimit().toString());
            }

            if(obj.getAttributes() != null) 
            {
                domElement.setAttributeNS(null, DAPSelect.ATT_ATTRIBUTES, obj.getAttributes());
            }

            if(obj.getTypesonly() != null) 
            {
                domElement.setAttributeNS(null, DAPSelect.ATT_TYPESONLY, OpenLibertyHelpers.stringFromBoolean(obj.getTypesonly(), OpenLibertyHelpers.FALSE_STR));
            }

            if(obj.getDerefaliases() != null) 
            {
                domElement.setAttributeNS(null, DAPSelect.ATT_DEREFALIASES, obj.getDerefaliases().toString());
            }
            
        }  
        
        @Override
        protected void marshallElementContent(XMLObject xmlObject, Element domElement) throws MarshallingException
        {
            // no element content
        }        
        
    }
    
    
    
    /**
     * Internal class that handles unmarshalling specific to 
     * DAPSelect
     * 
     * @author asa
     *
     */
    public static class Unmarshaller extends AbstractXMLObjectUnmarshaller
    {       
        
        @Override
        protected void processAttribute(XMLObject xmlObject, Attr attribute) throws UnmarshallingException
        {
            DAPSelect obj = (DAPSelect) xmlObject;
            
            String attLocalName = attribute.getLocalName();
            
            if (attLocalName.equals(DAPSelect.ATT_SCOPE)) 
            {
                obj.setScope(OpenLibertyHelpers.integerFromString(attribute.getValue()));
            }   
            else if (attLocalName.equals(DAPSelect.ATT_SIZELIMIT)) 
            {
                obj.setSizelimit(OpenLibertyHelpers.integerFromString(attribute.getValue()));
            }   
            else if (attLocalName.equals(DAPSelect.ATT_TIMELIMIT)) 
            {
                obj.setTimelimit(OpenLibertyHelpers.integerFromString(attribute.getValue()));
            }   
            else if (attLocalName.equals(DAPSelect.ATT_ATTRIBUTES)) 
            {
                obj.setAttributes(attribute.getValue());
            }   
            else if (attLocalName.equals(DAPSelect.ATT_TYPESONLY)) 
            {
                obj.setTypesonly(OpenLibertyHelpers.booleanFromString(attribute.getValue()));
            }   
            else if (attLocalName.equals(DAPSelect.ATT_DEREFALIASES)) 
            {
                obj.setDerefaliases(OpenLibertyHelpers.integerFromString(attribute.getValue()));
            }   
        }

        
        @Override
        protected void processChildElement(XMLObject parentXMLObject, XMLObject childXMLObject) throws UnmarshallingException
        {            
            DAPSelect obj = (DAPSelect) parentXMLObject;
            
            if(childXMLObject instanceof DN)
            {
                obj.setDn((DN)childXMLObject);
            }
            else if(childXMLObject instanceof Filter)
            {
                obj.setFilter((Filter)childXMLObject);
            }                       
        }
        
        @Override
        protected void processElementContent(XMLObject xmlObject, String elementContent) 
        {
            // no element content 
        }
        
    } 
    
    
    
    
}

/*

  <xs:complexType name="SelectType">
    <xs:sequence>
      <xs:element name="dn" minOccurs="0" maxOccurs="1" type="xs:string"/>
      <xs:element name="filter" minOccurs="0" maxOccurs="1" type="xs:string"/>
    </xs:sequence>
    <xs:attribute name="scope" use="optional" type="xs:integer" default="0"/>
    <xs:attribute name="sizelimit" use="optional" type="xs:integer" default="0"/>
    <xs:attribute name="timelimit" use="optional" type="xs:integer" default="0"/>
    <xs:attribute name="attributes" use="optional" type="xs:string"/>
    <xs:attribute name="typesonly" use="optional" type="xs:boolean" default="false"/>
    <xs:attribute name="derefaliases" use="optional" type="xs:integer" default="0"/>
  </xs:complexType>

*/
