package org.openliberty.xmltooling.idsis.dap;

import java.util.List;

import org.openliberty.xmltooling.Konstantz;
import org.opensaml.core.xml.AbstractXMLObjectBuilder;
import org.opensaml.core.xml.XMLObject;
import org.opensaml.core.xml.schema.impl.XSStringImpl;
import org.opensaml.core.xml.schema.impl.XSStringMarshaller;
import org.opensaml.core.xml.schema.impl.XSStringUnmarshaller;

/**
 * Contents of SortType string is a dollar-separated (ASCII 0x24) list of sort keys in descending order of
 * importance, where each sort key is a comma-separated list of three elements:<br />
 * <ol>
 * <li>Order: "a" == ascending (reverse order false, the default), "d" == descending (reverse order true),</li>
 * <li>Ordering rule (empty means default for the attribute),</li>
 * <li>Attribute name.</li>
 * </ol>
 * <p>
 * These elements are interpreted in the sense of [RFC2891], Section 1.1 "Request Control."
 * </p>
 * <p>
 * Example:<br/>
 * <pre>
 *      a,,cn$d,,fn$d,,sn
 * </pre>
 * </p>
 * <p>
 * An implementation that does not support sorting MAY ignore the sort specification. An implementation that
 * partially supports sorting, SHOULD make best effort to satisfy the sort criteria, but need not adhere to it literally.
 * Such implementations MAY ignore LDAP control-criticality specifications regarding the sort control.
 * </p>
 * <p>
 * Sorting is not supported for subscriptions.
 * </p>
 * <pre>
 *   &lt;xs:complexType name="SortType"&gt;
 *     &lt;xs:simpleContent&gt;
 *       &lt;xs:extension base="xs:string"/&gt;
 *     &lt;/xs:simpleContent&gt;
 *   &lt;/xs:complexType&gt;
 * </pre>
 * @author asa
 *
 */
public class DAPSort extends XSStringImpl
{
    public static final String LOCAL_NAME = "Sort";

    protected DAPSort()
    {
        super(Konstantz.DAP_NS, LOCAL_NAME, Konstantz.DAP_PREFIX);
    }

    protected DAPSort(String namespaceURI, String elementLocalName, String namespacePrefix)
    {
        super(namespaceURI, elementLocalName, namespacePrefix);
    }
    
    /**
     * This method is used to build the DAPSort string.  It is called for each "Sort Element" 
     * that a user is interested in specifying
     * <p>
     * A sort element is in the form "sortOrder,orderingRule,attributeName"  
     * 
     * @param attributeName required name of the attribute such as "cn" "fn" etc...
     * @param sortOrder default ascending
     * @param orderingRule optional
     */
    public void addSortElement(String attributeName, SortOrder sortOrder, String orderingRule)
    {
        if(attributeName!=null)
        {            
            String value = super.getValue();
            StringBuffer valueBuff = new StringBuffer(value!=null?value:"");
            if(valueBuff.length()>0) valueBuff.append("$");
            valueBuff.append(SortOrder.value(sortOrder)).append(",").append(orderingRule!=null?orderingRule:"").append(",").append(attributeName);
            super.setValue(valueBuff.toString());
        }

    }

    @Override
	public List<XMLObject> getOrderedChildren()
    {
        return null;
    }

    public static class Marshaller extends XSStringMarshaller
    {        
    }

    public static class Unmarshaller extends XSStringUnmarshaller
    {        
    }


    public static class Builder extends AbstractXMLObjectBuilder<DAPSort>
    {
        @Override
        public DAPSort buildObject(String namespaceURI, String localName, String namespacePrefix)
        {
            return new DAPSort(namespaceURI, localName, namespacePrefix);
        }  
    }

    public enum SortOrder
    {
        ASCENDING, DESCENDING;

        public static String value(SortOrder sortOrder)        
        {
            if(sortOrder!=null && sortOrder == DESCENDING) return "d";
            else return "a";
        }

    }


}

