package org.openliberty.xmltooling.pp;

import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.UUID;

import org.joda.time.DateTime;
import org.openliberty.xmltooling.Konstantz;
import org.openliberty.xmltooling.dst2_1.CommonAttributes;
import org.openliberty.xmltooling.pp.dst2_1.ct.C;
import org.openliberty.xmltooling.pp.dst2_1.ct.L;
import org.openliberty.xmltooling.pp.dst2_1.ct.LL;
import org.openliberty.xmltooling.pp.dst2_1.ct.LPostalAddress;
import org.openliberty.xmltooling.pp.dst2_1.ct.LSt;
import org.openliberty.xmltooling.pp.dst2_1.ct.PostalAddress;
import org.openliberty.xmltooling.pp.dst2_1.ct.PostalCode;
import org.openliberty.xmltooling.pp.dst2_1.ct.St;
import org.openliberty.xmltooling.utility_2_0.Extension;
import org.opensaml.core.xml.XMLObject;
import org.opensaml.core.xml.util.XMLObjectChildrenList;
import org.opensaml.xmlsec.signature.AbstractSignableXMLObject;

/**
 * Commonly used group of postal address fields
 * <p>
 * AP SHOULD implement permissions enforcement that allows only C 
 * and PostalCode attributes to be provided to those SPs that do not 
 * need the entire address.
 *  
 * @author asa
 *
 */
public class Address extends AbstractSignableXMLObject
{

    public static String LOCAL_NAME = "Address";

    // Attributes
    private CommonAttributes attributes;

    // Elements
    private PostalAddress postalAddress;
    private XMLObjectChildrenList<LPostalAddress> lPostalAddresses;  
    private PostalCode postalCode;
    private L l;
    private XMLObjectChildrenList<LL> lLs;
    private St st;
    private XMLObjectChildrenList<LSt> lSts;
    private C c;
    private Extension extension;

    /**
     * Easily make a US Address
     * 
     * @param street
     * @param city
     * @param state
     * @param zip
     * @param country
     */
    public Address(String street, String city, String state, String zip, String country)
    {
        super(Konstantz.PP_NS, LOCAL_NAME, Konstantz.PP_PREFIX);
        this.setPostalAddress(new PostalAddress(street));
        this.setL(new L(city));
        this.setSt(new St(state));
        this.setPostalCode(new PostalCode(zip));
        this.setC(new C(country));        
        attributes = new CommonAttributes();
    }

    public String getCountryValue()
    {
        return null!=c ? c.getValue() : null;
    }

    public void setCountryValue(String value)
    {
        if(c==null) setC(new C(value));
        else c.setValue(value);
    }
    
    public String getPostalCodeValue()
    {
        return null!=postalCode ? postalCode.getValue() : null;
    }

    public void setPostalCodeValue(String value)
    {
        if(postalCode==null) setPostalCode(new PostalCode(value));
        else postalCode.setValue(value);
    }
    
    public String getStateOrProvinceValue()
    {
        return null!=st ? st.getValue() : null;
    }

    public void setStateOrProvinceValue(String value)
    {
        if(st==null) setSt(new St(value));
        else st.setValue(value);
    }
    
    public String getCityOrLocaleValue()
    {
        return null!=l ? l.getValue() : null;
    }

    public void setCityOrLocaleValue(String value)
    {
        if(l==null) setL(new L(value));
        else l.setValue(value);
    }
    
    public String getStreetOrPostalAddressValue()
    {
        return null!=postalAddress ? postalAddress.getValue() : null;
    }

    public void setStreetOrPostalAddressValue(String value)
    {
        if(postalAddress==null) setPostalAddress(new PostalAddress(value));
        else postalAddress.setValue(value);
    }


    
    /**
     * If createChildren is set to true,  this constructor initializes all of the direct children of the
     * Address and sets the id attribute
     * 
     */
    public Address(boolean createChildren)
    {
        super(Konstantz.PP_NS, LOCAL_NAME, Konstantz.PP_PREFIX);
        attributes = new CommonAttributes();
        if(createChildren)
        {
            attributes.setModificationTime(new DateTime(), this);            
            attributes.setId(UUID.randomUUID().toString(), this); 
            this.setPostalAddress(new PostalAddress());
            this.setL(new L());
            this.setSt(new St());
            this.setPostalCode(new PostalCode());
            this.setC(new C());        
        }
    }


    public Address(String namespaceURI, String elementLocalName, String namespacePrefix) 
    {
        super(namespaceURI, elementLocalName, namespacePrefix);     
        attributes = new CommonAttributes();
    }

    // Attributes
    public CommonAttributes attributes()
    {
        return attributes;
    }

    // Elements
    public void setPostalAddress(PostalAddress postalAddress) 
    {
        this.postalAddress = prepareForAssignment(this.postalAddress, postalAddress);
    }

    public PostalAddress getPostalAddress() 
    { 
        return postalAddress; 
    }   

    public XMLObjectChildrenList<LPostalAddress> getLPostalAddresses()
    {
        if(null==lPostalAddresses) lPostalAddresses = new XMLObjectChildrenList<LPostalAddress>(this);
        return lPostalAddresses;
    }

    public void setPostalCode(PostalCode postalCode) 
    {
        this.postalCode = prepareForAssignment(this.postalCode, postalCode);
    }

    public PostalCode getPostalCode() 
    { 
        return postalCode; 
    }    

    public void setL(L l) 
    {
        this.l = prepareForAssignment(this.l, l);
    }

    public L getL() 
    { 
        return l; 
    } 

    public XMLObjectChildrenList<LL> getLLs()
    {
        if(null==lLs) lLs = new XMLObjectChildrenList<LL>(this);
        return lLs;
    }

    public void setSt(St st) 
    {
        this.st = prepareForAssignment(this.st, st);
    }

    public St getSt() 
    { 
        return st; 
    } 

    public XMLObjectChildrenList<LSt> getLSts()
    {
        if(null==lSts) lSts = new XMLObjectChildrenList<LSt>(this);
        return lSts;
    }

    public void setC(C c) 
    {
        this.c = prepareForAssignment(this.c, c);
    }

    public C getC() 
    { 
        return c; 
    }    

    public void setExtension(Extension extension) 
    {
        this.extension = prepareForAssignment(this.extension, extension);
    }

    public Extension getExtension() 
    { 
        return extension; 
    }    


    public List<XMLObject> getOrderedChildren() 
    {
        List<XMLObject> children = new LinkedList<XMLObject>();

        children.add(postalAddress);        
        if(null!=lPostalAddresses) children.addAll(lPostalAddresses); 
        children.add(postalCode);        
        children.add(l);       
        if(null!=lLs) children.addAll(lLs);   
        children.add(st);
        if(null!=lSts) children.addAll(lSts);   
        children.add(c);  
        children.add(extension);  

        return Collections.unmodifiableList(children);
    }   

}

/*
<xs:element name="Address" type="AddressType"/>
<xs:complexType name="AddressType">
    <xs:sequence>
        <xs:element ref="PostalAddress" minOccurs="0"/>
        <xs:element ref="LPostalAddress" minOccurs="0" maxOccurs="unbounded"/>
        <xs:element ref="PostalCode" minOccurs="0"/>
        <xs:element ref="L" minOccurs="0"/>
        <xs:element ref="LL" minOccurs="0" maxOccurs="unbounded"/>
        <xs:element ref="St" minOccurs="0"/>
        <xs:element ref="LSt" minOccurs="0" maxOccurs="unbounded"/>
        <xs:element ref="C" minOccurs="0"/>
        <xs:element ref="Extension" minOccurs="0"/>
    </xs:sequence>
    <xs:attributeGroup ref="commonAttributes"/>
</xs:complexType>
 */
