package org.openliberty.xmltooling.pp;

import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.UUID;

import org.joda.time.DateTime;
import org.openliberty.xmltooling.Konstantz;
import org.openliberty.xmltooling.dst2_1.CommonAttributes;
import org.openliberty.xmltooling.pp.dst2_1.ct.AddrType;
import org.openliberty.xmltooling.pp.dst2_1.ct.LComment;
import org.openliberty.xmltooling.pp.dst2_1.ct.LNick;
import org.openliberty.xmltooling.pp.dst2_1.ct.Nick;
import org.openliberty.xmltooling.utility_2_0.Extension;
import org.opensaml.core.xml.XMLObject;
import org.opensaml.core.xml.util.XMLObjectChildrenList;
import org.opensaml.xmlsec.signature.AbstractSignableXMLObject;

/**
 * An address card data structure for ID-SIS-PP
 * 
 * @author asa
 *
 */
public class AddressCard extends AbstractSignableXMLObject
{
    public static String LOCAL_NAME = "AddressCard";
    public static String XPATH_SELECTOR = "/pp:PP/pp:AddressCard";

    // Attributes
    private CommonAttributes attributes;

    // Elements
    private XMLObjectChildrenList<AddrType> addrTypes;  
    private Address address;
    private Nick nick;
    private XMLObjectChildrenList<LNick> lNicks;
    private LComment lComment;
    private Extension extension;

    /**
     * If createChildren is set to true,  this constructor initializes all of the direct children of the
     * AddressCard and sets the id attribute
     * 
     */
    public AddressCard(boolean createChildren)
    {
        super(Konstantz.PP_NS, LOCAL_NAME, Konstantz.PP_PREFIX);
        attributes = new CommonAttributes();        
        if(createChildren)
        {
            attributes.setModificationTime(new DateTime(), this);            
            attributes.setId(UUID.randomUUID().toString(), this);    
            this.setNick(new Nick());
            this.setAddress(new Address(true));
            this.setLComment(new LComment());            
        }
    }

    public AddressCard()
    {
        super(PP.NAMESPACE_URI, AddressCard.LOCAL_NAME, PP.NAMESPACE_PREFIX);
        attributes = new CommonAttributes();
    }


    protected AddressCard(String namespaceURI, String elementLocalName, String namespacePrefix) 
    {
        super(namespaceURI, elementLocalName, namespacePrefix);     
        attributes = new CommonAttributes();
    }

    // Attributes
    public CommonAttributes attributes()
    {
        return attributes;
    }

    // Elements
    public XMLObjectChildrenList<AddrType> getAddrTypes()
    {
        if(null==addrTypes) addrTypes = new XMLObjectChildrenList<AddrType>(this);
        return addrTypes;
    }

    public void setAddress(Address address) 
    {
        this.address = prepareForAssignment(this.address, address);
    }

    public Address getAddress() 
    { 
        return address; 
    }   

    public void setNick(Nick nick) 
    {
        this.nick = prepareForAssignment(this.nick, nick);
    }

    /**
     * Nickname SHOULD NOT be printed in address label. The nickname can not and 
     * SHOULD NOT be used as machine-readable identification of any data item. Nick 
     * names are not unique.
     * 
     * @return
     */
    public Nick getNick() 
    { 
        return nick; 
    }   

    public XMLObjectChildrenList<LNick> getLNicks()
    {
        if(null==lNicks) lNicks = new XMLObjectChildrenList<LNick>(this);
        return lNicks;
    }

    public void setLComment(LComment lComment) 
    {
        this.lComment = prepareForAssignment(this.lComment, lComment);
    }

    public LComment getLComment() 
    { 
        return lComment; 
    }   

    public void setExtension(Extension extension) 
    {
        this.extension = prepareForAssignment(this.extension, extension);
    }

    public Extension getExtension() 
    { 
        return extension; 
    }    

    public List<XMLObject> getOrderedChildren() 
    {
        List<XMLObject> children = new LinkedList<XMLObject>();

        if(null!=addrTypes) children.addAll(addrTypes); 
        children.add(address);        
        children.add(nick);       
        if(null!=lNicks) children.addAll(lNicks);   
        children.add(lComment);
        children.add(extension);  

        return Collections.unmodifiableList(children);
    }   



}

/*
<xs:element name="AddressCard" type="AddressCardType"/>
<xs:complexType name="AddressCardType">
    <xs:sequence>
        <xs:element ref="AddrType" minOccurs="0" maxOccurs="unbounded"/>
        <xs:element ref="Address" minOccurs="0"/>
        <xs:element ref="Nick" minOccurs="0"/>
        <xs:element ref="LNick" minOccurs="0" maxOccurs="unbounded"/>
        <xs:element ref="LComment" minOccurs="0"/>
        <xs:element ref="Extension" minOccurs="0"/>
    </xs:sequence>
    <xs:attributeGroup ref="commonAttributes"/>
</xs:complexType>
 */