package org.openliberty.xmltooling.ps;

import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

import javax.xml.namespace.QName;

import org.joda.time.DateTime;
import org.openliberty.xmltooling.Konstantz;
import org.openliberty.xmltooling.OpenLibertyHelpers;
import org.opensaml.core.xml.AbstractXMLObject;
import org.opensaml.core.xml.AbstractXMLObjectBuilder;
import org.opensaml.core.xml.ElementExtensibleXMLObject;
import org.opensaml.core.xml.XMLObject;
import org.opensaml.core.xml.io.AbstractXMLObjectMarshaller;
import org.opensaml.core.xml.io.AbstractXMLObjectUnmarshaller;
import org.opensaml.core.xml.io.MarshallingException;
import org.opensaml.core.xml.io.UnmarshallingException;
import org.opensaml.core.xml.util.IndexedXMLObjectChildrenList;
import org.opensaml.core.xml.util.XMLObjectChildrenList;
import org.w3c.dom.Attr;
import org.w3c.dom.Element;
// import org.openliberty.xmltooling.security.Token;
/**
 * PSObject is the basic data object in the People Service representing both Collections and Entities
 * <p>
 * From section 2.1 of the ID-WSF People Service Specification v1.0:
 * <p>
 * Both individual users and the groups to which they may belong are represented as &lt;Object&gt; elements - whether an
 * &lt;Object&gt; refers to a group or a user (or perhaps some other individual entity) is distinguished by a NodeType attribute
 * with values of urn:liberty:ps:collection or urn:liberty:ps:entity respectively (see Section 2.1.1 for exact definition).
 * </p>
 * <p>
 * The &lt;Object> element has {@link DisplayName} elements to carry a human-readable name for the &lt;Object&gt; (see
 * Section 2.1.5).
 * </p>
 * <p>
 * The &lt;ObjectID&gt; element uniquely labels each &lt;Object&gt; (see Section 2.1.4).
 * The optional CreatedDateTime and ModifiedDateTime attributes express the time at which an Object was
 * created and last modified respectively (see Section 2.1.2).
 * </p>
 * <p>
 * To account for nested Objects, an &lt;Object&gt; element can have multiple &lt;Object&gt; and/or &lt;ObjectRef&gt; elements
 * to refer to other Objects.
 * </p>
 * <pre>
 * &lt;xs:element name="Object" type="ObjectType" /&gt;
 * 
 * &lt;xs:complexType name="ObjectType"&gt;
 *  &lt;xs:sequence&gt;
 *      &lt;xs:element ref="ObjectID" minOccurs="0" /&gt;
 *      &lt;xs:element name="DisplayName" type="LocalizedDisplayNameType" minOccurs="1" maxOccurs="unbounded" /&gt;
 *      &lt;xs:element name="Tag" type="TagType" minOccurs="0" maxOccurs="unbounded" /&gt;
 *      &lt;xs:element ref="Object" minOccurs="0" maxOccurs="unbounded" /&gt;
 *      &lt;xs:element name="ObjectRef" type="ObjectIDType" minOccurs="0" maxOccurs="unbounded" /&gt;
 *  &lt;/xs:sequence&gt;
 *  &lt;xs:attribute name="NodeType" type="xs:anyURI" use="required" /&gt;
 *  &lt;xs:attribute name="CreatedDateTime" type="xs:dateTime" use="optional" /&gt;
 *  &lt;xs:attribute name="ModifiedDateTime" type="xs:dateTime" use="optional" /&gt;
 * &lt;/xs:complexType&gt;
 * </pre>
 * @author asa
 *
 */
public class PSObject extends AbstractXMLObject implements ElementExtensibleXMLObject
{

    public static String LOCAL_NAME = "Object";

    // ATTRIBUTE NAMES
    public static String ATT_NODE_TYPE = "NodeType";
    public static String ATT_CREATED_DATE_TIME = "CreatedDateTime";		// optional
    public static String ATT_MODIFIED_DATE_TIME = "ModifiedDateTime";	// optional

    // ATTRIBUTES
    private NodeType nodeType;
    private DateTime createdDateTime;								// optional
    private DateTime modifiedDateTime;							// optional    
    
    // ELEMENTS
    private ObjectID objectID;									// optional
    private XMLObjectChildrenList<DisplayName> displayNames;	// at least one required
    private XMLObjectChildrenList<Tag> tags;					// optional
    private XMLObjectChildrenList<PSObject> objects;			// only if collection
    private XMLObjectChildrenList<ObjectRef> objectRefs;		
    // private Token token;
    
    private IndexedXMLObjectChildrenList<XMLObject> unknownXMLObjects;



    public PSObject()
    {
        super(Konstantz.PS_NS, LOCAL_NAME, Konstantz.PS_PREFIX);
        unknownXMLObjects = new IndexedXMLObjectChildrenList<XMLObject>(this);
        displayNames = new XMLObjectChildrenList<DisplayName>(this);
    }

    public PSObject(String displayName)
    {
        super(Konstantz.PS_NS, LOCAL_NAME, Konstantz.PS_PREFIX);
        
        unknownXMLObjects = new IndexedXMLObjectChildrenList<XMLObject>(this);
        displayNames = new XMLObjectChildrenList<DisplayName>(this);

        if(null!=displayName)
        {
        	DisplayName _d = new DisplayName(displayName, DisplayName.DEFAULT_LOCALE);
//        	_d.setValue(displayName);
//        	_d.setLocale(DisplayName.DEFAULT_LOCALE);
        	displayNames.add(_d);   
        }
    }
    
    protected PSObject(String namespaceURI, String elementLocalName, String namespacePrefix) 
    {
        super(namespaceURI, elementLocalName, namespacePrefix);		
        unknownXMLObjects = new IndexedXMLObjectChildrenList<XMLObject>(this);
        displayNames = new XMLObjectChildrenList<DisplayName>(this);
    }


    public static PSObject createEntity(String displayName)
    {
    	PSObject object = new PSObject(displayName);
    	object.setNodeType(NodeType.ENTITY);
    	return object; 	
    }
    
    public static PSObject createCollection(String displayName)
    {
    	PSObject object = new PSObject(displayName);
    	object.setNodeType(NodeType.COLLECTION);
    	return object; 	
    }


    public boolean isCollection()
    {
    	return nodeType==NodeType.COLLECTION;
    }
    
    public boolean isEntity()
    {
    	return nodeType==NodeType.ENTITY;
    }

    public void print()
    {
    	PSObject.printPSObject(this);
    }

	public static void printPSObject(PSObject object)
	{
		printPSObject(object, 0);
	}
	
	private static void printPSObject(PSObject object, int depth)
	{
		for(int i=0; i<depth;i++)
		{
			System.err.print("\t");
		}

		System.err.print(object.getNodeType().urn()+": "+object.getName()+" / "+object.getObjectID().getValue());		
		List<Tag> tags = object.getTags();
		if(null!=tags && tags.size()>0)
		{
			System.err.print(" { ");
			for(Tag tag : tags)
			{
				System.err.print(tag.getValue()+" ");
			}
			System.err.print("}");
		}		
		System.err.print("\n");
		
		if(object.getNodeType()==NodeType.COLLECTION) 
		{
			List<PSObject> children = object.getObjects();
			for(PSObject child : children)
			{
				printPSObject(child, depth+1);				
			}
		}

	}
	
    

    // helper methods
    
    /**
     * This method returns a PSObjectRef only if there is a PSObjectID present in the PSObject
     * 
	 * @return a reference to the object or null if one cannot be created
	 */
    public ObjectRef createObjectRef()
    {
        ObjectRef objectRef = null;
    	
        ObjectID objectId = this.getObjectID();
		if(objectId!=null && objectId.getValue()!=null)
		{
			objectRef = new ObjectRef();
			objectRef.setValue(objectId.getValue());
		}

		return objectRef;
    }
    
    /**
     * This method returns a TargetObjectID only if there is a PSObjectID present in the PSObject
     * 
	 * @return a TargetObjectID referring to the object or null if one cannot be created
	 */
    public TargetObjectID createTargetObjectID()
    {
        TargetObjectID targetObjectID = null;
    	
        ObjectID objectId = this.getObjectID();
		if(objectId!=null && objectId.getValue()!=null)
		{
			
			targetObjectID = new TargetObjectID();
			targetObjectID.setValue(objectId.getValue());
		}
		
    	return targetObjectID;
    } 
    
    /**
     * 
     * @return the value of the first DisplayName
     */
    public String getName()
    {
    	if(null!=displayNames && displayNames.size()>0)
    	{
    		// first look for the default
    		for(DisplayName dname : displayNames)
    		{
    			if(dname.isDefault())
    			{
    				return dname.getValue();
    			}
    		}
    		
    		// otherwise send the first one
    		String name = displayNames.get(0).getValue();
    		if(name!=null) return name;
    	}
    	return "";
    }

    
    /**
     * Sets the value of the first DisplayName, or creates one if it does not exist
     * 
     * @return the value of the first DisplayName
     */
    public void setName(String displayNameValue)
    {
    	if(null!=displayNames && displayNames.size()>0)
    	{
    		displayNames.get(0).setValue(displayNameValue);
    	}
    	else
    	{
    		// Add a DisplayName element 
    		DisplayName displayName = new DisplayName(displayNameValue, null);
    		// displayName.setValue(displayNameValue);
    		getDisplayNames().add(displayName);
    	}
    }


    
    
    
    // Attributes

    public NodeType getNodeType() 
    {
        return nodeType;
    }

    public void setNodeType(NodeType nodeType) 
    {
        this.nodeType = nodeType;
    }	

    public void setCreatedDateTime(DateTime createdDateTime) 
    {
        this.createdDateTime = prepareForAssignment(this.createdDateTime, createdDateTime);
    }

    public DateTime getCreatedDateTime() 
    { 
        return createdDateTime; 
    }

    public void setModifiedDateTime(DateTime modifiedDateTime) 
    {
        this.modifiedDateTime = prepareForAssignment(this.modifiedDateTime, modifiedDateTime);
    }

    public DateTime getModifiedDateTime() 
    { 
        return modifiedDateTime; 
    }

    
    // Elements
    
    public void setObjectID(ObjectID objectID) 
    {
        this.objectID = prepareForAssignment(this.objectID, objectID);
    }

    public ObjectID getObjectID() 
    { 
        return this.objectID; 
    }

    /*
    public void setToken(Token token) 
    {
        this.token = prepareForAssignment(this.token, token);
    }

    public Token getToken() 
    { 
        return this.token; 
    }
     */
    
	public XMLObjectChildrenList<DisplayName> getDisplayNames() 
	{				
        
		return displayNames;
	}		

	public XMLObjectChildrenList<Tag> getTags() 
	{        
		if(null==tags) tags = new XMLObjectChildrenList<Tag>(this);
		return tags;
	}	
	
	public XMLObjectChildrenList<PSObject> getObjects() 
	{				
		if(null==objects) objects = new XMLObjectChildrenList<PSObject>(this);
		return objects;
	}
	
	public XMLObjectChildrenList<ObjectRef> getObjectRefs() 
	{				
		if(null==objectRefs) objectRefs = new XMLObjectChildrenList<ObjectRef>(this);
		return objectRefs;
	}		
    

    

    // Unknowns and children
    
    public List<XMLObject> getUnknownXMLObjects() 
    {		
        return unknownXMLObjects;
    }
    
    @SuppressWarnings("unchecked")
    public List<XMLObject> getUnknownXMLObjects(QName typeOrName)
    {
        return (List<XMLObject>) unknownXMLObjects.subList(typeOrName);
    }

    public List<XMLObject> getOrderedChildren() 
    {
        List<XMLObject> children = new LinkedList<XMLObject>();

        children.add(objectID);
        children.addAll(displayNames);
        if(null!=tags) children.addAll(tags);
        if(null!=objects) children.addAll(objects);
        if(null!=objectRefs) children.addAll(objectRefs);
        // children.add(token);        
        children.addAll(unknownXMLObjects);

        return Collections.unmodifiableList(children);
    }	

    
    
	public enum NodeType
	{
		COLLECTION("urn:liberty:ps:collection"),
		ENTITY("urn:liberty:ps:entity");

		String urn;

		NodeType(String urn)
		{
			this.urn = urn;
		}

		public String urn()
		{
			return urn;
		}
		
		static NodeType forUrn(String urn)
		{
			if(null!=urn && urn.equals(COLLECTION.urn)) return COLLECTION;
			else return ENTITY;
		}

	}
    
    
    
    
    /**
     * Static Builder for PSObject
     * 
     * @author asa
     *
     */
    public static class Builder extends AbstractXMLObjectBuilder<PSObject> 
    {
        @Override
		public PSObject buildObject(String namespaceURI, String localName, String namespacePrefix)
        {
            return new PSObject(namespaceURI, localName, namespacePrefix);
        }
        
    }
    
    /**
     * Static Marshaller for PSObject
     * 
     * @author asa
     *
     */
    public static class Marshaller extends AbstractXMLObjectMarshaller
    {
        
        @Override
        protected void marshallAttributes(XMLObject xmlObject, Element domElement) throws MarshallingException 
        {
            PSObject o = (PSObject)xmlObject;

            if(o.getNodeType() != null) 
            {
                domElement.setAttributeNS(null, PSObject.ATT_NODE_TYPE, o.getNodeType().urn);
            }
            
            if(o.getCreatedDateTime() != null)
            {
                domElement.setAttributeNS(null, PSObject.ATT_CREATED_DATE_TIME, OpenLibertyHelpers.stringForDateTime(o.getCreatedDateTime()));
            }
            
            if(o.getModifiedDateTime() != null)
            {
                domElement.setAttributeNS(null, PSObject.ATT_MODIFIED_DATE_TIME, OpenLibertyHelpers.stringForDateTime(o.getModifiedDateTime()));
            }

        }


        @Override
        protected void marshallElementContent(XMLObject xmlObject, Element domElement) throws MarshallingException 
        {
            // NO TEXT CONTENT
        }
        
    }
    

    
    /**
     * Ststic Unmarshaller for PSObject
     * 
     * @author asa
     *
     */
    public static class Unmarshaller extends AbstractXMLObjectUnmarshaller
    {
        
        @Override
        protected void processAttribute(XMLObject xmlObject, Attr attribute) throws UnmarshallingException 
        {
            PSObject o = (PSObject) xmlObject;
            
            String localName = attribute.getLocalName();
            
            if (localName.equals(PSObject.ATT_NODE_TYPE)) 
            {
                o.setNodeType(NodeType.forUrn( attribute.getValue() ));
            }
            else if (localName.equals(PSObject.ATT_CREATED_DATE_TIME)) 
            {
                o.setCreatedDateTime(OpenLibertyHelpers.dateTimeForString(attribute.getValue()));
            }
            else if (localName.equals(PSObject.ATT_MODIFIED_DATE_TIME)) 
            {
                o.setModifiedDateTime(OpenLibertyHelpers.dateTimeForString(attribute.getValue()));
            }           
        }

        
        
        @Override
        protected void processChildElement(XMLObject parentXMLObject, XMLObject childXMLObject) throws UnmarshallingException 
        {   
            PSObject o = (PSObject) parentXMLObject;        
            
            if (childXMLObject instanceof ObjectID) 
            {
                o.setObjectID((ObjectID) childXMLObject);
            }
            else if(childXMLObject instanceof Tag)
            {
                o.getTags().add((Tag) childXMLObject);
            }
            else if(childXMLObject instanceof DisplayName)
            {
                o.getDisplayNames().add((DisplayName) childXMLObject);
            }
            else if(childXMLObject instanceof PSObject)
            {
                o.getObjects().add((PSObject) childXMLObject);
            }
            else if(childXMLObject instanceof ObjectRef)
            {
                o.getObjectRefs().add((ObjectRef) childXMLObject);
            }        
            else
            {
                o.getUnknownXMLObjects().add(childXMLObject);
            }   
                    
        }
        
        @Override
        protected void processElementContent(XMLObject xmlObject, String elementContent) 
        {
            // no element content
        }

    }
    
}
