/**
 * 
 */
package org.openliberty.xmltooling.sasl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.xml.namespace.QName;

import org.opensaml.core.xml.AttributeExtensibleXMLObject;
import org.opensaml.core.xml.XMLObject;
import org.opensaml.core.xml.util.AttributeMap;
import org.opensaml.saml.saml2.core.RequestedAuthnContext;
import org.opensaml.xmlsec.signature.AbstractSignableXMLObject;

/**
 * @author tguion
 *
 */
public class SASLRequest extends AbstractSignableXMLObject implements AttributeExtensibleXMLObject {
	
	public static String DEFAULT_ELEMENT_LOCAL_NAME = "SASLRequest";

    /** Default element name. */
    public static final QName DEFAULT_ELEMENT_NAME = new QName(SASLConstants.NAMESPACE, DEFAULT_ELEMENT_LOCAL_NAME, SASLConstants.NAMESPACE_PREFIX);
	
    /** Name for the attribute which defines the mechanism. */
    public final static String MECHANISM_ATTRIB_NAME = "mechanism";
    
    /** Name for the attribute which defines the authzID. */
    public final static String AUTHZID_ATTRIB_NAME = "authzID";
        
    /** Name for the attribute which defines the advisoryAuthnID. */
    public final static String ADVISORYAUTHNID_ATTRIB_NAME = "advisoryAuthnID";
	
	/**
	 * Used to convey a list of one-or-more client-supported SASL mechanism
	 * names to the server, or to signal the server if the client wishes to abort
	 * the exchange. It is included on all <SASLRequest> messages sent by the
	 * client. [Required]
	 */
	private String mechanism;
	
	/**
	 * The authzID, also known as user identifier or username or Principal, that
	 * the client wishes to establish as the "authorization identity" per
	 * [RFC4422]. [Optional]
	 */
	private String authzID;
	
	/**
	 * The advisoryAuthnID may be used to advise the server what authentication
	 * identity will be asserted by the client via the selected SASL mechanism;
	 * i.e., it is a "hint." The advisoryAuthnID provides a means for server
	 * implementations to optimize their behavior on a per authentication
	 * identity basis. E.g. if a client requests to execute a certain SASL
	 * mechanism on behalf of some given authentication identity (represented by
	 * advisoryAuthnID) and authorization identity (represented by authzID)
	 * pair, the server can decide whether to proceed without having to execute
	 * the SASL mechanism (execution of which might involve more than a single
	 * round-trip). Server implementations that make use of the optional
	 * advisoryAuthnID attribute SHOULD be capable of processing initial
	 * <SASLRequest> messages that do not include the advisoryAuthnID attribute.
	 * [Optional]
	 */
	private String advisoryAuthnID;
	
	private AttributeMap otherAttributes;
	
	/**
	 * base64Binary data
	 */
	private Data data;
	
	private RequestedAuthnContext requestedAuthnContext;
	
    /** Extensions child elements (see org.opensaml.saml2.core.impl.RequestAbstractTypeImpl) */
    private Extensions extensions;

	/**
	 * @param namespaceURI
	 * @param elementLocalName
	 * @param namespacePrefix
	 */
	public SASLRequest(String namespaceURI, String elementLocalName,
			String namespacePrefix) {
		super(namespaceURI, elementLocalName, namespacePrefix);
		otherAttributes = new AttributeMap(this);
	}

	/**
	 * @return the advisoryAuthnID
	 */
	public String getAdvisoryAuthnID() {
		return advisoryAuthnID;
	}

	/**
	 * @param advisoryAuthnID the advisoryAuthnID to set
	 */
	public void setAdvisoryAuthnID(String advisoryAuthnID) {
		this.advisoryAuthnID = prepareForAssignment(this.advisoryAuthnID, advisoryAuthnID);
	}

	/**
	 * @return the authzID
	 */
	public String getAuthzID() {
		return authzID;
	}

	/**
	 * @param authzID the authzID to set
	 */
	public void setAuthzID(String authzID) {
		this.authzID = prepareForAssignment(this.authzID, authzID);
	}

	/**
	 * @return the data
	 */
	public Data getData() {
		return data;
	}

	/**
	 * @param data the data to set
	 */
	public void setData(Data data) {
		this.data = prepareForAssignment(this.data, data);
	}

	/**
	 * @return the mechanism
	 */
	public String getMechanism() {
		return mechanism;
	}

	/**
	 * @param mechanism the mechanism to set
	 */
	public void setMechanism(String mechanism) {
		this.mechanism = prepareForAssignment(this.mechanism, mechanism);
	}

	/**
	 * @return the requestedAuthnContext
	 */
	public RequestedAuthnContext getRequestedAuthnContext() {
		return requestedAuthnContext;
	}

	/**
	 * @param requestedAuthnContext the requestedAuthnContext to set
	 */
	public void setRequestedAuthnContext(RequestedAuthnContext requestedAuthnContext) {
		this.requestedAuthnContext = prepareForAssignment(this.requestedAuthnContext, requestedAuthnContext);
	}

    public Extensions getExtensions() {
        return this.extensions;
    }

    public void setExtensions(Extensions newExtensions) {
        this.extensions = prepareForAssignment(this.extensions, newExtensions);
    }

	/* (non-Javadoc)
	 * @see org.opensaml.xml.AttributeExtensibleXMLObject#getUnknownAttributes()
	 */
	public AttributeMap getUnknownAttributes() {
		return otherAttributes;
	}

	/* (non-Javadoc)
	 * @see org.opensaml.xml.XMLObject#getOrderedChildren()
	 */
	public List<XMLObject> getOrderedChildren() {
        ArrayList<XMLObject> children = new ArrayList<XMLObject>();

        if (null != data)
        {
        	children.add(data);
        }
        if (null != requestedAuthnContext)
        {
        	children.add(requestedAuthnContext);
        }
        if (null != extensions) 
        {
            children.add(extensions);
        }

        if (children.size() == 0) {
            return null;
        }

        return Collections.unmodifiableList(children);
	}
	

}
