package org.openliberty.xmltooling.soapbinding;

import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Map.Entry;

import javax.xml.namespace.QName;

import net.shibboleth.utilities.java.support.xml.AttributeSupport;
import net.shibboleth.utilities.java.support.xml.QNameSupport;

import org.openliberty.xmltooling.Konstantz;
import org.openliberty.xmltooling.OpenLibertyHelpers;
import org.opensaml.core.xml.AbstractXMLObjectBuilder;
import org.opensaml.core.xml.AttributeExtensibleXMLObject;
import org.opensaml.core.xml.XMLObject;
import org.opensaml.core.xml.config.XMLObjectProviderRegistrySupport;
import org.opensaml.core.xml.io.AbstractXMLObjectMarshaller;
import org.opensaml.core.xml.io.AbstractXMLObjectUnmarshaller;
import org.opensaml.core.xml.io.MarshallingException;
import org.opensaml.core.xml.io.UnmarshallingException;
import org.opensaml.core.xml.util.AttributeMap;
import org.opensaml.core.xml.util.XMLObjectChildrenList;
import org.opensaml.xmlsec.signature.AbstractSignableXMLObject;
import org.w3c.dom.Attr;
import org.w3c.dom.Element;


/**
 * A WSC that interacts with a user (typically through a web-site offered by the WSC) may need to indicate its readiness
 * to redirect the user agent of the user, or its readiness to pose questions to the user on behalf of other parties (such
 * as WSPs). The &lt;UserInteraction&gt; header block provides a means by which a WSC can indicate its preferences
 * and capabilities for interactions with requesting principals and, additionally, a SOAP fault message and HTTP redirect
 * profile that enables the WSC and WSP to cooperate in redirecting the requesting principal to the WSP and, after
 * browser interaction, back to the WSC.
 * 
 * <p>
 * <pre>
 *   &lt;xs:complexType name="UserInteractionHeaderType"&gt;
 *       &lt;xs:sequence&gt;
 *           &lt;xs:element name="InteractionService" type="wsa:EndpointReferenceType" minOccurs="0" maxOccurs="unbounded"/&gt;
 *       &lt;/xs:sequence&gt;
 *       &lt;xs:attribute name="interact" type="xs:string" use="optional" default="interactIfNeeded"/&gt;
 *       &lt;xs:attribute name="language" type="xs:NMTOKENS" use="optional"/&gt;
 *       &lt;xs:attribute name="redirect" type="xs:boolean" use="optional" default="0"/&gt;
 *       &lt;xs:attribute name="maxInteractTime" type="xs:integer" use="optional"/&gt;
 *       &lt;xs:anyAttribute namespace="##other" processContents="lax"/&gt;
 *   &lt;/xs:complexType&gt;
 * 
 *   &lt;xs:element name="UserInteraction" type="UserInteractionHeaderType"/&gt;   
 * </pre>
 * </p>
 * 
 * @author asa
 *
 */
public class UserInteraction extends AbstractSignableXMLObject implements AttributeExtensibleXMLObject
{

    public static final String LOCAL_NAME = "UserInteraction";

    // Elements
    /**
     * If present, this element MUST describe an interaction service hosted by the sender. This indicates that the
     * sender can process messages defined for the interaction service [LibertyInteract], posing questions from the
     * recipient of the message to the Principal.
     */
    private XMLObjectChildrenList<InteractionService> interactionServices;

    // Attributes
    /**
     * Indicates any preference that the sender has about interactions between the receiver and the requesting
     * principal. The value is a string, for which we define the values in {@link Interact} 
     */
    private String interact;            // default "interactIfNeeded"
    /**
     * This attribute indicates languages that the user is likely able to process. The value of this attribute is a space
     * separated list of language identification tags ([RFC3066]). The WSC can obtain this information from the 
     * HTTP ([RFC2616]) Accept-Language header, or by other means, for example from a personal profile service.
     */
    private String language;
    /**
     * An optional attribute to indicate that the sender supports the &lt;RedirectRequest&gt; element that a WSP may
     * include in a message to the WSC. The value is true or false. When absent the default behavior will be as if
     * false.
     */
    private Boolean redirect;           // optional default "0"
    /**
     * This is used to indicate the maximum time in seconds that the sender regards as reasonable for any possible
     * interaction. The receiver is not expected to start any interaction if it has reason to assume that such an
     * interaction is likely to take more time. In case an interaction is started and does seem to take longer the
     * receiver is expected to respond with a message that contains a InteractionTimeout status code to the sender.
     */
    private Integer maxInteractTime; 
    private AttributeMap unknownAttributes;

    // Attribute Names
    public static final String ATT_INTERACT = "interact";
    public static final String ATT_LANGUAGE = "language";
    public static final String ATT_REDIRECT = "redirect";
    public static final String ATT_MAX_INTERACT_TIME = "maxInteractTime";


    public UserInteraction()
    {
        super(Konstantz.SB_NS, LOCAL_NAME, Konstantz.SB_PREFIX);
    }

    public UserInteraction(String namespaceURI, String elementLocalName, String namespacePrefix)
    {
        super(namespaceURI, elementLocalName, namespacePrefix);
    }

    public String getInteract()
    {
        return interact;
    }

    public void setInteract(String interact)
    {
        this.interact = prepareForAssignment(this.interact, interact);
    }

    public String getLanguage()
    {
        return language;
    }

    public void setLanguage(String language)
    {
        this.language = prepareForAssignment(this.language, language);
    }

    public Boolean getRedirect()
    {
        return redirect;
    }

    public void setRedirect(Boolean redirect)
    {
        this.redirect = prepareForAssignment(this.redirect, redirect);
    }

    public Integer getMaxInteractTime()
    {
        return maxInteractTime;
    }

    public void setMaxInteractTime(Integer maxInteractTime)
    {
        this.maxInteractTime = prepareForAssignment(this.maxInteractTime, maxInteractTime);
    }

    public XMLObjectChildrenList<InteractionService> getInteractionServices()
    {
        if(null==interactionServices) interactionServices = new XMLObjectChildrenList<InteractionService>(this);
        return interactionServices;
    }

    public List<XMLObject> getOrderedChildren()
    {
        List<XMLObject> children = new LinkedList<XMLObject>();

        if(null!=interactionServices) children.addAll(interactionServices);

        return Collections.unmodifiableList(children);
    }

    public AttributeMap getUnknownAttributes()
    {
        if(null==unknownAttributes)unknownAttributes = new AttributeMap(this);
        return unknownAttributes;
    }

    /**
     * Implementors may choose to define additional values to indicate finer grained control over the user interactions.
     * 
     * @author asa
     *
     */
    public enum Interact
    {
        /**
         * Indicates to the recipient that it should interact with the requesting principal if needed to satisfy
         * the ID-WSF request. This is the default.
         */
        INTERACT_IF_NEEDED("InteractIfNeeded"),
        /**
         * Indicates to the recipient that it MUST NOT interact with the requesting principal, either directly
         * or indirectly. The sender prefers to receive an error response over the situation where the requesting principal
         * would be distracted by an interaction.
         */
        DO_NOT_INTERACT("DoNotInteract"),
        /**
         * Indicates to the recipient that it MAY interact with the requesting principal only if an
         * explicit policy for the offered service so requires. The sender prefers to receive an error response over the situation
         * where the WSP would obtain service response data (e.g. Personal Profile data) from the resource owner, but the
         * sender does prefer to obtain a positive service response even if that requires policy-related interaction for e.g.
         * obtaining consent.
         */
        DO_NOT_INTERACT_FOR_DATA("DoNotInteractForData");

        private String code;

        private Interact(String code)
        {
            this.code = code;
        }

        public Interact getInteractForCode(String code)
        {
            if(null!=code)
            {
                for(Interact interact : Interact.values())
                {
                    if(interact.getCode().equals(code)) return interact;
                }               
            }
            return null;
        }

        public String getCode()
        {
            return code;
        }

    }

    


    /**
     * Marshall attributes into the DOM
     * 
     * @author asa
     *
     */
    public static class Marshaller extends AbstractXMLObjectMarshaller
    {

        @Override
        protected void marshallAttributes(XMLObject xmlObject, Element domElement) throws MarshallingException
        {
            UserInteraction obj = (UserInteraction)xmlObject;

            if(obj.getInteract() != null) 
            {
                domElement.setAttributeNS(null, UserInteraction.ATT_INTERACT, obj.getInteract());
            } 
            
            if(obj.getLanguage() != null) 
            {
                domElement.setAttributeNS(null, UserInteraction.ATT_LANGUAGE, obj.getLanguage());
            } 

            if(obj.getMaxInteractTime() != null) 
            {
                domElement.setAttributeNS(null, UserInteraction.ATT_MAX_INTERACT_TIME, obj.getMaxInteractTime().toString());
            } 
            
            if(obj.getRedirect() != null) 
            {
                domElement.setAttributeNS(null, UserInteraction.ATT_REDIRECT, OpenLibertyHelpers.stringFromBoolean(obj.getRedirect(), OpenLibertyHelpers.FALSE_STR));
            } 

            // marshall the unknown attributes
            Attr attr;
            for(Entry<QName, String> entry: obj.getUnknownAttributes().entrySet())
            {
                attr = AttributeSupport.constructAttribute(domElement.getOwnerDocument(), entry.getKey());
                attr.setValue(entry.getValue());
                domElement.setAttributeNodeNS(attr);
                if (XMLObjectProviderRegistrySupport.isIDAttribute(entry.getKey()) || obj.getUnknownAttributes().isIDAttribute(entry.getKey())) 
                {
                    attr.getOwnerElement().setIdAttributeNode(attr, true);
                }
            }

        }


        @Override
        protected void marshallElementContent(XMLObject xmlObject, Element domElement) throws MarshallingException 
        {
            // no element content
        }

    }


    /**
     * Internal class that handles unmarshalling from the DOM
     * 
     * @author asa
     *
     */
    public static class Unmarshaller extends AbstractXMLObjectUnmarshaller
    {       

        @Override
        protected void processChildElement(XMLObject parentXMLObject, XMLObject childXMLObject) throws UnmarshallingException
        {            
            UserInteraction obj = (UserInteraction) parentXMLObject;

            if(childXMLObject instanceof InteractionService)
            {
                obj.getInteractionServices().add((InteractionService)childXMLObject);                   
            }
        }

        @Override
        protected void processAttribute(XMLObject xmlObject, Attr attribute) throws UnmarshallingException
        {
            UserInteraction obj = (UserInteraction) xmlObject;

            String attLocalName = attribute.getLocalName();
            if(attLocalName.equals(UserInteraction.ATT_INTERACT)) 
            {
                obj.setInteract(attribute.getValue());
            } 
            else if(attLocalName.equals(UserInteraction.ATT_LANGUAGE)) 
            {
                obj.setLanguage(attribute.getValue());
            }
            else if(attLocalName.equals(UserInteraction.ATT_MAX_INTERACT_TIME)) 
            {
                obj.setMaxInteractTime(OpenLibertyHelpers.integerFromString(attribute.getValue()));
            }
            else if(attLocalName.equals(UserInteraction.ATT_REDIRECT)) 
            {
                obj.setRedirect(OpenLibertyHelpers.booleanFromString(attribute.getValue()));
            }
            // xs:anyAttribute
            else 
            {
                QName attribQName = QNameSupport.getNodeQName(attribute);
                if (attribute.isId()) 
                {
                    obj.getUnknownAttributes().registerID(attribQName);
                }
                obj.getUnknownAttributes().put(attribQName, attribute.getValue());
            }                           
        }

        @Override
        protected void processElementContent(XMLObject xmlObject, String elementContent)
        {
            // no element content            
        }

    }




    /**
     * Static Internal Builder
     * 
     * @author asa
     *
     */
    public static class Builder extends AbstractXMLObjectBuilder<UserInteraction> 
    {

        @Override
		public UserInteraction buildObject(String namespaceURI, String localName, String namespacePrefix) 
        {
            return new UserInteraction(namespaceURI, localName, namespacePrefix);
        }

    }


    
}
