package org.openliberty.xmltooling.subs;

import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

import org.joda.time.DateTime;
import org.openliberty.xmltooling.Konstantz;
import org.openliberty.xmltooling.OpenLibertyHelpers;
import org.openliberty.xmltooling.utility_2_0.Extension;
import org.opensaml.core.xml.AbstractXMLObject;
import org.opensaml.core.xml.AbstractXMLObjectBuilder;
import org.opensaml.core.xml.XMLObject;
import org.opensaml.core.xml.io.AbstractXMLObjectMarshaller;
import org.opensaml.core.xml.io.AbstractXMLObjectUnmarshaller;
import org.opensaml.core.xml.io.MarshallingException;
import org.opensaml.core.xml.io.UnmarshallingException;
import org.opensaml.core.xml.util.XMLObjectChildrenList;
import org.w3c.dom.Attr;
import org.w3c.dom.Element;

/**
 * The &lt;Subscription&gt; element contains all the parameters for a subscription. It defines what data a WSC wants to have, 
 * where it should be sent, when a subscription expires, which events should trigger notifications, etc. 
 * <p>
 * <pre>
   <xs:complexType name="SubscriptionType">
    <xs:sequence>
      <xs:element ref="subs:RefItem"    minOccurs="0" maxOccurs="unbounded"/>
      <xs:element ref="lu:Extension"    minOccurs="0" maxOccurs="unbounded"/>
    </xs:sequence>
    <xs:attribute name="subscriptionID" use="required" type="lu:IDType"/>
    <xs:attribute name="notifyToRef" use="required" type="xs:anyURI"/>
    <xs:attribute name="adminNotifyToRef" use="optional" type="notifyToRef"/>
    <xs:attribute name="starts" use="optional" type="xs:dateTime"/>
    <xs:attribute name="expires" use="optional" type="xs:dateTime"/>
    <xs:attribute name="id" use="optional" type="xs:ID"/>
    
    <xs:attribute name="includeData" use="optional">
      <xs:simpleType>
        <xs:restriction base="xs:string">
          <xs:enumeration value="Yes"/>
          <xs:enumeration value="No"/>
          <xs:enumeration value="YesWithCommonAttributes"/>
        </xs:restriction>
      </xs:simpleType>
    </xs:attribute>
  </xs:complexType>
  </pre>
 * @author asa
 *
 */
public class Subscription extends AbstractXMLObject
{
    
    public static String LOCAL_NAME = "Subscription";
    
    // Attributes
    private String subscriptionID;          // lu:IDType, required
    private String notifyToRef;             // xs:anyURI, required
    private String adminNotifyToRef;        // notifyToRef, oprional
    private DateTime starts;                // optional
    private DateTime expires;               // optional
    private String id;                      // xs:ID, optional
    private IncludeDataEnum includeData;    //     
    
    // Attribute names
    public static final String ATT_SUBSCRIPTION_ID = "subscriptionID";
    public static final String ATT_NOTIFY_TO_REF = "notifyToRef";
    public static final String ATT_ADMIN_NOTIFY_TO_REF = "adminNotifyToRef";
    public static final String ATT_STARTS = "starts";
    public static final String ATT_EXPIRES = "expires";
    public static final String ATT_ID = "id";
    public static final String ATT_INCLUDE_DATA = "includeData";
    
    // Elements
    private XMLObjectChildrenList<Extension> extensions;
    private XMLObjectChildrenList<RefItem> refItems;

    
    public Subscription()
    {
        super(Konstantz.SUBS_NS, LOCAL_NAME, Konstantz.SUBS_PREFIX);
    }
        
    protected Subscription(String namespaceURI, String elementLocalName, String namespacePrefix)
    {
        super(namespaceURI, elementLocalName, namespacePrefix);
    }

    
    // Attributes

    public String getNotifyToRef()
    {
        return notifyToRef;
    }


    public void setNotifyToRef(String notifyToRef)
    {
        this.notifyToRef = prepareForAssignment(this.notifyToRef, notifyToRef);;
    }


    public String getAdminNotifyToRef()
    {
        return adminNotifyToRef;
    }


    public void setAdminNotifyToRef(String adminNotifyToRef)
    {
        this.adminNotifyToRef = prepareForAssignment(this.adminNotifyToRef, adminNotifyToRef);
    }


    public DateTime getStarts()
    {
        return starts;
    }


    public void setStarts(DateTime starts)
    {
        this.starts = prepareForAssignment(this.starts, starts);
    }


    public DateTime getExpires()
    {
        return expires;
    }


    public void setExpires(DateTime expires)
    {
        this.expires = prepareForAssignment(this.expires, expires);
    }


    public String getId()
    {
        return id;
    }


    public void setId(String id)
    {
        this.id = prepareForAssignment(this.id, id);
    }


    public IncludeDataEnum getIncludeData()
    {
        return includeData;
    }


    public void setIncludeData(IncludeDataEnum includeData)
    {
        this.includeData = prepareForAssignment(this.includeData, includeData);
    }


    /**
     * The subscriptionID XML attribute MUST be unique within all subscriptions a WSC has at a WSP 
     * 
     * @param subscriptionID
     */
    public void setSubscriptionID(String subscriptionID)
    {
        this.subscriptionID = prepareForAssignment(this.subscriptionID, subscriptionID);
    }
    
    /**
     * A unique id for this WSC
     * 
     * @return
     */
    public String getSubscriptionID()
    {
        return subscriptionID;
    }
    
    
    

    public XMLObjectChildrenList<RefItem> getRefItems()
    {
        if(null==refItems) refItems = new XMLObjectChildrenList<RefItem>(this);
        return refItems;
    }
    

    public XMLObjectChildrenList<Extension> getExtensions()
    {
        if(null==extensions) extensions = new XMLObjectChildrenList<Extension>(this);
        return extensions;
    }
    
    
    public List<XMLObject> getOrderedChildren()
    {
        List<XMLObject> children = new LinkedList<XMLObject>();
        
        if(null!=extensions) children.addAll(extensions); 
        if(null!=refItems) children.addAll(refItems);
        
        return Collections.unmodifiableList(children);
    }

    
    
    public enum IncludeDataEnum
    {
        YES("Yes"),
        NO("No"),
        YES_WITH_COMMON_ATTRIBUTES("YesWithCommonAttributes");
        
        private String value;
        
        private IncludeDataEnum(String value)
        {
            this.value = value;
        }
        
        public String getValue()
        {
            return this.value;
        }
 
        public static IncludeDataEnum getIncludeDataEnumForValue(String value)
        {
            if(value!=null)
            {
                if(value.equals(YES.value)) return YES;
                else if(value.equals(NO.value)) return NO;
                else if(value.equals(YES_WITH_COMMON_ATTRIBUTES.value)) return YES_WITH_COMMON_ATTRIBUTES;
            }
            return null;
        }
        
        
    }
 
    /**
     * Internal builder class simplifies the file structure
     * 
     * @author asa
     *
     */
    public static class Builder extends AbstractXMLObjectBuilder<Subscription> 
    {

        @Override
		public Subscription buildObject(String namespaceURI, String localName, String namespacePrefix) 
        {
            return new Subscription(namespaceURI, localName, namespacePrefix);
        }

    }
    
    /**
     * Static Marshaller
     * 
     * @author asa
     *
     */
    public static class Marshaller extends AbstractXMLObjectMarshaller
    {    

        @Override
        protected void marshallAttributes(XMLObject xmlObject, Element domElement) throws MarshallingException 
        {            
            Subscription object = (Subscription)xmlObject;

            if(object.getSubscriptionID() != null) 
            {
                domElement.setAttributeNS(null, Subscription.ATT_SUBSCRIPTION_ID, object.getSubscriptionID());
            }
            
            if(object.getNotifyToRef() != null)
            {
                domElement.setAttributeNS(null, Subscription.ATT_NOTIFY_TO_REF, object.getNotifyToRef());
            }

            if(object.getAdminNotifyToRef() != null)
            {
                domElement.setAttributeNS(null, Subscription.ATT_ADMIN_NOTIFY_TO_REF, object.getAdminNotifyToRef());
            }
            
            if(object.getStarts() != null)
            {
                domElement.setAttributeNS(null, Subscription.ATT_STARTS, OpenLibertyHelpers.stringForDateTime( object.getStarts() ));
            }
        
            if(object.getExpires() != null)
            {
                domElement.setAttributeNS(null, Subscription.ATT_EXPIRES, OpenLibertyHelpers.stringForDateTime( object.getExpires() ));
            }

            if(object.getId() != null)
            {
                domElement.setAttributeNS(null, Subscription.ATT_ID, object.getId());
            }

            if(object.getIncludeData() != null)
            {
                domElement.setAttributeNS(null, Subscription.ATT_INCLUDE_DATA, object.getIncludeData().getValue());
            }
            
        }


        @Override
        protected void marshallElementContent(XMLObject xmlObject, Element domElement) throws MarshallingException 
        {
            // NO TEXT CONTENT
        }


    }
    
    
    /**
     * Static Unmarshaller
     * 
     * @author asa
     *
     */
    public static class Unmarshaller extends AbstractXMLObjectUnmarshaller
    {
                
        @Override
        protected void processAttribute(XMLObject xmlObject, Attr attribute) throws UnmarshallingException 
        {
            Subscription object = (Subscription) xmlObject;

            if (attribute.getLocalName().equals(Subscription.ATT_SUBSCRIPTION_ID)) 
            {
                object.setSubscriptionID(attribute.getValue());
            }
            else if (attribute.getLocalName().equals(Subscription.ATT_NOTIFY_TO_REF)) 
            {
                object.setNotifyToRef(attribute.getValue());
            }
            else if (attribute.getLocalName().equals(Subscription.ATT_ADMIN_NOTIFY_TO_REF)) 
            {
                object.setAdminNotifyToRef(attribute.getValue());
            }
            else if (attribute.getLocalName().equals(Subscription.ATT_STARTS)) 
            {
                object.setStarts(OpenLibertyHelpers.dateTimeForString( attribute.getValue() ));
            }
            else if (attribute.getLocalName().equals(Subscription.ATT_EXPIRES)) 
            {
                object.setExpires(OpenLibertyHelpers.dateTimeForString( attribute.getValue() ));
            }
            else if (attribute.getLocalName().equals(Subscription.ATT_ID)) 
            {
                object.setId(attribute.getValue());
            }
            else if (attribute.getLocalName().equals(Subscription.ATT_INCLUDE_DATA)) 
            {
                object.setIncludeData( Subscription.IncludeDataEnum.getIncludeDataEnumForValue(attribute.getValue()));
            }


        }

        
        @Override
        protected void processChildElement(XMLObject parentXMLObject, XMLObject childXMLObject) throws UnmarshallingException 
        {   
            Subscription object = (Subscription) parentXMLObject;        
            
            if (childXMLObject instanceof Extension) 
            {
                object.getExtensions().add((Extension) childXMLObject);
            }
            else if(childXMLObject instanceof RefItem)
            {
                object.getRefItems().add((RefItem) childXMLObject);
            }
                    
        }

        
        @Override
        protected void processElementContent(XMLObject xmlObject, String elementContent) 
        {
            // NO CONTENT
        }


    }
    
    
}
