package org.openliberty.xmltooling.wsa;

import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

import javax.xml.namespace.QName;

import org.joda.time.DateTime;
import org.openliberty.xmltooling.Konstantz;
import org.openliberty.xmltooling.disco.SecurityContext;
import org.openliberty.xmltooling.disco.SecurityMechID;
import org.opensaml.core.xml.ElementExtensibleXMLObject;
import org.opensaml.core.xml.XMLObject;
import org.opensaml.core.xml.util.IndexedXMLObjectChildrenList;
import org.opensaml.xmlsec.signature.AbstractSignableXMLObject;

public class EndpointReference extends AbstractSignableXMLObject implements ElementExtensibleXMLObject
{

    public static String LOCAL_NAME = "EndpointReference";

    public static String ATT_NOT_ON_OR_AFTER = "notOnOrAfter";
    public static String ATT_ID = "id";

    private String id;
    private DateTime notOnOrAfter;

    // private String value;
    private Address address;
    private Metadata metadata;
    private ReferenceParameters referenceParameters;

    private IndexedXMLObjectChildrenList<XMLObject> unknownXMLObjects;

    public EndpointReference()
    {
        super(Konstantz.WSA_NS, LOCAL_NAME, Konstantz.WSA_PREFIX); 
        unknownXMLObjects = new IndexedXMLObjectChildrenList<XMLObject>(this);
    }

    protected EndpointReference(String namespaceURI, String elementLocalName, String namespacePrefix) 
    {
        super(namespaceURI, elementLocalName, namespacePrefix);
        unknownXMLObjects = new IndexedXMLObjectChildrenList<XMLObject>(this);
    }

    public EndpointReference freshEndpointReference()
    {
    	return this;
    }
    
    @Override
    public String toString()
    {        
        return simplePrettyDescription();
    }

    /**
     * Returns a simple representation of this EndpointReference
     * 
     * @return
     */
    public String simplePrettyDescription()
    {
        StringBuffer descBuff = new StringBuffer();
        Metadata metadata = this.getMetadata();
        if(null!=metadata)
        {
            if(null!=metadata.getMetadataAbstract())
            {
                String abstrakt = metadata.getMetadataAbstract().getValue();
                if(null!=abstrakt)
                {
                    descBuff.append(abstrakt);
                }
            }

            if(descBuff.length()==0)
            {
                if(null!=getAddress())
                {
                    descBuff.append(getAddress().getValue());
                }
                else
                {
                    descBuff.append(super.toString());
                }
            }

            descBuff.append(" (");
            for(SecurityContext securityContext : this.getMetadata().getSecurityContexts())
            {
                for(SecurityMechID securityMechID : securityContext.getSecurityMechIDs())
                {
                    descBuff.append(securityMechID.getValue()).append(" ");
                }
            }
            descBuff.append(")");
            return descBuff.toString();
        }

        if(null!=getAddress())
        {
            return getAddress().getValue();
        }
        else
        {
            return super.toString();
        }

    }


    public List<XMLObject> getUnknownXMLObjects() 
    {		
        return unknownXMLObjects;
    }

    @SuppressWarnings("unchecked")
    public List<XMLObject> getUnknownXMLObjects(QName typeOrName)
    {
        return (List<XMLObject>) unknownXMLObjects.subList(typeOrName);
    }

    public List<XMLObject> getOrderedChildren() 
    {
        List<XMLObject> children = new LinkedList<XMLObject>();

        children.add(address);
        children.add(metadata);        
        children.add(referenceParameters);        
        if(null!=unknownXMLObjects) children.addAll(unknownXMLObjects);

        return Collections.unmodifiableList(children);
    }	


//  Attributes

    public String getId() 
    {
        return id;
    }

    public void setId(String id) 
    {
        registerOwnID(this.id, id);
        this.id = id;
    }	

    public void setNotOnOrAfter(DateTime notOnOrAfter) 
    {
        this.notOnOrAfter = prepareForAssignment(this.notOnOrAfter, notOnOrAfter);
    }

    public DateTime getNotOnOrAfter() 
    { 
        return notOnOrAfter; 
    }

    public void setAddress(Address address) 
    {
        this.address = prepareForAssignment(this.address, address);
    }

    public Address getAddress() 
    { 
        return this.address; 
    }

    public void setMetadata(Metadata metadata) 
    {
        this.metadata = prepareForAssignment(this.metadata, metadata);
    }

    public Metadata getMetadata() 
    { 
        return this.metadata; 
    }

    public void setReferenceParameters(ReferenceParameters referenceParameters) 
    {
        this.referenceParameters = prepareForAssignment(this.referenceParameters, referenceParameters);
    }

    public ReferenceParameters getReferenceParameters()
    {
        return referenceParameters;
    }



    public static String[] securityMechIDStringsFromEPR(EndpointReference epr)
    {        
        List<SecurityContext> securityContexts = epr.getMetadata().getSecurityContexts();

        if(null!=securityContexts)
        {
            List<String> secMechs = new ArrayList<String>();

            for(SecurityContext securityContext : securityContexts)
            {
                List<SecurityMechID> securityMechIDs =  securityContext.getSecurityMechIDs();
                if(null!=securityMechIDs)
                {
                    for(SecurityMechID securityMechID : securityMechIDs)
                    {
                        secMechs.add(securityMechID.getValue());
                    }
                }
            }

            if(secMechs.size()>0)
            {
                String[] secMechArray = new String[secMechs.size()];      
                secMechs.toArray(secMechArray);
                return secMechArray;
            }
        }

        return null;
    }



}
