/**
 * 
 */
package org.openliberty.xmltooling.wsa;

import java.util.List;
import java.util.Map.Entry;
import java.util.UUID;

import javax.xml.namespace.QName;

import net.shibboleth.utilities.java.support.security.SecureRandomIdentifierGenerationStrategy;
import net.shibboleth.utilities.java.support.xml.AttributeSupport;
import net.shibboleth.utilities.java.support.xml.ElementSupport;
import net.shibboleth.utilities.java.support.xml.QNameSupport;

import org.apache.log4j.Logger;
import org.openliberty.xmltooling.Konstantz;
import org.opensaml.core.xml.AbstractXMLObjectBuilder;
import org.opensaml.core.xml.AttributeExtensibleXMLObject;
import org.opensaml.core.xml.XMLObject;
import org.opensaml.core.xml.config.XMLObjectProviderRegistrySupport;
import org.opensaml.core.xml.io.AbstractXMLObjectMarshaller;
import org.opensaml.core.xml.io.AbstractXMLObjectUnmarshaller;
import org.opensaml.core.xml.io.MarshallingException;
import org.opensaml.core.xml.io.UnmarshallingException;
import org.opensaml.core.xml.util.AttributeMap;
import org.opensaml.xmlsec.signature.AbstractSignableXMLObject;
import org.w3c.dom.Attr;
import org.w3c.dom.Element;

/**
 * Values of the &lt;wsa:MessageID&gt; header block MUST satisfy the following property:
 * <blockquote>
 * <tt>
 * Any party that assigns a value to a <wsa:MessageID> header block MUST ensure that there is
 * negligible probability that that party or any other party will accidentally assign the same identifier
 * to any other message.
 * </tt>
 * </blockquote>
 * <p>
 * The mechanism by which SOAP-based ID-* senders or receivers ensure that an identifier is unique is left to
 * implementations. In the case that a pseudo-random technique is employed, the above requirement MAY be met by
 * randomly choosing a value 160 bits in length.
 * 
 * @author tguion
 * @author asa
 *
 */
public class MessageID extends AbstractSignableXMLObject implements AttributeExtensibleXMLObject 
{
    private static Logger log = Logger.getLogger(MessageID.class);
    
    private static SecureRandomIdentifierGenerationStrategy idGenerator;
    
    public static final String DEFAULT_ELEMENT_LOCAL_NAME = "MessageID"; 

    public static final QName DEFAULT_ELEMENT_NAME = QNameSupport.constructQName(Konstantz.WSA_NS, DEFAULT_ELEMENT_LOCAL_NAME, Konstantz.WSA_PREFIX);

    private String value;

    /**
     * Support for AttributeExtensibleXMLObject interface
     */
    private AttributeMap otherAttributes;

    /**
     * Constructor that automatically sets the value of the message id.  This constructor will
     * only be called by users of the client library who wish to skip calling the MessageID$Builder
     * for the same purpose.
     */
    public MessageID()
    {
        super(Konstantz.WSA_NS, DEFAULT_ELEMENT_LOCAL_NAME, Konstantz.WSA_PREFIX);   
        
        if(null==idGenerator) idGenerator = new SecureRandomIdentifierGenerationStrategy();            
        
        if(null!=idGenerator) this.setValue("olid:"+idGenerator.generateIdentifier());
        else this.setValue("uuid:"+UUID.randomUUID().toString());
        
    }

    /**
     * Constructor
     * 
     * @param namespaceURI the namespace the element is in
     * @param elementLocalName the local name of the XML element this Object represents
     * @param namespacePrefix the prefix for the given namespace
     */
    protected MessageID(String namespaceURI, String elementLocalName, String namespacePrefix) 
    {
        super(namespaceURI, elementLocalName, namespacePrefix);
        otherAttributes = new AttributeMap(this);
    }



    /**
     * {@inheritDoc}
     */
    public String getValue() 
    {
        return value;
    }

    /**
     * {@inheritDoc}
     */
    public void setValue(String newValue) 
    {
        value = prepareForAssignment(value, newValue);
    }

    /* (non-Javadoc)
     * @see org.opensaml.xml.AttributeExtensibleXMLObject#getUnknownAttributes()
     */
    public AttributeMap getUnknownAttributes() 
    {   
        // lazy initialization
        if(null==otherAttributes) otherAttributes = new AttributeMap(this);
        return otherAttributes;
    }

    /* (non-Javadoc)
     * @see org.opensaml.xml.XMLObject#getOrderedChildren()
     */
    public List<XMLObject> getOrderedChildren() 
    {
        return null;
    }


    
    /**
     * Internal Unmarshaller
     * @author asa
     *
     */
    public static class Unmarshaller extends AbstractXMLObjectUnmarshaller 
    {

        /* (non-Javadoc)
         * @see org.opensaml.xml.io.AbstractXMLObjectUnmarshaller#processAttribute(org.opensaml.xml.XMLObject, org.w3c.dom.Attr)
         */
        @Override
        protected void processAttribute(XMLObject xmlObject, Attr attribute) throws UnmarshallingException 
        {
            MessageID obj = (MessageID) xmlObject;
            QName attribQName = QNameSupport.getNodeQName(attribute);
            if (attribute.isId()) 
            {
                obj.getUnknownAttributes().registerID(attribQName);
            }
            obj.getUnknownAttributes().put(attribQName, attribute.getValue());
        }

        /* (non-Javadoc)
         * @see org.opensaml.xml.io.AbstractXMLObjectUnmarshaller#processChildElement(org.opensaml.xml.XMLObject, org.opensaml.xml.XMLObject)
         */
        @Override
        protected void processChildElement(XMLObject parentXMLObject, XMLObject childXMLObject) throws UnmarshallingException 
        {
            // no children
        }

        /* (non-Javadoc)
         * @see org.opensaml.xml.io.AbstractXMLObjectUnmarshaller#processElementContent(org.opensaml.xml.XMLObject, java.lang.String)
         */
        @Override
        protected void processElementContent(XMLObject xmlObject, String elementContent) 
        {
            MessageID obj = (MessageID) xmlObject;
            
            if(elementContent != null)
            {
                obj.setValue(elementContent.trim());
            }
        }

    }

    
    

    /**
     * Internal Marshaller
     * @author asa
     *
     */
    public static class Marshaller extends AbstractXMLObjectMarshaller 
    {

        /* (non-Javadoc)
         * @see org.opensaml.xml.io.AbstractXMLObjectMarshaller#marshallAttributes(org.opensaml.xml.XMLObject, org.w3c.dom.Element)
         */
        @Override
        protected void marshallAttributes(XMLObject xmlObject, Element domElement) throws MarshallingException 
        {
            MessageID obj = (MessageID) xmlObject;    

            Attr attr;
            for(Entry<QName, String> entry: obj.getUnknownAttributes().entrySet())
            {
                attr = AttributeSupport.constructAttribute(domElement.getOwnerDocument(), entry.getKey());
                attr.setValue(entry.getValue());
                domElement.setAttributeNodeNS(attr);
                if (XMLObjectProviderRegistrySupport.isIDAttribute(entry.getKey()) || obj.getUnknownAttributes().isIDAttribute(entry.getKey())) 
                {
                    attr.getOwnerElement().setIdAttributeNode(attr, true);
                }
            }
        }

        /* (non-Javadoc)
         * @see org.opensaml.xml.io.AbstractXMLObjectMarshaller#marshallElementContent(org.opensaml.xml.XMLObject, org.w3c.dom.Element)
         */
        @Override
        protected void marshallElementContent(XMLObject xmlObject, Element domElement) throws MarshallingException 
        {
            MessageID obj = (MessageID) xmlObject;
            ElementSupport.appendTextContent(domElement, obj.getValue());
        }

    }


    /**
     * Internal Builder
     * 
     * @author asa
     *
     */
    public static class Builder extends AbstractXMLObjectBuilder<MessageID> 
    {
        @Override
        public MessageID buildObject(String namespaceURI, String localName, String namespacePrefix) 
        {
            return new MessageID(namespaceURI, localName, namespacePrefix);
        }

    }





}
