/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.servlet.impl;

import javax.annotation.Nonnull;

import jakarta.servlet.http.HttpServletResponse;
import net.shibboleth.shared.primitive.NonnullSupplier;

/**
 * An implementation of {@link NonnullSupplier} which looks up the current thread-local
 * servlet response obtained from {@link HttpServletRequestResponseContext}, raising an
 * {@link IllegalStateException} if null.
 */
public class ThreadLocalHttpServletResponseSupplier implements NonnullSupplier<HttpServletResponse> {

    /**
     * {@inheritDoc}
     * Get the current HttpServletResponse from ThreadLocal storage.
     *
     * @return the current response
     */
    @Nonnull public HttpServletResponse get() {
        final HttpServletResponse response = HttpServletRequestResponseContext.getResponse();
        if (response == null) {
            throw new IllegalStateException(
                    "Current HttpServletResponse has not been loaded via HttpServletRequestResponseContext");
        }
        
        return response;
    }

}