/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.spring.custom;

import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.slf4j.Logger;

import org.springframework.beans.factory.config.BeanDefinitionHolder;
import org.springframework.beans.factory.support.BeanDefinitionRegistry;
import org.springframework.beans.factory.xml.AbstractSingleBeanDefinitionParser;

import net.shibboleth.shared.annotation.constraint.NotEmpty;
import net.shibboleth.shared.primitive.LoggerFactory;

/**
 * This is the base parser for all our custom syntax.
 *   
 * It is a trivial extension of {@link AbstractCustomBeanDefinitionParser}, but allows
 * for a single point of change to our behaviors. 
 */
public class AbstractCustomBeanDefinitionParser extends AbstractSingleBeanDefinitionParser {

    /**
     * Location to look for {@link Properties} that may affect custom parsing.
     */
    @Nonnull @NotEmpty
    public static final String CUSTOM_PARSER_PROPERTIES_LOCATION = "META-INF/net/shibboleth/spring/parser.properties";

    /** Property bad for custom parser properties to expose to subclasses. */
    @Nullable private static Properties customProperties;
    
    /** Logger. */
    @Nonnull private static final Logger LOG = LoggerFactory.getLogger(AbstractCustomBeanDefinitionParser.class);

    /** {@inheritDoc}
     * The override is to warn if there is an ID clash within the same context.
     * */
    protected void registerBeanDefinition(@Nonnull final BeanDefinitionHolder definition,
            @Nonnull final BeanDefinitionRegistry registry) {
        if (registry.containsBeanDefinition(definition.getBeanName())) {
            final String claz = definition.getBeanDefinition().getBeanClassName();
            LOG.warn("Duplicate Definition '{}' of type '{}'", definition.getBeanName(), claz);
        }
        super.registerBeanDefinition(definition, registry);
    }

    /**
     * Get a custom parser property, or return the default value specified.
     * 
     * @param name property name
     * @param defaultValue default value
     * 
     * @return the property value or the default if not set
     * 
     * @since 9.0.0
     */
    @Nullable public static String getCustomProperty(@Nonnull final String name, @Nullable final String defaultValue) {
        
        synchronized (AbstractCustomBeanDefinitionParser.class) {
            if (customProperties == null) {
                final Properties props = new Properties();
                try (final InputStream is = Thread.currentThread().getContextClassLoader().getResourceAsStream(
                                CUSTOM_PARSER_PROPERTIES_LOCATION)) {
                    // NOTE: in this invocation style via class loader, resource should NOT have a leading slash
                    // because all names are absolute. This is unlike Class.getResourceAsStream 
                    // where a leading slash is required for absolute names.
                    if (is != null) {
                        props.load(is);
                    } else {
                        LOG.debug("No custom Spring parser configuration properties loaded from {}",
                                CUSTOM_PARSER_PROPERTIES_LOCATION);
                    }
                } catch (final IOException e) {
                    LOG.warn("Problem attempting to load custom Spring parser configuration properties '" 
                            + CUSTOM_PARSER_PROPERTIES_LOCATION + "' from classpath", e);
                } finally {
                    customProperties = props;
                }
            }
        }
        
        assert customProperties != null;
        return customProperties.getProperty(name, defaultValue);
    }

}