/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.spring.factory;

import javax.annotation.Nullable;

import org.springframework.beans.factory.BeanCreationException;

import net.shibboleth.shared.component.DestructableComponent;
import net.shibboleth.shared.component.InitializableComponent;
import net.shibboleth.shared.primitive.LoggerFactory;

/**
 * A Factory bean which is aware of the component lifecycle interfaces.
 * 
 * @param <T> The type to implement
 */
public abstract class AbstractComponentAwareFactoryBean<T> extends AbstractFactoryBean<T> {
    
    /** Flag controlling null behavior. */
    private boolean throwIfNull;
    
    /**
     * Constructor.
     */
    public AbstractComponentAwareFactoryBean() {
        throwIfNull = true;
    }

    /**
     * {@inheritDoc}
     * 
     * <p>Call our destroy method if aposite.</p>
     */
    @Override protected void destroyInstance(@Nullable final T instance) throws Exception {
        super.destroyInstance(instance);
        if (instance instanceof DestructableComponent dc) {
            dc.destroy();
        }
    }
    
    /**
     * Sets whether to raise an exception if a null is returned from {@link #doCreateInstance}.
     * 
     * <p>Defaults to true.</p>
     * 
     * @param flag
     * 
     * @since 9.1.0
     */
    public void setThrowIfNull(final boolean flag) {
        throwIfNull = flag;
    }

    /**
     * <p>Call the parent class to create the object, then initialize it aposite.</p>
     * 
     * {@inheritDoc}.
     */
    @Override
    @Nullable protected final T createInstance() throws Exception {
        if (!isSingleton()) {
            LoggerFactory.getLogger(AbstractComponentAwareFactoryBean.class).error(
                    "Configuration error: {} should not be used to create prototype beans."
                            + "  Destroy is never called for prototype beans", AbstractComponentAwareFactoryBean.class);
            throw new BeanCreationException("Do not use AbstractComponentAwareFactoryBean to create prototype beans");
        }
        
        final T theBean = doCreateInstance();
        
        if (throwIfNull && theBean == null) {
            throw new BeanCreationException("Bean was null and throwIfNull was set");
        }
        
        if (theBean instanceof InitializableComponent ic) {
            ic.initialize();
        }
        return theBean;
    }

    /**
     * Call the parent class to create the instance.
     * 
     * @return the bean.
     * @throws Exception if needed.
     */
    @Nullable protected abstract T doCreateInstance() throws Exception;

}