/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.fs.s3;

import com.amazonaws.AmazonClientException;
import com.amazonaws.AmazonServiceException;
import com.amazonaws.auth.AWSCredentials;
import com.amazonaws.auth.BasicAWSCredentials;
import com.amazonaws.services.s3.AmazonS3Client;
import com.amazonaws.services.s3.model.Bucket;
import com.amazonaws.services.s3.model.ObjectListing;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.Owner;
import com.amazonaws.services.s3.model.S3ObjectSummary;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLDecoder;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import org.apache.flink.configuration.GlobalConfiguration;
import org.apache.flink.core.fs.BlockLocation;
import org.apache.flink.core.fs.FSDataInputStream;
import org.apache.flink.core.fs.FSDataOutputStream;
import org.apache.flink.core.fs.FileStatus;
import org.apache.flink.core.fs.FileSystem;
import org.apache.flink.core.fs.Path;
import org.apache.flink.runtime.fs.s3.S3BlockLocation;
import org.apache.flink.runtime.fs.s3.S3BucketObjectPair;
import org.apache.flink.runtime.fs.s3.S3DataInputStream;
import org.apache.flink.runtime.fs.s3.S3DataOutputStream;
import org.apache.flink.runtime.fs.s3.S3DirectoryStructure;
import org.apache.flink.runtime.fs.s3.S3FileStatus;
import org.apache.flink.util.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class S3FileSystem
extends FileSystem {
    private static final Logger LOG = LoggerFactory.getLogger(S3FileSystem.class);
    public static final String S3_HOST_KEY = "fs.s3.host";
    public static final String S3_PORT_KEY = "fs.s3.port";
    public static final String S3_RRS_KEY = "fs.s3.rrs";
    public static final String S3_ACCESS_KEY_KEY = "fs.s3.accessKey";
    public static final String S3_SECRET_KEY_KEY = "fs.s3.secretKey";
    private static final String DEFAULT_S3_HOST = "s3.amazonaws.com";
    private static final boolean DEFAULT_S3_RRS = true;
    private static final int DEFAULT_S3_PORT = 80;
    private static final String HTTP_PREFIX = "http";
    private static final int HTTP_RESOURCE_NOT_FOUND_CODE = 404;
    private static final char S3_DIRECTORY_SEPARATOR = '/';
    public static final String S3_SCHEME = "s3";
    private static final String URL_ENCODE_CHARACTER = "UTF-8";
    private String host = null;
    private int port = -1;
    private URI s3Uri = null;
    private AmazonS3Client s3Client = null;
    private S3DirectoryStructure directoryStructure = null;
    private final boolean useRRS = GlobalConfiguration.getBoolean((String)"fs.s3.rrs", (boolean)true);

    public S3FileSystem() {
        LOG.info("Creating new S3 file system binding with Reduced Redundancy Storage " + (this.useRRS ? "enabled" : "disabled"));
    }

    public Path getWorkingDirectory() {
        return new Path(this.s3Uri);
    }

    public Path getHomeDirectory() {
        return new Path(this.s3Uri);
    }

    public URI getUri() {
        return this.s3Uri;
    }

    public void initialize(URI name) throws IOException {
        String[] splits;
        this.host = name.getHost();
        if (this.host == null) {
            LOG.debug("Provided URI does not provide a host to connect to, using configuration...");
            this.host = GlobalConfiguration.getString((String)S3_HOST_KEY, (String)DEFAULT_S3_HOST);
        }
        this.port = name.getPort();
        if (this.port == -1) {
            LOG.debug("Provided URI does not provide a port to connect to, using configuration...");
            this.port = GlobalConfiguration.getInteger((String)S3_PORT_KEY, (int)80);
        }
        String userInfo = name.getUserInfo();
        String awsAccessKey = null;
        String awsSecretKey = null;
        if (userInfo != null && (splits = userInfo.split(":")).length > 1) {
            awsAccessKey = URLDecoder.decode(splits[0], URL_ENCODE_CHARACTER);
            awsSecretKey = URLDecoder.decode(splits[1], URL_ENCODE_CHARACTER);
        }
        if (awsAccessKey == null) {
            LOG.debug("Provided URI does not provide an access key to Amazon S3, using configuration...");
            awsAccessKey = GlobalConfiguration.getString((String)S3_ACCESS_KEY_KEY, null);
            if (awsAccessKey == null) {
                throw new IOException("Cannot determine access key to Amazon S3. Please make sure to configure it by setting the configuration key 'fs.s3.accessKey'.");
            }
        }
        if (awsSecretKey == null) {
            LOG.debug("Provided URI does not provide a secret key to Amazon S3, using configuration...");
            awsSecretKey = GlobalConfiguration.getString((String)S3_SECRET_KEY_KEY, null);
            if (awsSecretKey == null) {
                throw new IOException("Cannot determine secret key to Amazon S3. Please make sure to configure it by setting the configuration key 'fs.s3.secretKey'.");
            }
        }
        BasicAWSCredentials credentials = new BasicAWSCredentials(awsAccessKey, awsSecretKey);
        this.s3Client = new AmazonS3Client((AWSCredentials)credentials);
        this.initializeDirectoryStructure(name);
    }

    private void initializeDirectoryStructure(URI name) throws IOException {
        String basePath = name.getPath();
        while (true) {
            try {
                String endpoint = new URL(HTTP_PREFIX, this.host, this.port, basePath).toString();
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Trying S3 endpoint " + endpoint);
                }
                this.s3Client.setEndpoint(endpoint);
                Owner owner = this.s3Client.getS3AccountOwner();
                LOG.info("Successfully established connection to Amazon S3 using the endpoint " + endpoint);
                LOG.info("Amazon S3 user is " + owner.getDisplayName());
            }
            catch (MalformedURLException e) {
                throw new IOException(StringUtils.stringifyException((Throwable)e));
            }
            catch (AmazonClientException e) {
                if (basePath.isEmpty()) {
                    throw new IOException("Cannot establish connection to Amazon S3: " + StringUtils.stringifyException((Throwable)e));
                }
                int pos = basePath.lastIndexOf("/");
                if (pos < 0) {
                    basePath = "";
                    continue;
                }
                basePath = basePath.substring(0, pos);
                continue;
            }
            break;
        }
        try {
            this.s3Uri = new URI(S3_SCHEME, null, this.host, this.port, basePath, null, null);
        }
        catch (URISyntaxException e) {
            throw new IOException(StringUtils.stringifyException((Throwable)e));
        }
        this.directoryStructure = new S3DirectoryStructure(basePath);
    }

    public FileStatus getFileStatus(Path f) throws IOException {
        S3BucketObjectPair bop = this.directoryStructure.toBucketObjectPair(f);
        if (!bop.hasBucket() && !bop.hasObject()) {
            return new S3FileStatus(f, 0L, true, 0L, 0L);
        }
        try {
            if (bop.hasBucket() && !bop.hasObject()) {
                List buckets = this.s3Client.listBuckets();
                for (Bucket bucket : buckets) {
                    if (!bop.getBucket().equals(bucket.getName())) continue;
                    long creationDate = S3FileSystem.dateToLong(bucket.getCreationDate());
                    return new S3FileStatus(f, 0L, true, creationDate, 0L);
                }
                throw new FileNotFoundException("Cannot find " + f.toUri());
            }
            try {
                ObjectMetadata om = this.s3Client.getObjectMetadata(bop.getBucket(), bop.getObject());
                long modificationDate = S3FileSystem.dateToLong(om.getLastModified());
                if (this.objectRepresentsDirectory(bop.getObject(), om.getContentLength())) {
                    return new S3FileStatus(f, 0L, true, modificationDate, 0L);
                }
                return new S3FileStatus(f, om.getContentLength(), false, modificationDate, 0L);
            }
            catch (AmazonServiceException e) {
                if (e.getStatusCode() == 404) {
                    throw new FileNotFoundException("Cannot find " + f.toUri());
                }
                throw e;
            }
        }
        catch (AmazonClientException e) {
            throw new IOException(StringUtils.stringifyException((Throwable)e));
        }
    }

    private static long dateToLong(Date date) {
        if (date == null) {
            return 0L;
        }
        return date.getTime();
    }

    public BlockLocation[] getFileBlockLocations(FileStatus file, long start, long len) throws IOException {
        if (start + len > file.getLen()) {
            return null;
        }
        S3BlockLocation bl = new S3BlockLocation(this.host, file.getLen());
        return new BlockLocation[]{bl};
    }

    public FSDataInputStream open(Path f, int bufferSize) throws IOException {
        return this.open(f);
    }

    public FSDataInputStream open(Path f) throws IOException {
        FileStatus fileStatus = this.getFileStatus(f);
        if (fileStatus.isDir()) {
            throw new IOException("Cannot open " + f.toUri() + " because it is a directory");
        }
        S3BucketObjectPair bop = this.directoryStructure.toBucketObjectPair(f);
        if (!bop.hasBucket() || !bop.hasObject()) {
            throw new IOException(f.toUri() + " cannot be opened");
        }
        return new S3DataInputStream(this.s3Client, bop.getBucket(), bop.getObject());
    }

    public FileStatus[] listStatus(Path f) throws IOException {
        S3BucketObjectPair bop = this.directoryStructure.toBucketObjectPair(f);
        try {
            if (!bop.hasBucket()) {
                List list = this.s3Client.listBuckets();
                FileStatus[] array = new S3FileStatus[list.size()];
                Iterator it = list.iterator();
                int i = 0;
                while (it.hasNext()) {
                    Bucket bucket = (Bucket)it.next();
                    long creationDate = S3FileSystem.dateToLong(bucket.getCreationDate());
                    S3FileStatus status = new S3FileStatus(S3FileSystem.extendPath(f, bucket.getName() + '/'), 0L, true, creationDate, 0L);
                    array[i++] = status;
                }
                return array;
            }
            if (bop.hasBucket() && !bop.hasObject()) {
                if (!this.s3Client.doesBucketExist(bop.getBucket())) {
                    throw new FileNotFoundException("Cannot find " + f.toUri());
                }
                return this.listBucketContent(f, bop);
            }
            ObjectMetadata omd = this.s3Client.getObjectMetadata(bop.getBucket(), bop.getObject());
            if (this.objectRepresentsDirectory(bop.getObject(), omd.getContentLength())) {
                return this.listBucketContent(f, bop);
            }
            S3FileStatus fileStatus = new S3FileStatus(f, omd.getContentLength(), false, S3FileSystem.dateToLong(omd.getLastModified()), 0L);
            return new FileStatus[]{fileStatus};
        }
        catch (AmazonClientException e) {
            throw new IOException(StringUtils.stringifyException((Throwable)e));
        }
    }

    private S3FileStatus[] listBucketContent(Path f, S3BucketObjectPair bop) throws IOException {
        int depth;
        ObjectListing listing = null;
        ArrayList<S3FileStatus> resultList = new ArrayList<S3FileStatus>();
        int n = depth = bop.hasObject() ? S3FileSystem.getDepth(bop.getObject()) + 1 : 0;
        do {
            listing = listing == null ? (bop.hasObject() ? this.s3Client.listObjects(bop.getBucket(), bop.getObject()) : this.s3Client.listObjects(bop.getBucket())) : this.s3Client.listNextBatchOfObjects(listing);
            List list = listing.getObjectSummaries();
            for (S3ObjectSummary os : list) {
                String key = os.getKey();
                int childDepth = S3FileSystem.getDepth(os.getKey());
                if (childDepth != depth) continue;
                if (bop.hasObject()) {
                    if (key.startsWith(bop.getObject())) {
                        key = key.substring(bop.getObject().length());
                    }
                    if (key.isEmpty()) continue;
                }
                long modificationDate = S3FileSystem.dateToLong(os.getLastModified());
                S3FileStatus fileStatus = this.objectRepresentsDirectory(os) ? new S3FileStatus(S3FileSystem.extendPath(f, key), 0L, true, modificationDate, 0L) : new S3FileStatus(S3FileSystem.extendPath(f, key), os.getSize(), false, modificationDate, 0L);
                resultList.add(fileStatus);
            }
        } while (listing.isTruncated());
        return resultList.toArray(new S3FileStatus[0]);
    }

    private static int getDepth(String key) {
        int sepPos;
        int depth = 0;
        int nextStartPos = 0;
        int length = key.length();
        while (nextStartPos < length && (sepPos = key.indexOf(47, nextStartPos)) >= 0) {
            ++depth;
            nextStartPos = sepPos + 1;
        }
        if (length > 0 && key.charAt(length - 1) == '/') {
            --depth;
        }
        return depth;
    }

    public boolean delete(Path f, boolean recursive) throws IOException {
        try {
            FileStatus fileStatus = this.getFileStatus(f);
            S3BucketObjectPair bop = this.directoryStructure.toBucketObjectPair(f);
            if (fileStatus.isDir()) {
                boolean retVal = false;
                FileStatus[] dirContent = this.listStatus(f);
                if (dirContent.length > 0) {
                    if (!recursive) {
                        throw new IOException("Found non-empty directory " + f + " while performing non-recursive delete");
                    }
                    for (FileStatus entry : dirContent) {
                        if (!this.delete(entry.getPath(), true)) continue;
                        retVal = true;
                    }
                }
                if (!bop.hasBucket()) {
                    return retVal;
                }
                if (!bop.hasObject()) {
                    this.s3Client.deleteBucket(bop.getBucket());
                } else {
                    this.s3Client.deleteObject(bop.getBucket(), bop.getObject());
                }
            } else {
                this.s3Client.deleteObject(bop.getBucket(), bop.getObject());
            }
        }
        catch (AmazonClientException e) {
            throw new IOException(StringUtils.stringifyException((Throwable)e));
        }
        return true;
    }

    public boolean mkdirs(Path f) throws IOException {
        boolean retCode;
        block10: {
            S3BucketObjectPair bop = this.directoryStructure.toBucketObjectPair(f);
            if (!bop.hasBucket() && !bop.hasObject()) {
                return false;
            }
            retCode = false;
            try {
                if (bop.hasBucket() && !this.s3Client.doesBucketExist(bop.getBucket())) {
                    this.s3Client.createBucket(bop.getBucket());
                    retCode = true;
                }
                if (!bop.hasObject()) break block10;
                String object = bop.getObject();
                if (!object.isEmpty() && object.charAt(object.length() - 1) != '/') {
                    object = object.concat(Character.toString('/'));
                }
                while (true) {
                    try {
                        this.s3Client.getObjectMetadata(bop.getBucket(), object);
                    }
                    catch (AmazonServiceException e) {
                        if (e.getStatusCode() == 404) {
                            int nextPos;
                            this.createEmptyObject(bop.getBucket(), object);
                            if (object.length() > 1 && (nextPos = object.lastIndexOf(47, object.length() - 2)) >= 0) {
                                object = object.substring(0, nextPos + 1);
                                continue;
                            }
                            break;
                        }
                        throw e;
                    }
                    break;
                }
            }
            catch (AmazonClientException e) {
                throw new IOException(StringUtils.stringifyException((Throwable)e));
            }
        }
        return retCode;
    }

    private void createEmptyObject(String bucketName, String objectName) {
        InputStream im = new InputStream(){

            @Override
            public int read() throws IOException {
                return -1;
            }
        };
        ObjectMetadata om = new ObjectMetadata();
        om.setContentLength(0L);
        this.s3Client.putObject(bucketName, objectName, im, om);
    }

    public FSDataOutputStream create(Path f, boolean overwrite, int bufferSize, short replication, long blockSize) throws IOException {
        if (!overwrite && this.exists(f)) {
            throw new IOException(f.toUri() + " already exists");
        }
        S3BucketObjectPair bop = this.directoryStructure.toBucketObjectPair(f);
        if (!bop.hasBucket() || !bop.hasObject()) {
            throw new IOException(f.toUri() + " is not a valid path to create a new file");
        }
        if (bufferSize < 0x500000) {
            throw new IOException("Provided buffer must be at least 5242880 bytes");
        }
        byte[] buf = new byte[bufferSize];
        return new S3DataOutputStream(this.s3Client, bop.getBucket(), bop.getObject(), buf, this.useRRS);
    }

    public FSDataOutputStream create(Path f, boolean overwrite) throws IOException {
        return this.create(f, overwrite, 0x500000, (short)1, 1024L);
    }

    private boolean objectRepresentsDirectory(S3ObjectSummary os) {
        return this.objectRepresentsDirectory(os.getKey(), os.getSize());
    }

    private boolean objectRepresentsDirectory(String name, long size) {
        if (name.isEmpty()) {
            return false;
        }
        return name.charAt(name.length() - 1) == '/' && size == 0L;
    }

    static Path extendPath(Path parent, String extension) throws IOException {
        URI parentUri = parent.toUri();
        if (extension.isEmpty()) {
            return parent;
        }
        String path = parentUri.getPath();
        String extendedPath = path.isEmpty() ? (extension.charAt(0) == '/' ? extension : "/" + extension) : (path.charAt(path.length() - 1) == '/' ? (extension.charAt(0) == '/' ? (extension.length() > 1 ? path + extension.substring(1) : path) : path + extension) : (extension.charAt(0) == '/' ? path + extension : path + "/" + extension));
        try {
            URI extendedUri = new URI(parentUri.getScheme(), parentUri.getAuthority() != null ? parentUri.getAuthority() : "", extendedPath, parentUri.getQuery(), parentUri.getFragment());
            return new Path(extendedUri);
        }
        catch (URISyntaxException e) {
            throw new IOException(StringUtils.stringifyException((Throwable)e));
        }
    }

    public boolean rename(Path src, Path dst) throws IOException {
        throw new UnsupportedOperationException("This method is not yet implemented");
    }

    public boolean initOutPathDistFS(Path outPath, FileSystem.WriteMode writeMode, boolean createDirectory) throws IOException {
        if (createDirectory) {
            outPath = outPath.suffix("/");
        }
        return super.initOutPathDistFS(outPath, writeMode, createDirectory);
    }

    public boolean isDistributedFS() {
        return true;
    }
}

