/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.net;

import java.io.IOException;
import java.net.Inet4Address;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.NetworkInterface;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketAddress;
import java.util.Enumeration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NetUtils {
    private static final Logger LOG = LoggerFactory.getLogger(NetUtils.class);
    private static final long MIN_SLEEP_TIME = 50L;
    private static final long MAX_SLEEP_TIME = 20000L;

    public static InetAddress resolveAddress(InetSocketAddress jobManagerAddress) throws IOException {
        AddressDetectionState strategy = jobManagerAddress != null ? AddressDetectionState.ADDRESS : AddressDetectionState.HEURISTIC;
        while (true) {
            Enumeration<NetworkInterface> e = NetworkInterface.getNetworkInterfaces();
            while (e.hasMoreElements()) {
                NetworkInterface n = e.nextElement();
                Enumeration<InetAddress> ee = n.getInetAddresses();
                block13: while (ee.hasMoreElements()) {
                    InetAddress i = ee.nextElement();
                    switch (strategy) {
                        case ADDRESS: {
                            if (!NetUtils.hasCommonPrefix(jobManagerAddress.getAddress().getAddress(), i.getAddress()) || !NetUtils.tryToConnect(i, jobManagerAddress, strategy.getTimeout(), true)) continue block13;
                            LOG.info("Determined {} as the machine's own IP address", (Object)i);
                            return i;
                        }
                        case FAST_CONNECT: 
                        case SLOW_CONNECT: {
                            boolean correct = NetUtils.tryToConnect(i, jobManagerAddress, strategy.getTimeout(), true);
                            if (!correct) continue block13;
                            LOG.info("Determined {} as the machine's own IP address", (Object)i);
                            return i;
                        }
                        case HEURISTIC: {
                            if (LOG.isDebugEnabled()) {
                                LOG.debug("ResolveAddress using heuristic strategy for " + i + " with" + " isLinkLocalAddress:" + i.isLinkLocalAddress() + " isLoopbackAddress:" + i.isLoopbackAddress() + ".");
                            }
                            if (i.isLinkLocalAddress() || i.isLoopbackAddress() || !(i instanceof Inet4Address)) continue block13;
                            LOG.warn("Hostname " + InetAddress.getLocalHost().getHostName() + " resolves to " + "loopback address. Using instead " + i.getHostAddress() + " on network " + "interface " + n.getName() + ".");
                            return i;
                        }
                    }
                    throw new RuntimeException("Unknown address detection strategy: " + (Object)((Object)strategy));
                }
            }
            switch (strategy) {
                case ADDRESS: {
                    strategy = AddressDetectionState.FAST_CONNECT;
                    break;
                }
                case FAST_CONNECT: {
                    strategy = AddressDetectionState.SLOW_CONNECT;
                    break;
                }
                case SLOW_CONNECT: {
                    if (!InetAddress.getLocalHost().isLoopbackAddress()) {
                        LOG.info("Heuristically taking " + InetAddress.getLocalHost() + " as own " + "IP address.");
                        return InetAddress.getLocalHost();
                    }
                    strategy = AddressDetectionState.HEURISTIC;
                    break;
                }
                case HEURISTIC: {
                    throw new RuntimeException("Unable to resolve own inet address by connecting to address (" + jobManagerAddress + ").");
                }
            }
            if (!LOG.isDebugEnabled()) continue;
            LOG.debug("Defaulting to detection strategy " + (Object)((Object)strategy));
        }
    }

    public static InetAddress findConnectingAddress(InetSocketAddress targetAddress, long maxWaitMillis, long startLoggingAfter) throws IOException {
        if (targetAddress == null) {
            throw new NullPointerException("targetAddress must not be null");
        }
        if (maxWaitMillis <= 0L) {
            throw new IllegalArgumentException("Max wait time must be positive");
        }
        long startTime = System.currentTimeMillis();
        long currentSleepTime = 50L;
        long elapsedTime = 0L;
        while (elapsedTime < maxWaitMillis) {
            boolean logging;
            AddressDetectionState strategy = AddressDetectionState.ADDRESS;
            boolean bl = logging = elapsedTime >= startLoggingAfter;
            if (logging) {
                LOG.info("Trying to connect to " + targetAddress);
            }
            do {
                InetAddress address;
                if ((address = NetUtils.findAddressUsingStrategy(strategy, targetAddress, logging)) != null) {
                    return address;
                }
                switch (strategy) {
                    case ADDRESS: {
                        strategy = AddressDetectionState.FAST_CONNECT;
                        break;
                    }
                    case FAST_CONNECT: {
                        strategy = AddressDetectionState.SLOW_CONNECT;
                        break;
                    }
                    case SLOW_CONNECT: {
                        strategy = null;
                        break;
                    }
                    default: {
                        throw new RuntimeException("Unsupported strategy: " + (Object)((Object)strategy));
                    }
                }
            } while (strategy != null);
            elapsedTime = System.currentTimeMillis() - startTime;
            long toWait = Math.min(maxWaitMillis - elapsedTime, currentSleepTime);
            if (toWait > 0L) {
                if (logging) {
                    LOG.info("Could not connect. Waiting for {} msecs before next attempt", (Object)toWait);
                } else {
                    LOG.debug("Could not connect. Waiting for {} msecs before next attempt", (Object)toWait);
                }
                try {
                    Thread.sleep(toWait);
                }
                catch (InterruptedException e) {
                    throw new IOException("Connection attempts have been interrupted.");
                }
            }
            currentSleepTime = Math.min(2L * currentSleepTime, 20000L);
        }
        LOG.warn("Could not connect to {}. Selecting a local address using heuristics.", (Object)targetAddress);
        InetAddress heuristic = NetUtils.findAddressUsingStrategy(AddressDetectionState.HEURISTIC, targetAddress, true);
        if (heuristic != null) {
            return heuristic;
        }
        LOG.warn("Could not find any IPv4 address that is not loopback or link-local. Using localhost address.");
        return InetAddress.getLocalHost();
    }

    private static InetAddress findAddressUsingStrategy(AddressDetectionState strategy, InetSocketAddress targetAddress, boolean logging) throws IOException {
        byte[] targetAddressBytes = targetAddress.getAddress().getAddress();
        Enumeration<NetworkInterface> e = NetworkInterface.getNetworkInterfaces();
        while (e.hasMoreElements()) {
            NetworkInterface netInterface = e.nextElement();
            Enumeration<InetAddress> ee = netInterface.getInetAddresses();
            block6: while (ee.hasMoreElements()) {
                InetAddress interfaceAddress = ee.nextElement();
                switch (strategy) {
                    case ADDRESS: {
                        if (!NetUtils.hasCommonPrefix(targetAddressBytes, interfaceAddress.getAddress())) continue block6;
                        LOG.debug("Target address {} and local address {} share prefix - trying to connect.", (Object)targetAddress, (Object)interfaceAddress);
                        if (!NetUtils.tryToConnect(interfaceAddress, targetAddress, strategy.getTimeout(), logging)) continue block6;
                        return interfaceAddress;
                    }
                    case FAST_CONNECT: 
                    case SLOW_CONNECT: {
                        LOG.debug("Trying to connect to {} from local address {} with timeout {}", new Object[]{targetAddress, interfaceAddress, strategy.getTimeout()});
                        if (!NetUtils.tryToConnect(interfaceAddress, targetAddress, strategy.getTimeout(), logging)) continue block6;
                        return interfaceAddress;
                    }
                    case HEURISTIC: {
                        if (LOG.isDebugEnabled()) {
                            LOG.debug("Checking address {} using heuristics: linkLocal: {} loopback: {}", new Object[]{interfaceAddress, interfaceAddress.isLinkLocalAddress(), interfaceAddress.isLoopbackAddress()});
                        }
                        if (!(interfaceAddress instanceof Inet4Address) || interfaceAddress.isLinkLocalAddress() || interfaceAddress.isLoopbackAddress()) continue block6;
                        return interfaceAddress;
                    }
                }
                throw new RuntimeException("Unsupported strategy: " + (Object)((Object)strategy));
            }
        }
        return null;
    }

    private static boolean hasCommonPrefix(byte[] address, byte[] address2) {
        return address[0] == address2[0] && address[1] == address2[1];
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static boolean tryToConnect(InetAddress fromAddress, SocketAddress toSocket, int timeout, boolean logFailed) throws IOException {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Trying to connect to (" + toSocket + ") from local address " + fromAddress + " with timeout " + timeout);
        }
        Socket socket = new Socket();
        try {
            InetSocketAddress bindP = new InetSocketAddress(fromAddress, 0);
            socket.bind(bindP);
            socket.connect(toSocket, timeout);
            boolean bl = true;
            return bl;
        }
        catch (Exception ex) {
            String message = "Failed to connect from address '" + fromAddress + "': " + ex.getMessage();
            if (LOG.isDebugEnabled()) {
                LOG.debug(message, (Throwable)ex);
            } else if (logFailed) {
                LOG.info(message);
            }
            boolean bl = false;
            return bl;
        }
        finally {
            socket.close();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static int getAvailablePort() {
        for (int i = 0; i < 50; ++i) {
            ServerSocket serverSocket = null;
            try {
                serverSocket = new ServerSocket(0);
                int port = serverSocket.getLocalPort();
                if (port == 0) continue;
                int n = port;
                return n;
            }
            catch (IOException e) {
                if (!LOG.isDebugEnabled()) continue;
                LOG.debug("Unable to allocate port " + e.getMessage(), (Throwable)e);
                continue;
            }
            finally {
                if (serverSocket != null) {
                    try {
                        serverSocket.close();
                    }
                    catch (Throwable throwable) {}
                }
            }
        }
        throw new RuntimeException("Could not find a free permitted port on the machine.");
    }

    private static enum AddressDetectionState {
        ADDRESS(50),
        FAST_CONNECT(50),
        SLOW_CONNECT(1000),
        HEURISTIC(0);

        private int timeout;

        private AddressDetectionState(int timeout) {
            this.timeout = timeout;
        }

        public int getTimeout() {
            return this.timeout;
        }
    }
}

